﻿using Nintendo.G3dTool.Entities;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.toollib;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Serialization;

namespace TextureCompositor
{
    public class SourceTexture
    {
        public string TexturePath { get; set; }
        public Channel Channel { get; set; }
    }
    public class CompositeTextureArgs
    {
        public CompositeTextureArgs()
        {
            for (int i = 0; i < 4; ++i)
            {
                this.SourceTextures[i] = null;
            }
        }

        public string DestinationTexPath { get; set; } = string.Empty;
        public int Width { get; set; } = 1;
        public int Height { get; set; } = 1;

        internal SourceTexture[] SourceTextures { get; } = new SourceTexture[4];

        public SourceTexture SourceTextureR
        {
            get
            {
                return SourceTextures[0];
            }

            set
            {
                SourceTextures[0] = value;
            }
        }
        public SourceTexture SourceTextureG
        {
            get
            {
                return SourceTextures[1];
            }

            set
            {
                SourceTextures[1] = value;
            }
        }
        public SourceTexture SourceTextureB
        {
            get
            {
                return SourceTextures[2];
            }

            set
            {
                SourceTextures[2] = value;
            }
        }
        public SourceTexture SourceTextureA
        {
            get
            {
                return SourceTextures[3];
            }

            set
            {
                SourceTextures[3] = value;
            }
        }
    }

    public delegate void CompositeTexture(CompositeTextureArgs args);

    public enum Channel
    {
        R,
        G,
        B,
        A,
    }

    // @@ id="_m0" hint="merged0" source="_s0.R,_x0.R"

    [Serializable]
    public class CompositeSource
    {
        public string SamplerName { get; set; }

        public Channel Channel { get; set; }
    }

    [Serializable]
    public class CompositeSampler
    {
        public string SamplerName { get; set; }
        public int SourceCount
        {
            get
            {
                return new CompositeSource[] { SourceR, SourceG, SourceB, SourceA }.Count(x => x != null);
            }
        }
        public CompositeSource SourceR { get; set; }
        public CompositeSource SourceG { get; set; }
        public CompositeSource SourceB { get; set; }
        public CompositeSource SourceA { get; set; }
        public ComponentSelector CompSel { get; set; }
        public texture_info_quantize_typeType? Format { get; set; } = null;
        public Bool4 Linear { get; set; } = null;
    }

    [Serializable]
    public class CompositeSamplerInfo : ToolDataObject
    {
        [XmlArrayItem("CompositeSampler")]
        public List<CompositeSampler> CompositeSamplers { get; } = new List<CompositeSampler>();

        public string ShadingModelName { get; set; }
    }

    [Serializable]
    public class SourceSampler
    {
        public string SamplerName { get; set; }

        public int ResizeX { get; set; }

        public int ResizeY { get; set; }
    }

    [Serializable]
    public class SourceSamplerInfo : ToolDataObject
    {
        [XmlArrayItem("SourceSampler")]
        public List<SourceSampler> SourceSamplers { get; } = new List<SourceSampler>();
    }

    [Serializable]
    public class SamplerTextureConversionInfo : ToolDataObject
    {
        public int? ResizeWidth { get; set; }
        public int? ResizeHeight { get; set; }
    }

    public class TextureSamplerCompositor
    {
        public TextureSamplerCompositor()
        {
        }

        public string OutputFolderPath { get; set; } = string.Empty;

        public List<string> TextureReferencePaths { get; } = new List<string>();

        public Action<string> WriteDebugMessageAction { get; set; } = x => { };

        public void CompositeMaterialSamplers(
            Material material,
            CompositeSamplerInfo compositeSamplerInfo,
            ShaderDefinition shaderDef)
        {
            Ensure.Argument.NotNull(compositeSamplerInfo);
            foreach (var compSampler in compositeSamplerInfo.CompositeSamplers)
            {
                if (material.Samplers.Any(x => x.Name == compSampler.SamplerName))
                {
                    // 合成後のサンプラーが存在する場合はとりあえずエラー
                    throw new Exception($"合成後のサンプラー{compSampler.SamplerName}が既に存在しています");
                }

                // サンプラーアサインがなければ追加する
                {
                    var samplerAssign = material.ShaderAssign.SamplerAssigns.FirstOrDefault(x => x.SamplerName == compSampler.SamplerName);
                    if (samplerAssign == null)
                    {
                        samplerAssign = new SamplerAssign() { Id = compSampler.SamplerName };
                        material.ShaderAssign.SamplerAssigns.Add(samplerAssign);
                    }

                    samplerAssign.SamplerName = compSampler.SamplerName;
                }

                // TODO: 合成後のテクスチャー名をどうするか
                string compTexName = $"{material.Name}_{compSampler.SamplerName}";
                string sourceTexNameR = string.Empty;
                string sourceTexNameG = string.Empty;
                string sourceTexNameB = string.Empty;
                string sourceTexNameA = string.Empty;

                // 合成後のサンプラーを追加し、元のサンプラーを削除する
                {
                    var targetSampler = new Sampler() { Name = compSampler.SamplerName };
                    targetSampler.TexName = compTexName;

                    material.Samplers.Add(targetSampler);

                    List<Sampler> removeSamplerList = new List<Sampler>();
                    if (compSampler.SourceR != null)
                    {
                        var sourceSamplerR = material.Samplers.FirstOrDefault(x => x.Name == compSampler.SourceR.SamplerName);
                        sourceTexNameR = sourceSamplerR.TexName;
                        removeSamplerList.Add(sourceSamplerR);
                    }

                    if (compSampler.SourceG != null)
                    {
                        var sourceSamplerG = material.Samplers.FirstOrDefault(x => x.Name == compSampler.SourceG.SamplerName);
                        sourceTexNameG = sourceSamplerG.TexName;
                        removeSamplerList.Add(sourceSamplerG);
                    }

                    if (compSampler.SourceB != null)
                    {
                        var sourceSamplerB = material.Samplers.FirstOrDefault(x => x.Name == compSampler.SourceB.SamplerName);
                        sourceTexNameB = sourceSamplerB.TexName;
                        removeSamplerList.Add(sourceSamplerB);
                    }

                    if (compSampler.SourceA != null)
                    {
                        var sourceSamplerA = material.Samplers.FirstOrDefault(x => x.Name == compSampler.SourceA.SamplerName);
                        sourceTexNameA = sourceSamplerA.TexName;
                        removeSamplerList.Add(sourceSamplerA);
                    }

                    foreach (var removeSampler in removeSamplerList)
                    {
                        material.Samplers.Remove(removeSampler);
                    }
                }

                // テクスチャーのチャンネル合成
                {
                    CompositeTextureArgs args = new CompositeTextureArgs();
                    args.DestinationTexPath = System.IO.Path.Combine(OutputFolderPath, "textures", $"{compTexName}.ftxb");
                    args.Width = 1024;
                    args.Height = 1024;
                    if (compSampler.SourceR != null)
                    {
                        args.SourceTextureR = new SourceTexture();
                        args.SourceTextureR.TexturePath = TextureUtility.FindTexturePath(sourceTexNameR, this.TextureReferencePaths);
                        args.SourceTextureR.Channel = compSampler.SourceR.Channel;
                    }
                    if (compSampler.SourceG != null)
                    {
                        args.SourceTextureG = new SourceTexture();
                        args.SourceTextureG.TexturePath = TextureUtility.FindTexturePath(sourceTexNameG, this.TextureReferencePaths);
                        args.SourceTextureG.Channel = compSampler.SourceG.Channel;
                    }
                    if (compSampler.SourceB != null)
                    {
                        args.SourceTextureB = new SourceTexture();
                        args.SourceTextureB.TexturePath = TextureUtility.FindTexturePath(sourceTexNameB, this.TextureReferencePaths);
                        args.SourceTextureB.Channel = compSampler.SourceB.Channel;
                    }
                    if (compSampler.SourceA != null)
                    {
                        args.SourceTextureA = new SourceTexture();
                        args.SourceTextureA.TexturePath = TextureUtility.FindTexturePath(sourceTexNameA, this.TextureReferencePaths);
                        args.SourceTextureA.Channel = compSampler.SourceA.Channel;
                    }

                    CompositeTexture(args);
                }
            }
        }

        private static (int maxWidth, int maxHeight) FindMaxSourceTextureSize(CompositeTextureArgs args)
        {
            int maxWidth = 0;
            int maxHeight = 0;
            foreach (var source in args.SourceTextures)
            {
                if (source == null)
                {
                    continue;
                }

                var texFile = IfBinaryReadUtility.ReadIntermediateFileWithoutStreams(source.TexturePath, G3dToolUtility.GetXsdBasePath());
                var tex = texFile.GetRootEntity<Texture>();
                if (tex.TextureInfo.Width > maxWidth)
                {
                    maxWidth = tex.TextureInfo.Width;
                }
                if (tex.TextureInfo.Height > maxHeight)
                {
                    maxHeight = tex.TextureInfo.Height;
                }
            }

            return (maxWidth, maxHeight);
        }

        private static bool GetLinearFlag(Texture tex, Channel channel)
        {
            switch (channel)
            {
                case Channel.R: return tex.TextureInfo.Linear.X;
                case Channel.G: return tex.TextureInfo.Linear.Y;
                case Channel.B: return tex.TextureInfo.Linear.Z;
                case Channel.A: return tex.TextureInfo.Linear.W;
                default:
                    throw new Exception("Unexpected default.");
            }
        }

        /// <summary>
        /// テクスチャーのチャンネル合成を行います。圧縮やリサイズは行いません。
        /// </summary>
        /// <param name="args"></param>
        public static void CompositeTexture(CompositeTextureArgs args)
        {
            //using (var tmpFolderCreator = new ScopedDirectoryCreator())
            {
                //string tmpFolder = tmpFolderCreator.DirectoryPath;
                StringBuilder converterOptions = new StringBuilder();
                int channelCount = 0;
                Bool4 linear = new Bool4();
                if (args.SourceTextureR != null)
                {
                    //string unormTexPath = Path.Combine(tmpFolder, Path.GetFileName(args.SourceTextureR.TexturePath));
                    //TextureUtility.ConvertTexture($"{args.SourceTextureR.TexturePath} -o {unormTexPath} -f unorm_8_8_8_8");
                    converterOptions.Append($"-r {args.SourceTextureR.TexturePath}:{args.SourceTextureR.Channel.ToString().ToLower()} ");
                    var file = IfBinaryReadUtility.ReadIntermediateFileWithoutStreams(args.SourceTextureR.TexturePath, G3dToolUtility.GetXsdBasePath());
                    linear.X = GetLinearFlag(file.GetRootEntity<Texture>(), args.SourceTextureR.Channel);
                    channelCount = 1;
                }

                if (args.SourceTextureG != null)
                {
                    converterOptions.Append($"-g {args.SourceTextureG.TexturePath}:{args.SourceTextureG.Channel.ToString().ToLower()} ");
                    var file = IfBinaryReadUtility.ReadIntermediateFileWithoutStreams(args.SourceTextureR.TexturePath, G3dToolUtility.GetXsdBasePath());
                    linear.Y = GetLinearFlag(file.GetRootEntity<Texture>(), args.SourceTextureR.Channel);
                    channelCount = 2;
                }

                if (args.SourceTextureB != null)
                {
                    converterOptions.Append($"-b {args.SourceTextureB.TexturePath}:{args.SourceTextureB.Channel.ToString().ToLower()} ");
                    var file = IfBinaryReadUtility.ReadIntermediateFileWithoutStreams(args.SourceTextureR.TexturePath, G3dToolUtility.GetXsdBasePath());
                    linear.Z = GetLinearFlag(file.GetRootEntity<Texture>(), args.SourceTextureR.Channel);
                    channelCount = 3;
                }

                if (args.SourceTextureA != null)
                {
                    converterOptions.Append($"-a {args.SourceTextureA.TexturePath}:{args.SourceTextureA.Channel.ToString().ToLower()} ");
                    var file = IfBinaryReadUtility.ReadIntermediateFileWithoutStreams(args.SourceTextureR.TexturePath, G3dToolUtility.GetXsdBasePath());
                    linear.W = GetLinearFlag(file.GetRootEntity<Texture>(), args.SourceTextureR.Channel);
                    channelCount = 4;
                }

                if (channelCount == 0)
                {
                    // チャンネル合成なし
                    return;
                }

                {
                    // 合成のベースとなる出力先ファイル用にダミーファイルを作成
                    var outputTexFile = new IntermediateFile(IntermediateFileKind.Texture);
                    var tex = outputTexFile.GetRootEntity<Texture>();
                    (int width, int height) = FindMaxSourceTextureSize(args);
                    {
                        tex.TextureInfo.Width = width;
                        tex.TextureInfo.Height = height;
                        tex.TextureInfo.Dimension = nw.g3d.nw4f_3dif.texture_info_dimensionType.Item2d;
                        switch (channelCount)
                        {
                            case 1:
                                tex.TextureInfo.QuantizeType = nw.g3d.nw4f_3dif.texture_info_quantize_typeType.unorm_8;
                                tex.TextureInfo.CompSel.R = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.CompSel.G = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.CompSel.B = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.CompSel.A = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.Size = width * height;
                                tex.TextureInfo.Linear.X = linear.X;
                                tex.TextureInfo.Linear.Y = false;
                                tex.TextureInfo.Linear.Z = false;
                                tex.TextureInfo.Linear.W = false;
                                break;
                            case 2:
                                tex.TextureInfo.QuantizeType = nw.g3d.nw4f_3dif.texture_info_quantize_typeType.unorm_8_8;
                                tex.TextureInfo.CompSel.R = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.CompSel.G = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.CompSel.B = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.CompSel.A = nw.g3d.nw4f_3dif.texture_info_comp_selValue.g;
                                tex.TextureInfo.Size = width * height * 2;
                                tex.TextureInfo.Linear.X = linear.X;
                                tex.TextureInfo.Linear.Y = linear.Y;
                                tex.TextureInfo.Linear.Z = false;
                                tex.TextureInfo.Linear.W = false;
                                break;
                            case 3:
                                tex.TextureInfo.QuantizeType = nw.g3d.nw4f_3dif.texture_info_quantize_typeType.unorm_8_8_8_8;
                                tex.TextureInfo.CompSel.R = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.CompSel.G = nw.g3d.nw4f_3dif.texture_info_comp_selValue.g;
                                tex.TextureInfo.CompSel.B = nw.g3d.nw4f_3dif.texture_info_comp_selValue.b;
                                tex.TextureInfo.CompSel.A = nw.g3d.nw4f_3dif.texture_info_comp_selValue.Item1;
                                tex.TextureInfo.Size = width * height * 4;
                                tex.TextureInfo.Linear.X = linear.X;
                                tex.TextureInfo.Linear.Y = linear.Y;
                                tex.TextureInfo.Linear.Z = linear.Z;
                                tex.TextureInfo.Linear.W = false;
                                break;
                            case 4:
                                tex.TextureInfo.QuantizeType = nw.g3d.nw4f_3dif.texture_info_quantize_typeType.unorm_8_8_8_8;
                                tex.TextureInfo.CompSel.R = nw.g3d.nw4f_3dif.texture_info_comp_selValue.r;
                                tex.TextureInfo.CompSel.G = nw.g3d.nw4f_3dif.texture_info_comp_selValue.g;
                                tex.TextureInfo.CompSel.B = nw.g3d.nw4f_3dif.texture_info_comp_selValue.b;
                                tex.TextureInfo.CompSel.A = nw.g3d.nw4f_3dif.texture_info_comp_selValue.a;
                                tex.TextureInfo.Size = width * height * 4;
                                tex.TextureInfo.Linear.X = linear.X;
                                tex.TextureInfo.Linear.Y = linear.Y;
                                tex.TextureInfo.Linear.Z = linear.Z;
                                tex.TextureInfo.Linear.W = linear.W;
                                break;
                            default:
                                throw new Exception("Unexpected default");
                        }

                        var stream = new StreamByte();
                        byte[] streamData = new byte[tex.TextureInfo.Size];
                        Array.Clear(streamData, 0, streamData.Length);
                        stream.Values.Add(streamData);
                        tex.Streams.Add(stream);

                        tex.TextureInfo.Stream = stream;
                    }
                    {
                        var originalImage = new OriginalImage();
                        originalImage.Width = width;
                        originalImage.Height = height;
                        originalImage.Size = width * height * 4;
                        originalImage.Format = nw.g3d.nw4f_3dif.original_image_formatType.rgba8;
                        tex.OriginalImages.Add(originalImage);

                        var stream = new StreamByte();
                        byte[] streamData = new byte[originalImage.Size];
                        stream.Values.Add(streamData);
                        tex.Streams.Add(stream);

                        originalImage.Stream = stream;
                    }

                    string folder = Path.GetDirectoryName(args.DestinationTexPath);
                    if (!string.IsNullOrEmpty(folder) && !Directory.Exists(folder))
                    {
                        Directory.CreateDirectory(folder);
                    }

                    IfWriteUtility.WriteIntermediateFile(outputTexFile, args.DestinationTexPath, G3dToolUtility.GetXsdBasePath());
                }

                converterOptions.Append($"{args.DestinationTexPath}");
                TextureUtility.ConvertTexture(converterOptions.ToString());
            }
        }
    }
}
