﻿using Nintendo.Foundation.Contracts;
using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;
using nw.g3d.nw4f_3dif;
using nw.g3d.toollib;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace TextureCompositor
{
    public class Macro
    {
        public string Name { get; set; } = string.Empty;
        public string Value { get; set; } = "1";
    }

    public static class GlslAnalyzationUtility
    {
        public static bool IsSilent { get; set; } = false;

        /// <summary>
        /// 入力されたマテリアルのシェーダーバリエーションからサンプラーのアノテーション情報を抽出します。
        /// </summary>
        /// <param name="shaderDef"></param>
        /// <param name="material"></param>
        /// <returns></returns>
        public static CompositeSamplerInfo ExtractCompositeSamplerInfo(
            ShaderDefinition shaderDef,
            Material material)
        {
            if (!material.IsShaderAssigned)
            {
                return null;
            }

            var shadingModel = shaderDef.ShadingModels.FirstOrDefault(x => x.Name == material.ShaderAssign.ShadingModel);
            if (shadingModel == null)
            {
                throw new Exception($"マテリアルに割り当てられているシェーディングモデル {material.ShaderAssign.ShadingModel} が与えられたシェーダー定義に見つかりません。");
            }

            List<Macro> macros = new List<Macro>();
            foreach (var option in material.ShaderAssign.ShaderOptions)
            {
                var optionVar = shadingModel.OptionVars.FirstOrDefault(x => x.Id == option.Id);
                if (optionVar.Type != nw.g3d.nw4f_3dif.option_var_typeType.@static)
                {
                    // g3d ランタイムがサンプラー割り当ての動的変更を考慮した仕様になっていないので、static オプションのみサポート
                    continue;
                }

                macros.Add(new Macro() { Name = optionVar.VertexSymbol.Name, Value = option.Value });
                macros.Add(new Macro() { Name = optionVar.FragmentSymbol.Name, Value = option.Value });
            }

            return ExtractCompositeSamplerInfo(shaderDef, shadingModel, macros);
        }

        public static CompositeSamplerInfo ExtractCompositeSamplerInfo(
            ShaderDefinition shaderDef,
            ShadingModel shadingModel,
            IEnumerable<Macro> inputMacros)
        {
            Ensure.Argument.True(shaderDef.ShadingModels.Contains(shadingModel));
            string tmpFolder = Utility.GetTempFolderPath();
            try
            {
                // nw4f シェーダーコンバーターに渡すために全ソースコードをファイルに書き出し
                // TODO: 正式なものに置き換えたら消す
                Directory.CreateDirectory(tmpFolder);
                foreach(var src in shaderDef.ShaderSrcs)
                {
                    string fileName = Path.GetFileName(src.Path);

                    // ソースコードファイル名に重複があると nw4f シェーダーコンバーターのバグ(includeパスが効かない)で変換できないのでチェック
                    Ensure.Operation.Equals(1, shaderDef.ShaderSrcs.Count(x => Path.GetFileName(x.Path) == fileName), $"ソースコードのファイル名に重複があるため変換できません。{src.Path}");

                    string outputPath = Path.Combine(tmpFolder, fileName);
                    File.WriteAllText(outputPath, src.Stream.Value);
                }

                {
                    CompositeSamplerInfo info = new CompositeSamplerInfo();
                    info.ShadingModelName = shadingModel.Name;
                    if (shadingModel.VertexStage != null)
                    {
                        List<Macro> macros = new List<Macro>();
                        macros.Add(new Macro() { Name = "NN_G3D_VERTEX_SHADER", Value = "1" });
                        macros.AddRange(inputMacros);
                        CollectCombinerSamplerInfo(ref info, shadingModel.VertexStage.ShaderSrc.Path, tmpFolder, macros);
                    }

                    if (shadingModel.FragmentStage != null)
                    {
                        List<Macro> macros = new List<Macro>();
                        macros.Add(new Macro() { Name = "NN_G3D_FRAGMENT_SHADER", Value = "1" });
                        macros.AddRange(inputMacros);
                        CollectCombinerSamplerInfo(ref info, shadingModel.FragmentStage.ShaderSrc.Path, tmpFolder, macros);
                    }

                    info.CompositeSamplers.RemoveAll(x => x.SourceCount == 0);
                    if (info.CompositeSamplers.Count > 0)
                    {
                        return info;
                    }
                }
            }
            finally
            {
                if (Directory.Exists(tmpFolder))
                {
                    Directory.Delete(tmpFolder, true);
                }
            }

            return null;
        }

        private static void CollectCombinerSamplerInfo(
            ref CompositeSamplerInfo info,
            string mainSourcePath,
            string outputFolder,
            IEnumerable<Macro> additionalMacros)
        {
            string fileName = Path.GetFileName(mainSourcePath);
            string preprocessedSourcePath = Path.Combine(outputFolder, $"{fileName}.preprocess.glsl");

            // プリプロセスの適用
            {
                string inputSourcePath = Path.Combine(outputFolder, fileName);
                List<Macro> macros = new List<Macro>();
                macros.AddRange(additionalMacros);
                ApplyPreprocess(preprocessedSourcePath, inputSourcePath, macros);
            }

            // アノテーションの解析
            {
                var annotations = FindSamplerAnnotations(preprocessedSourcePath);
                foreach (var annotation in annotations)
                {
                    // とりあえず C スタイルコメントは考慮しない
                    string id = FindAnnotationValue("id", annotation);
                    if (string.IsNullOrEmpty(id))
                    {
                        id = FindAnnotationValue("sampler_id", annotation);
                        if (string.IsNullOrEmpty(id))
                        {
                            continue;
                        }
                    }

                    var sampler = info.CompositeSamplers.FirstOrDefault(x => x.SamplerName == id);
                    if (sampler == null)
                    {
                        sampler = new CompositeSampler()
                        {
                            SamplerName = id
                        };
                        info.CompositeSamplers.Add(sampler);
                    }

                    {
                        string annotationValue = FindAnnotationValue("source_r", annotation);
                        if (!string.IsNullOrEmpty(annotationValue))
                        {
                            (string name, Channel? channel) = ConvertToNameAndChannel(annotationValue);
                            sampler.SourceR = new CompositeSource();
                            sampler.SourceR.SamplerName = name;
                            sampler.SourceR.Channel = channel ?? channel.Value;
                        }
                    }
                    {
                        string annotationValue = FindAnnotationValue("source_g", annotation);
                        if (!string.IsNullOrEmpty(annotationValue))
                        {
                            (string name, Channel? channel) = ConvertToNameAndChannel(annotationValue);
                            sampler.SourceG = new CompositeSource();
                            sampler.SourceG.SamplerName = name;
                            sampler.SourceG.Channel = channel ?? channel.Value;
                        }
                    }
                    {
                        string annotationValue = FindAnnotationValue("source_b", annotation);
                        if (!string.IsNullOrEmpty(annotationValue))
                        {
                            (string name, Channel? channel) = ConvertToNameAndChannel(annotationValue);
                            sampler.SourceB = new CompositeSource();
                            sampler.SourceB.SamplerName = name;
                            sampler.SourceB.Channel = channel ?? channel.Value;
                        }
                    }
                    {
                        string annotationValue = FindAnnotationValue("source_a", annotation);
                        if (!string.IsNullOrEmpty(annotationValue))
                        {
                            (string name, Channel? channel) = ConvertToNameAndChannel(annotationValue);
                            sampler.SourceA = new CompositeSource();
                            sampler.SourceA.SamplerName = name;
                            sampler.SourceA.Channel = channel ?? channel.Value;
                        }
                    }
                    {
                        string annotationValue = FindAnnotationValue("comp_sel", annotation);
                        if (!string.IsNullOrEmpty(annotationValue))
                        {
                            sampler.CompSel = new ComponentSelector();
                            sampler.CompSel.FromString(annotationValue);
                        }
                    }
                    {
                        string annotationValue = FindAnnotationValue("format", annotation);
                        if (!string.IsNullOrEmpty(annotationValue))
                        {
                            texture_info_quantize_typeType format;
                            if (!Enum.TryParse(annotationValue, out format))
                            {
                                throw new Exception($"Failed to parse {annotationValue} to {typeof(texture_info_quantize_typeType).Name}.");
                            }

                            sampler.Format = format;
                        }
                    }
                    {
                        string annotationValue = FindAnnotationValue("linear", annotation);
                        if (!string.IsNullOrEmpty(annotationValue))
                        {
                            sampler.Linear = ConvertStringToLinear(annotationValue);
                        }
                    }
                }
            }
        }

        private static Bool4 ConvertStringToLinear(string source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.AreEqual(4, source.Length, $"invalid linear format {source}");
            var result = new Bool4();
            result.X = ConvertCharToLinear(source[0]);
            result.Y = ConvertCharToLinear(source[1]);
            result.Z = ConvertCharToLinear(source[2]);
            result.W = ConvertCharToLinear(source[3]);
            return result;
        }

        private static bool ConvertCharToLinear(char source)
        {
            switch (source)
            {
                case '0': return false;
                case '1': return true;
                default:
                    throw new Exception($"invalide linear format {source}");
            }
        }

        private static (string name, Channel? channel) ConvertToNameAndChannel(string sourceSamplerText)
        {
            string name = string.Empty;
            Channel? channel = null;
            var splited = sourceSamplerText.Split(':');
            name = splited[0];
            if (splited.Length > 1)
            {
                channel = ConvertStringToChannel(splited[1]);
            }

            return (name, channel);
        }

        private static Channel ConvertStringToChannel(string channelText)
        {
            switch (channelText.ToLower())
            {
                case "r": return Channel.R;
                case "g": return Channel.G;
                case "b": return Channel.B;
                case "a": return Channel.A;
                default:
                    throw new Exception("Unexpected default.");
            }
        }

        public static IEnumerable<string> FindSamplerAnnotations(string sourcePath)
        {
            var sourceCode = File.ReadAllText(sourcePath);
            List<string> annotations = new List<string>();
            string[] lines = sourceCode.Split('\n');
            foreach (string line in lines)
            {
                int commentStartIndex = line.IndexOf("//");
                if ((commentStartIndex != -1) &&
                    line.Contains("@@") &&
                    (line.Contains("sampler_id") || (line.Contains("sampler2D") && line.Contains("id"))))
                {
                    string annotationComment = line.Substring(commentStartIndex);
                    annotations.Add(annotationComment);
                }
            }

            return annotations;
        }

        public static void ApplyPreprocess(
            string outputSourcePath,
            string inputSourcePath,
            IEnumerable<Macro> macros)
        {
            string sdkRoot = Path.Combine(G3dToolUtility.GetG3dToolRootPath(), "../../../");
            string nw4fShaderConverterPath = Path.Combine(
                sdkRoot,
                @"Externals\nw4f_externals\Tools\Graphics\G3dTool\win64\NW4F_g3dshdrcvtr.exe");
            string sourceFolder = Path.GetDirectoryName(inputSourcePath);
            StringBuilder args = new StringBuilder();
            // バーテックスシェーダーとフラグメントシェーダー両方の入力が必須なのでダミーをさしておく
            string dummyOutputPath = Utility.GetTempFilePath(Path.GetFileName(inputSourcePath));
            args.Append($"-pp -ovs=\"{outputSourcePath}\" -ofs=\"{dummyOutputPath}\" -ivs=\"{inputSourcePath}\" -ifs=\"{inputSourcePath}\" -sc=\"GL_SOURCE\" -cp=65001 -ip=\"{sourceFolder}\" ");
            foreach (var macro in macros)
            {
                args.Append($"-dm=\"{macro.Name}={macro.Value}\" ");
            }

            if (IsSilent)
            {
                args.Append("-s ");
            }

            string currentDir = Directory.GetCurrentDirectory();
            try
            {
                // include パスがカレントディレクトリからの相対になってしまうのでここでセット
                Directory.SetCurrentDirectory(sourceFolder);
                DebugWriteLine(args.ToString());
                Utility.ExecuteProcess(nw4fShaderConverterPath, args.ToString());
            }
            finally
            {
                Directory.SetCurrentDirectory(currentDir);
                if (File.Exists(dummyOutputPath))
                {
                    File.Delete(dummyOutputPath);
                }
            }
        }

        [Conditional("DEBUG")]
        private static void DebugWriteLine(string message)
        {
            Console.WriteLine(message);
        }

        public static string FindAnnotationValue(string annotationName, string comment)
        {
            string[] spaceSplitedTokens = comment.Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);

            // ダブルクォーテーションで囲まれたトークンが切り離されているので結合する
            List<string> annotationTokens = new List<string>();
            for (int tokenIndex = 0; tokenIndex < spaceSplitedTokens.Length; ++tokenIndex)
            {
                string token = spaceSplitedTokens[tokenIndex];
                if ((token.Count(x => x == '"') == 1))
                {
                    StringBuilder combinedToken = new StringBuilder(token);
                    ++tokenIndex;
                    for (; tokenIndex < spaceSplitedTokens.Length; ++tokenIndex)
                    {
                        string subToken = spaceSplitedTokens[tokenIndex];
                        combinedToken.Append(' ' + subToken);
                        if ((subToken.Count(x => x == '"') == 1))
                        {
                            break;
                        }
                    }

                    annotationTokens.Add(combinedToken.ToString());
                }
                else
                {
                    annotationTokens.Add(token);
                }
            }

            // アノテーションの値を探す
            foreach (string annotationToken in annotationTokens)
            {
                if (annotationToken.StartsWith($"{annotationName}="))
                {
                    string[] idTokens = annotationToken.Split('=');
                    return idTokens[1].TrimEnd('\n').TrimEnd('\r').Trim('\"');
                }
            }

            return string.Empty;
        }
    }
}
