﻿using Nintendo.ToolFoundation.CommandLine;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace TextureCompositor
{
    public class CommandLineOptions
    {
        public bool IsSilent { get; set; } = false;

        public List<string> InputMaterialPaths { get; private set; } = new List<string>();
        public List<string> InputModelPaths { get; private set; } = new List<string>();

        public List<string> InputShaderDefinitionPaths { get; private set; } = new List<string>();

        public string OutputFolder { get; private set; }

        public string MaterialEditingScriptPath { get; private set; }

        public string TextureConvertScriptPath { get; private set; }

        public CommandLineOptions(string[] args)
        {
            // 引数パース
            var rootCommand = ActionCommand.CreateRootCommand(true);
            rootCommand.AddFlagOption('s', "silent", () => this.IsSilent = true)
                .GetBuilder()
                .SetDescription("ログを無効にします。");

            rootCommand
                .GetBuilder()
                .SetDescription("サンプラーやサンプラーに紐づくテクスチャーをチャンネル合成します。");

            rootCommand.AddValue(1, inputPath =>
            {
                string path = Environment.ExpandEnvironmentVariables(inputPath);
                if (!System.IO.File.Exists(path))
                {
                    throw new Exception($"指定されたファイルが存在しません\n\"{path}\"");
                }

                string ext = System.IO.Path.GetExtension(path);
                if (ext.Substring(1, 3) == "fmt")
                {
                    this.InputMaterialPaths.Add(path);
                }
                else if (ext.Substring(1, 3) == "fmd")
                {
                    this.InputModelPaths.Add(path);
                }
                else
                {
                    throw new Exception($"入力ファイルにはマテリアル中間ファイル(.fmtb)、もしくはモデル中間ファイル(.fmdb)を指定してください。\n\"{path}\"");
                }
            })
                .GetBuilder()
                .Require()
                .SetDescription("入力となるマテリアル中間ファイル、もしくはモデル中間ファイルを指定します。")
                .SetValueName("path");

            rootCommand.AddValueOption('o', "output", inputPath =>
            {
                string path = Environment.ExpandEnvironmentVariables(inputPath);
                this.OutputFolder = path;
            })
                .GetBuilder()
                .Require()
                .SetDescription("出力先のフォルダーを指定します。")
                .SetValueName("path");

            rootCommand.AddValueOption("shader-definition", inputPath =>
            {
                this.InputShaderDefinitionPaths.AddRange(Utility.ExpandFilePathPatterns(inputPath, "*.fsdb"));
            })
                .GetBuilder()
                .SetDescription("シェーダー定義ファイル(.fsdb)、もしくはファイルが格納されているフォルダーを指定します。セミコロン区切りで複数パスを入力することができます。")
                .SetValueName("paths");

            rootCommand.AddValueOption("script", inputPath =>
            {
                string path = Environment.ExpandEnvironmentVariables(inputPath);
                if (!System.IO.File.Exists(path))
                {
                    throw new Exception($"指定されたファイルが存在しません\n\"{path}\"");
                }

                string ext = System.IO.Path.GetExtension(path);
                if (ext == ".csx")
                {
                    this.TextureConvertScriptPath = path;
                    this.MaterialEditingScriptPath = path;
                }
                else
                {
                    throw new Exception($"スクリプトには C# スクリプト(.csx)を指定してください。\n\"{path}\"");
                }
            })
                .GetBuilder()
                .SetDescription("マテリアルプリプロセスやテクスチャー変換を行う C# スクリプト(.csx)を指定します。")
                .SetValueName("path");

            CommandLine.ParseArgs(args, rootCommand, new ParseSettings()
            {
                ErrorAction = message =>
                {
                    throw new Exception($"{message}\n指定された引数: {string.Join(" ", args)}\n{CommandLine.GetHelpText(args, rootCommand)}");
                },
                HelpWriter = Console.WriteLine
            });
        }
    }
}
