﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Diagnostics;
using System.Text;

namespace nw.g3d.nw4f_3dif
{
    // 中間ファイルデータフォーマッタユーティリティ
    public static class G3dDataFormatter
    {
        #region user_data
        // ユーザーデータ配列のデータ列をフォーマットします
        public static void FormatUserDataArrayData(
            user_data_arrayType user_data_array, string indent)
        {
            if (user_data_array == null) { return; }
            foreach (user_dataType user_data in user_data_array.Enumerate())
            {
                G3dDataFormatter.FormatUserDataData(user_data, indent);
            }
        }

        // ユーザーデータのデータ列をフォーマットします
        public static void FormatUserDataData(user_dataType user_data, string indent)
        {
            if (user_data.Item is user_intType)
            {
                G3dDataFormatter.FormatUserIntData(user_data.Item as user_intType, indent);
            }
            else if (user_data.Item is user_floatType)
            {
                G3dDataFormatter.FormatUserFloatData(user_data.Item as user_floatType, indent);
            }
        }

        // 整数型ユーザーデータのデータ列をフォーマットします
        public static void FormatUserIntData(user_intType user_int, string indent)
        {
            int[] vals = G3dDataParser.ParseIntArray(user_int.Value);
            if (vals == null) { return; }

            StringBuilder builder = new StringBuilder();
            builder.AppendLine();
            for (int i = 0; i < vals.Length; i++)
            {
                if ((i % 4) == 0) { builder.AppendFormat("{0}\t\t\t", indent); }

                int strlen = G3dDataFormatter.AppendInt(builder, vals[i]);

                // 最終項目なら何もしない
                if (i == (vals.Length - 1)) { continue; }

                if ((i % 4) != (4 - 1))
                {
                    // 末尾でなければタブ挿入
                    G3dDataFormatter.AppendTab4(builder, strlen);
                }
                else
                {
                    // 末尾なら改行
                    builder.AppendLine();
                }
            }

            builder.AppendLine();
            builder.AppendFormat("{0}\t\t", indent);

            user_int.Value = builder.ToString();
        }

        // 実数型ユーザーデータのデータ列をフォーマットします
        public static void FormatUserFloatData(user_floatType user_float, string indent)
        {
            float[] vals = G3dDataParser.ParseFloatArray(user_float.Value);
            if (vals == null) { return; }

            StringBuilder builder = new StringBuilder();
            builder.AppendLine();

            for (int i = 0; i < vals.Length; i++)
            {
                if ((i % 4) == 0) { builder.AppendFormat("{0}\t\t\t", indent); }

                int strlen = G3dDataFormatter.AppendFloat(builder, vals[i]);

                // 最終項目なら何もしない
                if (i == (vals.Length - 1)) { continue; }

                if ((i % 4) != (4 - 1))
                {
                    // 末尾でなければタブ挿入
                    G3dDataFormatter.AppendTab4(builder, strlen);
                }
                else
                {
                    // 末尾なら改行
                    builder.AppendLine();
                }
            }

            builder.AppendLine();
            builder.AppendFormat("{0}\t\t", indent);

            user_float.Value = builder.ToString();
        }
        #endregion

        #region stream
        // ストリーム配列のデータ列をフォーマットします
        public static void FormatStreamArrayData(stream_arrayType stream_array)
        {
            if (stream_array == null) { return; }

            // 大きなデータが多いので並列化する
            G3dParallel.ForEach(stream_array.stream, delegate(streamType stream)
            {
                G3dDataFormatter.FormatStreamData(stream);
            });
        }

        // ストリームのデータ列をフォーマットします
        public static void FormatStreamData(streamType stream)
        {
            if (stream.type == stream_typeType.@float)
            {
                float[] vals = G3dDataParser.ParseFloatArray(stream.Value);
                if (vals == null) { return; }
                FormatStreamData(stream, vals);
            }
            else if (stream.type == stream_typeType.@int)
            {
                int[] vals = G3dDataParser.ParseIntArray(stream.Value);
                if (vals == null) { return; }
                FormatStreamData(stream, vals);
            }
            else if (stream.type == stream_typeType.@byte)
            {
                byte[] vals = G3dDataParser.ParseByteArray(stream.Value);
                if (vals == null) { return; }
                FormatStreamData(stream, vals);
            }
            else if ((stream.type == stream_typeType.@string) ||
                (stream.type == stream_typeType.wstring))
            {
                // 何もしない
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {stream.type}");
            }
        }

        // ストリームのデータ列をフォーマットします
        public static void FormatStreamData(streamType stream, float[] vals)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((stream != null) && (stream.type == stream_typeType.@float));
            stream.Value = G3dDataFormatter.FormatStreamData(vals, stream.column);
        }

        // ストリームのデータ列をフォーマットします
        public static string FormatStreamData(float[] vals, int column)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(vals != null);

            StringBuilder builder = new StringBuilder();
            builder.AppendLine();

            for (int i = 0; i < vals.Length; i++)
            {
                if ((i % column) == 0) { builder.Append("\t\t"); }

                int strlen = G3dDataFormatter.AppendFloat(builder, vals[i]);

                // 最終項目なら何もしない
                if (i == (vals.Length - 1)) { continue; }

                if ((i % column) != (column - 1))
                {
                    // 末尾でなければタブ挿入
                    G3dDataFormatter.AppendTab4(builder, strlen);
                }
                else
                {
                    // 末尾なら改行
                    builder.AppendLine();
                }
            }

            builder.AppendLine();
            builder.Append("\t");
            return builder.ToString();
        }

        // ストリームのデータ列をフォーマットします
        public static void FormatStreamData(streamType stream, int[] vals)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((stream != null) && (stream.type == stream_typeType.@int));
            stream.Value = G3dDataFormatter.FormatStreamData(vals, stream.column);
        }

        // ストリームのデータ列をフォーマットします
        public static string FormatStreamData(int[] vals, int column)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(vals != null);

            StringBuilder builder = new StringBuilder();
            builder.AppendLine();

            for (int i = 0; i < vals.Length; i++)
            {
                if ((i % column) == 0) { builder.Append("\t\t"); }

                int strlen = G3dDataFormatter.AppendInt(builder, vals[i]);

                // 最終項目なら何もしない
                if (i == (vals.Length - 1)) { continue; }

                if ((i % column) != (column - 1))
                {
                    // 末尾でなければタブ挿入
                    G3dDataFormatter.AppendTab4(builder, strlen);
                }
                else
                {
                    // 末尾なら改行
                    builder.AppendLine();
                }
            }

            builder.AppendLine();
            builder.Append("\t");
            return builder.ToString();
        }

        // ストリームのデータ列をフォーマットします
        public static void FormatStreamData(streamType stream, byte[] vals)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((stream != null) && (stream.type == stream_typeType.@byte));
            stream.Value = G3dDataFormatter.FormatStreamData(vals, stream.column);
        }

        // ストリームのデータ列をフォーマットします
        public static string FormatStreamData(byte[] vals, int column)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(vals != null);

            StringBuilder builder = new StringBuilder();
            builder.AppendLine();

            for (int i = 0; i < vals.Length; i++)
            {
                if ((i % column) == 0) { builder.Append("\t\t"); }

                builder.AppendFormat("{0:X2}", vals[i]);

                // 最終項目なら何もしない
                if (i == (vals.Length - 1)) { continue; }

                if ((i % column) != (column - 1))
                {
                    // 末尾でなければスペース挿入
                    builder.Append(' ');

                    // 4 byte 区切りでスペース二つ
                    if ((column == 16) && ((i % 4) == 3)) { builder.Append(' '); }

                    // 3 byte 区切りでスペース二つ
                    if ((column == 12) && ((i % 3) == 2)) { builder.Append(' '); }
                }
                else
                {
                    // 末尾なら改行
                    builder.AppendLine();
                }
            }

            builder.AppendLine();
            builder.Append("\t");
            return builder.ToString();
        }
        #endregion

        //=====================================================================
        #region utility
        // int の追加
        public static int AppendInt(StringBuilder builder, int val)
        {
            // 符号分のスペース挿入
            int strlen = 0;
            if (val >= 0)
            {
                builder.Append(' ');
                strlen++;
            }

            string valStr = val.ToString();
            builder.Append(valStr);
            strlen += valStr.Length;
            return strlen;
        }

        // uint の追加
        public static int AppendUInt(StringBuilder builder, uint val)
        {
            string valStr = val.ToString();
            builder.Append(valStr);
            return valStr.Length;
        }

        // float の追加
        public static int AppendFloat(StringBuilder builder, float val)
        {
            // 符号分のスペース挿入
            int strlen = 0;
            if (val >= 0f)
            {
                builder.Append(' ');
                strlen++;
            }

            // 同じ値に戻ることを保障する
            string valStr = val.ToString("R");
            builder.Append(valStr);
            strlen += valStr.Length;
            return strlen;
        }

        // 4タブ幅のタブ挿入
        public static void AppendTab4(StringBuilder builder, int strlen)
        {
            builder.Append('\t');
            if (strlen < 4) { builder.Append('\t'); }
            if (strlen < 8) { builder.Append('\t'); }
            if (strlen < 12) { builder.Append('\t'); }
        }
        #endregion
    }
}
