﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;

namespace nw.g3d.nw4f_3dif
{
    /// <summary>
    /// データ列を格納するストリームです。
    /// </summary>
    public partial class G3dStream
    {
        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public G3dStream() { }

        /// <summary>
        /// 整数型ストリームのコンストラクタです。
        /// </summary>
        /// <param name="data">ストリームデータです。</param>
        /// <param name="column">列数です。</param>
        public G3dStream(int[] data, int column)
        {
            this.type = stream_typeType.@int;
            this.column = column;
            this.IntData.AddRange(data);
        }

        /// <summary>
        /// 浮動小数点数型ストリームのコンストラクタです。
        /// </summary>
        /// <param name="data">ストリームデータです。</param>
        /// <param name="column">列数です。</param>
        public G3dStream(float[] data, int column)
        {
            this.type = stream_typeType.@float;
            this.column = column;
            this.FloatData.AddRange(data);
        }

        /// <summary>
        /// テキストストリームを元にしたコンストラクタです。
        /// </summary>
        /// <param name="source">元にするテキストストリームです。</param>
        public G3dStream(streamType source) { SetText(source); }

        /// <summary>
        /// XML ストリームを元にしたコンストラクタです。
        /// </summary>
        /// <param name="source">元にする XML ストリームを指定します。</param>
        public G3dStream(XmlElement source) { SetElement(source); }

        //---------------------------------------------------------------------
        // テキストの設定
        public virtual void SetText(streamType source)
        {
            this.type = source.type;
            this.column = source.column;

            this.FloatData.Clear();
            this.IntData.Clear();
            this.ByteData.Clear();
            this.StringData = null;

            if (this.type == stream_typeType.@float)
            {
                float[] data = G3dDataParser.ParseFloatArray(source.Value);
                if (data != null) { this.FloatData.AddRange(data); }
            }
            else if (this.type == stream_typeType.@int)
            {
                int[] data = G3dDataParser.ParseIntArray(source.Value);
                if (data != null) { this.IntData.AddRange(data); }
            }
            else if (this.type == stream_typeType.@byte)
            {
                byte[] data = G3dDataParser.ParseByteArray(source.Value);
                if (data != null) { this.ByteData.AddRange(data); }
            }
            else if ((this.type == stream_typeType.@string) ||
                (this.type == stream_typeType.wstring))
            {
                this.StringData = source.Value;
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {this.type}");
            }
        }

        // テキストの取得
        public virtual streamType GetText(int index)
        {
            streamType stream = new streamType();
            stream.stream_index = index;
            stream.type = this.type;
            stream.column = this.column;
            if (this.type == stream_typeType.@float)
            {
                stream.count = this.FloatData.Count;
                G3dDataFormatter.FormatStreamData(stream, this.FloatData.ToArray());
            }
            else if (this.type == stream_typeType.@int)
            {
                stream.count = this.IntData.Count;
                G3dDataFormatter.FormatStreamData(stream, this.IntData.ToArray());
            }
            else if (this.type == stream_typeType.@byte)
            {
                stream.count = this.ByteData.Count;
                G3dDataFormatter.FormatStreamData(stream, this.ByteData.ToArray());
            }
            else if ((this.type == stream_typeType.@string) ||
                (this.type == stream_typeType.wstring))
            {
                stream.count = 1;
                stream.Value = this.StringData;
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {this.type}");
            }
            return stream;
        }

        //---------------------------------------------------------------------
        // 要素の設定
        public virtual void SetElement(XmlElement source)
        {
            this.FloatData.Clear();
            this.IntData.Clear();
            this.ByteData.Clear();
            this.StringData = null;

            string type = source.GetAttribute("type");
            if (type == "float")
            {
                this.type = stream_typeType.@float;
                float[] data = G3dDataParser.ParseFloatArray(source.InnerText);
                if (data != null) { this.FloatData.AddRange(data); }
            }
            else if (type == "int")
            {
                this.type = stream_typeType.@int;
                int[] data = G3dDataParser.ParseIntArray(source.InnerText);
                if (data != null) { this.IntData.AddRange(data); }
            }
            else if (type == "byte")
            {
                this.type = stream_typeType.@byte;
                byte[] data = G3dDataParser.ParseByteArray(source.InnerText);
                if (data != null) { this.ByteData.AddRange(data); }
            }
            else if (type == "string")
            {
                this.type = stream_typeType.@string;
                this.StringData = source.InnerText;
            }
            else if (type == "wstring")
            {
                this.type = stream_typeType.wstring;
                this.StringData = source.InnerText;
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {this.type}");
            }
            this.column = int.Parse(source.GetAttribute("column"));
        }

        // 要素の取得
        public virtual XmlElement GetElement(XmlDocument document, int index)
        {
            XmlElement element = document.CreateElement("stream");
            element.SetAttribute("stream_index", index.ToString());
            if (this.type == stream_typeType.@float)
            {
                element.SetAttribute("type", "float");
                element.SetAttribute("count", this.FloatData.Count.ToString());
                element.InnerText = G3dDataFormatter.FormatStreamData(
                    this.FloatData.ToArray(), this.column);
            }
            else if (this.type == stream_typeType.@int)
            {
                element.SetAttribute("type", "int");
                element.SetAttribute("count", this.IntData.Count.ToString());
                element.InnerText = G3dDataFormatter.FormatStreamData(
                    this.IntData.ToArray(), this.column);
            }
            else if (this.type == stream_typeType.@byte)
            {
                element.SetAttribute("type", "byte");
                element.SetAttribute("count", this.ByteData.Count.ToString());
                element.InnerText = G3dDataFormatter.FormatStreamData(
                    this.ByteData.ToArray(), this.column);
            }
            else if (this.type == stream_typeType.@string)
            {
                element.SetAttribute("type", "string");
                element.SetAttribute("count", "1");
                element.InnerText = this.StringData;
            }
            else if (this.type == stream_typeType.wstring)
            {
                element.SetAttribute("type", "wstring");
                element.SetAttribute("count", "1");
                element.InnerText = this.StringData;
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {this.type}");
            }
            element.SetAttribute("column", this.column.ToString());
            return element;
        }

        //---------------------------------------------------------------------
        // 読み込み
        public virtual void Read(BinaryReader rd)
        {
            ulong chunkID = rd.ReadUInt64();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(chunkID == G3dConstant.StreamSubChunk);
            this.type = (stream_typeType)rd.ReadInt32();
            int count = rd.ReadInt32();
            this.column = rd.ReadInt32();
            int size = rd.ReadInt32();

            // パディング読み飛ばし
            rd.ReadInt32();
            rd.ReadInt32();

            this.FloatData.Clear();
            this.IntData.Clear();
            this.ByteData.Clear();
            this.StringData = null;

            if (this.type == stream_typeType.@float)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(size == (count * sizeof(float)));
                this.FloatData.Capacity = count;
                for (int i = 0; i < count; i++)
                {
                    this.FloatData.Add(rd.ReadSingle());
                }
            }
            else if (this.type == stream_typeType.@int)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(size == (count * sizeof(int)));
                this.IntData.Capacity = count;
                for (int i = 0; i < count; i++)
                {
                    this.IntData.Add(rd.ReadInt32());
                }
            }
            else if (this.type == stream_typeType.@byte)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(size == (count * sizeof(byte)));
                // メモリ使用量の増加を抑えるために、一定量ずつコピー
                this.ByteData.Capacity = count;
                const int BlockSize = 100 * 1024;
                var remain = count % BlockSize;
                var repeat = (count - remain) / BlockSize;
                if (repeat > 0)
                {
                    var byteArray = new byte[BlockSize];
                    for (var i = 0; i < repeat; i++)
                    {
                        rd.Read(byteArray, 0, BlockSize);
                        this.ByteData.AddRange(byteArray);
                    }
                }
                if (remain > 0)
                {
                    this.ByteData.AddRange(rd.ReadBytes(remain));
                }
            }
            else if (this.type == stream_typeType.@string)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(count == 1);
                // null 終端を読まない
                byte[] stringBytes = rd.ReadBytes(size - 1);
                rd.ReadByte();
                // 空文字列は this.StringData == null として扱う
                if (stringBytes.Length > 0)
                {
                    string stringDataLF = Encoding.ASCII.GetString(stringBytes);
                    this.StringData = stringDataLF.Replace("\n", "\r\n");
                }
            }
            else if (this.type == stream_typeType.wstring)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(count == 1);
                // null 終端を読まない
                byte[] stringBytes = rd.ReadBytes(size - 2);
                rd.ReadChar();
                // 空文字列は this.StringData == null として扱う
                if (stringBytes.Length > 0)
                {
                    string stringDataLF = Encoding.Unicode.GetString(stringBytes);
                    this.StringData = stringDataLF.Replace("\n", "\r\n");
                }
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {this.type}");
            }
        }

        // 書き込み
        public virtual void Write(BinaryWriter wt)
        {
            // 文字列データの準備
            byte[] stringBytes = new byte[0];
            int nullSize = 0;
            if (this.type == stream_typeType.@string)
            {
                if (this.StringData != null)
                {
                    string stringDataLF = this.StringData.Replace("\r\n", "\n");
                    stringDataLF = stringDataLF.Replace('\r', '\n');
                    stringBytes = Encoding.ASCII.GetBytes(stringDataLF);
                }
                nullSize = 1;
            }
            else if (this.type == stream_typeType.wstring)
            {
                if (this.StringData != null)
                {
                    string stringDataLF = this.StringData.Replace("\r\n", "\n");
                    stringDataLF = stringDataLF.Replace('\r', '\n');
                    stringBytes = Encoding.Unicode.GetBytes(stringDataLF);
                }
                nullSize = 2;
            }

            wt.Write(G3dConstant.StreamSubChunk);
            wt.Write((int)this.type);
            if (this.type == stream_typeType.@float)
            {
                wt.Write(this.FloatData.Count);
                wt.Write(this.column);
                wt.Write(this.FloatData.Count * sizeof(float));
            }
            else if (this.type == stream_typeType.@int)
            {
                wt.Write(this.IntData.Count);
                wt.Write(this.column);
                wt.Write(this.IntData.Count * sizeof(int));
            }
            else if (this.type == stream_typeType.@byte)
            {
                wt.Write(this.ByteData.Count);
                wt.Write(this.column);
                wt.Write(this.ByteData.Count * sizeof(byte));
            }
            else if ((this.type == stream_typeType.@string) ||
                (this.type == stream_typeType.wstring))
            {
                // 文字列の count は必ず 1
                wt.Write(1);
                wt.Write(this.column);
                // null 文字分を足す
                wt.Write(stringBytes.Length + nullSize);
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {this.type}");
            }

            // 32byte アライメント
            int padding = 0;
            wt.Write(padding);
            wt.Write(padding);

            if (this.type == stream_typeType.@float)
            {
                foreach (float value in this.FloatData) { wt.Write(value); }
            }
            else if (this.type == stream_typeType.@int)
            {
                foreach (int value in this.IntData) { wt.Write(value); }
            }
            else if (this.type == stream_typeType.@byte)
            {
                wt.Write(this.ByteData.ToArray());
            }
            else if ((this.type == stream_typeType.@string) ||
                (this.type == stream_typeType.wstring))
            {
                wt.Write(stringBytes);
                for (int i = 0; i < nullSize; i++) { wt.Write((byte)0); }
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {this.type}");
            }
        }

        //---------------------------------------------------------------------
        public readonly List<float> FloatData = new List<float>();
        public readonly List<int> IntData = new List<int>();
        public readonly List<byte> ByteData = new List<byte>();
        public string StringData { get; set; }

        public stream_typeType type { get; set; }
        public int column { get; set; }
    }
}
