﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using System.Collections;
using System;

namespace nw.g3d.nw4f_3dif
{
    /// <summary>
    /// 配列要素
    /// </summary>
    [Serializable]
    public abstract class G3dArrayElement<TType>
        where TType : IG3dIndexHintElement
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public abstract TType[] Items { get; set; }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public abstract int length_hint { get; set; }

        //-----------------------------------------------------------------
        /// <summary>
        /// ヒントの更新
        /// </summary>
        public void UpdateHint()
        {
            int length = this.Count;
            this.length_hint = length;
            for (int i = 0; i < length; i++) { this.Items[i].index_hint = i; }
        }

        //-----------------------------------------------------------------
        /// <summary>
        /// 配列の列挙
        /// </summary>
        /// <returns>列挙子</returns>
        public IEnumerable<TType> Enumerate()
        {
            if (this.Items != null)
            {
                foreach (TType item in this.Items) { yield return item; }
            }
        }

        //-----------------------------------------------------------------
        /// <summary>
        /// 配列の長さを返します。
        /// </summary>
        public int Capacity
        {
            get { return (this.Items != null) ? this.Items.Length : 0; }
        }

        /// <summary>
        /// 配列が充填した状態にあるかどうかを確認します。
        /// </summary>
        /// <returns>
        /// null 要素が無ければ、true を返します。
        /// </returns>
        public bool IsFilled
        {
            get
            {
                int capacity = this.Capacity;
                for (int i = 0; i < capacity; i++)
                {
                    IG3dIndexHintElement item = this.Items[i];
                    if (item == null) { return false; }
                }
                return true;
            }
        }

        /// <summary>
        /// 配列を充填した状態にします。
        /// </summary>
        public void Trim()
        {
            if (this.Items == null) { return; }

            int count = 0;
            foreach (TType item in this.Items) { if (item != null) { count++; } }

            if (count == 0)
            {
                this.Items = null;
                return;
            }

            if (this.Items.Length == count) { return; }

            TType[] array = new TType[count];
            int index = 0;
            foreach (TType item in this.Items)
            {
                if (item != null)
                {
                    array[index] = item;
                    index++;
                }
            }
            this.Items = array;
        }

        /// <summary>
        /// 配列の要素数を返します。
        /// 配列が充填した状態で利用する必要があります。
        /// </summary>
        public int Count
        {
            get
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(this.IsFilled);
                return this.Capacity;
            }
        }

        //-----------------------------------------------------------------
        //-----------------------------------------------------------------
        //インデックスの管理はここでは行わない
        //補正にはドキュメント自体が必要、別クラスにすべき
        //各インスタンスをこれ以上拡張しない範囲で拡張する

        //IsValid null があったら例外を投げる、インデックスがそろっていなかったら例外を投げる

        //TODO: Array と List のインターフェースを参考に拡張する
        //Array<> にあるものは用意しないのも手？取り回すだけ取り回す？リストのように気軽に使いたい
        //TODO: Add、AddRange、BinarySearch、Clear、Contains、CopyTo、
        //Exists、Find、FindIndex、FindAll、FindLast、FindLastIndex、ForEach
        //IndexOf、LastIndexOf、Insert、InsertRange、
        //Remove、RemoveAt、RemoveAll、RemoveRange、Sort、IsSorted、Reverse
        //Trim
        //TrueForAll
        //引数をダウンキャストした際の null チェックも行う
    }

    //=====================================================================
    public partial class process_log_arrayType :
        G3dArrayElement<process_logType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override process_logType[] Items
        {
            get { return this.process_log; }
            set { this.process_log = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class macro_arrayType :
        G3dArrayElement<macroType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override macroType[] Items
        {
            get { return this.macro; }
            set { this.macro = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class user_data_arrayType :
        G3dArrayElement<user_dataType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override user_dataType[] Items
        {
            get { return this.user_data; }
            set { this.user_data = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class stream_arrayType :
        G3dArrayElement<streamType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override streamType[] Items
        {
            get { return this.stream; }
            set { this.stream = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class material_arrayType :
        G3dArrayElement<materialType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override materialType[] Items
        {
            get { return this.material; }
            set { this.material = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class render_info_arrayType :
        G3dArrayElement<render_infoType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override render_infoType[] Items
        {
            get { return this.render_info; }
            set { this.render_info = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class sampler_arrayType :
        G3dArrayElement<samplerType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override samplerType[] Items
        {
            get { return this.sampler; }
            set { this.sampler = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class shader_option_arrayType :
        G3dArrayElement<shader_optionType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shader_optionType[] Items
        {
            get { return this.shader_option; }
            set { this.shader_option = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class sampler_assign_arrayType :
        G3dArrayElement<sampler_assignType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override sampler_assignType[] Items
        {
            get { return this.sampler_assign; }
            set { this.sampler_assign = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class shader_param_arrayType :
        G3dArrayElement<shader_paramType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shader_paramType[] Items
        {
            get { return this.shader_param; }
            set { this.shader_param = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class attrib_assign_arrayType :
        G3dArrayElement<attrib_assignType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override attrib_assignType[] Items
        {
            get { return this.attrib_assign; }
            set { this.attrib_assign = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class bone_arrayType :
        G3dArrayElement<boneType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override boneType[] Items
        {
            get { return this.bone; }
            set { this.bone = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class vertex_arrayType :
        G3dArrayElement<vertexType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override vertexType[] Items
        {
            get { return this.vertex; }
            set { this.vertex = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class vtx_attrib_arrayType :
        G3dArrayElement<vtx_attribType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override vtx_attribType[] Items
        {
            get { return this.vtx_attrib; }
            set { this.vtx_attrib = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class vtx_buffer_arrayType :
        G3dArrayElement<vtx_bufferType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override vtx_bufferType[] Items
        {
            get { return this.vtx_buffer; }
            set { this.vtx_buffer = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class input_arrayType :
        G3dArrayElement<inputType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override inputType[] Items
        {
            get { return this.input; }
            set { this.input = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class shape_arrayType :
        G3dArrayElement<shapeType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shapeType[] Items
        {
            get { return this.shape; }
            set { this.shape = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class mesh_arrayType :
        G3dArrayElement<meshType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override meshType[] Items
        {
            get { return this.mesh; }
            set { this.mesh = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class submesh_arrayType :
        G3dArrayElement<submeshType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override submeshType[] Items
        {
            get { return this.submesh; }
            set { this.submesh = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class key_shape_arrayType :
        G3dArrayElement<key_shapeType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override key_shapeType[] Items
        {
            get { return this.key_shape; }
            set { this.key_shape = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class target_attrib_arrayType :
        G3dArrayElement<target_attribType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override target_attribType[] Items
        {
            get { return this.target_attrib; }
            set { this.target_attrib = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class original_material_arrayType :
        G3dArrayElement<original_materialType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override original_materialType[] Items
        {
            get { return this.original_material; }
            set { this.original_material = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class original_color_arrayType :
        G3dArrayElement<original_colorType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override original_colorType[] Items
        {
            get { return this.original_color; }
            set { this.original_color = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class original_texsrt_arrayType :
        G3dArrayElement<original_texsrtType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override original_texsrtType[] Items
        {
            get { return this.original_texsrt; }
            set { this.original_texsrt = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class original_image_arrayType :
        G3dArrayElement<original_imageType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override original_imageType[] Items
        {
            get { return this.original_image; }
            set { this.original_image = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class bone_anim_arrayType :
        G3dArrayElement<bone_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override bone_animType[] Items
        {
            get { return this.bone_anim; }
            set { this.bone_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class per_material_anim_arrayType :
        G3dArrayElement<per_material_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override per_material_animType[] Items
        {
            get { return this.per_material_anim; }
            set { this.per_material_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class shader_param_anim_arrayType :
        G3dArrayElement<param_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override param_animType[] Items
        {
            get { return this.param_anim; }
            set { this.param_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class tex_pattern_anim_arrayType :
        G3dArrayElement<pattern_anim_targetType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override pattern_anim_targetType[] Items
        {
            get { return this.pattern_anim; }
            set { this.pattern_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class shader_param_mat_anim_arrayType :
        G3dArrayElement<shader_param_mat_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shader_param_mat_animType[] Items
        {
            get { return this.shader_param_mat_anim; }
            set { this.shader_param_mat_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class param_anim_arrayType :
        G3dArrayElement<param_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override param_animType[] Items
        {
            get { return this.param_anim; }
            set { this.param_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class original_per_material_anim_arrayType
        : G3dArrayElement<original_material_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override original_material_animType[] Items
        {
            get { return this.original_per_material_anim; }
            set { this.original_per_material_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class original_material_anim_arrayType :
        G3dArrayElement<original_material_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override original_material_animType[] Items
        {
            get { return this.original_material_anim; }
            set { this.original_material_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class original_color_anim_arrayType :
        G3dArrayElement<original_color_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override original_color_animType[] Items
        {
            get { return this.original_color_anim; }
            set { this.original_color_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class original_texsrt_anim_arrayType :
        G3dArrayElement<original_texsrt_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override original_texsrt_animType[] Items
        {
            get { return this.original_texsrt_anim; }
            set { this.original_texsrt_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class tex_pattern_arrayType :
        G3dArrayElement<tex_patternType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override tex_patternType[] Items
        {
            get { return this.tex_pattern; }
            set { this.tex_pattern = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class tex_pattern_mat_anim_arrayType :
        G3dArrayElement<tex_pattern_mat_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override tex_pattern_mat_animType[] Items
        {
            get { return this.tex_pattern_mat_anim; }
            set { this.tex_pattern_mat_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class bone_vis_bone_anim_arrayType :
        G3dArrayElement<bone_vis_bone_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override bone_vis_bone_animType[] Items
        {
            get { return this.bone_vis_bone_anim; }
            set { this.bone_vis_bone_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class mat_vis_mat_anim_arrayType :
        G3dArrayElement<mat_vis_mat_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override mat_vis_mat_animType[] Items
        {
            get { return this.mat_vis_mat_anim; }
            set { this.mat_vis_mat_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class vertex_shape_anim_arrayType :
        G3dArrayElement<vertex_shape_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override vertex_shape_animType[] Items
        {
            get { return this.vertex_shape_anim; }
            set { this.vertex_shape_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class camera_anim_arrayType :
        G3dArrayElement<camera_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override camera_animType[] Items
        {
            get { return this.camera_anim; }
            set { this.camera_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class light_anim_arrayType :
        G3dArrayElement<light_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override light_animType[] Items
        {
            get { return this.light_anim; }
            set { this.light_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class fog_anim_arrayType :
        G3dArrayElement<fog_animType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override fog_animType[] Items
        {
            get { return this.fog_anim; }
            set { this.fog_anim = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class include_path_arrayType :
        G3dArrayElement<include_pathType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override include_pathType[] Items
        {
            get { return this.include_path; }
            set { this.include_path = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class force_include_file_arrayType :
        G3dArrayElement<force_include_fileType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override force_include_fileType[] Items
        {
            get { return this.force_include_file; }
            set { this.force_include_file = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class shader_arrayType :
        G3dArrayElement<shaderType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shaderType[] Items
        {
            get { return this.shader; }
            set { this.shader = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class variation_arrayType :
        G3dArrayElement<variationType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override variationType[] Items
        {
            get { return this.variation; }
            set { this.variation = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class force_include_arrayType :
        G3dArrayElement<force_includeType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override force_includeType[] Items
        {
            get { return this.force_include; }
            set { this.force_include = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class shading_model_arrayType :
        G3dArrayElement<shading_modelType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shading_modelType[] Items
        {
            get { return this.shading_model; }
            set { this.shading_model = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class option_var_arrayType :
    G3dArrayElement<option_varType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override option_varType[] Items
        {
            get { return this.option_var; }
            set { this.option_var = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class attrib_var_arrayType :
        G3dArrayElement<attrib_varType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override attrib_varType[] Items
        {
            get { return this.attrib_var; }
            set { this.attrib_var = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class sampler_var_arrayType :
    G3dArrayElement<sampler_varType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override sampler_varType[] Items
        {
            get { return this.sampler_var; }
            set { this.sampler_var = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class block_var_arrayType :
    G3dArrayElement<block_varType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override block_varType[] Items
        {
            get { return this.block_var; }
            set { this.block_var = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class uniform_var_arrayType :
    G3dArrayElement<uniform_varType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override uniform_varType[] Items
        {
            get { return this.uniform_var; }
            set { this.uniform_var = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class textblock_arrayType : G3dArrayElement<textblockType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override textblockType[] Items
        {
            get { return this.textblock; }
            set { this.textblock = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class shader_storage_block_var_arrayType : G3dArrayElement<shader_storage_block_varType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shader_storage_block_varType[] Items
        {
            get { return this.shader_storage_block_var; }
            set { this.shader_storage_block_var = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class buffer_var_arrayType : G3dArrayElement<buffer_varType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override buffer_varType[] Items
        {
            get { return this.buffer_var; }
            set { this.buffer_var = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class render_info_slot_arrayType :
        G3dArrayElement<render_info_slotType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override render_info_slotType[] Items
        {
            get { return this.render_info_slot; }
            set { this.render_info_slot = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class page_arrayType :
    G3dArrayElement<pageType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override pageType[] Items
        {
            get { return this.page; }
            set { this.page = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class interleave_arrayType :
        G3dArrayElement<interleaveType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override interleaveType[] Items
        {
            get { return this.interleave; }
            set { this.interleave = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class group_arrayType :
        G3dArrayElement<groupType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override groupType[] Items
        {
            get { return this.group; }
            set { this.group = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class shader_src_arrayType :
        G3dArrayElement<shader_srcType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shader_srcType[] Items
        {
            get { return this.shader_src; }
            set { this.shader_src = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    //=====================================================================
    public partial class target_shader_arrayType :
        G3dArrayElement<target_shaderType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override target_shaderType[] Items
        {
            get { return this.target_shader; }
            set { this.target_shader = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class shader_program_arrayType :
    G3dArrayElement<shader_programType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override shader_programType[] Items
        {
            get { return this.shader_program; }
            set { this.shader_program = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class option_arrayType :
        G3dArrayElement<optionType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override optionType[] Items
        {
            get { return this.option; }
            set { this.option = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class connection_arrayType : G3dArrayElement<connectionType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override connectionType[] Items
        {
            get { return this.connection; }
            set { this.connection = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class block_arrayType : G3dArrayElement<blockType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override blockType[] Items
        {
            get { return this.block; }
            set { this.block = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }

    public partial class uniform_arrayType : G3dArrayElement<uniformType>
    {
        /// <summary>
        /// 配列
        /// </summary>
        [XmlIgnore]
        public override uniformType[] Items
        {
            get { return this.uniform; }
            set { this.uniform = value; }
        }

        /// <summary>
        /// 配列長のヒント
        /// </summary>
        [XmlIgnore]
        public override int length_hint
        {
            get { return this.length; }
            set { this.length = value; }
        }
    }
}
