﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace nw.g3d.nw4f_3dif
{
    /// <summary>
    /// 処理頂点数カウンタ
    /// </summary>
    public static class G3dProcessingVertexCounter
    {
        /// <summary>
        /// トライアングルズの処理頂点数を返します。
        /// </summary>
        /// <param name="indexStream">インデックス列挙子</param>
        /// <returns>処理頂点数</returns>
        public static int CountTriangles(IEnumerable<int> indexStream)
        {
            Queue<int> vertexCache = new Queue<int>();
            int processingVertexCount = 0;
            foreach (int index in indexStream)
            {
                if (vertexCache.Contains(index)) { continue; }
                processingVertexCount++;
                vertexCache.Enqueue(index);
                if (vertexCache.Count > G3dConstant.VertexCacheSize) { vertexCache.Dequeue(); }
            }
            return processingVertexCount;
        }

        public static int CountTrianglesNX(IEnumerable<int> indexStream)
        {
            const int MaxPrimitiveCount = 32;
            const int MaxUniqueVertexCount = 32;

            List<int> batchIndex = new List<int>();
            List<int> uniqueIndex = new List<int>();
            int transformCount = 0;
            int primitiveCount = 0;
            // プリミティブ単位のループ
            for (int index = 0; index < indexStream.Count(); index += 3)
            {
                uniqueIndex.Clear();
                // 処理しようとするプリミティブの頂点でバッチ対象に含まれていない頂点があるかチェックする
                for (int indexStreamIndex = index; indexStreamIndex < index + 3; ++indexStreamIndex)
                {
                    int vertexIndex = indexStream.ElementAt(indexStreamIndex);
                    if (!batchIndex.Contains(vertexIndex) && !uniqueIndex.Contains(vertexIndex))
                    {
                        uniqueIndex.Add(vertexIndex);
                    }
                }

                // 1バッチに含めることが出来る頂点数の上限に達するかをチェック
                if (batchIndex.Count() + uniqueIndex.Count() > MaxUniqueVertexCount)
                {
                    transformCount += batchIndex.Count();
                    batchIndex.Clear();
                    primitiveCount = 0;
                }

                // 処理しようとするプリミティブの頂点でバッチ対象に含まれていない頂点を追加する
                for (int indexStreamIndex = index; indexStreamIndex < index + 3; ++indexStreamIndex)
                {
                    int vertexIndex = indexStream.ElementAt(indexStreamIndex);
                    if (!batchIndex.Contains(vertexIndex))
                    {
                        batchIndex.Add(vertexIndex);
                    }
                }

                ++primitiveCount;

                // 1バッチに含めることが出来るプリミティブ数の上限に達しているかをチェック
                if (primitiveCount == MaxPrimitiveCount || index + 3 >= indexStream.Count())
                {
                    transformCount += batchIndex.Count();
                    batchIndex.Clear();
                    primitiveCount = 0;
                }
            }

            return transformCount;
        }
    }
}
