﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;

namespace nw.g3d.nw4f_3dif
{
    /// <summary>
    /// 中間ファイルのパス
    /// </summary>
    public static class G3dPath
    {
        /// <summary>
        /// プラットフォーム
        /// </summary>
        public const string Platform = "f";

        /// <summary>
        /// テキスト中間ファイル
        /// </summary>
        public const string Text = "a";

        /// <summary>
        /// バイナリ中間ファイル
        /// </summary>
        public const string Binary = "b";

        //=====================================================================
        /// <summary>
        /// モデル
        /// </summary>
        public const string Model = "md";

        /// <summary>
        /// モデル中間ファイル拡張子
        /// </summary>
        public const string ModelExtension =
            "." + G3dPath.Platform + G3dPath.Model;

        /// <summary>
        /// テキストモデル中間ファイル拡張子
        /// </summary>
        public const string ModelTextExtension =
            G3dPath.ModelExtension + G3dPath.Text;

        /// <summary>
        /// バイナリモデル中間ファイル拡張子
        /// </summary>
        public const string ModelBinaryExtension =
            G3dPath.ModelExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// マテリアル
        /// </summary>
        public const string Material = "mt";

        /// <summary>
        /// マテリアル中間ファイル拡張子
        /// </summary>
        public const string MaterialExtension =
            "." + G3dPath.Platform + G3dPath.Material;

        /// <summary>
        /// テキストマテリアル中間ファイル拡張子
        /// </summary>
        public const string MaterialTextExtension =
            G3dPath.MaterialExtension + G3dPath.Text;

        /// <summary>
        /// バイナリマテリアル中間ファイル拡張子
        /// </summary>
        public const string MaterialBinaryExtension =
            G3dPath.MaterialExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// テクスチャ
        /// </summary>
        public const string Texture = "tx";

        /// <summary>
        /// テクスチャ中間ファイル拡張子
        /// </summary>
        public const string TextureExtension =
            "." + G3dPath.Platform + G3dPath.Texture;

        /// <summary>
        /// テキストテクスチャ中間ファイル拡張子
        /// </summary>
        public const string TextureTextExtension =
            G3dPath.TextureExtension + G3dPath.Text;

        /// <summary>
        /// バイナリテクスチャ中間ファイル拡張子
        /// </summary>
        public const string TextureBinaryExtension =
            G3dPath.TextureExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// スケルタルアニメーション
        /// </summary>
        public const string SkeletalAnim = "sk";

        /// <summary>
        /// スケルタルアニメーション中間ファイル拡張子
        /// </summary>
        public const string SkeletalAnimExtension =
            "." + G3dPath.Platform + G3dPath.SkeletalAnim;

        /// <summary>
        /// テキストスケルタルアニメーション中間ファイル拡張子
        /// </summary>
        public const string SkeletalAnimTextExtension =
            G3dPath.SkeletalAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリスケルタルアニメーション中間ファイル拡張子
        /// </summary>
        public const string SkeletalAnimBinaryExtension =
            G3dPath.SkeletalAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// マテリアルアニメーション
        /// </summary>
        public const string MaterialAnim = "ma";

        /// <summary>
        /// マテリアルアニメーション中間ファイル拡張子
        /// </summary>
        public const string MaterialAnimExtension =
            "." + G3dPath.Platform + G3dPath.MaterialAnim;

        /// <summary>
        /// テキストマテリアルアニメーション中間ファイル拡張子
        /// </summary>
        public const string MaterialAnimTextExtension =
            G3dPath.MaterialAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリマテリアルアニメーション中間ファイル拡張子
        /// </summary>
        public const string MaterialAnimBinaryExtension =
            G3dPath.MaterialAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// シェーダーパラメーターアニメーション
        /// </summary>
        public const string ShaderParamAnim = "sp";

        /// <summary>
        /// シェーダーパラメーターアニメーション中間ファイル拡張子
        /// </summary>
        public const string ShaderParamAnimExtension =
            "." + G3dPath.Platform + G3dPath.ShaderParamAnim;

        /// <summary>
        /// テキストシェーダーパラメーターアニメーション中間ファイル拡張子
        /// </summary>
        public const string ShaderParamAnimTextExtension =
            G3dPath.ShaderParamAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリシェーダーパラメーターアニメーション中間ファイル拡張子
        /// </summary>
        public const string ShaderParamAnimBinaryExtension =
            G3dPath.ShaderParamAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// カラーアニメーション
        /// </summary>
        public const string ColorAnim = "cl";

        /// <summary>
        /// カラーアニメーション中間ファイル拡張子
        /// </summary>
        public const string ColorAnimExtension =
            "." + G3dPath.Platform + G3dPath.ColorAnim;

        /// <summary>
        /// テキストカラーアニメーション中間ファイル拡張子
        /// </summary>
        public const string ColorAnimTextExtension =
            G3dPath.ColorAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリカラーアニメーション中間ファイル拡張子
        /// </summary>
        public const string ColorAnimBinaryExtension =
            G3dPath.ColorAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// テクスチャ SRT アニメーション
        /// </summary>
        public const string TexSrtAnim = "ts";

        /// <summary>
        /// テクスチャ SRT アニメーション中間ファイル拡張子
        /// </summary>
        public const string TexSrtAnimExtension =
            "." + G3dPath.Platform + G3dPath.TexSrtAnim;

        /// <summary>
        /// テキストテクスチャ SRT アニメーション中間ファイル拡張子
        /// </summary>
        public const string TexSrtAnimTextExtension =
            G3dPath.TexSrtAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリテクスチャ SRT アニメーション中間ファイル拡張子
        /// </summary>
        public const string TexSrtAnimBinaryExtension =
            G3dPath.TexSrtAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// テクスチャパターンアニメーション
        /// </summary>
        public const string TexPatternAnim = "tp";

        /// <summary>
        /// テクスチャパターンアニメーション中間ファイル拡張子
        /// </summary>
        public const string TexPatternAnimExtension =
            "." + G3dPath.Platform + G3dPath.TexPatternAnim;

        /// <summary>
        /// テキストテクスチャパターンアニメーション中間ファイル拡張子
        /// </summary>
        public const string TexPatternAnimTextExtension =
            G3dPath.TexPatternAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリテクスチャパターンアニメーション中間ファイル拡張子
        /// </summary>
        public const string TexPatternAnimBinaryExtension =
            G3dPath.TexPatternAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// ボーンビジビリティアニメーション
        /// </summary>
        public const string BoneVisibilityAnim = "vb";

        /// <summary>
        /// ボーンビジビリティアニメーション中間ファイル拡張子
        /// </summary>
        public const string BoneVisibilityAnimExtension =
            "." + G3dPath.Platform + G3dPath.BoneVisibilityAnim;

        /// <summary>
        /// テキストボーンビジビリティアニメーション中間ファイル拡張子
        /// </summary>
        public const string BoneVisibilityAnimTextExtension =
            G3dPath.BoneVisibilityAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリボーンビジビリティアニメーション中間ファイル拡張子
        /// </summary>
        public const string BoneVisibilityAnimBinaryExtension =
            G3dPath.BoneVisibilityAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// マテリアルビジビリティアニメーション
        /// </summary>
        public const string MatVisibilityAnim = "vm";

        /// <summary>
        /// マテリアルビジビリティアニメーション中間ファイル拡張子
        /// </summary>
        public const string MatVisibilityAnimExtension =
            "." + G3dPath.Platform + G3dPath.MatVisibilityAnim;

        /// <summary>
        /// テキストマテリアルビジビリティアニメーション中間ファイル拡張子
        /// </summary>
        public const string MatVisibilityAnimTextExtension =
            G3dPath.MatVisibilityAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリマテリアルビジビリティアニメーション中間ファイル拡張子
        /// </summary>
        public const string MatVisibilityAnimBinaryExtension =
            G3dPath.MatVisibilityAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// シェイプアニメーション
        /// </summary>
        public const string ShapeAnim = "sh";

        /// <summary>
        /// シェイプアニメーション中間ファイル拡張子
        /// </summary>
        public const string ShapeAnimExtension =
            "." + G3dPath.Platform + G3dPath.ShapeAnim;

        /// <summary>
        /// テキストシェイプアニメーション中間ファイル拡張子
        /// </summary>
        public const string ShapeAnimTextExtension =
            G3dPath.ShapeAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリシェイプアニメーション中間ファイル拡張子
        /// </summary>
        public const string ShapeAnimBinaryExtension =
            G3dPath.ShapeAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// シーンアニメーション
        /// </summary>
        public const string SceneAnim = "sn";

        /// <summary>
        /// シーンアニメーション中間ファイル拡張子
        /// </summary>
        public const string SceneAnimExtension =
            "." + G3dPath.Platform + G3dPath.SceneAnim;

        /// <summary>
        /// テキストシーンアニメーション中間ファイル拡張子
        /// </summary>
        public const string SceneAnimTextExtension =
            G3dPath.SceneAnimExtension + G3dPath.Text;

        /// <summary>
        /// バイナリシーンアニメーション中間ファイル拡張子
        /// </summary>
        public const string SceneAnimBinaryExtension =
            G3dPath.SceneAnimExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// シェーダー設定
        /// </summary>
        public const string ShaderConfig = "sc";

        /// <summary>
        /// シェーダー設定中間ファイル拡張子
        /// </summary>
        public const string ShaderConfigExtension =
            "." + G3dPath.Platform + G3dPath.ShaderConfig;

        /// <summary>
        /// テキストシェーダー設定中間ファイル拡張子
        /// </summary>
        public const string ShaderConfigTextExtension =
            G3dPath.ShaderConfigExtension + G3dPath.Text;

        /// <summary>
        /// バイナリシェーダー設定中間ファイル拡張子
        /// </summary>
        public const string ShaderConfigBinaryExtension =
            G3dPath.ShaderConfigExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// シェーダー定義
        /// </summary>
        public const string ShaderDefinition = "sd";

        /// <summary>
        /// シェーダー定義中間ファイル拡張子
        /// </summary>
        public const string ShaderDefinitionExtension =
            "." + G3dPath.Platform + G3dPath.ShaderDefinition;

        /// <summary>
        /// テキストシェーダー定義中間ファイル拡張子
        /// </summary>
        public const string ShaderDefinitionTextExtension =
            G3dPath.ShaderDefinitionExtension + G3dPath.Text;

        /// <summary>
        /// バイナリシェーダー定義中間ファイル拡張子
        /// </summary>
        public const string ShaderDefinitionBinaryExtension =
            G3dPath.ShaderDefinitionExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// シェーダーバリエーション
        /// </summary>
        public const string ShaderVariation = "sv";

        /// <summary>
        /// シェーダーバリエーション中間ファイル拡張子
        /// </summary>
        public const string ShaderVariationExtension =
            "." + G3dPath.Platform + G3dPath.ShaderVariation;

        /// <summary>
        /// テキストシェーダーバリエーション中間ファイル拡張子
        /// </summary>
        public const string ShaderVariationTextExtension =
            G3dPath.ShaderVariationExtension + G3dPath.Text;

        /// <summary>
        /// バイナリシェーダーバリエーション中間ファイル拡張子
        /// </summary>
        public const string ShaderVariationBinaryExtension =
            G3dPath.ShaderVariationExtension + G3dPath.Binary;

        //---------------------------------------------------------------------
        /// <summary>
        /// コンバイナーシェーダー
        /// </summary>
        public const string CombinerShader = "cm";

        /// <summary>
        /// シェーダーバリエーション中間ファイル拡張子
        /// </summary>
        public const string CombinerShaderExtension =
            "." + G3dPath.Platform + G3dPath.CombinerShader;

        /// <summary>
        /// テキストコンバイナーシェーダー中間ファイル拡張子
        /// </summary>
        public const string CombinerShaderTextExtension =
            G3dPath.CombinerShaderExtension + G3dPath.Text;

        /// <summary>
        /// バイナリコンバイナーシェーダー中間ファイル拡張子
        /// </summary>
        public const string CombinerShaderBinaryExtension =
            G3dPath.CombinerShaderExtension + G3dPath.Binary;

        //=====================================================================
        /// <summary>
        /// テキスト拡張子テーブル
        /// </summary>
        public static readonly string[] TextExtensions = new string[]
        {
            G3dPath.ModelTextExtension,
            G3dPath.TextureTextExtension,
            G3dPath.SkeletalAnimTextExtension,
            G3dPath.MaterialAnimTextExtension,
            G3dPath.ShaderParamAnimTextExtension,
            G3dPath.ColorAnimTextExtension,
            G3dPath.TexSrtAnimTextExtension,
            G3dPath.TexPatternAnimTextExtension,
            G3dPath.BoneVisibilityAnimTextExtension,
            G3dPath.MatVisibilityAnimTextExtension,
            G3dPath.ShapeAnimTextExtension,
            G3dPath.SceneAnimTextExtension,
            G3dPath.ShaderConfigTextExtension,
            G3dPath.ShaderDefinitionTextExtension,
            G3dPath.ShaderVariationTextExtension,
        };

        /// <summary>
        /// バイナリ拡張子テーブル
        /// </summary>
        public static readonly string[] BinaryExtensions = new string[]
        {
            G3dPath.ModelBinaryExtension,
            G3dPath.TextureBinaryExtension,
            G3dPath.SkeletalAnimBinaryExtension,
            G3dPath.MaterialAnimBinaryExtension,
            G3dPath.ShaderParamAnimBinaryExtension,
            G3dPath.ColorAnimBinaryExtension,
            G3dPath.TexSrtAnimBinaryExtension,
            G3dPath.TexPatternAnimBinaryExtension,
            G3dPath.BoneVisibilityAnimBinaryExtension,
            G3dPath.MatVisibilityAnimBinaryExtension,
            G3dPath.ShapeAnimBinaryExtension,
            G3dPath.SceneAnimBinaryExtension,
            G3dPath.ShaderConfigBinaryExtension,
            G3dPath.ShaderDefinitionBinaryExtension,
            G3dPath.ShaderVariationBinaryExtension,
        };

        //---------------------------------------------------------------------
        /// <summary>
        /// 要素名からテキスト拡張子を取得します。
        /// fcla、ftsa には非対応です。
        /// </summary>
        /// <param name="elementName">要素名</param>
        /// <returns>要素名に対応する拡張子</returns>
        /// <detail></detail>
        public static string GetTextExtensionByElement(string elementName)
        {
            int index = Array.IndexOf<string>(G3dConstant.ElementNames, elementName);
            Nintendo.Foundation.Contracts.Assertion.Operation.True(index != -1);
            return G3dPath.TextExtensions[index];
        }

        /// <summary>
        /// 要素からバイナリ拡張子を取得します。
        /// fclb、ftsb には非対応です。
        /// </summary>
        /// <param name="elementName">要素名</param>
        /// <returns>要素名に対応する拡張子</returns>
        public static string GetBinaryExtensionByElement(string elementName)
        {
            int index = Array.IndexOf<string>(G3dConstant.ElementNames, elementName);
            Nintendo.Foundation.Contracts.Assertion.Operation.True(index != -1);
            return G3dPath.BinaryExtensions[index];
        }

        //=====================================================================
        /// <summary>
        /// 中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>中間ファイルのパスであれば true</returns>
        public static bool IsPath(string filePath)
        {
            string extension = Path.GetExtension(filePath).ToLower();
            if (extension.Length != 5) { return false; }
            if (extension[0] != '.') { return false; }
            if (extension[1] != G3dPath.Platform[0]) { return false; }
            if ((extension[4] != G3dPath.Binary[0]) &&
                (extension[4] != G3dPath.Text[0])) { return false; }
            string type = extension.Substring(2, 2);
            switch (type)
            {
                case G3dPath.Model:
                case G3dPath.Material:
                case G3dPath.Texture:
                case G3dPath.SkeletalAnim:
                case G3dPath.MaterialAnim:
                case G3dPath.ShaderParamAnim:
                case G3dPath.ColorAnim:
                case G3dPath.TexSrtAnim:
                case G3dPath.TexPatternAnim:
                case G3dPath.BoneVisibilityAnim:
                case G3dPath.MatVisibilityAnim:
                case G3dPath.ShapeAnim:
                case G3dPath.SceneAnim:
                case G3dPath.ShaderConfig:
                case G3dPath.ShaderDefinition:
                case G3dPath.ShaderVariation:
                case G3dPath.CombinerShader:
                    return true;
                default:
                    break;
            }
            return false;
        }

        /// <summary>
        /// テキスト中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>テキスト中間ファイルのパスであれば true</returns>
        public static bool IsTextPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.EndsWith(G3dPath.Text);
        }

        /// <summary>
        /// バイナリ中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>バイナリ中間ファイルのパスであれば true</returns>
        public static bool IsBinaryPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.EndsWith(G3dPath.Binary);
        }

        /// <summary>
        /// ストリームを持つ中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>ストリームを持つ中間ファイルのパスであれば true</returns>
        public static bool IsStreamBinaryPath(string filePath)
        {
            if (!G3dPath.IsBinaryPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            switch (extension)
            {
                case G3dPath.ModelBinaryExtension:
                case G3dPath.MaterialBinaryExtension:
                case G3dPath.TextureBinaryExtension:
                case G3dPath.SkeletalAnimBinaryExtension:
                case G3dPath.MaterialAnimBinaryExtension:
                case G3dPath.ShaderParamAnimBinaryExtension:
                case G3dPath.ColorAnimBinaryExtension:
                case G3dPath.TexSrtAnimBinaryExtension:
                case G3dPath.TexPatternAnimBinaryExtension:
                case G3dPath.BoneVisibilityAnimBinaryExtension:
                case G3dPath.MatVisibilityAnimBinaryExtension:
                case G3dPath.ShapeAnimBinaryExtension:
                case G3dPath.SceneAnimBinaryExtension:
                case G3dPath.ShaderConfigBinaryExtension:
                case G3dPath.ShaderDefinitionBinaryExtension:
                case G3dPath.ShaderVariationBinaryExtension:
                case G3dPath.CombinerShaderBinaryExtension:
                    return true;
            }
            return false;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// モデル中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>モデル中間ファイルのパスであれば true</returns>
        public static bool IsModelPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.ModelExtension);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// マテリアル中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>マテリアル中間ファイルのパスであれば true</returns>
        public static bool IsMaterialPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.MaterialExtension);
        }

        /// <summary>
        /// テクスチャ中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>テクスチャ中間ファイルのパスであれば true</returns>
        public static bool IsTexturePath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.TextureExtension);
        }

        /// <summary>
        /// スケルタルアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>スケルタルアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsSkeletalAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.SkeletalAnimExtension);
        }

        /// <summary>
        /// マテリアルアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>マテリアルアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsMaterialAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.MaterialAnimExtension);
        }

        /// <summary>
        /// マテリアルアニメーションに変換できる中間ファイルパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>マテリアルアニメーションに変換できる中間ファイルのパスであれば true</returns>
        public static bool IsMaterialAnimConvertiblePath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return IsShaderParamAnimGroupPath(filePath)
                    || IsMatVisibilityAnimPath(filePath)
                    || IsTexPatternAnimPath(filePath);
        }

        /// <summary>
        /// シェーダーパラメーターアニメーション中間ファイルグループのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>シェーダーパラメーターアニメーション中間ファイルグループのパスであれば true</returns>
        public static bool IsShaderParamAnimGroupPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return
                extension.StartsWith(G3dPath.ShaderParamAnimExtension) ||
                extension.StartsWith(G3dPath.ColorAnimExtension) ||
                extension.StartsWith(G3dPath.TexSrtAnimExtension);
        }

        /// <summary>
        /// シェーダーパラメーターアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>シェーダーパラメーターアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsShaderParamAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.ShaderParamAnimExtension);
        }

        /// <summary>
        /// カラーアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>カラーアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsColorAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.ColorAnimExtension);
        }

        /// <summary>
        /// テクスチャ SRT アニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>テクスチャ SRT アニメーション中間ファイルのパスであれば true</returns>
        public static bool IsTexSrtAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.TexSrtAnimExtension);
        }

        /// <summary>
        /// テクスチャパターンアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>テクスチャパターンアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsTexPatternAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.TexPatternAnimExtension);
        }

        /// <summary>
        /// ボーンビジビリティアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>ボーンビジビリティアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsBoneVisibilityAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.BoneVisibilityAnimExtension);
        }

        /// <summary>
        /// マテリアルビジビリティアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>マテリアルビジビリティアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsMatVisibilityAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.MatVisibilityAnimExtension);
        }

        /// <summary>
        /// シェイプアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>シェイプアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsShapeAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.ShapeAnimExtension);
        }

        /// <summary>
        /// シーンアニメーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>シーンアニメーション中間ファイルのパスであれば true</returns>
        public static bool IsSceneAnimPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.SceneAnimExtension);
        }

        /// <summary>
        /// シェーダー設定中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>シェーダー設定中間ファイルのパスであれば true</returns>
        public static bool IsShaderConfigPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.ShaderConfigExtension);
        }

        /// <summary>
        /// シェーダー定義中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>シェーダー定義中間ファイルのパスであれば true</returns>
        public static bool IsShaderDefinitionPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.ShaderDefinitionExtension);
        }

        /// <summary>
        /// シェーダーバリエーション中間ファイルのパスかどうかを確認します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>シェーダーバリエーション中間ファイルのパスであれば true</returns>
        public static bool IsShaderVariationPath(string filePath)
        {
            if (!G3dPath.IsPath(filePath)) { return false; }
            string extension = Path.GetExtension(filePath).ToLower();
            return extension.StartsWith(G3dPath.ShaderVariationExtension);
        }

        //=====================================================================
        /// <summary>
        /// テキスト中間ファイルパスをバイナリ中間ファイルパスに変換します。
        /// </summary>
        /// <param name="textPath">テキスト中間ファイルパス</param>
        /// <returns>バイナリ中間ファイルパス</returns>
        public static string ToBinaryPath(string textPath)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(IsTextPath(textPath));
            string extension = Path.GetExtension(textPath).ToLower();
            extension = extension.Remove(extension.Length - 1) + G3dPath.Binary;
            return Path.ChangeExtension(textPath, extension);
        }

        /// <summary>
        /// バイナリ中間ファイルパスをテキスト中間ファイルパスに変換します。
        /// </summary>
        /// <param name="binaryPath">バイナリ中間ファイルパス</param>
        /// <returns>テキスト中間ファイルパス</returns>
        public static string ToTextPath(string binaryPath)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(IsBinaryPath(binaryPath));
            string extension = Path.GetExtension(binaryPath).ToLower();
            extension = extension.Remove(extension.Length - 1) + G3dPath.Text;
            return Path.ChangeExtension(binaryPath, extension);
        }

        //=====================================================================
    }
}
