﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//#define NW4F_3DIF_USE_PARALLEL

using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using System.Threading;
using System.Diagnostics;

namespace nw.g3d.nw4f_3dif
{
    /// <summary>
    /// 並列処理
    /// </summary>
    public static class G3dParallel
    {
        /// <summary>
        /// 実行タスクに割り当てるコンテキスト
        /// </summary>
        public abstract class TaskContext : IDisposable
        {
            /// <summary>
            /// 親コンテキスト
            /// </summary>
            public readonly TaskContext Parent;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public TaskContext()
                : this(null)
            { }

            /// <summary>
            /// コンストラクタ
            /// 親コンテキストを指定して子コンテキストを作成する。
            /// </summary>
            /// <param name="parent">親コンテキスト</param>
            protected TaskContext(TaskContext parent)
            {
                Parent = parent;
            }

            /// <summary>
            /// リソース解放
            /// </summary>
            public abstract void Dispose();

            /// <summary>
            /// 新規タスク用の子コンテキストを作成する
            /// </summary>
            /// <returns></returns>
            public abstract TaskContext CreateChildContext();
        }

        /// <summary>
        /// タスクコンテキストを G3dParallel に登録/解除するためのクラス
        /// アプリケーション開始時などの G3dParallel 実行前に、ルートスレッドのコンテキストを登録することで
        /// それ以降の G3dParallel 実行時にタスク毎にコンテキストが割り当てられる。
        /// </summary>
        public sealed class TaskContextStack : IDisposable
        {
            private readonly Thread thread_;
            private readonly Stack<TaskContext> stack_;

            /// <summary>
            /// スレッドの実行タスクに割り当てられたコンテキスト
            /// System.Threading.Tasks.Parallel での並列化粒度が不明なのでスタック化してコンテキストを保持する (ひとつのワーカースレッドで複数タスクが実行されることを想定)
            /// </summary>
            private static readonly System.Collections.Concurrent.ConcurrentDictionary<Thread, Stack<TaskContext>> stacks_ = new System.Collections.Concurrent.ConcurrentDictionary<Thread, Stack<TaskContext>>();

            /// <summary>
            /// コンストラクタ
            /// 現在のスレッドにコンテキストを割り当てる。
            /// </summary>
            /// <param name="ctx">割り当てるコンテキスト</param>
            public TaskContextStack(TaskContext ctx)
            {
                if (ctx != null)
                {
                    thread_ = Thread.CurrentThread;

                    // 辞書からスタックを取得もしくは、辞書に追加する。
                    var stack = stacks_.GetOrAdd(thread_, (key) => { return new Stack<TaskContext>(); });

                    // スタックはスレッド専用なので排他制御不要。
                    stack.Push(ctx);
                    stack_ = stack;
                }
                else
                {
                    thread_ = null;
                    stack_ = null;
                }
            }

            /// <summary>
            /// リソース解放
            /// コンストラクタで割り当てたコンテキストを解除する
            /// コンテキストの解放は行わない。
            /// ※コンストラクタ呼び出しと同じスレッドから呼び出すこと。別スレッドから呼び出した場合の動作は不定。
            /// </summary>
            public void Dispose()
            {
                if (stack_ != null)
                {
                    // コンストラクタ呼び出しと別スレッドからの呼び出しは禁止。
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(thread_ == Thread.CurrentThread);

                    // スタックはスレッド専用なので排他制御不要。
                    stack_.Pop();
                    if (stack_.Count == 0)
                    {
                        // 辞書からスタックを取り除く。
                        Stack<TaskContext> stack;
                        stacks_.TryRemove(thread_, out stack);
                    }
                }
            }

            /// <summary>
            /// カレントスレッドで実行中のタスクに割り当てられているコンテキストを取得する。
            /// 割り当てられていない場合は null を返す。
            /// </summary>
            public static TaskContext Current
            {
                get
                {
                    Stack<TaskContext> stack;
                    return (stacks_.TryGetValue(Thread.CurrentThread, out stack) && (stack.Count > 0)) ? stack.Peek() : null;
                }
            }
        }

        /// <summary>
        /// 並列処理数
        /// 並列処理中の変更には対応していません。
        /// 0 以下はプロセッサ数
        /// </summary>
        public static int Job { get; set; } = 0;

        /// <summary>
        /// for 文による並列処理
        /// </summary>
        /// <param name="fromInclusive">値を含む開始値</param>
        /// <param name="toExclusive">値を含まない終了値</param>
        /// <param name="body">処理メソッド</param>
        public static void For(int fromInclusive, int toExclusive, Action<int> body)
        {
            var currCtx = TaskContextStack.Current;
            if (G3dParallel.Job != 1)
            {
#if NW4F_3DIF_USE_PARALLEL
                Parallel.For(
                    fromInclusive,
                    toExclusive,
                    (int i) =>
                    {
                        using (var childCtx = (currCtx != null) ? currCtx.CreateChildContext() : null)
                        using (var stack = new TaskContextStack(childCtx))
                        {
                            body(i);
                        }
                    });
#else
                using (ForChannel forChannel = new ForChannel(currCtx))
                {
                    forChannel.For(fromInclusive, toExclusive, body);
                }
#endif
            }
            else
            {
                for (int i = fromInclusive; i < toExclusive; i++)
                {
                    using (var childCtx = (currCtx != null) ? currCtx.CreateChildContext() : null)
                    using (var stack = new TaskContextStack(childCtx))
                    {
                        body(i);
                    }
                }
            }
        }

        /// <summary>
        /// foreach 文による並列処理
        /// </summary>
        /// <typeparam name="TSource">列挙する型パラメーター</typeparam>
        /// <param name="source">列挙値</param>
        /// <param name="body">処理メソッド</param>
        public static void ForEach<TSource>(IEnumerable<TSource> source, Action<TSource> body)
        {
            var currCtx = TaskContextStack.Current;
            if (G3dParallel.Job != 1)
            {
#if NW4F_3DIF_USE_PARALLEL
                Parallel.ForEach<TSource>(
                    source,
                    (TSource target) =>
                    {
                        using (var childCtx = (currCtx != null) ? currCtx.CreateChildContext() : null)
                        using (var stack = new TaskContextStack(childCtx))
                        {
                            body(target);
                        }
                    });
#else
                using (ForEachChannel<TSource> forEachChannel = new ForEachChannel<TSource>(currCtx))
                {
                    forEachChannel.ForEach(source, body);
                }
#endif
            }
            else
            {
                foreach (TSource target in source)
                {
                    using (var childCtx = (currCtx != null) ? currCtx.CreateChildContext() : null)
                    using (var stack = new TaskContextStack(childCtx))
                    {
                        body(target);
                    }
                }
            }
        }

#if !NW4F_3DIF_USE_PARALLEL
        public static void Cleanup(bool wait = false)
        {
            while (WorkerPool.Count > 0)
            {
                var worker = WorkerPool.Pop();
                if (worker != null)
                {
                    worker.Stop();
                }
            }

            if (wait)
            {
                while (WorkerPool.Count > 0)
                {
                    if (Thread.Yield() == false)
                    {
                        Thread.Sleep(1);
                    }
                }
            }
        }
#endif

#if !NW4F_3DIF_USE_PARALLEL
        //---------------------------------------------------------------------
        // For チャンネル
        private class ForChannel : Channel
        {
            public ForChannel(TaskContext ctx)
                : base(ctx)
            { }

            // For の実行
            public void For(int fromInclusive, int toExclusive, Action<int> body)
            {
                this.Count = fromInclusive;
                this.ToExclusive = toExclusive;
                this.Body = body;
                Run();
            }

            // リクエストの取得
            public override Action GetRequest()
            {
                // Worker で Channel を lock している
                if (this.Count == this.ToExclusive) { return null; }
                int index = this.Count;
                Action result = delegate { this.Body(index); };
                this.Count++;
                return result;
            }

            private volatile int Count;
            private volatile int ToExclusive;
            private volatile Action<int> Body;
        }

        // ForEach チャンネル
        private class ForEachChannel<TSource> : Channel
        {
            public ForEachChannel(TaskContext ctx)
                : base(ctx)
            { }

            // ForEach の実行
            public void ForEach(IEnumerable<TSource> source, Action<TSource> body)
            {
                this.Enumerator = source.GetEnumerator();
                this.Body = body;
                Run();
            }

            // リクエストの取得
            public override Action GetRequest()
            {
                // Worker で Channel を lock している
                if (this.Enumerator.MoveNext())
                {
                    TSource source = this.Enumerator.Current;
                    return delegate { this.Body(source); };
                }
                return null;
            }

            private volatile IEnumerator<TSource> Enumerator;
            private volatile Action<TSource> Body;
        }

        // チャンネル
        private abstract class Channel : IDisposable
        {
            // コンストラクタ
            public Channel(TaskContext ctx)
            {
                TaskContext = ctx;
                int count = Job > 1 ? Job : System.Environment.ProcessorCount;
                this.Workers = new Worker[count];
                // ワーカーをプールから借りる
                lock (G3dParallel.WorkerPool)
                {
                    for (int i = 0; i < count; i++) { this.Workers[i] = PopWorker(); }
                }
            }

            // 破棄
            public void Dispose()
            {
                // ワーカーをプールに帰す
                lock (G3dParallel.WorkerPool)
                {
                    for (int i = 0; i < this.Workers.Length; i++)
                    {
                        PushWorker(this.Workers[i]);
                        this.Workers[i] = null;
                    }
                }
            }

            // 実行
            protected void Run()
            {
                // 全ワーカーを起動
                foreach (Worker worker in this.Workers) { worker.Resume(this); }

                // 全ワーカーの終了待ち
                while (true)
                {
                    bool finished = true;
                    foreach (Worker worker in this.Workers)
                    {
                        if (!worker.IsSuspended()) { finished = false; }
                    }
                    if (finished) { break; }
                    Thread.Sleep(1);
                }

                // 例外があれば AggregateException を投げる
                if (Exceptions != null)
                {
                    throw new System.AggregateException(Exceptions);
                }
            }

            // リクエストの取得
            public abstract Action GetRequest();

            // タスクコンテキストの取得
            public readonly TaskContext TaskContext;

            protected readonly Worker[] Workers;

            protected List<Exception> Exceptions { get; set; }

            // 例外の追加
            public void AddException(Exception exception)
            {
                // Worker で Channel を lock している

                if (Exceptions == null)
                {
                    Exceptions = new List<Exception>();
                }

                Exceptions.Add(exception);
            }
        }

        //---------------------------------------------------------------------
        // ワーカー
        private class Worker
        {
            // コンストラクタ
            public Worker()
            {
                this.Thread = new Thread(this.Run);
                this.Thread.CurrentCulture = Thread.CurrentThread.CurrentCulture;
                this.Thread.CurrentUICulture = Thread.CurrentThread.CurrentUICulture;
                this.Thread.Name = string.Format("G3dParallel[{0}] Thread", GetType().Name);
                this.Thread.IsBackground = true;
                this.Thread.Priority = ThreadPriority.Lowest;
                this.Thread.Start();
            }

            // 再開
            public void Resume(Channel channel)
            {
                this.Channel = channel;
                lock (this) { Monitor.Pulse(this); }
            }

            private bool stopRequested;
            // 止まる
            public void Stop()
            {
                lock (this)
                {
                    stopRequested = true;
                    Monitor.Pulse(this);
                }
            }

            // 実行
            private void Run()
            {
                while (true)
                {
                    if (RunMain())
                    {
                        break;
                    }
                }
            }

            private bool RunMain()
            {
                lock (this)
                {
                    Monitor.Wait(this);
                    if (stopRequested)
                    {
                        return true;
                    }
                }

                Channel channel = this.Channel;
                TaskContext parentCtx;
                Action run;
                while (true)
                {
                    lock (channel)
                    {
                        run = channel.GetRequest();
                        parentCtx = channel.TaskContext;
                    }

                    if (run == null)
                    {
                        break;
                    }

                    using (var childCtx = (parentCtx != null) ? parentCtx.CreateChildContext() : null)
                    using (var stack = new TaskContextStack(childCtx))
                    {
#if !DEBUG
                        try
                        {
                            run();
                        }
                        catch (Exception exception)
                        {
                            lock (channel)
                            {
                                channel.AddException(exception);
                            }
                        }
#else
                        // デバッグ版ではエラーハンドリングしない
                        run();
#endif
                    }
                }
                this.Channel = null;
                channel = null;
                return false;
            }

            // 一時停止状態かどうか
            public bool IsSuspended()
            {
                System.Threading.ThreadState state = this.Thread.ThreadState;
                return
                    (this.Channel == null) &&
                    ((state & System.Threading.ThreadState.WaitSleepJoin) != 0);
            }

            private volatile Channel Channel;
            private readonly Thread Thread;
        }

        //---------------------------------------------------------------------
        // ワーカープールからポップ
        private static Worker PopWorker()
        {
            // G3dParallel.WorkerPool は Channel で lock する
            if (G3dParallel.WorkerPool.Count > 0)
            {
                Worker worker = G3dParallel.WorkerPool.Pop();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(worker.IsSuspended());
                return worker;
            }
            else
            {
                G3dParallel.WorkerCount++;
                Worker worker = new Worker();
                // 生成したワーカーが待ち状態になるまで待つ
                while (true)
                {
                    if (worker.IsSuspended()) { break; }
                    Thread.Sleep(1);
                }
                return worker;
            }
        }

        // ワーカープールへプッシュ
        private static void PushWorker(Worker worker)
        {
            // G3dParallel.WorkerPool は Channel で lock する
            Nintendo.Foundation.Contracts.Assertion.Operation.True(worker.IsSuspended());
            G3dParallel.WorkerPool.Push(worker);
            //Debug.WriteLine("{0} / {1}", G3dParallel.WorkerPool.Count, WorkerCount);
        }

        private static Stack<Worker> WorkerPool = new Stack<Worker>();
        private static volatile int WorkerCount = 0;
#endif
    }
}
