﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Globalization;

namespace nw.g3d.nw4f_3dif
{
    // 中間ファイルデータ列アクセスユーティリティ
    public static class G3dDataParser
    {
        // デリミタ
        public const string Delimiter = " \t\r\n";

        // トークン化
        public static string[] Tokenize(string source)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(source != null);
            return source.Split(Delimiter.ToCharArray(),
                StringSplitOptions.RemoveEmptyEntries);
        }

        //=====================================================================
        // bool 配列のパース
        public static bool[] ParseBoolArray(string source)
        {
            if (source == null) { return null; }
            string[] valueStrs = Tokenize(source);
            if (valueStrs.Length == 0) { return null; }

            bool[] result = new bool[valueStrs.Length];
            for (int i = 0; i < valueStrs.Length; i++)
            {
                result[i] = bool.Parse(valueStrs[i]);
            }
            return result;
        }

        // int 型 bool 配列のパース
        public static bool[] ParseInt2BoolArray(string source)
        {
            if (source == null) { return null; }
            string[] valueStrs = Tokenize(source);
            if (valueStrs.Length == 0) { return null; }

            bool[] result = new bool[valueStrs.Length];
            for (int i = 0; i < valueStrs.Length; i++)
            {
                int intValue = int.Parse(valueStrs[i]);
                result[i] = (intValue != 0);
            }
            return result;
        }

        // byte 配列のパース
        public static byte[] ParseByteArray(string source)
        {
            if (source == null) { return null; }
            string[] valueStrs = Tokenize(source);
            if (valueStrs.Length == 0) { return null; }

            byte[] result = new byte[valueStrs.Length];
            for (int i = 0; i < valueStrs.Length; i++)
            {
                result[i] = byte.Parse(valueStrs[i], NumberStyles.HexNumber);
            }
            return result;
        }

        // int 配列のパース
        public static int[] ParseIntArray(string source)
        {
            if (source == null) { return null; }
            string[] valueStrs = Tokenize(source);
            if (valueStrs.Length == 0) { return null; }

            int[] result = new int[valueStrs.Length];
            for (int i = 0; i < valueStrs.Length; i++)
            {
                result[i] = int.Parse(valueStrs[i]);
            }
            return result;
        }

        // uint 配列のパース
        public static uint[] ParseUIntArray(string source)
        {
            if (source == null) { return null; }
            string[] valueStrs = Tokenize(source);
            if (valueStrs.Length == 0) { return null; }

            uint[] result = new uint[valueStrs.Length];
            for (int i = 0; i < valueStrs.Length; i++)
            {
                result[i] = uint.Parse(valueStrs[i]);
            }
            return result;
        }

        // float 配列のパース
        public static float[] ParseFloatArray(string source)
        {
            if (source == null) { return null; }
            string[] valueStrs = Tokenize(source);
            if (valueStrs.Length == 0) { return null; }

            float[] result = new float[valueStrs.Length];
            for (int i = 0; i < valueStrs.Length; i++)
            {
                result[i] = float.Parse(valueStrs[i]);
            }
            return result;
        }
    }
}
