﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Data.SqlTypes;
using System.Diagnostics;
using System.Linq;
using System.Runtime.Serialization;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    using WERsList = ResizableList<WingedEdge>;
    using WEKey = Tuple<SourceType, string>;

    /// <summary>
    /// WingedEdge データ構造
    /// </summary>
    public partial class WingedEdge : ICloneable, IComparable
    {
#if DEBUG
        public WingedEdge Test { get; set; } = null;
#endif
        private bool mBorder;
        private bool mUnMovable;
        private int mFace0;
        private int mFace1;
        private int mVertex0;
        private int mVertex1;
        private int mId;
        private SourceKey mSpaceKey = new SourceKey(SourceType.Position, null);
        private Dictionary<SourceKey, WERsList> mChildEdges = new Dictionary<SourceKey, WERsList>();
        private WERsList mParentEdges = new WERsList();

        // ICloneable.Cloneの実装
        public object Clone()
        {
            return MemberwiseClone();
        }

        internal WingedEdge()
        {
            mUnMovable = false;
            mBorder = false;
            mFace0 = -1;
            mFace1 = -1;

            mVertex0 = -1;
            mVertex1 = -1;
            mId = -1;
        }

        internal WingedEdge(WingedEdge right)
        {
            mUnMovable = right.mUnMovable;
            mBorder = right.mBorder;
            mFace0 = right.mFace0;
            mFace1 = right.mFace1;
            mVertex0 = right.mVertex0;
            mVertex1 = right.mVertex1;
            mId = right.mId;
        }

        /// <summary>
        /// 参照エッジを取得
        /// </summary>
        public WERsList GetChildEdges(SourceType type, string name)
        {
            SourceKey key = new SourceKey(type, name);
            if (mChildEdges.ContainsKey(key))
            {
                return mChildEdges[key];
            }
            WERsList edges = new WERsList();
            mChildEdges.Add(key, edges);
            return edges;
        }

        /// <summary>
        ///
        /// </summary>
        public Dictionary<SourceKey, WERsList> ChildEdges
        {
            get { return mChildEdges; }
            set { mChildEdges = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public WERsList ParentEdges
        {
            get { return mParentEdges; }
            set { mParentEdges = value; }
        }

        /// <summary>
        /// エッジが所属する空間タイプを取得変更します
        /// </summary>
        public SourceKey SpaceKey
        {
            get { return mSpaceKey; }
            set { mSpaceKey = value; }
        }

        /// <summary>
        /// 動かしてもいいエッジであるかどうか？
        /// </summary>
        public bool UnMovable
        {
            get { return mUnMovable; }
            set { mUnMovable = value; }
        }

        /// <summary>
        /// 境界線かどうか？を取得・設定
        /// </summary>
        public bool Boder
        {
            get { return mBorder; }
            set { mBorder = value; }
        }

        /// <summary>
        /// 開放端エッジかどうか？を取得
        /// </summary>
        public bool IsOpen
        {
            get { return (mFace0 == int.MaxValue || mFace1 == int.MaxValue); }
        }

        /// <summary>
        /// エッジの両側に存在するフェースを設定・取得
        /// </summary>
        public int Face0
        {
            get { return mFace0; }
            set { mFace0 = value; }
        }

        /// <summary>
        /// エッジの両側に存在するフェースを設定・取得
        /// </summary>
        public int Face1
        {
            get { return mFace1; }
            set { mFace1 = value; }
        }

        /// <summary>
        /// エッジの両端の頂点
        /// </summary>
        public int ID
        {
            get { return mId; }
            set { mId = value; }
        }

        /// <summary>
        /// エッジの両端の頂点
        /// </summary>
        public int Vertex0
        {
            get { return mVertex0; }
            set { mVertex0 = value; }
        }
        /// <summary>
        /// エッジの両端の頂点
        /// </summary>
        public int Vertex1
        {
            get { return mVertex1; }
            set { mVertex1 = value; }
        }

        /// <summary>
        /// 親エッジから自分自身を取り除く
        /// </summary>
        public void RemoveFromParent()
        {
            int parentN = mParentEdges.Count;
            for (int pi = 0; pi < parentN; pi++)
            {
                if (mParentEdges[pi].mChildEdges[mSpaceKey].Contains(this))
                {
                    mParentEdges[pi].mChildEdges[mSpaceKey].Remove(this);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="edge"></param>
        public void AddChildEdgesFrom(WingedEdge edge)
        {
            if (edge.mChildEdges.Count == 0) { return; }

            // 親エッジの参照を除去する
            // 新たな親エッジとの参照を追加する
            foreach (var container in edge.mChildEdges)
            {
                WERsList cedges = container.Value;
                int childN = cedges.Count;
                for (int i = 0; i < childN; i++)
                {
                    cedges[i].ParentEdges.Remove(edge);
                    if (!cedges[i].ParentEdges.Contains(this))
                    {
                        cedges[i].ParentEdges.Add(this);
                    }
                }
            }

            foreach (var container in edge.mChildEdges)
            {
                WERsList cedges = null;
                if (ChildEdges.TryGetValue(container.Key, out cedges))
                {
                    int childN = container.Value.Count;
                    cedges.AddRange(container.Value);
                }
            }
        }

        public bool IsVerticesEqual(WingedEdge right)
        {
            if (right == null)
            {
                return false;
            }
            return Vertex0 == right.Vertex0 && Vertex1 == right.Vertex1;
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public int CompareTo(object obj)
        {
            var c = obj as WingedEdge;
            if (c == null)
            {
                return -1;
            }
            if (ReferenceEquals(this, obj))
            {
                return 0;
            }

            if (c.Vertex0 == Vertex0)
            {
                if (c.Vertex1 == Vertex1)
                {
                    if (c.Face0 == Face0)
                    {
                        if (c.Face1 == Face1)
                        {
                            return string.Compare(SpaceKey.name, c.SpaceKey.name, StringComparison.Ordinal);
                        }
                        return Face1.CompareTo(c.Face1);
                    }
                    return Face0.CompareTo(c.Face0);
                }
                return Vertex1.CompareTo(c.Vertex1);
            }
            return Vertex0.CompareTo(c.Vertex0);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public void Initialize()
        {
            mUnMovable = false;
            mBorder = false;
            mFace0 = int.MaxValue;
            mFace1 = int.MaxValue;
            mVertex0 = int.MaxValue;
            mVertex1 = int.MaxValue;
        }
    }

    /// <summary>
    /// Brep構造
    /// 頂点/エッジ リストとしても機能する
    /// </summary>
    public partial class WingedEdgeMesh
    {
        private Mesh mMesh = null;
        private SourceKey mRepresentativeKey = new SourceKey(SourceType.Position, null);   //縮退時の代表エッジ

        private Dictionary<SourceKey, WERsList> mWingedEdges = new Dictionary<SourceKey, WERsList>(new SourceKeyComparer());
        private Dictionary<SourceKey, WERsList> mNonmaniEdges = new Dictionary<SourceKey, WERsList>(new SourceKeyComparer());

        public IEnumerable<SourceKey> SourceKeys
        {
            get { return mWingedEdges.Keys; }
        }

        /// <summary>
        /// 指定キーのエッジ数を取得する
        /// </summary>
        /// <param name="type"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public int GetWingedEdgeN(SourceType type = SourceType.Position, string name = null)
        {
            var key = new SourceKey(type, name);
            if (mWingedEdges.ContainsKey(key))
            {
                return mWingedEdges[key].Count;
            }
            return 0;
        }

        /// <summary>
        /// 指定キーのエッジリストを取得する
        /// </summary>
        /// <param name="type"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public WERsList GetWingedEdges(SourceType type = SourceType.Position, string name = null)
        {
            var key = new SourceKey(type, name);
            if (mWingedEdges.ContainsKey(key))
            {
                return mWingedEdges[key];
            }
            return null;
        }

        /// <summary>
        /// 指定キーの非多様体エッジリストを取得する
        /// </summary>
        /// <param name="type"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public WERsList GetNonmanifoldEdges(SourceType type = SourceType.Position, string name = null)
        {
            var key = new SourceKey(type, name);
            if (mNonmaniEdges.ContainsKey(key))
            {
                return mNonmaniEdges[key];
            }
            return null;
        }

        /// <summary>
        /// 削除対象になる代表エッジリスト
        /// </summary>
        public WERsList RepresentativeEdges
        {
            get { return GetWingedEdges(mRepresentativeKey.type, mRepresentativeKey.name); }
        }

        /// <summary>
        /// 代表エッジリストのエッジ数
        /// </summary>
        public int RepresentativeEdgeN
        {
            get { return GetWingedEdgeN(mRepresentativeKey.type, mRepresentativeKey.name); }
        }

        /// <summary>
        /// 削除対象になる代表エッジリスト内の非多様体エッジ
        /// </summary>
        public WERsList RepresentativeNmfdEdges
        {
            get { return GetNonmanifoldEdges(mRepresentativeKey.type, mRepresentativeKey.name); }
        }

        /// <summary>
        /// 終了
        /// </summary>
        public void Destroy()
        {
            mWingedEdges.Clear();
            mNonmaniEdges.Clear();
            mMesh = null;
        }

        /// <summary>
        /// 最大の必要なメモリサイズを取得
        /// </summary>
        /// <param name="mesh"></param>
        /// <returns></returns>
        public static int GetRequiredMemSize(Mesh mesh)
        {
            if (mesh == null)
            {
                return 0;
            }
            try
            {
                int faceN = mesh.FaceN;
                return faceN * 3 * sizeof(int) * 6;
            }
            catch (Exception ex)
            {
                ExcepHandle.CreateException(ex);
                throw;
            }
        }

        /// <summary>
        /// 両翼の面がなす角度を求める
        /// </summary>
        /// <returns></returns>
        public double GetWingAngle(WingedEdge we)
        {
            if (we.Face0 == int.MaxValue || we.Face1 == int.MaxValue) { return 0; }

            //角度で特徴稜線を
            Vector3 nml0 = mMesh.mFaceNormal[we.Face0];
            Vector3 nml1 = mMesh.mFaceNormal[we.Face1];
            return nml0.GetAngle(nml1, false);
        }

        /// <summary>
        /// 全ての空間でエッジを作成する
        /// </summary>
        /// <param name="mesh"></param>
        public void CreateAll(Mesh mesh)
        {
            foreach (var source in mesh.Sources)
            {
                if (source.Type == SourceType.Position
                    //|| source.Type == SourceType.Normal
                    //|| source.Type == SourceType.Color
                    || source.Type == SourceType.Texture)
                {
                    Create(mesh, source.Type, source.Name);
                    CreateVEL(source.Type, source.Name);
                }
            }
        }

        /// <summary>
        /// チェッカ関数、重複エッジが（存在しないはずのタイミングで）
        /// 存在していることを発見したら例外
        /// </summary>
        /// <param name="type"></param>
        /// <param name="name"></param>
        [Conditional("DEBUG")]
        private void CheckDuplicateEdges(SourceType type, string name, int v0 = int.MaxValue, int v1 = int.MaxValue)
        {
            WERsList wingedEdges = GetWingedEdges(type, name);
            WERsList nonmaniEdges = GetNonmanifoldEdges(type, name);
            // ソートする
            wingedEdges.Sort();
            int edgeN = wingedEdges.Count;
            for (int i = 1; i < edgeN - 1; i++)
            {
                if (!wingedEdges[i].UnMovable && !wingedEdges[i - 1].UnMovable)
                {
                    if (wingedEdges[i].IsVerticesEqual(wingedEdges[i - 1]))
                    {
                        if (!nonmaniEdges.Contains(wingedEdges[i]))
                        {
                            throw new Exception("find duplicate edge");
                        }
                    }
                }
            }
        }

        private void InitialConstructEdges(WERsList wingedEdges, WERsList degEdgesNeighbors, SourceType type, string name)
        {
            int faceN = mMesh.FaceN;
            wingedEdges.Resize(faceN * 3, null);

            int sourceN = mMesh.Sources.Count;
            int so = mMesh.GetSourceOfset(type, name);
            ResizableList<int> vnums = mMesh.VNums;
            ResizableList<int> vofset = mMesh.VOffset;
            ResizableList<int> vbuffer = mMesh.VBuffer;
            ResizableList<int> degenerateVertices = new ResizableList<int>();

            int edgeN = 0;
            for (int f = 0; f < faceN; f++)
            {
                // Degenerate してるフェースは対象としない
                if (mMesh.IsInvalidateFace(f)) { continue; }

                int vn = vnums[f];
                int vofs = vofset[f];
                for (int v = 0; v < vn; v++)
                {
                    int vid0 = vbuffer[(vofs + (v + 0) % vn) * sourceN + so];
                    int vid1 = vbuffer[(vofs + (v + 1) % vn) * sourceN + so];

                    if (wingedEdges[edgeN] == null)
                    {
                        wingedEdges[edgeN] = new WingedEdge();
                        wingedEdges[edgeN].Initialize();
                    }

                    // 同じインデックスを指しているエッジが存在する
                    // その場合は、周辺へのケアが必要になる
                    if (vid0 == vid1)
                    {
                        degenerateVertices.Add(vid0);
                    }

                    wingedEdges[edgeN].Vertex0 = Math.Min(vid0, vid1);
                    wingedEdges[edgeN].Vertex1 = Math.Max(vid0, vid1);
                    wingedEdges[edgeN].Face0 = f;
                    wingedEdges[edgeN].SpaceKey.type = type;
                    wingedEdges[edgeN].SpaceKey.name = name;
                    edgeN++;
                }
            }
            wingedEdges.Resize(edgeN, null);

            // 既に縮退済みエッジ周辺はきな臭いのでストックする
            degEdgesNeighbors.Clear();
            if (degenerateVertices.Any())
            {
                for (int ei = 0; ei < edgeN; ei++)
                {
                    int vi0 = wingedEdges[ei].Vertex0;
                    int vi1 = wingedEdges[ei].Vertex1;

                    if (degenerateVertices.Contains(vi0) || degenerateVertices.Contains(vi1))
                    {
                        wingedEdges[ei].UnMovable = true;
                        degEdgesNeighbors.Add(wingedEdges[ei]);
                    }
                }
            }
        }

        /// <summary>
        /// 縮退済みエッジとその周辺にたいする特殊処理
        /// </summary>
        /// <param name="degEdgesNeighbors"></param>
        private void ProcessForDegeneratedEdges(WERsList degEdgesNeighbors)
        {
            List<WingedEdge> dgeEdges = new List<WingedEdge>();

            foreach (WingedEdge we in degEdgesNeighbors)
            {
                if (we.Vertex0 == we.Vertex1)
                {
                    dgeEdges.Add(we);
                }
            }

            int degEdgeN = dgeEdges.Count;
            for (int dei = 0; dei < degEdgeN; dei++)
            {
                WingedEdge dwe = dgeEdges[dei];
                int faceId = dwe.Face0; // このフェースをウイングに持つエッジは

                int degENN = degEdgesNeighbors.Count;
                for (int ej = 0; ej < degENN; ej++)
                {
                    if (degEdgesNeighbors[ej].Face0 == faceId)
                    {
                        // 消すための準備として、所属フェースなしにする
                        degEdgesNeighbors[ej].Face0 = int.MaxValue;
                        degEdgesNeighbors.RemoveAt(ej);
                        ej = 0;
                        degENN--;
                    }
                }
            }
        }

        /// <summary>
        /// 重複したエッジを削除します
        /// </summary>
        /// <param name="wingedEdges"></param>
        /// <param name="nonmaniEdges"></param>
        private void EraseDuplicateEdges(WERsList wingedEdges, WERsList nonmaniEdges)
        {
            WingedEdge prev = null;
            int edgeN = 0;
            int hitCnt = 0;
            bool[] erasable = new bool[wingedEdges.Count];

            // ソートする
            wingedEdges.Sort();

            // 重複したエッジを削って反対側に、Face１を格納する
            // 同時に非多様体エッジ検出を行う
            foreach (WingedEdge curr in wingedEdges)
            {
                erasable[edgeN] = false;
                if (curr.Face0 == int.MaxValue && curr.Face1 == int.MaxValue)
                {
                    erasable[edgeN] = true;
                }
                else
                {
                    if (prev != null)
                    {
                        if (prev.IsVerticesEqual(curr))
                        {
                            hitCnt++;
                            if (prev.Face1 == int.MaxValue)
                            {
                                prev.Face1 = curr.Face0;
                                erasable[edgeN] = true;
                            }
                            // 非多様体エッジ
                            // X および　Yウイングになっている
                            if (hitCnt >= 2)
                            {
                                // 非多様体エッジを蓄える
                                for (int hi = hitCnt; hi >= 0; hi--)
                                {
                                    erasable[edgeN - hi] = false;
                                    if (wingedEdges[edgeN - hi].UnMovable == false)
                                    {
                                        wingedEdges[edgeN - hi].UnMovable = true;
                                        if (nonmaniEdges.BinarySearch(wingedEdges[edgeN - hi]) == -1)
                                        {
                                            nonmaniEdges.Add(wingedEdges[edgeN - hi]);
                                        }
                                    }
                                }
                            }
                        }
                        else
                        {
                            hitCnt = 0;
                        }
                    }
                    prev = curr;
                }
                edgeN++;
            }

            int enableN = 0;
            // 重複したエッジを削除して綺麗に並べる
            enableN = 0;
            for (int t = 0; t < wingedEdges.Count; t++)
            {
                if (!erasable[t])
                {
                    wingedEdges[enableN] = wingedEdges[t];
                    enableN++;
                }
            }
            wingedEdges.Resize(enableN, null);
            // ソートする
            wingedEdges.Sort();
            enableN = 0;
            foreach (var wingedEdge in wingedEdges)
            {
                wingedEdges[enableN].ID = enableN;
                enableN++;
            }
        }

        /// <summary>
        /// ウイングドエッジを作成する
        /// </summary>
        public void Create(Mesh mesh, SourceType type = SourceType.Position, string name = null)
        {
            if (mesh == null) { return; }
            try
            {
                mMesh = mesh;

                // 既にエッジがあるなら流用
                WERsList wingedEdges = GetWingedEdges(type, name);
                if (wingedEdges == null)
                {
                    wingedEdges = new WERsList();
                    mWingedEdges.Add(new SourceKey(type, name), wingedEdges);
                }
                wingedEdges.Clear();

                // 既にエッジがあるなら流用
                WERsList nonmaniEdges = GetNonmanifoldEdges(type, name);
                if (nonmaniEdges == null)
                {
                    nonmaniEdges = new WERsList();
                    mNonmaniEdges.Add(new SourceKey(type, name), nonmaniEdges);
                }
                nonmaniEdges.Clear();

                WERsList degEdges = new WERsList(); // 縮退済みだった値とその周辺が溜められる
                // エッジの初期構築
                InitialConstructEdges(wingedEdges, degEdges, type, name);
                // インデックスで一致している、縮退エッジとその周辺を除去したい
                ProcessForDegeneratedEdges(degEdges);
                // 重複したエッジの削除を行う
                EraseDuplicateEdges(wingedEdges, nonmaniEdges);

                // 境界エッジの設定
                foreach (WingedEdge curr in wingedEdges)
                {
                    if (curr.Face0 == int.MaxValue && curr.Face1 == int.MaxValue)
                    {
                        throw ExcepHandle.CreateException("Create :: Illegal edge was found.");
                    }

                    if (curr.Face0 == int.MaxValue || curr.Face1 == int.MaxValue)
                    {
                        curr.Boder = true;
                    }
                }

                CheckDuplicateEdges(type, name);
            }
            catch (Exception ex)
            {
                Destroy();
                ExcepHandle.CreateException(ex);
                throw;
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="type"></param>
        /// <param name="name"></param>
        public void SetRepresentativeKey(SourceType type, string name)
        {
            if (mRepresentativeKey == null)
            {
                mRepresentativeKey = new SourceKey(type, name);
            }
            else
            {
                mRepresentativeKey.type = type;
                mRepresentativeKey.name = name;
            }
        }

        /// <summary>
        /// 代表エッジから他のエッジへの間の参照関係を構築します
        /// </summary>
        public void BuildEdgeReference()
        {
            try
            {
                int faceN = mMesh.FaceN;
                int sourceN = mMesh.Sources.Count;
                int rpo = mMesh.GetSourceOfset(mRepresentativeKey.type, mRepresentativeKey.name);

                ResizableList<int> vnums = mMesh.VNums;
                ResizableList<int> vofset = mMesh.VOffset;
                ResizableList<int> vbuffer = mMesh.VBuffer;

                int edgeN = 0;
                for (int f = 0; f < faceN; f++)
                {
                    // Degenerate してるフェースは対象としない
                    if (mMesh.IsInvalidateFace(f)) { continue; }

                    int vn = vnums[f];
                    int vofs = vofset[f];
                    for (int v = 0; v < vn; v++)
                    {
                        for (int so = 0; so < sourceN; so++)
                        {
                            MeshSourceBase source = mMesh.Sources[so];
                            SourceType type = source.Type;
                            string name = source.Name;
                            if (mRepresentativeKey.CompareTo(type, name) == 0) { continue; }

                            int rpv0 = vbuffer[(vofs + (v + 0) % vn) * sourceN + rpo];
                            int rpv1 = vbuffer[(vofs + (v + 1) % vn) * sourceN + rpo];
                            List<WingedEdge> rpedge = FindRepEdge(rpv0, rpv1);

                            int vid0 = vbuffer[(vofs + (v + 0) % vn) * sourceN + so];
                            int vid1 = vbuffer[(vofs + (v + 1) % vn) * sourceN + so];
                            List<WingedEdge> edge = FindEdge(vid0, vid1, type, name);

                            if (edge != null && rpedge != null)
                            {
                                for (int ei = 0; ei < rpedge.Count; ei++)
                                {
                                    WERsList refEdges = rpedge[ei].GetChildEdges(type, name);
                                    for (int ej = 0; ej < edge.Count; ej++)
                                    {
                                        refEdges.Add(edge[ej]);
                                        if (!edge[ej].ParentEdges.Contains(rpedge[ei]))
                                        {
                                            edge[ej].ParentEdges.Add(rpedge[ei]);
                                        }
                                    }
                                }
                            }
                        }
                        edgeN++;
                    }
                }
                RemoveDuplicateEdgeReference();
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /// <summary>
        /// 重複したエッジ参照を削除
        /// （構築時には、何度も検索したくない）
        /// </summary>
        private void RemoveDuplicateEdgeReference()
        {
            try
            {
                int cntN = RepresentativeEdges.Count;

                for (int ei = 0; ei < cntN; ei++)
                {
                    int refDictN = RepresentativeEdges[ei].ChildEdges.Count;
                    for (int rdi = 0; rdi < refDictN; rdi++)
                    {
                        KeyValuePair<SourceKey, WERsList> refEdgesContainer = RepresentativeEdges[ei].ChildEdges.ElementAt(rdi);
                        WERsList refEdges = refEdgesContainer.Value;

                        refEdges.Sort();
                        int refN = refEdges.Count;
                        WingedEdge prev = null;
                        bool[] erasable = new bool[refN];
                        // 重複エッジの検出を行う
                        for (int ej = 0; ej < refN; ej++)
                        {
                            WingedEdge curr = refEdges[ej];
                            erasable[ej] = false;
                            if (prev != null)
                            {
                                if (prev.IsVerticesEqual(curr))
                                {
                                    erasable[ej] = true;
                                }
                            }
                            prev = curr;
                        }

                        // 重複エッジの除去
                        WERsList tmpList = new WERsList();
                        for (int ej = 0; ej < refN; ej++)
                        {
                            if (!erasable[ej])
                            {
                                tmpList.Add(refEdges[ej]);
                            }
                        }
                        RepresentativeEdges[ei].ChildEdges[refEdgesContainer.Key] = tmpList;
                    }
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public void GetOrderedChildIndices(WingedEdge childedge, ref int uv0, ref int uv1)
        {
            var parentEdge = childedge.ParentEdges.First();
            var offset = mMesh.VOffset[childedge.Face0];
            var vnum = mMesh.VNums[childedge.Face0];
            var sourceN = mMesh.SourceN;
            var po = mMesh.GetSourceOfset(mRepresentativeKey.type, mRepresentativeKey.name);
            var vo = mMesh.GetSourceOfset(childedge.SpaceKey.type, childedge.SpaceKey.name);
            for (int i = 0; i < vnum; i++)
            {
                var pid = mMesh.VBuffer[(offset + i) * sourceN + po];
                var vid = mMesh.VBuffer[(offset + i) * sourceN + vo];

                if (pid == parentEdge.Vertex0)
                {
                    uv0 = vid;
                }
                if (pid == parentEdge.Vertex1)
                {
                    uv1 = vid;
                }
            }
        }
    }
}
