﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    //using SourceKey = Tuple<string, SourceType>;
    using KeyedIndices = Dictionary<string, List<int>>;
    using IndexedIndices = SortedDictionary<int, List<int>>;

    using UsedSourceLists = Dictionary<SourceKey, ResizableList<int>>;
    using ReverseSourceMap = Dictionary<SourceKey, Dictionary<int, int>>;

    public class ShapeSplitter
    {
        private Mesh mSrcMesh;

        private ResizableList<Mesh> mOriginalMeshList = new ResizableList<Mesh>();
        private Dictionary<string, Mesh> mDstMeshList = new Dictionary<string, Mesh>();

        public ShapeSplitter(Mesh mesh, ResizableList<Mesh> origMeshList)
        {
            mSrcMesh = mesh;
            mOriginalMeshList = origMeshList;
        }

        /// <summary>
        /// 分割される予定のメッシュの数をカウントアップします
        /// </summary>
        /// <returns></returns>
        private KeyedIndices AssembleSplittingMeshes()
        {
            KeyedIndices faceGroupCount = new KeyedIndices();

            for (int fi = 0; fi < mSrcMesh.FaceN; fi++)
            {
                int fgid = mSrcMesh._FaceMaterialShapeID[fi];
                if (mSrcMesh.MaterialShapeInformationList[fgid].Index != fgid)
                {
                    throw ExcepHandle.CreateException("Index doesn't correspond.");
                }
                string key = mSrcMesh.MaterialShapeInformationList[fgid].ShapeName;
                if (!faceGroupCount.ContainsKey(key))
                {
                    faceGroupCount.Add(key, new List<int>());
                }
                faceGroupCount[key].Add(fi);
            }
            return faceGroupCount;
        }

        /// <summary>
        /// 指定の出力メッシュに、ソースと同じ数のソースを作成します。
        /// </summary>
        /// <param name="dstMesh"></param>
        private void CreateDstMeshSource(Mesh dstMesh)
        {
            if (dstMesh == null)
            {
                throw ExcepHandle.CreateException("dstMesh is null");
            }
            foreach (var sourceBase in mSrcMesh.Sources)
            {
                dstMesh.AddSource(sourceBase.Name, sourceBase.Type, sourceBase.Stride);
            }
        }

        /// <summary>
        /// 指定のfaceListが使用している、ソースのインデックスリストを集めます
        /// </summary>
        /// <param name="faceList"></param>
        private UsedSourceLists AssembleDstSourceIndexList(List<int> faceList)
        {
            try
            {
                var usedIndices = new UsedSourceLists();
                var usedIndexFlag = new Dictionary<SourceKey, ResizableList<bool>>();
                // 各入れ物を作成
                foreach (var sourceBase in mSrcMesh.Sources)
                {
                    SourceKey key = new SourceKey(sourceBase.Type, sourceBase.Name);
                    usedIndices[key] = new ResizableList<int>();
                    usedIndexFlag[key] = new ResizableList<bool>();
                    usedIndexFlag[key].Resize(sourceBase.Count, false);
                }

                int srcSN = mSrcMesh.SourceN;
                ResizableList<int> srcVoff = mSrcMesh.mVoffs;
                ResizableList<int> srcVnum = mSrcMesh.mVnums;
                ResizableList<int> srcVbuff = mSrcMesh.mVbuffs;

                foreach (var fi in faceList)
                {
                    int svo = srcVoff[fi];
                    int svn = srcVnum[fi];
                    for (int vi = 0; vi < svn; vi++)
                    {
                        for (int si = 0; si < srcSN; si++)
                        {
                            string name = mSrcMesh.Sources[si].Name;
                            SourceType type = mSrcMesh.Sources[si].Type;
                            SourceKey key = new SourceKey(type, name);

                            int svi = srcVbuff[(svo + vi) * srcSN + si];

                            if (!usedIndexFlag[key][svi])
                            {
                                usedIndices[key].Add(svi);
                                usedIndexFlag[key][svi] = true;
                            }
                        }
                    }
                }

                return usedIndices;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Fail to create SourceIndexList ", ex);
            }
        }

        /// <summary>
        /// ソースメッシュに同じ名前のものがある場合は、致命的なエラー
        /// </summary>
        [Conditional("DEBUG")]
        private void DestIndicesSumTest(Dictionary<string, UsedSourceLists> dstIndices)
        {
            var dstSourceCount = new Dictionary<SourceKey, int>();

            foreach (var sourceBase in mSrcMesh.Sources)
            {
                var key = new SourceKey(sourceBase.Type, sourceBase.Name);
                dstSourceCount[key] = 0;
            }

            // 各ソースの数を合計する
            foreach (var indicesMap in dstIndices)
            {
                foreach (var node in indicesMap.Value)
                {
                    if (!dstSourceCount.ContainsKey(node.Key))
                    {
                        throw new Exception("Unknown source is found in list.");
                    }
                    dstSourceCount[node.Key] += node.Value.Count;
                }
            }
            foreach (var node in dstSourceCount)
            {
                MeshSourceBase srcSource = mSrcMesh.GetSource(node.Key.type, node.Key.name);
                if (srcSource.Count != node.Value)
                {
                    string msg = string.Format("{0}:total indices count is not equal to original source's count", srcSource.Name);
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(msg);
                    //throw new Exception(msg);
                }
            }
        }

        /// <summary>
        /// 指定メッシュのソースの構築を行います
        /// </summary>
        /// <param name="mesh"></param>
        /// <param name="usedIndices"></param>
        private void SetupDstSource(Mesh mesh, Mesh originalMesh, UsedSourceLists usedIndices)
        {
            foreach (var node in usedIndices)
            {
                MeshSourceBase dstBase = mesh.GetSource(node.Key.type, node.Key.name);
                MeshSourceBase orgBase = originalMesh.GetSource(node.Key.type, node.Key.name);

                if (dstBase == null && orgBase != null)
                {
                    MeshSourceBase sourceBase = mSrcMesh.GetSource(node.Key.type, node.Key.name);
                    if (sourceBase == null)
                    {
                        throw ExcepHandle.CreateException("Tried to create source what does not exist in source.");
                    }
                    mesh.AddSource(node.Key.name, node.Key.type, orgBase.Stride);
                }
            }
            foreach (var node in usedIndices)
            {
                MeshSourceBase orgBase = originalMesh.GetSource(node.Key.type, node.Key.name);
                if (orgBase == null) { continue; }

                MeshSourceBase dstBase = mesh.GetSource(node.Key.type, node.Key.name);
                if (dstBase == null)
                {
                    throw ExcepHandle.CreateException("DstSouce cant find : " + node.Key.name + ":" + node.Key.type);
                }

                MeshSourceBase srcBase = mSrcMesh.GetSource(node.Key.type, node.Key.name);
                if (srcBase == null)
                {
                    throw ExcepHandle.CreateException("SrcSouce cant find : " + node.Key.name + ":" + node.Key.type);
                }

                int stride = srcBase.Stride;
                int dstSn = 0;
                dstBase.SetLength(node.Value.Count, false);
                foreach (var sid in node.Value)
                {
                    try
                    {
                        for (int sti = 0; sti < dstBase.Stride; sti++)
                        {
                            object value = srcBase.GetRowValue(sid * stride + sti);
                            dstBase.SetRowValue(dstSn * dstBase.Stride + sti, value);
                        }
                        dstSn++;
                    }
                    catch (Exception ex)
                    {
                        string msg = string.Format(" Failt to set value {0}[{1}] from {2}[{3}]",
                            dstBase.Name, dstSn, srcBase.Name, sid);
                        throw ExcepHandle.CreateException(msg, ex);
                    }
                }
            }
        }

        /// <summary>
        /// 指定のメッシュのフェースリストを構成します
        /// </summary>
        /// <param name="mesh"></param>
        /// <param name="usedFaces"></param>
        private SortedDictionary<int, int> SetupDstFaces(Mesh mesh, List<int> usedFaces, ReverseSourceMap revMap)
        {
            int dstFN = usedFaces.Count;
            int dstSN = mesh.SourceN;
            int srcSN = mSrcMesh.SourceN;
            mesh.ResizeFaces(dstFN, dstFN * 3 * dstSN);

            ResizableList<int> srcVoff = mSrcMesh.mVoffs;
            ResizableList<int> srcVnum = mSrcMesh.mVnums;
            ResizableList<int> srcVbuff = mSrcMesh.mVbuffs;

            ResizableList<int> dstVoff = mesh.mVoffs;
            ResizableList<int> dstVnum = mesh.mVnums;
            ResizableList<int> dstVbuff = mesh.mVbuffs;

            int dstVO = 0;
            dstFN = 0;
            SortedDictionary<int, int> revFaceMap = new SortedDictionary<int, int>();
            foreach (var fi in usedFaces)
            {
                int svo = srcVoff[fi];
                int svn = srcVnum[fi];

                dstVnum[dstFN] = svn;
                dstVoff[dstFN] = dstVO;
                for (int vi = 0; vi < svn; vi++)
                {
                    for (int si = 0; si < dstSN; si++)
                    {
                        string name = mSrcMesh.Sources[si].Name;
                        SourceType type = mSrcMesh.Sources[si].Type;
                        int svi = srcVbuff[(svo + vi) * srcSN + si];
                        SourceKey key = new SourceKey(type, name);

                        if (!revMap.ContainsKey(key))
                        {
                            throw ExcepHandle.CreateException("Source key :" + key.name + " is not contain.");
                        }
                        if (!revMap[key].ContainsKey(svi))
                        {
                            throw ExcepHandle.CreateException("Index key :" + svi.ToString() + " is not contain in revmap.");
                        }

                        dstVbuff[(dstVO + vi) * dstSN + si] = revMap[key][svi];
                    }
                }
                revFaceMap[fi] = dstFN;
                dstFN++;
                dstVO += svn;
            }

            return revFaceMap;
        }

        /// <summary>
        /// 逆引きインデックスを作成します
        /// </summary>
        /// <param name="usedIndices"></param>
        /// <returns></returns>
        private ReverseSourceMap CreateRevIndecesMap(UsedSourceLists usedIndices)
        {
            var map = new ReverseSourceMap();
            foreach (var node in usedIndices)
            {
                map[node.Key] = new Dictionary<int, int>();
                for (int i = 0; i < node.Value.Count; i++)
                {
                    map[node.Key][node.Value[i]] = i;
                }
            }
            return map;
        }

        /// <summary>
        ///
        /// </summary>
        /// <returns></returns>
        private Dictionary<string, UsedSourceLists> AssembleDestMeshInfo(Dictionary<string, List<int>> faceGroupCount)
        {
            try
            {
                int i = 0;
                foreach (var node in faceGroupCount)
                {
                    string name = node.Key;
                    if (!mDstMeshList.ContainsKey(name))
                    {
                        mDstMeshList.Add(name, new Mesh());
                        mDstMeshList[name].MeshName = name;

                        Mesh.MaterialShapeInfo si = new Mesh.MaterialShapeInfo();
                        si.Index = 0;
                        si.ShapeName = name;
                        mDstMeshList[name].MaterialShapeInformationList.Add(si);
                    }
                    //CreateDstMeshSource(mDstMeshList[i]);
                    i++;
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Fail at dst mesh and source objects", ex);
            }

            // インデックスリストの作成
            var dstIndices = new Dictionary<string, UsedSourceLists>();
            foreach (var node in faceGroupCount)
            {
                try
                {
                    UsedSourceLists usedList = AssembleDstSourceIndexList(node.Value);
                    dstIndices.Add(node.Key, usedList);
                }
                catch (Exception ex)
                {
                    throw ExcepHandle.CreateException("Fail at dst mesh and source objects.", ex);
                }
            }

            //-------------------------------------------------------------
            // インデックスリストの合計が、元のソースの要素数と一致するのか？
            // をチェックします。一致しない場合は、アウトです。
            //-------------------------------------------------------------
            DestIndicesSumTest(dstIndices);
            //-------------------------------------------------------------

            return dstIndices;
        }

        /// <summary>
        /// サブメッシュ情報を集める
        /// </summary>
        /// <returns></returns>
        private Dictionary<string, IndexedIndices> AssembleSubmeshInfo()
        {
            var list = new Dictionary<string, IndexedIndices>();
            for (int fi = 0; fi < mSrcMesh.FaceN; fi++)
            {
                int fgid = mSrcMesh._FaceMaterialShapeID[fi];
                int fsmid = mSrcMesh.mFaceSubMeshID[fi];

                string key = mSrcMesh.MaterialShapeInformationList[fgid].ShapeName;
                if (!list.ContainsKey(key))
                {
                    list.Add(key, new IndexedIndices());
                }

                if (!list[key].ContainsKey(fsmid))
                {
                    list[key].Add(fsmid, new List<int>());
                }
                list[key][fsmid].Add(fi);
            }
            return list;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="facegroup"></param>
        /// <param name="submesh"></param>
        [Conditional("DEBUG")]
        private void AssembledFaceInfoTest(KeyedIndices facegroup, Dictionary<string, IndexedIndices> submesh)
        {
            foreach (var smnode in submesh)
            {
                if (!facegroup.ContainsKey(smnode.Key))
                {
                    throw new Exception("FaceGroupIndex doesnt share: " + smnode.Key.ToString());
                }
            }

            // フェースグループ内での個数が一致してないか？
            foreach (var smnode in submesh)
            {
                string key = smnode.Key;
                int fnsum = 0;
                foreach (var faceListNode in smnode.Value)
                {
                    fnsum += faceListNode.Value.Count;
                }
                if (fnsum != facegroup[key].Count)
                {
                    throw new Exception("Face Count is not equal in face group: " + smnode.Key.ToString());
                }
            }

            // フェースグループ内のインデックス一致をチェック
            foreach (var smnode in submesh)
            {
                string groupName = smnode.Key;
                foreach (var faceListNode in smnode.Value)
                {
                    foreach (var fi in faceListNode.Value)
                    {
                        // 指定のIDのフェースが見つかるか？
                        if (facegroup[groupName].FindIndex((x) => x == fi) == -1)
                        {
                            string msg = string.Format("Face index[{0}] doues not find in face group list:{1} ", fi,
                                                       smnode.Key.ToString());
                            throw new Exception(msg);
                        }
                    }
                }
            }
        }

        private void SetupDstSubmeshes(Mesh mesh, IndexedIndices usedFaces, SortedDictionary<int, int> revFaceMap)
        {
            ResizableList<int> revSubMeshList = mesh.mFaceSubMeshID;
            Dictionary<int, int> smKeyMap = new Dictionary<int, int>();
            // オフセット未設定のサブメッシュ情報をセット
            int sumKey = 0;
            foreach (var subMeshNode in usedFaces)
            {
                if (!smKeyMap.ContainsKey(subMeshNode.Key) && !mesh.SubMeshInformationList.ContainsKey(sumKey))
                {
                    mesh.SubMeshInformationList.Add(sumKey, new Mesh.SubMeshInfo());
                    mesh.SubMeshInformationList[sumKey].subMeshOffset = 0;
                    mesh.SubMeshInformationList[sumKey].subMeshCount = 0;
                    mesh.SubMeshInformationList[sumKey].meshIndex = 0;
                    mesh.SubMeshInformationList[sumKey].subMeshindex = sumKey;
                    smKeyMap[subMeshNode.Key] = sumKey;
                    sumKey++;
                }
            }

            foreach (var subMeshNode in usedFaces)
            {
                foreach (var fi in subMeshNode.Value)
                {
                    if (!revFaceMap.ContainsKey(fi))
                    {
                        throw ExcepHandle.CreateException("revface map dont contain " + fi.ToString());
                    }
                    if (!revFaceMap.ContainsKey(fi))
                    {
                        throw ExcepHandle.CreateException("revSubMeshList");
                    }

                    revSubMeshList[revFaceMap[fi]] = smKeyMap[subMeshNode.Key];
                }
            }
            // 並べ替えを行う
            mesh.ReComputeSubMeshArray(true);
        }

        /// <summary>
        /// 出力先のメッシュの入れ物を作成します
        /// </summary>
        public Dictionary<string, Mesh> Execute()
        {
            // メッシュ毎にFaceGroupが幾つ存在するか？ポリゴンの構成がどうなっているか？を調べる
            var faceGroupCount = AssembleSplittingMeshes();
            // 各FaceGroup内部に、サブメッシュが幾つ存在するか？を調べる
            var groupSubMeshCount = AssembleSubmeshInfo();

            // そもそも分離の必要がない、または分離不能
            if (faceGroupCount.Count == 1) { return null; }

            // 収集した情報が正常であることをテスト
            // 結構重いと思う
            AssembledFaceInfoTest(faceGroupCount, groupSubMeshCount);

            // 入れ物になる出力メッシュを作成する
            var dstIndices = AssembleDestMeshInfo(faceGroupCount);
            //-------------------------------------------------------------
            // インデックスリストから、出力メッシュのソースを作成
            //-------------------------------------------------------------
            foreach (var node in faceGroupCount)
            {
                string key = node.Key;
                try
                {
                    Mesh om = mOriginalMeshList.Find((mesh) => mesh.MeshName == key);
                    if (om == null)
                    {
                        throw ExcepHandle.CreateException("the list dont have the mesh " + key);
                    }
                    SetupDstSource(mDstMeshList[key], om, dstIndices[key]);
                }
                catch (Exception ex)
                {
                    throw ExcepHandle.CreateException("Souces cant set up.", ex);
                }
            }
            //-------------------------------------------------------------
            // faceのセットアップを行う
            //-------------------------------------------------------------
            foreach (var node in faceGroupCount)
            {
                string key = node.Key;
                try
                {
                    Mesh.MaterialShapeInfo sinfo = mSrcMesh.GetShapeInfo(key);

                    // 逆引きインデックスを作成して、FaceIndexのセットアップを行います
                    ReverseSourceMap revMap = CreateRevIndecesMap(dstIndices[key]);
                    if (!faceGroupCount.ContainsKey(key))
                    {
                        throw ExcepHandle.CreateException("Face grout doesnt find.");
                    }
                    SortedDictionary<int, int> revFaceMap = SetupDstFaces(mDstMeshList[key], faceGroupCount[key], revMap);

                    if (!groupSubMeshCount.ContainsKey(key))
                    {
                        throw new Exception("Face grout doesnt find.");
                    }
                    // サブメッシュ情報のセットアップ
                    SetupDstSubmeshes(mDstMeshList[key], groupSubMeshCount[key], revFaceMap);
                }
                catch (Exception ex)
                {
                    throw ExcepHandle.CreateException("Face indices cant set up.", ex);
                }
            }

            // アトリビュートの再アレンジを行う
            foreach (var mesh in mDstMeshList)
            {
                var rearranger = new FaceAttributeRearrange(mesh.Value);
                rearranger.Run();
            }

            // Splitterで作成した分割後のメッシュに対する
            // 法線、面積などの再計算が必要（分割によって消えてしまう）
            // 頂点法線などは分割後に再計算すると、ハードエッジの原因となるのでしない
            foreach (var mesh in mDstMeshList)
            {
                mesh.Value?.RecomputeFaceNormalsAndArea();
            }
            return mDstMeshList;
        }
    }
}
