﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.iflib;
using nw4f.tinymathlib;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    using MVpair = Tuple<string, int>;
    using MVtx = Tuple<string, int, Vector3>;

    using MVpairs = List<Tuple<string, int>>;
    using SVInfo = Tuple<Tuple<string, int>, List<Tuple<string, int>>>;
    using VtxMap = ResizableList<int>;
    /// <summary>
    /// シェイプ結合を行う
    /// </summary>
    public class ShapeMerger
    {
        private static double mPerc = 1e-6;

        private bool mbPostMergeVertex = false;
        private ResizableList<Mesh> mSourceMeshList = null;
        private ResizableList<Mesh> mCloneMeshList = null;

#if SMSPHASH
        private static int mHashTableSize = 1000;
        private ResizableList<SpacialHashList<int>> mMeshPosHashTable = null;
#elif SMUG
        private UniformGrid<MVtx> mUgrid = new UniformGrid<MVtx>();
#else
        private Octree<MVtx> mOctree = new Octree<MVtx>();
#endif
        private Dictionary<MVpair, MVpairs> mSharedVertices = new Dictionary<MVpair, MVpairs>();

        private Mesh mMargedMesh = null;

        private double Threshold { get; set; }
        public static double Percentage
        {
            get { return mPerc; }
            set
            {
                mPerc = value;
                mPerc = Math.Max(0.0, mPerc);
                mPerc = Math.Min(1.0, mPerc);
            }
        }

        public bool PostMergeVtx
        {
            get { return mbPostMergeVertex; }
            set { mbPostMergeVertex = value; }
        }

        /// <summary>
        /// 頂点結合に使用する閾値を、メッシュから求める
        /// </summary>
        private void ComputeThreshold()
        {
            var minLen = double.MaxValue;
            foreach (var mesh in mSourceMeshList)
            {
                minLen = Math.Min(minLen, mesh.ComputeAverageEdgeLength());
            }
            Threshold = minLen * Percentage;
        }

        public ShapeMerger(ResizableList<Mesh> sourceList)
        {
            mSourceMeshList = sourceList;
        }

        public ShapeMerger(ICollection<Mesh> sourceList)
        {
            mSourceMeshList.Clear();
            foreach (var mesh in sourceList)
            {
                mSourceMeshList.Add(mesh);
            }
        }

        /// <summary>
        /// ソースメッシュに同じ名前のものがある場合は、致命的なエラー
        /// </summary>
        [Conditional("DEBUG")]
        private void CheckSourceMeshListName()
        {
            int sizeN = mSourceMeshList.Count;

            for (int i = 0; i < sizeN - 1; i++)
            {
                Mesh lmesh = mSourceMeshList[i];
                for (int j = i + 1; j < sizeN; j++)
                {
                    Mesh rmesh = mSourceMeshList[j];
                    if (string.CompareOrdinal(lmesh.MeshName, rmesh.MeshName) == 0)
                    {
                        throw new Exception("Critical fault : exist 2 mesh what have same name.");
                    }
                }
            }
        }

        /// <summary>
        /// テスト関数
        /// 結合後メッシュのフェースへのアクセスをチェック。
        /// 各属性が妥当に設定できているか？
        /// </summary>
        /// <param name="faceId"></param>
        [Conditional("DEBUG")]
        private void FaceAccessibilityTest(int faceId, Mesh oriMesh, int ofId)
        {
            int mrgSN = mMargedMesh.SourceN;
            int orgSN = oriMesh.SourceN;

            int vN = mMargedMesh.VertexN;
            int fN = mMargedMesh.FaceN;

            ResizableList<int> margeVnum = mMargedMesh.mVnums;
            ResizableList<int> margeVofs = mMargedMesh.mVoffs;
            ResizableList<int> margeVbuff = mMargedMesh.mVbuffs;
            ResizableList<int> margeFGroup = mMargedMesh._FaceMaterialShapeID;
            ResizableList<int> margeSMId = mMargedMesh.mFaceSubMeshID;

            ResizableList<int> origVnum = oriMesh.mVnums;
            ResizableList<int> oriVofs = oriMesh.mVoffs;
            ResizableList<int> oriVbuff = oriMesh.mVbuffs;

            int vn = margeVnum[faceId];
            int vo = margeVofs[faceId];

            int ovn = origVnum[ofId];
            int ovo = oriVofs[ofId];

            if (fN <= faceId)
            {
                throw new Exception("Face id is not legal value.");
            }
            if (vn != 3)
            {
                throw new Exception("Face count is not legal value.");
            }
            if (vo > fN * 3 * mrgSN)
            {
                throw new Exception("Vertex offset is not legal value.");
            }
            if (margeFGroup[faceId] == int.MaxValue || margeFGroup[faceId] == -1)
            {
                throw new Exception("Face group is not legal value.");
            }
            if (margeSMId[faceId] == int.MaxValue || margeSMId[faceId] == -1)
            {
                throw new Exception("Submesh group is not legal value.");
            }
            if (ovn != vn)
            {
                throw new Exception("Vertex number is not equal.");
            }

            for (int vi = 0; vi < vn; vi++)
            {
                for (int si = 0; si < mrgSN; si++)
                {
                    MeshSourceBase srcBase = mMargedMesh.Sources[si];
                    MeshSourceBase osrcBase = oriMesh.GetSource(srcBase.Type, srcBase.Name);
                    if (osrcBase == null) { continue; }

                    int sj = oriMesh.Sources.IndexOf(osrcBase);

                    int vid = margeVbuff[(vo + vi) * mrgSN + si];
                    int ovid = oriVbuff[(ovo + vi) * orgSN + sj];

                    if (vid >= srcBase.Count)
                    {
                        throw new Exception("Property index is not legal value. : " + srcBase.Name +
                                            " : " + vid.ToString() +
                                            "/" + srcBase.Count.ToString());
                    }
                    if (osrcBase == null)
                    {
                        throw new Exception("exist no correspondent source.");
                    }
#if VERBOSE
                    if( srcBase.Stride != osrcBase.Stride  )
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(@"Stride count is not equal.");
                    }
#endif
                    // 各ソースの値が一致しない
                    for (int sti = 0; sti < osrcBase.Stride; sti++)
                    {
                        object v0 = srcBase.GetRowValue(vid * srcBase.Stride + sti);
                        object v1 = osrcBase.GetRowValue(ovid * osrcBase.Stride + sti);

                        if (v0 is double && v1 is double)
                        {
                            if (Math.Abs((double)v0 - (double)v1) > Threshold)
                            {
                                throw new Exception("Source value is not equal.");
                            }
                        }
                        else if (!v0.Equals(v1))
                        {
                            throw new Exception("Source value is not equal.");
                        }
                    }
                }
            }
        }

        [Conditional("DEBUG")]
        private void PropEqualityTest()
        {
            int mi = 0;
            VectorN[] vlist0 = new VectorN[3];
            VectorN[] vlist1 = new VectorN[3];
            for (int vi = 0; vi < 3; vi++)
            {
                vlist0[vi] = new VectorN((uint)mMargedMesh.GetPropDimension());
                vlist1[vi] = new VectorN((uint)mMargedMesh.GetPropDimension());
            }

            foreach (var mesh in mSourceMeshList)
            {
#if VERBOSE
                List<int> l0 = mMargedMesh.CoutUpUsedPropFG(SourceType.Position, null, mi);
                List<int> l1 = mesh.CoutUpUsedPropFG(SourceType.Position, null);

                // ユニークカウントは一致してなくてもいい、一致しない場合
                // 重複頂点が境界上にあったというだけの事になる
                /*
                if (l0.Count != l1.Count)
                {
                    //throw new Exception("unmatch used count");
                }
                */
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine("marged used pos count {0} / original used pos count {1}", l0, l1);
#endif

                List<int> flist = new List<int>();
                mMargedMesh.GetFacesFromGroupIndex(mi, flist);

                for (int fi = 0; fi < mesh.FaceN; fi++)
                {
                    mMargedMesh.GetFaceVertex(flist[fi], ref vlist0);
                    mesh.GetFaceVertex(fi, ref vlist1);

                    for (int vi = 0; vi < 3; vi++)
                    {
                        Vector3 dist = new Vector3(vlist0[vi][0] - vlist1[vi][0], vlist0[vi][1] - vlist1[vi][1], vlist0[vi][2] - vlist1[vi][2]);
                        if (dist.Norm() > Threshold)
                        {
                            throw new Exception("Found unmatched position");
                        }
                    }
                }
                mi++;
            }
            mMargedMesh.FacePropsCountEqualityTest();
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsSubmeshAreaZero(Mesh mesh)
        {
            ResizableList<double> submeshArea = new ResizableList<double>();
            int faceN = mesh.FaceN;
            mesh.RecomputeFaceNormalsAndArea();

            submeshArea.Resize(mesh.SubMeshInformationList.Count, 0.0);
            for (int fi = 0; fi < faceN; fi++)
            {
                double farea = mesh.GetFaceArea(fi);
                int subMeshId = mesh.mFaceSubMeshID[fi];
                if (subMeshId >= submeshArea.Count || subMeshId < 0)
                {
                    throw ExcepHandle.CreateException("SubmeshID is bigger than list length.");
                }
                submeshArea[subMeshId] += farea;
            }

            // サブメッシュの合計面積がゼロの場合は、そのサブメッシュは消滅することになる
            for (int i = 0; i < submeshArea.Count; i++)
            {
                if (submeshArea[i] <= double.Epsilon)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 結合前の準備
        /// </summary>
        private void PreProc()
        {
            try
            {
                foreach (var mesh in mSourceMeshList)
                {
                    mesh.DebugOutput(mesh.MeshName + ".bm.obj");
                }
                CheckSourceMeshListName();
                List<string> names = new List<string>();
                if (CountUpMaxSourceNum(SourceType.Position, ref names) > 1)
                {
                    throw ExcepHandle.CreateException("Some mesh have several veretex lists. Such mesh is not legal input for this class.");
                }
                //クローンリスト
                CloneList();
                //近接頂点の結合
                MergeClosedVertex();
                //ハッシュテーブルを作成
#if SMSPHASH
                CreatePosHashTable();
#elif SMUG
                CreateUniformGrid();
#else
                CreateOctree();
#endif
                ComputeThreshold();

                // メッシュの境界ボリュームを作成します
                ComputeMeshBoundingBox();
            }
            catch (Exception ex)
            {
                ExcepHandle.CreateException(ex);
                throw;
            }
        }

        /// <summary>
        /// 後処理
        /// </summary>
        private void PostProc()
        {
            try
            {
                if (mbPostMergeVertex)
                {
                    ClosedVtxMarger cvm = new ClosedVtxMarger(mMargedMesh);
                    cvm.SetSpaceInfo(10, 10, 10);
                    cvm.Run();
                }
                mMargedMesh.DebugOutput(mMargedMesh.MeshName + ".am.obj");
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /// <summary>
        /// 実行処理
        /// </summary>
        public Mesh Execute()
        {
            // サブメッシュの面積がゼロのShapeがある場合は結合処理を行わない
            foreach (var mesh in mSourceMeshList)
            {
                if (mesh != null && IsSubmeshAreaZero(mesh))
                {
                    return null;
                }
            }

            try
            {
                PreProc();
            }
            catch
            {
                return null;
            }

            if (mSourceMeshList.Any(o => o.FaceN == 0))
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Vanish_Mesh"));
                return null;
            }

            // 複数のメッシュ間で共有される頂点を抜き出します
            if (!PickSharedVertices())
            {
                RestoreCloneList();
                return null;
            }

            // 結合後のメッシュを作成します。
            try
            {
                // 一時保持用の頂点
                var revPosIndices = new Dictionary<string, VtxMap>();
                // 頂点も含めて統合して用いる逆引きソースリスト
                var revSrcIndices = new Dictionary<string, Dictionary<string, VtxMap>>();

                // サブメッシュの逆引きリスト
                var revSubMeshIndices = new Dictionary<Tuple<string, int>, int>();

                // 統合メッシュの入れ物を作成
                CreateUniteMesh();

                // 共有する頂点ソースを作成します
                CreateUnitePos(ref revPosIndices);
                // 結合後のメッシュの頂点以外のソースを作成します
                CreateUniteSources(ref revSrcIndices);
                // 頂点と他のソースの逆引きマップを結合
                MargeSourceMap(ref revSrcIndices, revPosIndices);

                //サブメッシュ情報の入れ物となるMapとShape情報を作成します
                CreateShapeAndSMeshInfo(ref revSubMeshIndices);

                // 結合後のメッシュのフェースを作成します
                CreateFaces(revSrcIndices, revSubMeshIndices);

                // データの整合性テスト
                PropEqualityTest();
            }
            catch (Exception ex)
            {
                throw ex;
            }

            try
            {
                PostProc();
            }
            catch
            {
                throw;
            }
            return mMargedMesh;
        }

        /// <summary>
        /// 統合メッシュにサブメッシュリストを作る
        /// </summary>
        /// <param name="revSubMeshIndices"></param>
        private void CreateShapeAndSMeshInfo(ref Dictionary<Tuple<string, int>, int> revSubMeshIndices)
        {
            var subMeshMap = mMargedMesh.SubMeshInformationList;

            // Shape情報リストを作成
            mMargedMesh.MaterialShapeInformationList.Resize(mSourceMeshList.Count, null);

            // デフォルトサブメッシュ情報の追加
            var defSmKey = mSourceMeshList.Where(o => o.SubMeshInformationList.Count > 1).Sum(mesh => mesh.SubMeshInformationList.Count);
            var smCount = 0;
            subMeshMap.Add(defSmKey, Mesh.SubMeshInfo.Null);
            foreach (var mesh in mSourceMeshList)
            {
                var mid = mSourceMeshList.IndexOf(mesh);

                // Shape情報を設定
                mMargedMesh.MaterialShapeInformationList[mid] = new Mesh.MaterialShapeInfo
                {
                    Index = mid,
                    ShapeName = mesh.MeshName,
                    IsExcludeForSimplification = false
                };
                foreach (var meshSrc in mesh.Sources)
                {
                    mMargedMesh.MaterialShapeInformationList[mid].refSources.Add
                        (
                            Tuple.Create(meshSrc.Type, meshSrc.Name));
                }
                foreach (var node in mesh.SubMeshInformationList)
                {
                    if (node.Value.subMeshCount == 0)
                    {
                        throw ExcepHandle.CreateException("Illegal submesh information is found in mesh:" + mesh.MeshName);
                    }

                    var lsm = node.Value;
                    if (lsm.subMeshindex != node.Key)
                    {
                        throw ExcepHandle.CreateException("Illegal submesh information is found in mesh:" + mesh.MeshName);
                    }
                    var key = Tuple.Create(mesh.MeshName, node.Key);
                    // 一つのサブメッシュ情報でメッシュ全体がカバーされている場合は、設定しない
                    if (!mesh.CoveredMeshbyASubMesh())
                    {
                        var usm = new Mesh.SubMeshInfo
                        {
                            subMeshindex = smCount,
                            meshIndex = mid
                        };
                        // 統合後のサブメッシュIDを設定
                        // メッシュインデックスを設定( FaceGroup のインデックスと同じでもある
                        subMeshMap.Add(smCount, usm);
                        revSubMeshIndices.Add(key, smCount);
                        smCount++;
                    }
                    else
                    {
                        // デフォルトサブメッシュを参照
                        revSubMeshIndices.Add(key, defSmKey);
                    }
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="srcMapDDict"></param>
        /// <param name="posMapDict"></param>
        private void MargeSourceMap(ref Dictionary<string, Dictionary<string, VtxMap>> srcMapDDict,
                           Dictionary<string, VtxMap> posMapDict)
        {
            if (srcMapDDict.Count == 0)
            {
                foreach (var mesh in mSourceMeshList)
                {
                    if (!posMapDict.ContainsKey(mesh.MeshName))
                    {
                        throw ExcepHandle.CreateException("posMapDict doesnt have key : " + mesh.MeshName);
                    }

                    MeshSourceBase source = mesh.GetSource(SourceType.Position, null);
                    srcMapDDict.Add(mesh.MeshName, new Dictionary<string, VtxMap>());
                    Dictionary<string, VtxMap> srcMapDict = srcMapDDict[mesh.MeshName];
                    if (srcMapDict == null)
                    {
                        throw ExcepHandle.CreateException("srcMapDDict doesnt have key : " + mesh.MeshName);
                    }

                    VtxMap posMap = posMapDict[mesh.MeshName];
                    srcMapDict.Add(source.Name, posMap);
                }
            }
            else
            {
                if (srcMapDDict.Count != posMapDict.Count)
                {
                    throw ExcepHandle.CreateException("Count of source maps and count of pos map are different.");
                }

                foreach (var mesh in mSourceMeshList)
                {
                    if (!posMapDict.ContainsKey(mesh.MeshName))
                    {
                        throw ExcepHandle.CreateException("posMapDict doesnt have key : " + mesh.MeshName);
                    }
                    if (!srcMapDDict.ContainsKey(mesh.MeshName))
                    {
                        throw ExcepHandle.CreateException("srcMapDDict doesnt have key : " + mesh.MeshName);
                    }

                    MeshSourceBase source = mesh.GetSource(SourceType.Position, null);

                    VtxMap posMap = posMapDict[mesh.MeshName];
                    Dictionary<string, VtxMap> srcMapDict = srcMapDDict[mesh.MeshName];

                    if (srcMapDict.ContainsKey(source.Name))
                    {
                        srcMapDict[source.Name] = posMap;
                    }
                }
            }
        }

        /// <summary>
        /// 面を設定していきます
        /// </summary>
        /// <param name="posMap"></param>
        /// <param name="srcMap"></param>
        private void CreateFaces(Dictionary<string, Dictionary<string, VtxMap>> srcMapDDict, Dictionary<Tuple<string, int>, int> smMap)
        {
            var sumfaceN = CountUpFaceNum();
            var msN = mMargedMesh.SourceN;
            // フェースの入れ物を作成
            mMargedMesh.ResizeFaces(sumfaceN, sumfaceN * 3 * msN);

            var margeVnum = mMargedMesh.mVnums;
            var margeVofs = mMargedMesh.mVoffs;
            var margeVbuff = mMargedMesh.mVbuffs;
            var margeFGroup = mMargedMesh._FaceMaterialShapeID;
            var margeSMId = mMargedMesh.mFaceSubMeshID;

            var faceCounter = 0;
            foreach (var mesh in mSourceMeshList)
            {
                var lsN = mesh.SourceN;
                if (!srcMapDDict.ContainsKey(mesh.MeshName))
                {
                    throw ExcepHandle.CreateException("srcMapDDict doesnt have key : " + mesh.MeshName);
                }

                var srcMapDict = srcMapDDict[mesh.MeshName];
                // MaterialShapeInfo を、メッシュの名前から探す。なければエラー
                var shapeInfo = mMargedMesh.GetShapeInfo(mesh.MeshName);
                if (shapeInfo == null)
                {
                    throw ExcepHandle.CreateException("Shape information : " + mesh.MeshName + " can not be found.");
                }

                // 各メッシュのソースの名前を取得
                var srcNamese = mesh.Sources.Select(source => source.Name).ToList();

                var faceN = mesh.FaceN;
                for (var fi = 0; fi < faceN; fi++)
                {
                    var vo = mesh.mVoffs[fi];
                    var mvo = faceCounter * 3;
                    var smid = mesh.mFaceSubMeshID[fi];

                    margeVnum[faceCounter] = 3;
                    margeVofs[faceCounter] = mvo;
                    margeFGroup[faceCounter] = shapeInfo.Index;
                    margeSMId[faceCounter] = (smid != -1) ? smMap[Tuple.Create(mesh.MeshName, smid)] : 0;

                    for (var vi = 0; vi < 3; vi++)
                    {
                        for (var si = 0; si < lsN; si++)
                        {
                            if ((vo + vi) * lsN + si >= mesh.mVbuffs.Count)
                            {
                                throw ExcepHandle.CreateException("Index is over mesh.mVbuffs.count");
                            }

                            var sid = mesh.mVbuffs[(vo + vi) * lsN + si];
                            var key = srcNamese[si];
                            var stype = mesh.Sources[si].Type;

                            if (!srcMapDict.ContainsKey(key))
                            {
                                throw ExcepHandle.CreateException("src dict doesnt have key :: " + key);
                            }

                            var msi = mMargedMesh.GetSourceOfset(stype, stype != SourceType.Position ? key : null);
                            if (msi == int.MaxValue)
                            {
                                throw ExcepHandle.CreateException("Source ofset is not legal value.");
                            }

                            // 統合メッシュのソースのIDを取得して、フェースに設定する
                            var index = srcMapDict[key][sid];
                            if (index == int.MaxValue)
                            {
                                index = mMargedMesh.Sources[si].Count - 1;
                            }
                            margeVbuff[(mvo + vi) * msN + msi] = index;
                        }

                        var prevIndex = 0;
                        // 未使用ソースのインデックスを、既定値で埋める
                        for (var sj = 0; sj < msN; sj++)
                        {
                            if (margeVbuff[(mvo + vi) * msN + sj] == int.MaxValue)
                            {
                                margeVbuff[(mvo + vi) * msN + sj] = prevIndex;
                            }
                            else
                            {
                                prevIndex = margeVbuff[(mvo + vi) * msN + sj];
                            }
                        }
                    }
                    //----------------------------------------------
                    // 各フェースが元のメッシュと完全に一致しているか？
                    // データへのアクセスは保全されているか？
                    // などをテストする
                    FaceAccessibilityTest(faceCounter, mesh, fi);
                    //----------------------------------------------
                    faceCounter++;
                }
            }

            //----------------------------------------------
            // SubMesh の並べ替え
            // その他のフェースに付属するプロパティも並べ替え
            mMargedMesh.ReComputeSubMeshArray(true);
        }

        /// <summary>
        /// 結合後のフェースの数を数え上げます
        /// </summary>
        /// <returns> </returns>
        private int CountUpFaceNum()
        {
            int maxFace = 0;
            foreach (var mesh in mSourceMeshList)
            {
                maxFace += mesh.FaceN;
            }
            return maxFace;
        }

        /// <summary>
        /// 共有頂点を全てのShapeの間で検出します
        /// </summary>
        private bool PickSharedVertices()
        {
            ResizableList<ResizableList<int>> neighborList = null;

            try
            {
                PickNeighborMesh(ref neighborList);
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("PickSharedVertices:fail to pick neighbor shape", ex);
            }

#if SMSPHASH
            ResizableList<ResizableList<bool>> vtxUsedFlag = new ResizableList<ResizableList<bool>>();
            vtxUsedFlag.Resize(mSourceMeshList.Count,null);
            for (int i = 0; i < mSourceMeshList.Count; i++)
            {
                vtxUsedFlag[i] = new ResizableList<bool>();
                vtxUsedFlag[i].Resize(mSourceMeshList[i].VertexN,false);
            }

            bool bHit = false;

            for (int i = 0; i < mSourceMeshList.Count - 1; i++)
            {
                for (int j = i + 1; j < mSourceMeshList.Count; j++)
                {
                    // バウンディングボックスが交差している場合
                    //if (neighborList[i].Contains(j))
                    {
                        try
                        {
                            bHit |= ExistCorrespondVertices(i, j, vtxUsedFlag[i], vtxUsedFlag[j]);
                        }
                        catch (Exception ex)
                        {
                            string msg = string.Format("PickSharedVertices:fail to pick correspond vertex mesh:{0} and mesh:{1}", i, j);
                            throw new Exception(msg, ex);
                        }
                    }
                }
            }
#else

            Dictionary<string, ResizableList<bool>> vtxUsedFlag = new Dictionary<string, ResizableList<bool>>();
            for (int i = 0; i < mSourceMeshList.Count; i++)
            {
                string name = mSourceMeshList[i].MeshName;
                vtxUsedFlag.Add(name, new ResizableList<bool>());
                vtxUsedFlag[name].Resize(mSourceMeshList[i].VertexN, false);
            }

            bool bHit = false;

            for (int i = 0; i < mSourceMeshList.Count; i++)
            {
                try
                {
                    bHit |= ExistCorrespondVertices(i, vtxUsedFlag);
                }
                catch (Exception ex)
                {
                    string msg = string.Format("PickSharedVertices:fail to pick correspond vertex mesh:{0}", i);
                    throw ExcepHandle.CreateException(msg, ex);
                }
            }
#endif
            return bHit;
        }

        /// <summary>
        /// 統合メッシュの入れ物そのものを作成します
        /// </summary>
        private void CreateUniteMesh()
        {
            mMargedMesh = new Mesh();
            mMargedMesh.MeshName = "margedMesh";
        }

        /// <summary>
        /// 共有頂点をソースに追加します
        /// </summary>
        /// <param name="posList"></param>
        /// <param name="reverceIndices"></param>
        private void AddSharedPos(MeshSourceBase posList, ref Dictionary<string, VtxMap> reverceIndices)
        {
            // 逆引きインデックスリストを作成します
            if (reverceIndices == null)
            {
                reverceIndices = new Dictionary<string, VtxMap>();
            }
            else
            {
                reverceIndices.Clear();
            }
            for (int i = 0; i < mSourceMeshList.Count; i++)
            {
                reverceIndices[mSourceMeshList[i].MeshName] = new VtxMap();
                reverceIndices[mSourceMeshList[i].MeshName].Resize(mSourceMeshList[i].VertexN, int.MaxValue);
            }

            // オリジナルメッシュのソースをリストアップします
            var origPosList = new Dictionary<string, MeshSourceBase>(mSourceMeshList.Count);
            for (int i = 0; i < mSourceMeshList.Count; i++)
            {
                origPosList[mSourceMeshList[i].MeshName] = mSourceMeshList[i].GetSource(SourceType.Position, null);
            }

            // 最大の位置座標数を取得
            int maxVtxN = 0;
            for (int i = 0; i < mSourceMeshList.Count; i++)
            {
                maxVtxN += mSourceMeshList[i].VertexN;
            }
            posList.SetLength(maxVtxN, false);

            // 共有頂点を共有メッシュのソースに加える
            int vtxN = 0;
            Vector3 tmpV3 = new Vector3();
            foreach (var sharedVertece in mSharedVertices)
            {
                string name = sharedVertece.Key.Item1;
                int index = sharedVertece.Key.Item2;

                // 頂点位置の設定
                try
                {
                    MeshSourceDblCtrler.GetAsVec3(origPosList[name], index, ref tmpV3);
                    MeshSourceDblCtrler.SetAsVec3(posList, vtxN, tmpV3.x, tmpV3.y, tmpV3.z);
                }
                catch (Exception ex)
                {
                    string msg = string.Format("Fail to pick position from {0}[{1}]", name, index);
                    throw new Exception(msg, ex);
                }

                foreach (var mvpair in sharedVertece.Value)
                {
                    string name2 = mvpair.Item1;
                    int index2 = mvpair.Item2;
                    // 逆引きインデックスの作成
                    try
                    {
                        if (reverceIndices[name][index] == int.MaxValue)
                        {
                            reverceIndices[name][index] = vtxN;
                        }
                        if (reverceIndices[name2][index2] == int.MaxValue)
                        {
                            reverceIndices[name2][index2] = vtxN;
                        }
                    }
                    catch (Exception ex)
                    {
                        string msg = string.Format("Fail to set revert index {0}[{1}]", name, index);
                        throw ExcepHandle.CreateException(msg, ex);
                    }
                }

                vtxN++;
            }
        }

        /// <summary>
        /// 各メッシュの頂点を追加
        /// </summary>
        /// <param name="posList"></param>
        /// <param name="revIndices"></param>
        private void AddMeshPos(MeshSourceBase posList, ref Dictionary<string, VtxMap> revIndices)
        {
            int totVtxN = mSharedVertices.Count;
            foreach (Mesh mesh in mSourceMeshList)
            {
                MeshSourceBase meshPos = mesh.GetSource(SourceType.Position, null);
                int vtxN = mesh.VertexN;
                Vector3 tv3 = new Vector3();

                for (int vi = 0; vi < vtxN; vi++)
                {
                    MeshSourceDblCtrler.GetAsVec3(meshPos, vi, ref tv3);

                    if (!revIndices.ContainsKey(mesh.MeshName))
                    {
                        throw ExcepHandle.CreateException("Could not find reverce verteces key");
                    }

                    if (revIndices[mesh.MeshName][vi] == int.MaxValue)
                    {
                        MeshSourceDblCtrler.SetAsVec3(posList, totVtxN, tv3.x, tv3.y, tv3.z);
                        revIndices[mesh.MeshName][vi] = totVtxN;
                        totVtxN++;
                    }
                }
            }
        }

        /// <summary>
        /// 統合メッシュのソースを作成する
        /// </summary>
        private void CreateUnitePos(ref Dictionary<string, VtxMap> revIndices)
        {
            if (revIndices == null) { throw new ArgumentNullException("revIndices"); }

            string posname = null;
            foreach (var mesh in mSourceMeshList)
            {
                MeshSourceBase source = mesh.GetSource(SourceType.Position, null);

                if (posname == null)
                {
                    posname = source.Name;
                }
                else if (posname != source.Name)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(@"A mesh have some position list");
                }
            }

            // 共有頂点を統合メッシュのインデックスに追加します
            MeshSourceBase posList = mMargedMesh.GetSource(SourceType.Position, null);
            if (posList == null)
            {
                posList = mMargedMesh.AddSource(posname, SourceType.Position, 3);
            }
            try
            {
                // 共有頂点をソースに加える
                AddSharedPos(posList, ref revIndices);
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Fail to add shared vtx in pos list.", ex);
            }

            try
            {
                //各シェープの共有頂点を加える
                AddMeshPos(posList, ref revIndices);
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Fail to add vtx in pos list.", ex);
            }
        }

        /// <summary>
        /// 指定したソースが各メッシュでいくつあるか？
        /// </summary>
        /// <param name="type"></param>
        private int CountUpMaxSourceNum(SourceType type, ref List<string> uniqueName)
        {
            if (uniqueName == null)
            {
                throw ExcepHandle.CreateException("uniqueName is NULL");
            }

            int maxN = int.MinValue;
            foreach (Mesh mesh in mSourceMeshList)
            {
                List<MeshSourceBase> slist = mesh.GetSources(type);
                if (slist != null)
                {
                    foreach (MeshSourceBase source in slist)
                    {
                        if (!uniqueName.Contains(source.Name))
                        {
                            uniqueName.Add(source.Name);
                        }
                    }
                }
            }
            maxN = uniqueName.Count;
            return maxN;
        }

        /// <summary>
        /// 指定したソースのストライドを各メッシュで調べて最大値を返す
        /// </summary>
        /// <param name="type"></param>
        private int GetSourceStride(SourceType type, string name)
        {
            int maxN = int.MinValue;
            foreach (Mesh mesh in mSourceMeshList)
            {
                MeshSourceBase sourceBase = mesh.GetSource(type, name);
                if (sourceBase != null)
                {
                    maxN = Math.Max(maxN, sourceBase.Stride);
                }
            }
            return maxN;
        }

        /// <summary>
        ///
        /// </summary>
        private void SetUniteSourceValues(Mesh mesh, ref Dictionary<string, VtxMap> revIndices)
        {
            int maxCnt = 0;
            int checkCnt = 0;

            foreach (var s in mesh.Sources)
            {
                if (s.Type != SourceType.Position)
                {
                    checkCnt = s.Count;
                }
                maxCnt = Math.Max(maxCnt, s.Count);
            }

            // 結合後のソース全体の入れ物を作成
            foreach (var source in mMargedMesh.Sources)
            {
                revIndices.Add(source.Name, new VtxMap());
                revIndices[source.Name].Resize(maxCnt, int.MaxValue);
            }

            foreach (var msource in mMargedMesh.Sources)
            {
                if (msource.Type == SourceType.Position) { continue; }

                // 指定のソースが、現在のメッシュに存在する場合
                MeshSourceBase source = mesh.GetSource(msource.Type, msource.Name);
                if (source != null)
                {
                    int srcValN = source.Count;
                    // 各メッシュのソースの値を、統合メッシュのソースに加えていく
                    for (int ri = 0; ri < srcValN; ri++)
                    {
                        int dstIndex = msource.Count;
                        for (int sti = 0; sti < msource.Stride; sti++)
                        {
                            if (sti < source.Stride)
                            {
                                int rowIndex = ri * source.Stride + sti;
                                msource.AddRowValue(source.GetRowValue(rowIndex));
                            }
                            else
                            {
                                if (source.Type == SourceType.EtceteraInt)
                                {
                                    msource.AddRowValue(0);
                                }
                                else
                                {
                                    msource.AddRowValue(0.0);
                                }
                            }
                        }
                        revIndices[source.Name][ri] = dstIndex;
                    }
                }
                // 存在しない場合にも同数追加する
                else
                {
                    for (int ri = 0; ri < checkCnt; ri++)
                    {
                        int dstIndex = msource.Count;
                        for (int sti = 0; sti < msource.Stride; sti++)
                        {
                            if (msource.Type == SourceType.EtceteraInt)
                            {
                                msource.AddRowValue(0);
                            }
                            else
                            {
                                msource.AddRowValue(0.0);
                            }
                        }
                        revIndices[msource.Name][ri] = dstIndex;
                    }
                }
            }
        }

        /// <summary>
        /// 頂点以外のソースを,統合メッシュに作成します
        /// </summary>
        private void AddUniteSources()
        {
            // ソースを追加する
            foreach (SourceType t in Enum.GetValues(typeof(SourceType)))
            {
                if (t == SourceType.Position) { continue; }

                List<string> uniqueKey = new List<string>();

                // メッシュリストから、指定のソースタイプの
                // 重複しない名前のものを数え上げる
                CountUpMaxSourceNum(t, ref uniqueKey);

                // 重複なしの分だけ追加
                foreach (var key in uniqueKey)
                {
                    // 最大のストライドを求める（ストライドが異なることなんで基本的にはないはず）
                    int maxstride = GetSourceStride(t, key);
                    if (maxstride == int.MinValue)
                    {
                        throw ExcepHandle.CreateException("No source exist in MeshList." + key.ToString());
                    }
                    mMargedMesh.AddSource(key, t, maxstride);
                }
            }
        }

        /// <summary>
        /// 結合後のメッシュの頂点位置以外のソースを作成
        /// </summary>
        /// <param name="revIndicesList"></param>
        private void CreateUniteSources(ref Dictionary<string, Dictionary<string, VtxMap>> revIndicesList)
        {
            if (revIndicesList == null)
            {
                throw ExcepHandle.CreateException("revIndicisList is NULL");
            }

            // 頂点以外のソースを統合メッシュに加えます。
            AddUniteSources();

            // 各ソースの値を、加えていく
            foreach (var mesh in mSourceMeshList)
            {
                Dictionary<string, VtxMap> revIndices = new Dictionary<string, VtxMap>();
                SetUniteSourceValues(mesh, ref revIndices);

                revIndicesList.Add(mesh.MeshName, revIndices);
            }
        }

#if SMSPHASH
        private bool ExistCorrespondVertices(int left, int right, ResizableList<bool> leftFlags,ResizableList<bool> rightFlags )
        {
            if (mMeshPosHashTable.Count <= left)  throw new ArgumentException("over flow to count");
            if (mMeshPosHashTable.Count <= right) throw new ArgumentException("over flow to count");

            SpacialHashList<int> ltable = mMeshPosHashTable[left];
            SpacialHashList<int> rtable = mMeshPosHashTable[right];
            Mesh lmesh = null;
            Mesh rmesh = null;

            try
            {
                lmesh = mSourceMeshList[left];
                rmesh = mSourceMeshList[right];

                MeshSourceBase lposlist = lmesh.GetSource(SourceType.Position, null);
                MeshSourceBase rposlist = rmesh.GetSource(SourceType.Position, null);

                Vector3 lv3 = new Vector3();
                Vector3 rv3 = new Vector3();
                bool haveSv = false;

                for (int si = 0; si < mHashTableSize; si++)
                {
                    if (ltable[si].Count > 0 && rtable[si].Count > 0)
                    {
                        for (int pi = 0; pi < ltable[si].Count; pi++)
                        {
                            int lpid = ltable[si][pi];
                            MeshSourceDblCtrler.GetAsVec3(lposlist, lpid, ref lv3);

                            for (int pj = 0; pj < rtable[si].Count; pj++)
                            {
                                int rpid = rtable[si][pj];
                                MeshSourceDblCtrler.GetAsVec3(rposlist, rpid, ref rv3);

                                Vector3 dif = lv3 - rv3;
                                // 一致する頂点がある
                                if (dif.Norm() <= Threshold)
                                {
                                    // 各メッシュの共有頂点情報を作る
                                    MVpair ltuple = Tuple.Create(lmesh.Name, lpid);
                                    MVpair rtuple = Tuple.Create(rmesh.Name, rpid);
                                    // 共有頂点情報をリストに追加します。
                                    if (!mSharedVertices.ContainsKey(ltuple)){
                                        mSharedVertices.Add(ltuple, new MVpairs() );
                                    }
                                    mSharedVertices[ltuple].Add(rtuple);
                                    haveSv = true;
                                }
                            }
                        }
                    }
                }
                return haveSv;
            }
            catch (Exception ex)
            {
                string msg = string.Format(" mesh {0} - mesh {1} ", lmesh.Name, rmesh.Name);
                throw new Exception(msg,ex);
            }
        }
#elif SMUG
        /// <summary>
        /// 指定インデックスのメッシュの間に共有される頂点がないか？検査します
        /// </summary>
        /// <param name="left"></param>
        /// <param name="right"></param>
        /// <returns></returns>
        private bool ExistCorrespondVertices(int left, Dictionary<string, ResizableList<bool>> useflag)
        {
            Mesh lmesh = null;
            try
            {
                lmesh = mSourceMeshList[left];
                MeshSourceBase lposlist = lmesh.GetSource(SourceType.Position, null);

                Vector3 lv3 = new Vector3();
                Vector3 lunit = new Vector3(Threshold, Threshold, Threshold);
                bool haveSv = false;
                BoundingBox lbox = new BoundingBox();

                ResizableList<UniformGridNode<MVtx>> tmpNodes = new ResizableList<UniformGridNode<MVtx>>();
                for (int pi = 0; pi < lposlist.Count; pi++)
                {
                    MeshSourceDblCtrler.GetAsVec3(lposlist, pi, ref lv3);
                    lbox.Reset();
                    lbox.Update(lv3 - lunit);
                    lbox.Update(lv3 + lunit);

                    tmpNodes.Clear();
                    mUgrid.WindowQueryForBB(tmpNodes, lbox.Min, lbox.Max);
                    foreach (var otcreeNode in tmpNodes)
                    {
                        foreach (var mvtx in otcreeNode.Objects)
                        {
                            if (mvtx.Item1 == lmesh.Name || useflag[mvtx.Item1][mvtx.Item2]) { continue; }

                            Vector3 dif = lv3 - mvtx.Item3;
                            // 一致する頂点がある
                            if (dif.Norm() <= Threshold)
                            {
                                // 各メッシュの共有頂点情報を作る
                                MVpair ltuple = Tuple.Create(lmesh.Name, pi);
                                MVpair rtuple = Tuple.Create(mvtx.Item1, mvtx.Item2);

                                try
                                {
                                    // 共有頂点情報をリストに追加します。
                                    if (!mSharedVertices.ContainsKey(ltuple))
                                    {
                                        useflag[lmesh.Name][pi] = true;
                                        mSharedVertices.Add(ltuple, new MVpairs());
                                    }

                                    useflag[mvtx.Item1][mvtx.Item2] = true;
                                    mSharedVertices[ltuple].Add(rtuple);
                                }
                                catch (Exception ex)
                                {
                                    throw ex;
                                }
                                haveSv = true;
                            }
                        }
                    }
                }
                return haveSv;
            }
            catch (Exception ex)
            {
                string msg = string.Format("error at mesh {0}  ", lmesh.Name);
                throw new Exception(msg, ex);
            }
        }
#else
        /// <summary>
        /// 指定インデックスのメッシュの間に共有される頂点がないか？検査します
        /// </summary>
        /// <param name="left"></param>
        /// <param name="right"></param>
        /// <returns></returns>
        private bool ExistCorrespondVertices(int left, Dictionary<string, ResizableList<bool>> useflag)
        {
            Mesh lmesh = null;
            try
            {
                lmesh = mSourceMeshList[left];
                MeshSourceBase lposlist = lmesh.GetSource(SourceType.Position, null);

                Vector3 lv3 = new Vector3();
                Vector3 lunit = new Vector3(Threshold, Threshold, Threshold);
                bool haveSv = false;
                BoundingBox lbox = new BoundingBox();

                ResizableList<OctreeNode<MVtx>> tmpNodes = new ResizableList<OctreeNode<MVtx>>();
                for (int pi = 0; pi < lposlist.Count; pi++)
                {
                    MeshSourceDblCtrler.GetAsVec3(lposlist, pi, ref lv3);
                    lbox.Reset();
                    lbox.Update(lv3 - lunit);
                    lbox.Update(lv3 + lunit);

                    tmpNodes.Clear();
                    mOctree.WindowQueryForBB(tmpNodes, lbox.Min, lbox.Max);
                    foreach (var otcreeNode in tmpNodes)
                    {
                        foreach (var mvtx in otcreeNode.Objects)
                        {
                            if (mvtx.Item1 == lmesh.MeshName || useflag[mvtx.Item1][mvtx.Item2]) { continue; }

                            Vector3 dif = lv3 - mvtx.Item3;
                            // 一致する頂点がある
                            if (dif.Norm() <= Threshold)
                            {
                                // 各メッシュの共有頂点情報を作る
                                MVpair ltuple = Tuple.Create(lmesh.MeshName, pi);
                                MVpair rtuple = Tuple.Create(mvtx.Item1, mvtx.Item2);

                                try
                                {
                                    // 共有頂点情報をリストに追加します。
                                    if (!mSharedVertices.ContainsKey(ltuple))
                                    {
                                        useflag[lmesh.MeshName][pi] = true;
                                        mSharedVertices.Add(ltuple, new MVpairs());
                                    }

                                    useflag[mvtx.Item1][mvtx.Item2] = true;
                                    mSharedVertices[ltuple].Add(rtuple);
                                }
                                catch (Exception ex)
                                {
                                    throw ex;
                                }
                                haveSv = true;
                            }
                        }
                    }
                }
                return haveSv;
            }
            catch (Exception ex)
            {
                string msg = string.Format("Error is found while processing mesh {0}  ", lmesh.MeshName);
                throw ExcepHandle.CreateException(msg, ex);
            }
        }
#endif

        /// <summary>
        /// 各メッシュが、どのメッシュと隣接しているか？を取得します
        /// </summary>
        private void PickNeighborMesh(ref ResizableList<ResizableList<int>> neighborList)
        {
            try
            {
                neighborList = new ResizableList<ResizableList<int>>();
                neighborList.Resize(mSourceMeshList.Count, null);

                for (int i = 0; i < mSourceMeshList.Count; i++)
                {
                    neighborList[i] = new ResizableList<int>();
                }
            }
            catch (Exception ex)
            {
                string msg = string.Format("PickNeighborMesh:: fail to create neighbor mesh index list");
                throw ExcepHandle.CreateException(msg, ex);
            }

            try
            {
                // 隣接を調査する必要がない場合
                if (mSourceMeshList.Count == 1) { return; }

                for (int i = 0; i < mSourceMeshList.Count - 1; i++)
                {
                    Mesh imesh = mSourceMeshList[i];
                    for (int j = i + 1; j < mSourceMeshList.Count; j++)
                    {
                        Mesh jmesh = mSourceMeshList[j];

                        // バウンディングボックスが交差している場合
                        if (imesh.GetTotalBoundingBox().Intersect(jmesh.GetTotalBoundingBox()))
                        {
                            neighborList[i].Add(j);
                            neighborList[j].Add(i);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /// <summary>
        /// メッシュの境界ボリュームを作成します
        /// </summary>
        private void ComputeMeshBoundingBox()
        {
            try
            {
                for (int i = 0; i < mCloneMeshList.Count; i++)
                {
                    mCloneMeshList[i].ComputeBounds();
                }
            }
            catch
            {
                throw ExcepHandle.CreateException(@"Faile to Compute MeshBoundingBox");
            }
        }

        /// <summary>
        /// 全体のBoundingBoxを計算します
        /// </summary>
        /// <returns></returns>
        private BoundingBox ComputeTotalBB()
        {
            BoundingBox result = new BoundingBox();
            foreach (var mesh in mSourceMeshList)
            {
                if (mesh.FaceN != 0)
                {
                    mesh.ComputeMeshBounding();
                    result.Update(mesh.GetTotalBoundingBox().Min);
                    result.Update(mesh.GetTotalBoundingBox().Max);
                }
            }
            return result;
        }

#if SMSPHASH
        /// <summary>
        /// 各メッシュの位置座標を元にハッシュリスト作成
        /// </summary>
        private void CreatePosHashTable()
        {
            mMeshPosHashTable = new ResizableList<SpacialHashList<int>>();
            mMeshPosHashTable.Resize(mSourceMeshList.Count, null);
            for (int i = 0; i < mSourceMeshList.Count; i++){
                mMeshPosHashTable[i] = new SpacialHashList<int>();
                mMeshPosHashTable[i].Count = mHashTableSize;
            }

            // ハッシュリストを作る必要がない
            if (mSourceMeshList.Count == 1) { return; }

            for (int i = 0; i < mCloneMeshList.Count; i++)
            {
                try
                {
                    SpacialHashList<int> hashList = mMeshPosHashTable[i];
                    if (hashList == null)
                        throw new NullReferenceException("CreatePosHashTable :: hash table is null");

                    hashList.Clear();

                    Mesh mesh = mSourceMeshList[i];
                    MeshSourceBase source = mesh.GetSource(SourceType.Position, null);
                    Vector3 tv3 = new Vector3();
                    int posN = mesh.VertexN;

                    for (int vi = 0; vi < posN; vi++)
                    {
                        MeshSourceDblCtrler.GetAsVec3(source, vi, ref tv3);
                        hashList.Add(tv3, vi);
                    }
                }
                catch(Exception ex)
                {
                    //ex.Data.Add("smesh", mSourceMeshList[i]);
                    string msg = string.Format(@"Fail to compute hash for mesh {0}/{1}", i, mSourceMeshList[i].Name);
                    throw new Exception(msg,ex);
                }
            }
        }
#elif SMUG
        private void CreateUniformGrid()
        {
            BoundingBox fullBB = ComputeTotalBB();
            fullBB.Stretch(0.1);
            mUgrid.Initialize(fullBB,10,10,10);

            for (int i = 0; i < mCloneMeshList.Count; i++)
            {
                try
                {
                    Mesh mesh = mSourceMeshList[i];
                    MeshSourceBase source = mesh.GetSource(SourceType.Position, null);
                    int posN = mesh.VertexN;

                    BoundingBox lbox = new BoundingBox();
                    Vector3 lunit = new Vector3(Threshold, Threshold, Threshold);
                    for (int vi = 0; vi < posN; vi++)
                    {
                        Vector3 tv3 = new Vector3();
                        MeshSourceDblCtrler.GetAsVec3(source, vi, ref tv3);
                        mUgrid.AddObject(Tuple.Create(mesh.Name, vi, tv3), tv3);
                    }
                }
                catch (Exception ex)
                {
                    //ex.Data.Add("smesh", mSourceMeshList[i]);
                    string msg = string.Format(@"Fail to compute hash for mesh {0}/{1}", i, mSourceMeshList[i].Name);
                    throw new Exception(msg, ex);
                }
            }
        }
#else

        /// <summary>
        ///
        /// </summary>
        private void CreateOctree()
        {
            BoundingBox fullBB = ComputeTotalBB();
            mOctree.Create(5, fullBB.Min, fullBB.Max);

            for (int i = 0; i < mCloneMeshList.Count; i++)
            {
                try
                {
                    Mesh mesh = mSourceMeshList[i];
                    MeshSourceBase source = mesh.GetSource(SourceType.Position, null);
                    int posN = mesh.VertexN;

                    BoundingBox lbox = new BoundingBox();
                    Vector3 lunit = new Vector3(Threshold, Threshold, Threshold);
                    for (int vi = 0; vi < posN; vi++)
                    {
                        Vector3 tv3 = new Vector3();
                        MeshSourceDblCtrler.GetAsVec3(source, vi, ref tv3);
                        lbox.Reset();
                        lbox.Update(tv3 - lunit);
                        lbox.Update(tv3 + lunit);
                        mOctree.AddObject(Tuple.Create(mesh.MeshName, vi, tv3), lbox);
                    }
                }
                catch (Exception ex)
                {
                    //ex.Data.Add("smesh", mSourceMeshList[i]);
                    string msg = string.Format(@"Fail to compute hash for mesh {0}/{1}", i, mSourceMeshList[i].MeshName);
                    throw ExcepHandle.CreateException(msg, ex);
                }
            }
        }

#endif

        /// <summary>
        /// 頂点同士を結合します
        /// </summary>
        private void MergeClosedVertex()
        {
            foreach (var mesh in mSourceMeshList)
            {
                try
                {
                    ClosedVtxMarger cvm = new ClosedVtxMarger(mesh);
                    cvm.Percentage = Percentage;
                    cvm.SetSpaceInfo(10, 10, 10);
                    cvm.Run();
                }
                catch (Exception ex)
                {
                    int i = mSourceMeshList.IndexOf(mesh);
                    ex.Data.Add("smesh", mesh);
                    ex.Data.Add("cmesh", mCloneMeshList[i]);

                    string msg = string.Format("Failure MargeClosedVertex {0}[{1}]", i, mesh.MeshName);
                    throw ExcepHandle.CreateException(msg, ex);
                }
            }
        }

        /// <summary>
        /// クローンのリストを元に戻す
        /// </summary>
        private void RestoreCloneList()
        {
            try
            {
                mSourceMeshList.Clear();
                foreach (var mesh in mCloneMeshList)
                {
                    mSourceMeshList.Add(mesh);
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("CloneList", ex);
            }
        }

        /// <summary>
        /// メッシュのクローンリストを作成する
        /// </summary>
        /// <returns></returns>
        private void CloneList()
        {
            try
            {
                mCloneMeshList = new ResizableList<Mesh>();
                foreach (var mesh in mSourceMeshList)
                {
                    mCloneMeshList.Add(mesh.Clone());
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("CloneList", ex);
            }
        }

        /// <summary>
        /// 失敗時にクローンから元のメッシュに戻す
        /// </summary>
        /// <returns></returns>
        private void RevertCloneList()
        {
            try
            {
                int i = 0;
                foreach (var clonemesh in mCloneMeshList)
                {
                    mSourceMeshList[i] = clonemesh;
                    i++;
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("CloneList", ex);
            }
        }
    }
}
