﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using nw4f.meshlib;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public partial class QemSimp
    {
        private const double SmallDeviation = 1e-3;

        /// <summary>
        /// ブレンドウェイトの４変数の分散を求める
        /// １－３変数の場合も4変数扱いにして、残りを０とみなすことで
        /// 分散の値を大きくしたい狙い
        /// この場合、統計的な価値はないので偏差などに意味は無いが、重みとして利用する
        /// </summary>
        /// <param name="x1">ブレンドウェイト</param>
        /// <param name="x2">ブレンドウェイト</param>
        /// <param name="x3">ブレンドウェイト</param>
        /// <param name="x4">ブレンドウェイト</param>
        /// <param name="stride"> ブレンドウェイトの数 </param>
        /// <returns></returns>
        private double ComputeMotionVariation(double x1 = 0.0, double x2 = 0.0, double x3 = 0.0, double x4 = 0.0, int stride = 4)
        {
            if (stride == 0)
            {
                throw ExcepHandle.CreateException("Invalid stride");
            }

            double avg = (x1 + x2 + x3 + x4) / ((double)stride);
            if (avg <= 1e-5)
            {
                return 1.0;
            }

            double[] values = new double[4] { x1, x2, x3, x4 };
            double sum = 0;
            for (int i = 0; i < stride; i++)
            {
                double tmp = values[i] - avg;
                tmp = tmp * tmp;
                sum += tmp;
            }
            sum = sum / ((double)stride);
            if (sum < SmallDeviation)
            {
                sum = SmallDeviation;
            }
            return sum;
        }

        /// <summary>
        /// 引数で与えられたBlendWeightから、モーションの重みを分散を用いて計算する。
        /// </summary>
        /// <param name="x1"></param>
        /// <param name="x2"></param>
        /// <param name="x3"></param>
        /// <param name="x4"></param>
        /// <param name="stride"></param>
        /// <returns></returns>
        private double ComputeMotionWeight(double x1 = 0.0, double x2 = 0.0, double x3 = 0.0, double x4 = 0.0, int stride = 4)
        {
            double variation = ComputeMotionVariation(x1, x2, x3, x4, stride);
            variation = (1.0 / variation) + 10.0;
            //次数の最大値を抑えたい
            variation = Math.Max(1.0, Math.Log10(variation));

            double basenum = PropWaits[(int)SourceType.BlendWeight];
            if (basenum < SmallDeviation)
            {
                basenum = SmallDeviation;
            }
            if (basenum > Math.E)
            {
                basenum = Math.E;
            }

            double result = Math.Pow(basenum, variation) + Math.E;
            // 1以下にはしないが、最大重みを丸めておく
            // SmallDeviationを1に近づけるほど、最大重みは小さくなるので必要に応じて調整すればよい
            result = Math.Max(1.0, Math.Log(result));
            return result;
        }

        /// <summary>
        /// 指定頂点のBlendWeightを使った重みの計算を行う
        /// 重みは各フェース毎の最大値を使う
        /// （大体の場合は全て同じと思われるが、念のため全部計算する）
        /// </summary>
        /// <param name="vtm">共有フェースリスト</param>
        /// <param name="vid">頂点ID</param>
        /// <returns>重み</returns>
        private double ComputeMotionWeight(VtxTriMesh vtm, int vid)
        {
            double result = 0.0f;
            int sN = mMesh.SourceN;
            int po = mMesh.GetSourceOfset(SourceType.Position, null);

            List<MeshSourceBase> sources = mMesh.GetSources(SourceType.BlendWeight);
            // ない場合、0.0 が返るとリダクションしない
            if (sources.Count == 0)
            {
                return 1.0;
            }
            foreach (var fid in vtm.VTL[vid].TriList)
            {
                int vo = mMesh.mVoffs[fid];
                int vn = mMesh.mVnums[fid];
                foreach (var source in sources)
                {
                    int so = mMesh.GetSourceOfset(source.Type, source.Name);
                    for (int vi = 0; vi < vn; vi++)
                    {
                        int posid = mMesh.mVbuffs[(vo + vi) * sN + po];
                        if (posid == vid)
                        {
                            int sid = mMesh.mVbuffs[(vo + vi) * sN + so];
                            double[] tmp = new double[4] { 0.0, 0.0, 0.0, 0.0 };
                            for (int si = 0; si < source.Stride; si++)
                            {
                                double val = Convert.ToDouble(source.GetRowValue((source.Stride * sid) + si));
                                tmp[si] = val;
                            }
                            double weight = ComputeMotionWeight(tmp[0], tmp[1], tmp[2], tmp[3]);
                            result = Math.Max(weight, result);
                        }
                    }
                }
            }
            return result;
        }
    }
}
