﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using MS.Internal.Xml.XPath;
using nw4f.tinymathlib;
using System.Diagnostics;
using System.Linq;
using System.ServiceModel.Configuration;
using nw.g3d.iflib;
using Vector2 = nw4f.tinymathlib.Vector2;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    public partial class QemSimp
    {
        /// <summary>
        /// 最適化位置の計算が可能な関数であるかどうか？
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        private bool IsEnable_OptSpace_EdgeType(int type)
        {
            if (IsEdgeType(type, QemEdgeType.NonManifold) ||
                 IsEdgeType(type, QemEdgeType.TouchedHard) ||
                 IsEdgeType(type, QemEdgeType.HardEdgeOT))
            {
                return false;
            }
            return true;
        }

        private bool IsEnable_OptEdge_EdgeType(int type)
        {
            if (IsEdgeType(type, QemEdgeType.PropertyEdge) ||
                 IsEdgeType(type, QemEdgeType.HardEdge))
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// ZeroOne時の最適化位置を計算します
        /// </summary>
        /// <param name="quad"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="weight"></param>
        /// <param name="hd"></param>
        private void ComputeShapeOP_ZeroOne
            (
            Quadric quad,
            VectorN v0,
            VectorN v1,
            double weight,
            QemHeapData hd)
        {
            // ZeroOne でコストを求める
            // 境界線の場合、境界線の頂点をキープ側にしたい
            double c0 = double.MaxValue;
            double c1 = double.MaxValue;
            // 各頂点へとシュリンクするのにかかるコスト
            c0 = quad.Evaluate(v0, weight);
            if (c0 < 0)
            {
                c0 = double.MaxValue;
            }
            c1 = quad.Evaluate(v1, weight);
            if (c1 < 0)
            {
                c1 = double.MaxValue;
            }
            hd.VCost = Math.Min(c0, c1);

            if (c0 <= c1)
            {
                hd.KeepSide = 0;
                hd.InterpolateParam = 0.0f;
                hd.NewPosition0.Set(v0);
            }
            else
            {
                hd.InterpolateParam = 1.0;
                hd.KeepSide = 1;
                hd.NewPosition0.Set(v1);
            }
        }

        /// <summary>
        /// Optimized時に最適化位置を計算します
        /// </summary>
        /// <param name="quad"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="weight"></param>
        /// <param name="hd"></param>
        private void ComputeShapeOP_OptEdge
            (
            Quadric quad,
            VectorN v0,
            VectorN v1,
            double weight,
            QemHeapData hd)
        {
            double c0 = double.MaxValue;
            hd.InterpolateParam = 0.0f;
            hd.VCost = double.MaxValue;
            int maxV = 16;

            VectorN tv = new VectorN(v0.Dim);
            VectorN nv = new VectorN(v0.Dim);
            for (int i = 0; i <= maxV; i++)
            {
                double p = (double)i / maxV;

                for (int j = 0; j < v0.Dim; j++)
                {
                    tv[j] = v0[j] + (v1[j] - v0[j]) * p;
                }

                c0 = quad.Evaluate(tv, weight);
                if ((c0 < hd.VCost) && c0 >= 0)
                {
                    hd.VCost = c0;
                    hd.InterpolateParam = p;
                    nv.Set(tv);
                }
            }
            hd.NewPosition0.Set(nv);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="pos"></param>
        /// <param name="fid"></param>
        /// <param name="alpha"></param>
        /// <param name="beta"></param>
        /// <param name="gamma"></param>
        private void GetBarycentricPos(ref VectorN pos, WingedEdge edge, int fid, double alpha, double beta, double gamma)
        {
            int[] ids = new int[3];
            double[] coef = { alpha, beta, gamma };

            mMesh.GetFaceVtxIndices(fid, ref ids, edge.SpaceKey.type, edge.SpaceKey.name);

            MeshSourceBase source = mMesh.GetSource(edge.SpaceKey.type, edge.SpaceKey.name);
            VectorN tmp = new VectorN((uint)source.Stride);

            pos.SetZero();
            for (int i = 0; i < 3; i++)
            {
                MeshSourceDblCtrler.GetAsVecN(source, ids[i], source.Stride, ref tmp);
                pos += tmp * coef[i];
            }
        }

        /// <summary>
        /// 重心座標による最適化位置を求める
        /// </summary>
        /// <param name="quad"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="weight"></param>
        /// <param name="hd"></param>
        private void ComputeShapeOP_OptArea
             (
             Quadric quad,
             WingedEdge edge,
             VectorN v0,
             VectorN v1,
             double weight,
             QemHeapData hd)
        {
            double c0 = double.MaxValue;
            hd.InterpolateParam = 0.0f;
            hd.VCost = double.MaxValue;
            int maxV = 8;

            VectorN tv = new VectorN(v0.Dim);
            VectorN nv = new VectorN(v0.Dim);
            for (int i = 0; i <= maxV; i++)
            {
                double alpha = (double)i / (double)maxV;
                for (int j = 0; j <= maxV; j++)
                {
                    double beta = (1.0 - alpha) * ((double)j / (double)maxV);
                    double gamma = 1.0 - alpha - beta;
                    int[] fid = { edge.Face0, edge.Face1 };

                    for (int k = 0; k < 2; k++)
                    {
                        GetBarycentricPos(ref tv, edge, fid[k], alpha, beta, gamma);
                        c0 = quad.Evaluate(tv, weight);
                        if ((c0 < hd.VCost) && c0 >= 0)
                        {
                            hd.VCost = c0;
                            hd.InterpolateParam = 0.5;
                            nv.Set(tv);
                        }
                    }
                }
            }
            hd.NewPosition0.Set(nv);
        }

        /// <summary>
        /// 中心点に最適化位置を持ってくる
        /// </summary>
        /// <param name="quad"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="weight"></param>
        /// <param name="hd"></param>
        private void ComputeShapeOP_Cnt
            (
            Quadric quad,
            VectorN v0,
            VectorN v1,
            double weight,
            QemHeapData hd)
        {
            hd.InterpolateParam = 0.5;
            hd.NewPosition0.Set(v0 + ((v1 - v0) * hd.InterpolateParam));
            hd.VCost = _tmpQ.Evaluate(hd.NewPosition0, weight);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="quad"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="weight"></param>
        /// <param name="hd"></param>
        private void ComputeShapeOP_Cal
            (
            Quadric quad,
            VectorN kp,
            double weight,
            QemHeapData hd)
        {
            var c0 = double.MaxValue;
            VectorN tp = null;

            hd.InterpolateParam = 0.5;
            if (quad.Optimize(ref tp))
            {
                c0 = quad.Evaluate(tp, weight);
                hd.NewPosition0.Set(tp);
            }
            else
            {
                c0 = quad.Evaluate(kp, weight);
                hd.NewPosition0.Set(kp);
            }
            if (c0 < 0)
            {
                c0 = double.MaxValue;
            }
            hd.VCost = c0;
        }

        /// <summary>
        /// 指定された計算タイプに応じた最適化位置の計算を行います
        ///
        /// </summary>
        /// <param name="policy"></param>
        /// <param name="edge"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="kp"></param>
        /// <param name="weight"></param>
        /// <param name="hd"></param>
        private void ComputeOptimizedPos(
            QemOptPos policy,
            WingedEdge edge,
            int edgeType,
            VectorN v0,
            VectorN v1,
            VectorN kp,
            double weight,
            QemHeapData hd)
        {
            switch (policy)
            {
                case QemOptPos.ZeroOne:
                    ComputeShapeOP_ZeroOne(_tmpQ, v0, v1, weight, hd);
                    break;
                case QemOptPos.OptimizeEdge:
                    if (IsEnable_OptSpace_EdgeType(edgeType))
                    {
                        ComputeShapeOP_OptEdge(_tmpQ, v0, v1, weight, hd);
                    }
                    break;
                case QemOptPos.OptimizeArea:
                    if (IsEnable_OptSpace_EdgeType(edgeType))
                    {
                        // ハードエッジの縮退時には、エッジ上の縮退を行う
                        if (IsEnable_OptEdge_EdgeType(edgeType))
                        {
                            ComputeShapeOP_OptEdge(_tmpQ, v0, v1, weight, hd);
                        }
                        else
                        {
                            ComputeShapeOP_OptArea(_tmpQ, edge, v0, v1, weight, hd);
                        }
                    }
                    break;
                case QemOptPos.Center:
                    if (IsEnable_OptSpace_EdgeType(edgeType))
                    {
                        ComputeShapeOP_Cnt(_tmpQ, v0, v1, weight, hd);
                    }
                    break;
                case QemOptPos.CalcOptpos:
                    // ハードエッジの縮退時には、エッジ上の縮退を行う
                    if (IsEnable_OptEdge_EdgeType(edgeType))
                    {
                        ComputeShapeOP_OptEdge(_tmpQ, v0, v1, weight, hd);
                    }
                    else if (IsEnable_OptSpace_EdgeType(edgeType))
                    {
                        ComputeShapeOP_Cal(_tmpQ, kp, weight, hd);
                    }
                    break;
                default:
                    if (IsEnable_OptSpace_EdgeType(edgeType))
                    {
                        ComputeShapeOP_OptEdge(_tmpQ, v0, v1, weight, hd);
                    }
                    break;
            }
        }

        /// <summary>
        /// 特殊な条件下の場合の頂点およびコストの計算を事前に行う
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="border"></param>
        /// <param name="edgeType"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="weight"></param>
        /// <param name="hd"></param>
        /// <returns>true:最適化計算の必要がある。false:最適化計算の必要がない</returns>
        private bool PresetShapeOptimizedPos(
            WingedEdge edge,
            int[] border,
            int edgeType,
            VectorN v0,
            VectorN v1,
            double weight,
            ref QemHeapData hd)
        {
            // ZeroOne でコストを求める
            // 境界線の場合、境界線の頂点をキープ側にしたい
            hd.KeepSide = 0;
            hd.InterpolateParam = 0.0f;
            hd.VCost = double.MaxValue;

            if (IsEdgeType(edgeType, QemEdgeType.NonManifold))
            {
                hd.KeepSide = 0;
                hd.InterpolateParam = 0.0f;
                hd.VCost = double.MaxValue;
                hd.NewPosition0.Set(v0);
                return false;
            }
            else if (IsEdgeType(edgeType, QemEdgeType.TouchedHard) &&
                     !IsEdgeType(edgeType, QemEdgeType.HardEdge) &&
                     !IsEdgeType(edgeType, QemEdgeType.HardEdgeOT))
            {
                var and = border[edge.Vertex1] & border[edge.Vertex0];
                var fl0 = border[edge.Vertex0] & (~and);
                var fl1 = border[edge.Vertex1] & (~and);
                var cost0 = 0.0;
                var cost1 = 0.0;
                var quadv0 = Math.Abs(_tmpQ.Evaluate(v0, weight)) + 1e-7; // 0 にしないために下駄をはかせる
                var quadv1 = Math.Abs(_tmpQ.Evaluate(v1, weight)) + 1e-7; // 0 にしないために下駄をはかせる

                foreach (QemVertexType borderType in Enum.GetValues(typeof(QemVertexType)))
                {
                    if (borderType == QemVertexType.Inner)
                    {
                        continue;
                    }
                    if (((int)borderType & (int)QemVertexType.TouchMask) == 0)
                    {
                        continue;
                    }

                    if (IsBorder(fl1, borderType))
                    {
                        cost1 += quadv1;
                        if (borderType == QemVertexType.SubMesh || borderType == QemVertexType.Exclude)
                        {
                            cost1 = double.MaxValue / 10.0;
                        }
                    }
                    else if (IsBorder(fl0, borderType))
                    {
                        cost0 += quadv0;
                        if (borderType == QemVertexType.SubMesh || borderType == QemVertexType.Exclude)
                        {
                            cost0 = double.MaxValue / 10.0;
                        }
                    }
                }

                // いずれかの頂点が、どのような境界上の頂点でもない場合のみに強制的に頂点位置を決めていい
                if (cost0 < cost1 && cost0 < double.Epsilon)
                {
                    hd.KeepSide = 1;
                    hd.InterpolateParam = 1.0f;
                    hd.VCost = quadv1;
                    hd.NewPosition0.Set(v1);
                }
                else if (cost0 > cost1 && cost1 < double.Epsilon)
                {
                    hd.KeepSide = 0;
                    hd.InterpolateParam = 0.0f;
                    hd.VCost = quadv0;
                    hd.NewPosition0.Set(v0);
                }
                else
                {
                    hd.KeepSide = 0;
                    hd.InterpolateParam = 0.0f;
                    hd.VCost = double.MaxValue;
                    hd.EdgeType |= (int)QemEdgeType.NonManifold;
                    hd.NewPosition0.Set(v0);
                }
                return false;
            }
            else
            {
                double c0 = double.MaxValue;
                double c1 = double.MaxValue;

                // 各頂点へとシュリンクするのにかかるコスト
                c0 = _tmpQ.Evaluate(v0, weight);
                if (c0 < 0)
                {
                    c0 = double.MaxValue;
                }
                c1 = _tmpQ.Evaluate(v1, weight);
                if (c1 < 0)
                {
                    c1 = double.MaxValue;
                }
                hd.VCost = Math.Min(c0, c1);

                if (c0 <= c1)
                {
                    hd.KeepSide = 0;
                    hd.InterpolateParam = 0.0f;
                    hd.NewPosition0.Set(v0);
                }
                else
                {
                    hd.InterpolateParam = 1.0;
                    hd.KeepSide = 1;
                    hd.NewPosition0.Set(v1);
                }
            }
            return true;
        }

        private Quadric _tmpQ4F = null;
        /// <summary>
        /// 特徴的な稜線（法線考慮時）を考慮して
        /// コストと最適位置の再計算を行う。法線間の角度で重みづけを行う
        /// </summary>
        /// <param name="edge">エッジ</param>
        /// <param name="hd">簡略化用ヒープデータ</param>
        /// <param name="constraintAngle">閾値となる角度、コストの重みに利用する</param>
        private void ComputeFeatureCostAndPos
            (
                WingedEdge edge,
                int[] border,
                QemHeapData hd,
                double constraintAngle)
        {
            if (!IsNormalEnabled || Math.Abs(double.MaxValue - hd.VCost) < 1e-8)
            {
                return;
            }

            if (IsEdgeType(hd.EdgeType, QemEdgeType.HardEdge) ||
                IsEdgeType(hd.EdgeType, QemEdgeType.HardEdgeOT) ||
                IsEdgeType(hd.EdgeType, QemEdgeType.NonManifold) ||
                IsEdgeType(hd.EdgeType, QemEdgeType.TouchedHard))
            {
                return;
            }

            var vi0 = edge.Vertex0;
            var vi1 = edge.Vertex1;
            var gpv = new Vector3();
            var up = new Vector3();

            gpv.SetZero();
            // gradient を求める
            foreach (var tid in VtxTriMesh.VTL[vi0].TriList)
            {
                if (tid == edge.Face0 || tid == edge.Face1)
                {
                    continue;
                }
                var w = mMesh.mFaceArea[tid] / 3.0;
                gpv += mMesh.mFaceNormal[tid] * w;
            }

            foreach (var tid in VtxTriMesh.VTL[vi1].TriList)
            {
                if (tid == edge.Face0 || tid == edge.Face1)
                {
                    continue;
                }
                var w = mMesh.mFaceArea[tid] / 3.0;
                gpv += mMesh.mFaceNormal[tid] * w;
            }

            if (edge.Face0 != int.MaxValue)
            {
                var w = mMesh.mFaceArea[edge.Face0] / 3.0;
                gpv += mMesh.mFaceNormal[edge.Face0] * w;
            }
            if (edge.Face1 != int.MaxValue)
            {
                var w = mMesh.mFaceArea[edge.Face1] / 3.0;
                gpv += mMesh.mFaceNormal[edge.Face1] * w;
            }

            up[0] = hd.NewPosition0[0];
            up[1] = hd.NewPosition0[1];
            up[2] = hd.NewPosition0[2];
            var dp = -gpv.Dot(up);

            // gradient を追加して、最適位置とコストの再計算を行う
            if (_tmpQ4F == null)
            {
                _tmpQ4F = new Quadric(4, 0.0);
            }
            _tmpQ4F.Clear();
            _tmpQ4F.Set(_tmpQ);
            _tmpQ4F.AddFeature(gpv, -dp);

            VectorN newPos = new VectorN(4);
            if (OptPosPolicy == QemOptPos.CalcOptpos)
            {
                if (!_tmpQ4F.Optimize(ref newPos))
                {
                    newPos = new VectorN(4);
                    Array.Copy(hd.NewPosition0.Values, newPos.Values, 3);
                    newPos[3] = NormalWeight;
                }
                else
                {
                    // Normal Weight で、位置の大幅な変動は制御可能
                    Array.Copy(newPos.Values, hd.NewPosition0.Values, 3);
                    newPos[3] = NormalWeight;
                }
            }
            else
            {
                Array.Copy(hd.NewPosition0.Values, newPos.Values, 3);
                newPos[3] = NormalWeight;
            }

            var tmpCost = _tmpQ4F.Evaluate(newPos, 1.0);

            var weight = Math.Cosh(constraintAngle);
            if (tmpCost > 0.0)
            {
                hd.VCost = tmpCost * weight;
            }
        }

        /// <summary>
        /// 形状の体積変化のコストを計算します
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="border"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="hd"></param>
        private void ComputeShapeVolumeCostAndPos
            (
            WingedEdge edge,
            int[] border,
            double cangle,
            QemHeapData hd)
        {
            hd.VCost = double.MaxValue;
            if (edge.Vertex0 == int.MaxValue || edge.Vertex1 == int.MaxValue)
            {
                return;
            }

            var poslist = mMesh.GetSource(edge.SpaceKey.type, edge.SpaceKey.name);
            var tp0 = new VectorN(3);
            var tp1 = new VectorN(3);
            MeshSourceDblCtrler.GetAsVecN(poslist, edge.Vertex0, 3, ref tp0);
            MeshSourceDblCtrler.GetAsVecN(poslist, edge.Vertex1, 3, ref tp1);

            var weight = 1.0;
            if (!IsEdgeType(hd.EdgeType, QemEdgeType.NonManifold))
            {
                // Weight の計算
                // モーションに関するウェイトの計算
                if (IsRequiredComputingWeight)
                {
                    weight = ComputeQuadWeight(VtxTriMesh, edge.Vertex1) + ComputeQuadWeight(VtxTriMesh, edge.Vertex0);
                }

                // ハードエッジの場合の場合は接続状態で重みを計算する
                if (IsEdgeType(hd.EdgeType, QemEdgeType.HardEdge) || IsEdgeType(hd.EdgeType, QemEdgeType.TouchedHard))
                {
                    double cc = Math.Cosh(cangle * Math.PI / 180.0f);
                    weight *= cc;
                }
            }

            hd.VWeight = weight;
            _tmpQ = mQuads[edge.Vertex0] + mQuads[edge.Vertex1];
            var cont = PresetShapeOptimizedPos(
                  edge,
                  border,
                  hd.EdgeType,
                  tp0,
                  tp1,
                  weight,
                  ref hd);

            if (cont)
            {
                VectorN kp = (hd.KeepSide == 0) ? tp0 : tp1;
                ComputeOptimizedPos(
                    OptPosPolicy,
                    edge,
                    hd.EdgeType,
                    tp0,
                    tp1,
                    kp,
                    weight,
                    hd);
            }

            // ジオメトリの最大最小コストを得る
            var vCostBody = hd.VCost / weight;
            if (Math.Abs(hd.VCost - double.MaxValue) > double.Epsilon
                && !double.IsNaN(hd.VCost)
                && !double.IsInfinity(hd.VCost))
            {
                if (edge.Face0 != int.MaxValue)
                {
                    UpdateMinMaxCost(vCostBody, hd.EdgeType, edge.Face1);
                }
                if (edge.Face1 != int.MaxValue)
                {
                    UpdateMinMaxCost(vCostBody, hd.EdgeType, edge.Face1);
                }
            }
        }

        /// <summary>
        /// UVのコストを計算して、新しいUV座標値も同時に求める
        /// 新しいUV座標値の計算に成功すればTRUE、そうでないならFALSE
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="border"></param>
        /// <param name="hd"></param>
        /// <returns></returns>
        private bool ComputeUVCost(
            WingedEdge edge,
            List<int[]> border,
            QemHeapData hd)
        {
            hd.TexCost = 0.0;
            var ci = 0;
            foreach (var cep in edge.ChildEdges)
            {
                if (cep.Key.type == SourceType.Texture)
                {
                    foreach (var ce in cep.Value)
                    {
                        if (edge.Face0 == ce.Face0)
                        {
                            if (!ComputeUVEdgeCostAndNewPos(ce, border[ci], hd.NewPosition[0], hd))
                            {
                                return false;
                            }
                        }
                        else
                        {
                            if (!ComputeUVEdgeCostAndNewPos(ce, border[ci], hd.NewPosition[1], hd))
                            {
                                return false;
                            }
                        }
                    }
                }
                ci++;
            }
            return true;
        }

        /// <summary>
        /// 新規ＵＶ座標を、全体の新規位置座標に代入
        /// </summary>
        /// <param name="edge">エッジ、スペースキーの為に</param>
        /// <param name="nv">コピー元：ＵＶの新規座標</param>
        /// <param name="v0">コピー先：全体の新規座標</param>
        private void SetUVNewPosToGlobalNewPos
            (
            WingedEdge edge,
            VectorN nv,
            VectorN v0)
        {
            int sourceN = mMesh.SourceN;
            int di = 0;
            for (int s = 0; s < sourceN; s++)
            {
                MeshSourceBase meshsrc = (MeshSourceBase)mMesh.Sources[s];
                if (mPropuse[(int)meshsrc.Type])
                {
                    if (v0.Dim < (di + meshsrc.Stride))
                    {
                        break;
                    }
                    if (edge.SpaceKey.Equals(meshsrc.Key))
                    {
                        for (int st = 0; st < meshsrc.Stride; st++)
                        {
                            v0[di + st] = nv[st];
                        }
                    }
                    di += meshsrc.Stride;
                }
            }
        }

        private const double MinLength = 1e-5;

        /// <summary>
        /// 指定のエッジの上に、指定された頂点が乗ってるか？
        /// </summary>
        /// <param name="st"></param>
        /// <param name="ed"></param>
        /// <param name="p"></param>
        /// <returns></returns>
        public static bool ComputeCollision(Vector3 st, Vector3 ed, Vector3 p)
        {
            var edgeDir = ed - st;
            var dir0 = p - st;
            var dir1 = p - ed;

            // 端点に一致してるので、乗っかってる
            if (dir0.Norm() < MinLength || dir1.Norm() < MinLength)
            {
                return true;
            }
            edgeDir.Normalize();
            dir0.Normalize();
            dir1.Normalize();

            // そもそも一直線でないか、直線の外にいるか
            var dot0 = dir0.Dot(dir1);
            if (Math.Abs(dot0 + 1.0) < MinLength)
            {
                return false;
            }

            dot0 = edgeDir.Dot(dir1);
            return Math.Abs(dot0 - 1.0) < MinLength;
        }

        /// <summary>
        /// 指定の点(p)が三角形(abc)の内部にあるか？
        /// 内部は線上も含む
        /// </summary>
        public static bool IsIncludedPointInTri(Vector3 a, Vector3 b, Vector3 c, Vector3 p, bool enableOnEdge = false)
        {
            var ab = b - a;
            var bc = c - b;
            var ca = a - c;
            var ap = p - a;
            var bp = p - b;
            var cp = p - c;

            var v1 = ab.Cross(bp);
            var v2 = bc.Cross(cp);
            var v3 = ca.Cross(ap);

            var lv1 = v1.Norm();
            if (lv1 <= MinLength)
            {
                return enableOnEdge;
            }
            var dot1 = v1.Dot(v2) / lv1;
            var dot2 = v1.Dot(v3) / lv1;
            if (enableOnEdge)
            {
                if (ComputeCollision(a, b, p) || ComputeCollision(b, c, p) || ComputeCollision(c, a, p))
                {
                    return true;
                }
            }
            if (dot1 >= 0.0 && dot2 >= 0.0)
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// newposをポリゴンtidがなす平面上に投射して、位置を得る
        /// 投射出来ない場合はnull を返す
        /// </summary>
        /// <param name="newpos"> 投射したい位置座標 </param>
        /// <param name="tid">　ポリゴンID </param>
        /// <param name="checkInFace">trueならポリゴンに含まれてる事をチェックする </param>
        /// <returns></returns>
        private Vector3 ProjectOnPlane(VectorN newpos, int tid, bool checkInFace)
        {
            var indices = new int[3];
            var positions = new Vector3[3];
            positions[0] = new Vector3();
            positions[1] = new Vector3();
            positions[2] = new Vector3();

            var nml = mMesh.mFaceNormal[tid];

            mMesh.GetFaceVtxIndices(tid, ref indices);
            mMesh.GetVertexPos(indices[0], ref positions[0]);
            mMesh.GetVertexPos(indices[1], ref positions[1]);
            mMesh.GetVertexPos(indices[2], ref positions[2]);

            var offset = -nml.Dot(positions[0]);
            var tmpv3 = new Vector3(newpos[0], newpos[1], newpos[2]);
            var dist = nml.Dot(tmpv3) + offset;

            // 平面投射 を行って平面上の一点を取得する
            // ポリゴン内に含まれていなくてもいい場合は、内外判定しない
            tmpv3 = tmpv3 - nml * dist;
            if (!checkInFace ||
                IsIncludedPointInTri(positions[0], positions[1], positions[2], tmpv3, true))
            {
                return tmpv3;
            }
            return null;
        }

        /// <summary>
        /// 指定の頂点と、ポリゴンの情報から重心座標系内の座標値を求める
        /// </summary>
        /// <param name="newpos"></param>
        /// <param name="tid"></param>
        /// <returns></returns>
        private Vector3 ComputeBaryCentricParams(VectorN newpos, int tid)
        {
            var newPos = ProjectOnPlane(newpos, tid, true);
            if (newPos == null)
            {
                return null;
            }

            var indices = new int[3];
            var positions = new Vector3[3];
            positions[0] = new Vector3(); // a
            positions[1] = new Vector3(); // b
            positions[2] = new Vector3(); // c

            mMesh.GetFaceVtxIndices(tid, ref indices);
            mMesh.GetVertexPos(indices[0], ref positions[0]);
            mMesh.GetVertexPos(indices[1], ref positions[1]);
            mMesh.GetVertexPos(indices[2], ref positions[2]);

            var ap = positions[0] - newPos;
            var bp = positions[1] - newPos;
            var cp = positions[2] - newPos;

            var abc = (positions[1] - positions[0]).Cross(positions[2] - positions[0]).Norm() * 0.5;
            if (abc < double.Epsilon)
            {
                return null;
            }

            var apb = ap.Cross(cp).Norm() * 0.5;
            var bpc = bp.Cross(ap).Norm() * 0.5;
            var cpa = cp.Cross(bp).Norm() * 0.5;

            if (Math.Abs((cpa + apb + bpc) / abc - 1.0f) > MinLength)
            {
                return null;
            }

            return new Vector3
            {
                x = cpa / abc,
                y = apb / abc,
                z = bpc / abc
            };
        }

        /// <summary>
        /// 簡略化後の位置座標
        /// </summary>
        /// <param name="uvEdge"></param>
        /// <param name="newGeomPos"></param>
        /// <returns></returns>
        private Tuple<Vector2, double> ComputeNewUvFromNewPos(WingedEdge uvEdge, Vector2 uvPos0, Vector2 uvPos1, QemHeapData hd)
        {
            // UVエッジの構成がおかしい
            if (uvEdge.ParentEdges == null)
            {
                return null;
            }
            var pvIndex0 = uvEdge.ParentEdges[0].Vertex0;
            var pvIndex1 = uvEdge.ParentEdges[0].Vertex1;
            var distinctedList = new List<int>();
            var triList0 = VtxTriMesh.GetIncidentTrianglesForProperty(uvEdge, pvIndex0);
            var triList1 = VtxTriMesh.GetIncidentTrianglesForProperty(uvEdge, pvIndex1);

            distinctedList.AddRange(triList0);
            distinctedList.AddRange(triList1);

            var uvlist = mMesh.GetSource(uvEdge.SpaceKey.type, uvEdge.SpaceKey.name);
            var cost = 0.0;
            Vector2 result = new Vector2();
            var hit = false;
            foreach (var ti in distinctedList.Distinct())
            {
                var baryCrd = ComputeBaryCentricParams(hd.NewPosition[0], ti);
                if (baryCrd != null)
                {
                    var uvindices = new int[3];
                    var uvPositions = new Vector2[3];
                    mMesh.GetFaceVtxIndices(ti, ref uvindices, uvEdge.SpaceKey.type, uvEdge.SpaceKey.name);
                    for (var vi = 0; vi < 3; vi++)
                    {
                        uvPositions[vi] = new Vector2();
                        MeshSourceDblCtrler.GetAsVec2(uvlist, uvindices[vi], ref uvPositions[vi]);
                    }

                    result = uvPositions[0] * baryCrd[0] + uvPositions[1] * baryCrd[1] + uvPositions[2] * baryCrd[2];

                    var d0 = uvPos0 - result;
                    var d1 = uvPos1 - result;
                    cost = Math.Max(d0.Norm(), d1.Norm());
                    hit = true;
                    break;
                }
            }
            return hit ? Tuple.Create(result, cost) : null;
        }

        /// <summary>
        /// UV座標をエッジ上で補間する。
        /// 補間パラメータは、既に頂点位置計算で得られているものとする
        /// </summary>
        /// <param name="uvPos0"></param>
        /// <param name="uvPos1"></param>
        /// <param name="interpolateParam"></param>
        /// <returns></returns>
        private Tuple<Vector2, double> ComputeUvNewPosOnEdge(Vector2 uvPos0, Vector2 uvPos1, double interpolateParam)
        {
            interpolateParam = Math.Max(interpolateParam, 0.0);
            interpolateParam = Math.Min(interpolateParam, 1.0);

            var tmpV2 = new Vector2();
            tmpV2[0] = uvPos0[0] + (uvPos1[0] - uvPos0[0]) * interpolateParam;
            tmpV2[1] = uvPos0[1] + (uvPos1[1] - uvPos0[1]) * interpolateParam;
            var elen = 0.0;
            elen = Math.Max(elen, (uvPos0 - tmpV2).Norm());
            elen = Math.Max(elen, (uvPos1 - tmpV2).Norm());
            return Tuple.Create(tmpV2, elen * elen);
        }

        /// <summary>
        /// 新しいUV座標を計算する
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="uvPos0"></param>
        /// <param name="uvPos1"></param>
        /// <param name="newPos"></param>
        /// <param name="interpolateParam"></param>
        /// <returns></returns>
        private Tuple<Vector2, double> ComputeUvNewPos(WingedEdge edge, Vector2 uvPos0, Vector2 uvPos1, QemHeapData hd)
        {
            switch (OptPosPolicy)
            {
                // エッジ上/空間上の線形補間
                case QemOptPos.OptimizeArea:
                case QemOptPos.CalcOptpos:
                    {
                        return ComputeNewUvFromNewPos(edge, uvPos0, uvPos1, hd);
                    }
                // 決定済みの線形補間パラメーターを利用する
                default:
                    {
                        return ComputeUvNewPosOnEdge(uvPos0, uvPos1, hd.InterpolateParam);
                    }
            }
        }

        /// <summary>
        /// UVハードエッジ上での補間処理
        /// </summary>
        /// <param name="uvedge"></param>
        /// <param name="newTotalPos"></param>
        /// <param name="uvPos0"></param>
        /// <param name="uvPos1"></param>
        /// <returns></returns>
        private Tuple<Vector2, double> ComputeNewUvPosOnHardEdge(WingedEdge uvedge, VectorN newTotalPos, Vector2 uvPos0, Vector2 uvPos1)
        {
            var pvIndex0 = uvedge.ParentEdges[0].Vertex0;
            var pvIndex1 = uvedge.ParentEdges[0].Vertex1;
            Vector3 newGeomPos = new Vector3(newTotalPos[0], newTotalPos[1], newTotalPos[2]);
            var posList = mMesh.GetSource(uvedge.ParentEdges[0].SpaceKey.type, uvedge.ParentEdges[0].SpaceKey.name);

            var tempV0 = new Vector3();
            var tempV1 = new Vector3();
            MeshSourceDblCtrler.GetAsVec3(posList, pvIndex0, ref tempV0);
            MeshSourceDblCtrler.GetAsVec3(posList, pvIndex1, ref tempV1);

            var dir0 = tempV1 - tempV0;
            var len0 = dir0.Norm();
            dir0.Normalize();
            var dir1 = newGeomPos - tempV0;
            var dot = dir1.Dot(dir0);
            if (Math.Abs(dot) > 0.0)
            {
                dot = dot / len0;
            }
            dot = Math.Max(0.0, dot);
            dot = Math.Min(1.0, dot);
            var tmpV2 = uvPos0 + (uvPos1 - uvPos0) * dot;
            var elen = 0.0;
            elen = Math.Max(elen, (uvPos0 - tmpV2).Norm());
            elen = Math.Max(elen, (uvPos1 - tmpV2).Norm());
            return Tuple.Create(tmpV2, elen * dir1.Norm());
        }

        /// <summary>
        /// ＵＶエッジのコストと、新規座標を求める
        /// 新規座標が、ずれなく計算できている可能性が小さい場合はFalse
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="v0"></param>
        /// <param name="v1"></param>
        /// <param name="hd"></param>
        private bool ComputeUVEdgeCostAndNewPos
            (
            WingedEdge uvedge,
            int[] border,
            VectorN newTotalPos,
            QemHeapData hepData)
        {
            var parentEdge = uvedge.ParentEdges.First();
            var boundaryMetric = ComputeUVBoundaryMetric(uvedge);
            var uvList = mMesh.GetSource(uvedge.SpaceKey.type, uvedge.SpaceKey.name);
            var newUvPos = new VectorN(3);

            var tempPos0 = new Vector2();
            var tempPos1 = new Vector2();

            var onCorner0 = IsBorder(border[parentEdge.Vertex0], QemVertexType.UvCorner);
            var onCorner1 = IsBorder(border[parentEdge.Vertex1], QemVertexType.UvCorner);
            var uv0 = uvedge.Vertex0;
            var uv1 = uvedge.Vertex1;

            WingEdedgeMesh.GetOrderedChildIndices(uvedge, ref uv0, ref uv1);
            MeshSourceDblCtrler.GetAsVec2(uvList, uv0, ref tempPos0);
            MeshSourceDblCtrler.GetAsVec2(uvList, uv1, ref tempPos1);

            if (IsEdgeType(hepData.EdgeType, QemEdgeType.NonManifold))
            {
                newUvPos[0] = tempPos0[0];
                newUvPos[1] = tempPos0[1];
                newUvPos[2] = 0;
                hepData.TexCost = double.MaxValue;
            }
            else if (
                IsEdgeType(hepData.EdgeType, QemEdgeType.TouchedHard) ||
                (IsEdgeType(hepData.EdgeType, QemEdgeType.HardEdge) && boundaryMetric != null))
            {
                newUvPos[0] = tempPos0[0] + (tempPos1[0] - tempPos0[0]) * hepData.InterpolateParam;
                newUvPos[1] = tempPos0[1] + (tempPos1[1] - tempPos0[1]) * hepData.InterpolateParam;
                newUvPos[2] = 0;
                var elen = (tempPos1 - tempPos0).Norm();
                hepData.TexCost += elen;
                if (boundaryMetric != null)
                {
                    hepData.TexCost += boundaryMetric.Evaluate(newUvPos);
                }
            }
            else if (IsEdgeType(hepData.EdgeType, QemEdgeType.PropertyEdge))
            {
                var newPosAndCost = ComputeNewUvPosOnHardEdge(uvedge, newTotalPos, tempPos0, tempPos1);
                newUvPos[0] = newPosAndCost.Item1[0];
                newUvPos[1] = newPosAndCost.Item1[1];
                newUvPos[2] = 0;
                hepData.TexCost += newPosAndCost.Item2;
            }
            else if (onCorner0 && !onCorner1)
            {
                hepData.InterpolateParam = 0.0;
                newUvPos[0] = tempPos0[0];
                newUvPos[1] = tempPos0[1];
                newUvPos[2] = 0;
                if (boundaryMetric != null)
                {
                    hepData.TexCost += boundaryMetric.Evaluate(newUvPos);
                }
            }
            else if (!onCorner0 && onCorner1)
            {
                hepData.InterpolateParam = 1.0;
                newUvPos[0] = tempPos1[0];
                newUvPos[1] = tempPos1[1];
                newUvPos[2] = 0;
                if (boundaryMetric != null)
                {
                    hepData.TexCost += boundaryMetric.Evaluate(newUvPos);
                }
            }
            else
            {
                // 新規のUv座標を、頂点座標の投射によって求める
                // 正常に投射できず、補間が正しく行われてない場合にはnull
                // この時は処理失敗として扱う
                var newPosAndCost = ComputeUvNewPos(uvedge, tempPos0, tempPos1, hepData);
                if (newPosAndCost == null)
                {
                    hepData.TexCost = double.MaxValue;
                    return false;
                }
                else
                {
                    newUvPos[0] = newPosAndCost.Item1[0];
                    newUvPos[1] = newPosAndCost.Item1[1];
                    newUvPos[2] = 0;
                    hepData.TexCost += newPosAndCost.Item2;
                }
            }
            SetUVNewPosToGlobalNewPos(uvedge, newUvPos, newTotalPos);
            return true;
        }

        private double[] _tmpCangles = null;

        /// <summary>
        /// ジオメトリに関するコストの計算を行う
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="border"></param>
        /// <param name="data"></param>
        private void ComputeGeometryCost(WingedEdge edge, List<int[]> border, ref QemHeapData data)
        {
            // 形状コストと最適化位置を計算
            ComputeShapeVolumeCostAndPos(edge, border[0], _tmpCangles[(int)SourceType.Position], data);

            // Feature Cost は一旦最適頂点位置を求めた後に
            // 必要であれば位置の補正も行う
            if (IsNormalEnabled)
            {
                ComputeFeatureCostAndPos(edge, border[0], data, _tmpCangles[(int)SourceType.Normal]);
            }
            // 形状コストの補正を行う
            if (Policy == QemPolicy.AngleBased || Policy == QemPolicy.AreaAngleBased)
            {
                int[] tid = null;
                Vector3 p0 = new Vector3();
                Vector3 p1 = new Vector3();
                Vector3 p2 = new Vector3();

                MeshSourceBase poslist = mMesh.GetSource(SourceType.Position, null);

                double minQuality = double.MaxValue;
                foreach (var f in VtxTriMesh.VTL[edge.Vertex0].TriList)
                {
                    mMesh.GetFaceVtxIndices(f, ref tid);
                    if (tid[0] == edge.Vertex1 || tid[1] == edge.Vertex1 || tid[2] == edge.Vertex1) { continue; }
                    {
                        MeshSourceDblCtrler.GetAsVec3(poslist, tid[0], ref p0);
                        MeshSourceDblCtrler.GetAsVec3(poslist, tid[1], ref p1);
                        MeshSourceDblCtrler.GetAsVec3(poslist, tid[2], ref p2);
                        double mq = FaceQuality(p0, p1, p2);
                        if (minQuality > mq)
                        {
                            minQuality = mq;
                        }
                    }
                }

                {
                    if (0.3 > minQuality)
                    {
                        minQuality = 0.3;
                    }
                    data.VCost = data.VCost / minQuality;
                }
            }

            data.NewPosition1.Set(data.NewPosition0);
        }

        /// <summary>
        /// エッジのコストを計算します
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="border"></param>
        /// <param name="data"></param>
        /// <returns></returns>
        private double ComputeEdgeCost(WingedEdge edge, List<int[]> border, ref QemHeapData data)
        {
            // 初期化
            data.Clear();
            if (_tmpCangles == null)
            {
                _tmpCangles = new double[Enum.GetNames(typeof(SourceType)).Length];
            }
            Array.Clear(_tmpCangles, 0, Enum.GetNames(typeof(SourceType)).Length);

            // エッジタイプを最初に取得
            data.EdgeType = JudgeEdgeType(edge, border, _tmpCangles);

            // 形状情報から、コストと位置計算を行う
            ComputeGeometryCost(edge, border, ref data);
            // UVに関するコスト計算を行う
            if (IsUVEnabled)
            {
                data.TexCost = data.VCost;
                // 正常な計算によって、UV座標値が計算できなくなった場合
                // 頂点位置の計算をエッジ上補間にしたうえで再度計算して
                // 更にUV補間もやりなおす。
                if (!ComputeUVCost(edge, border, data))
                {
                    var tmpOptPosPolicy = OptPosPolicy;
                    OptPosPolicy = QemOptPos.OptimizeEdge;
                    ComputeGeometryCost(edge, border, ref data);
                    ComputeUVCost(edge, border, data);
                    OptPosPolicy = tmpOptPosPolicy;
                }
            }
            return data.VCost;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="wingededge"></param>
        /// <param name="border"></param>
        /// <param name="heapPos"></param>
        internal void BuildHeap(WingedEdgeMesh wingededge, List<int[]> border,
                                ref ResizableList<QemHeapData> heapPos)
        {
            try
            {
                Heap = null;
                Heap = new BucketHeapList<QemHeapData>();
                // テクスチャを考慮する割合の設定
                if (IsUVEnabled)
                {
                    Heap.PrimaryRate = Params.PropWeights[(int)SourceType.Texture];
                }
                else
                {
                    Heap.PrimaryRate = 0.0f;
                }
                // エッジについてコストを計算する
                ResizableList<WingedEdge> edges = wingededge.RepresentativeEdges;
                MaxCost = 0.0;
                MinCost = double.MaxValue;

                // 各エッジのコストを計算
                // 各頂点での最小コストのエッジを計算
                for (int e = 0; e < edges.Count; e++)
                {
                    QemHeapData data = new QemHeapData();
                    data.NewPosition0 = new VectorN((uint)DataDimension);
                    data.NewPosition1 = new VectorN((uint)DataDimension);
                    data.EdgeID = e;

                    if (data.EdgeID != edges[e].ID)
                    {
                        throw ExcepHandle.CreateException("Execute:: Edge iD was different");
                    }

                    // エッジ縮退による体積変動コストを計算します
                    ComputeEdgeCost(edges[e], border, ref data);

                    //Heap.Push(data);
                    Heap.AddToWaitList(data);

                    heapPos[e] = data;

                    if (e % 100 == 0)
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Edge_Cost_Notify", e, edges.Count));
                    }
                }
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Edge_Cost_Notify", edges.Count, edges.Count));
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(@"..");
                Heap.ConstructBucketHeaps();

                UpdateLimit();
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// ヒープ更新
        /// </summary>
        /// <param name="vid">更新したいエッジの起点となる頂点</param>
        /// <param name="vtmesh"></param>
        /// <param name="winedge">WingedEdge</param>
        /// <param name="border">境界</param>
        /// <param name="heapdata">ヒープ構造</param>
        private void UpdateHeap(int vid, List<int[]> border,
                                ResizableList<QemHeapData> heapdata)
        {
            if (VtxTriMesh.VTL[vid].TriList.Count == 0)
            {
                return;
            }

            ResizableList<WingedEdge> edges = WingEdedgeMesh.RepVEL[vid].EdgeList;
            for (int e = 0; e < edges.Count; e++)
            {
                if (edges[e].Vertex0 != edges[e].Vertex1)
                {
                    int eid = edges[e].ID;
                    QemHeapData data = heapdata[eid];
                    if (heapdata[eid] == null)
                    {
                        continue;
                    }
                    if (data.EdgeID != eid)
                    {
                        throw ExcepHandle.CreateException("UpdateHeap::Access to invalid heap data.");
                    }

                    Heap.Pop(heapdata[eid]);

                    ComputeEdgeCost(edges[e], border, ref data);

                    Heap.Push(heapdata[eid]);

                    // 最小、最大コストの更新
                    // エッジコストが、ローカルの閾値を超える場合は優先度の繰り下げを行う
                    if (Math.Abs(data.VCost - double.MaxValue) > double.Epsilon
                        && !double.IsNaN(data.VCost)
                        && !double.IsInfinity(data.VCost))
                    {
                        if (edges[e].Face0 != int.MaxValue)
                        {
                            UpdateLimit();
                        }
                        if (edges[e].Face1 != int.MaxValue)
                        {
                            UpdateLimit();
                        }
                    }
                }
                else
                {
                    var eid = edges[e].ID;
                    if (heapdata[eid] != null)
                    {
                        Heap.Pop(heapdata[eid]);
                        heapdata[eid].VCost = double.MaxValue;
                        heapdata[eid].KeepSide = -1;
                        heapdata[eid].EdgeType = (int)QemEdgeType.NonManifold;
                        Heap.Push(heapdata[eid]);
                    }
                }
            }
            //test_Heap_List();
        }

        /// <summary>
        /// 指定フェースが所属するグループの最小コスト、最大コストを更新
        /// </summary>
        /// <param name="data"></param>
        /// <param name="faceId"></param>
        private void UpdateMinMaxCost(double cost, int edgeType, int faceId)
        {
            if (!IsEdgeType(edgeType, QemEdgeType.NonManifold))
            {
                MaxCost = Math.Max(MaxCost, cost);
                MinCost = Math.Min(MinCost, cost);
            }
        }

        /// <summary>
        /// 指定フェースが所属するグループのリミットコストを更新
        /// </summary>
        private void UpdateLimit()
        {
            if (Math.Abs(MaxCost) >= double.MaxValue || Math.Abs(1.0 - Quality) < float.Epsilon)
            {
                LimitCost = double.MaxValue;
            }
            else
            {
                LimitCost = Math.Abs(MaxCost - MinCost) * Quality;
            }
        }
    }
}
