﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using nw4f.tinymathlib;
using System.Threading;
using nw.g3d.iflib;
using Vector2 = nw4f.tinymathlib.Vector2;
using Vector3 = nw4f.tinymathlib.Vector3;

// ReSharper disable PossibleIntendedRethrow
// ReSharper disable TooWideLocalVariableScope
// ReSharper disable RedundantAssignment
// ReSharper disable SuggestUseVarKeywordEvident
// ReSharper disable CompareOfFloatsByEqualityOperator
// ReSharper disable RedundantCast

namespace nw4f.meshlib
{
    public partial class QemSimp
    {
        private Vector3 tmpVec3 = new Vector3();

        /// <summary>
        /// 指定のフェースの指定の頂点IDのデータを取得
        /// リダクション時に考慮されている値のみ
        /// </summary>
        /// <param name="fid"></param>
        /// <param name="localId"></param>
        /// <param name="vpN"></param>
        private void GetFaceVertices(int fid, int localId, ref VectorN vpN)
        {
            int vo = mMesh.mVoffs[fid];
            int sn = mMesh.Sources.Count;
            int di = 0;
            int so = 0;

            foreach (MeshSourceBase source in mMesh.Sources)
            {
                int type = (int)source.Type;
                if (mPropuse[type])
                {
                    if (vpN.Dim < (di + source.Stride))
                    {
                        break;
                    }

                    int vid = mMesh.mVbuffs[(vo + localId) * sn + so];
                    for (int st = 0; st < source.Stride; st++, di++)
                    {
                        if (vpN.Dim <= di)
                        {
                            throw ExcepHandle.CreateException("GetFaceVertex:output array size is short against required length.");
                        }

                        vpN[di] = Convert.ToDouble(source.GetRowValue(vid * source.Stride + st));
                    }
                }
                so++;
            }
        }

        /// <summary>
        /// フェースの各頂点の値を取得する
        /// </summary>
        /// <param name="fid"></param>
        /// <param name="plist"></param>
        private void GetFaceVertices(int fid, ref VectorN[] plist)
        {
            try
            {
                int vn = mMesh.mVnums[fid];

                if (plist.Length < vn)
                {
                    throw ExcepHandle.CreateException(
                        "GetFaceVertex:output array size is short against required length.");
                }
                // 各頂点の値を取得して、頂点データとして返す
                for (int v = 0; v < vn; v++)
                {
                    GetFaceVertices(fid, v, ref plist[v]);
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// フェースの各頂点の値を取得する
        /// </summary>
        private void GetVertices(WingedEdge meshWedge, int side, ref VectorN v0, ref VectorN v1, bool fullget)
        {
            try
            {
                int vid0 = meshWedge.Vertex0;
                int vid1 = meshWedge.Vertex1;

                int fid = meshWedge.Face0;

                int sn = mMesh.Sources.Count;
                int po = mMesh.GetSourceOfset(SourceType.Position, null);

                if (meshWedge.Face0 == int.MaxValue && meshWedge.Face1 == int.MaxValue)
                {
                    throw ExcepHandle.CreateException("GetVertices: failed to get vertex from illegal wedge.");
                }

                v0.SetZero();
                v1.SetZero();

                //位置座標しかなかった場合に使っていた
                if (side == 0)
                {
                    fid = meshWedge.Face0;
                    if (fid == int.MaxValue)
                    {
                        fid = meshWedge.Face1;
                    }
                }
                else if (side == 1)
                {
                    fid = meshWedge.Face1;
                    if (fid == int.MaxValue)
                    {
                        fid = meshWedge.Face0;
                    }
                }

                // 頂点0側を取得します
                {
                    int vo = mMesh.mVoffs[fid];
                    bool hit = false;

                    for (int i = 0; i < 3; i++)
                    {
                        if (mMesh.mVbuffs[(vo + i) * sn + po] == vid0)
                        {
                            hit = true;
                            GetFaceVertices(fid, i, ref v0);
                            break;
                        }
                    }

                    if (!hit)
                    {
                        throw ExcepHandle.CreateException("GetVertices:Could not get vertex data.");
                    }
                }

                // 頂点1側を取得します
                {
                    int vo = mMesh.mVoffs[fid];
                    bool hit = false;
                    for (int i = 0; i < 3; i++)
                    {
                        if (mMesh.mVbuffs[(vo + i) * sn + po] == vid1)
                        {
                            hit = true;
                            GetFaceVertices(fid, i, ref v1);
                            break;
                        }
                    }
                    if (!hit)
                    {
                        throw ExcepHandle.CreateException("GetVertices:Could not get vertex data.");
                    }
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="keepSide"></param>
        /// <param name="newv"></param>
        private void SetVertices(WingedEdge edge, int keepSide, VectorN newv, int doffset)
        {
            var keepID = keepSide == 0 ? edge.Vertex0 : edge.Vertex1;
            var skey = edge.SpaceKey;
            var sourceN = mMesh.SourceN;

            if (doffset < 0)
            {
                doffset = 0;
                for (var s = 0; s < sourceN; s++)
                {
                    var source = (MeshSourceBase)mMesh.Sources[s];
                    if (mPropuse[(int)source.Type])
                    {
                        if (newv.Dim < (doffset + source.Stride))
                        {
                            break;
                        }
                        // キーが一致した
                        if (source.Key.Equals(skey))
                        {
                            break;
                        }
                        doffset += source.Stride;
                    }
                }
            }

            MeshSourceBase meshsrc = (MeshSourceBase)mMesh.GetSource(skey.type, skey.name);
            if (meshsrc != null)
            {
                for (int st = 0; st < meshsrc.Stride; st++)
                {
                    meshsrc.SetRowValue((keepID * meshsrc.Stride) + st, newv[doffset + st]);
                }
            }
        }

        private void SetVertices(SourceKey skey, int keepID, VectorN newv, int doffset)
        {
            var sourceN = mMesh.SourceN;

            if (doffset < 0)
            {
                doffset = 0;
                for (var s = 0; s < sourceN; s++)
                {
                    var source = (MeshSourceBase)mMesh.Sources[s];
                    if (mPropuse[(int)source.Type])
                    {
                        if (newv.Dim < (doffset + source.Stride))
                        {
                            break;
                        }
                        // キーが一致した
                        if (source.Key.Equals(skey))
                        {
                            break;
                        }
                        doffset += source.Stride;
                    }
                }
            }

            MeshSourceBase meshsrc = (MeshSourceBase)mMesh.GetSource(skey.type, skey.name);
            if (meshsrc != null)
            {
                for (int st = 0; st < meshsrc.Stride; st++)
                {
                    meshsrc.SetRowValue((keepID * meshsrc.Stride) + st, newv[doffset + st]);
                }
            }
        }

        /// <summary>
        /// 各プロパティの境界を超えていないか？チェックして
        /// 超えている場合は各プロパティの値を、妥当な値に更新する
        /// </summary>
        /// <param name="nv"></param>
        /// <param name="betterPos"></param>
        /// <param name="fid"></param>
        /// <returns></returns>
        private bool CheckBounds(ref VectorN nv, VectorN betterPos, int fid)
        {
            bool result = false;
            int dk = 0; // 仕様プロパティ次元
            int ti = 0;
            foreach (var source in mMesh.Sources)
            {
                int type = (int)source.Type;
                if (mPropuse[type])
                {
                    switch (source.Type)
                    {
                        case SourceType.Position:
                            result |= CheckPosBounds(ref nv, source, betterPos, dk, fid);
                            break;
                        case SourceType.Normal:
                            result |= CheckNormalBounds(ref nv, source, betterPos, dk);
                            break;
                        case SourceType.Texture:
                            result |= CheckTextureBounds(ref nv, source, betterPos, dk, fid, ti);
                            ti++;
                            break;
                        case SourceType.Color:
                            result |= CheckColorBounds(ref nv, source, betterPos, dk);
                            break;
                        case SourceType.BlendWeight:
                            result |= CheckBWeightBounds(ref nv, source, betterPos, dk);
                            break;
                    }
                    dk += source.Stride;
                }
            }
            return result;
        }

        /// <summary>
        /// 位置座標が境界を突破してないか？をチェックします
        /// </summary>
        /// <param name="nv"></param>
        private bool CheckPosBounds(ref VectorN nv, MeshSourceBase source, VectorN betterPos, int dk, int fid)
        {
            if (mMesh.IsInvalidateFace(fid))
            {
                throw ExcepHandle.CreateException("index : tried to check invalidate face");
            }
            if (mMesh.FaceMaterialShapeId.Count <= fid)
            {
                throw ExcepHandle.CreateException("index : out of range");
            }

            int fgid = mMesh.FaceMaterialShapeId[fid];
            if (fgid == int.MaxValue)
            {
                throw ExcepHandle.CreateException("the index of face group: out of range");
            }

            bool result = false;
            if (nv.Dim < (dk + source.Stride) || betterPos.Dim < (dk + source.Stride))
            {
                return false;
            }

            if (source.Type != SourceType.Position) { return false; }

            BoundingBox bb = mMesh.GetBoundingBox(fgid);
            bool error = false;
            for (int si = 0; si < source.Stride; si++)
            {
                if (nv[dk + si] < bb.Min[si] || nv[dk + si] > bb.Max[si])
                {
                    error = true;
                }
            }

            if (error)
            {
                result = true;
                for (int si = 0; si < source.Stride; si++)
                {
                    nv[dk + si] = betterPos[dk + si];
                }
            }
            return result;
        }

        /// <summary>
        /// 頂点法線が境界を突破してないか？をチェックします
        /// </summary>
        /// <param name="nv"></param>
        private bool CheckNormalBounds(ref VectorN nv, MeshSourceBase source, VectorN betterPos, int dk)
        {
            try
            {
                if (nv.Dim < (dk + source.Stride) || betterPos.Dim < (dk + source.Stride))
                {
                    return false;
                }
                // ノーマルだけは正規化して渡す
                if (source.Type == SourceType.Normal && OptPosPolicy != QemOptPos.ZeroOne)
                {
                    tmpVec3.SetZero();
                    for (int si = 0; si < source.Stride; si++)
                    {
                        tmpVec3[si] = nv[dk + si];
                    }

                    tmpVec3.Normalize();
                    for (int si = 0; si < source.Stride; si++)
                    {
                        nv[dk + si] = tmpVec3[si];
                    }
                }
                return false;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("CheckPropBounds Error", ex);
            }
        }

        /// <summary>
        /// 頂点カラーが境界を突破してないか？をチェックします
        /// </summary>
        /// <param name="nv"></param>
        private bool CheckColorBounds(ref VectorN nv, MeshSourceBase source, VectorN betterPos, int dk)
        {
            try
            {
                bool result = false;
                if (nv.Dim < (dk + source.Stride) || betterPos.Dim < (dk + source.Stride))
                {
                    return false;
                }
                if (source.Type != SourceType.Color) { return false; }

                bool error = false;
                for (int si = 0; si < source.Stride; si++)
                {
                    if (nv[dk + si] < 0 || nv[dk + si] > 255)
                    {
                        error = true;
                    }
                }

                if (error)
                {
                    result = true;
                    for (int si = 0; si < source.Stride; si++)
                    {
                        nv[dk + si] = betterPos[dk + si];
                    }
                }
                return result;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("CheckPropBounds Error", ex);
            }
        }

        /// <summary>
        /// ブレンドウェイトが境界を突破してないか？をチェックします
        /// </summary>
        /// <param name="nv"></param>
        private bool CheckBWeightBounds(ref VectorN nv, MeshSourceBase source, VectorN betterPos, int dk)
        {
            try
            {
                bool result = false;
                if (nv.Dim < (dk + source.Stride) || betterPos.Dim < (dk + source.Stride))
                {
                    return false;
                }
                if (source.Type != SourceType.Color) { return false; }

                bool error = false;
                for (int si = 0; si < source.Stride; si++)
                {
                    if (nv[dk + si] < 0 || nv[dk + si] > 1.0)
                    {
                        error = true;
                    }
                }

                if (error)
                {
                    result = true;
                    for (int si = 0; si < source.Stride; si++)
                    {
                        nv[dk + si] = betterPos[dk + si];
                    }
                }
                return result;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("CheckPropBounds Error", ex);
            }
        }
        /// <summary>
        /// テクスチャ座標が境界を突破してないか？をチェックします
        /// </summary>
        /// <param name="nv"></param>
        private bool CheckTextureBounds(ref VectorN nv, MeshSourceBase source, VectorN betterPos, int dk, int fid, int ti)
        {
            try
            {
                bool result = false;
                if (nv.Dim < (dk + source.Stride) || betterPos.Dim < (dk + source.Stride))
                {
                    return false;
                }
                // ノーマルだけは正規化して渡す
                if (source.Type == SourceType.Texture)
                {
                    int fgid = mMesh.FaceMaterialShapeId[fid];
                    if (fgid != int.MaxValue)
                    {
                        BoundingSquare bs = mMesh.GetUvBound(fgid, ti);
                        bool error = false;
                        for (int si = 0; si < source.Stride; si++)
                        {
                            if (nv[dk + si] < bs.Min[si] || nv[dk + si] > bs.Max[si])
                            {
                                error = true;
                            }
                        }

                        if (error)
                        {
                            result = true;
                            for (int si = 0; si < source.Stride; si++)
                            {
                                nv[dk + si] = betterPos[dk + si];
                            }
                        }
                    }
                }
                return result;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("CheckPropBounds Error", ex);
            }
        }

        /// <summary>
        /// 指定のエッジが四角錐の一辺になったかどうか？をチェックします
        /// </summary>
        /// <param name="wingededge"></param>
        /// <param name="edge"></param>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        /// <returns></returns>
        private bool IsPyramidShrink(WingedEdgeMesh wingededge, int keep, int kill)
        {
            Queue<int> queue = new Queue<int>();
            List<int> visitVertices = new List<int>();
            Dictionary<int, int> calldepth = new Dictionary<int, int>();
            if (!queue.Contains(keep))
            {
                queue.Enqueue(keep);
                calldepth[keep] = 0;
            }
            Func<int, bool> visitAct =
            (int vid) =>
                {
                    if (calldepth[vid] >= 2)
                    {
                        return false;
                    }
                    if (!visitVertices.Contains(vid))
                    {
                        visitVertices.Add(vid);
                    }
                    foreach (var e in wingededge.RepVEL[vid].EdgeList)
                    {
                        int ovid = e.Vertex0 == vid ? e.Vertex1 : e.Vertex0;
                        if (!visitVertices.Contains(ovid))
                        {
                            visitVertices.Add(ovid);
                            calldepth[ovid] = calldepth[vid] + 1;
                            queue.Enqueue(ovid);
                        }
                    }
                    return true;
                };

            while (queue.Count > 0)
            {
                if (!visitAct(queue.Dequeue()))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// エッジ縮退をしてもいいかどうか？をテストします
        /// 非多様体と、穴が閉じてしまうか？のテスト
        /// </summary>
        /// <param name="wingededge">エッジ情報</param>
        /// <param name="keep">保持頂点ID</param>
        /// <param name="kill">削除頂点ID</param>
        /// <returns> </returns>
        private bool TestSafeEdgeCollapse(WingedEdgeMesh wingededge, WingedEdge edge, int keep, int kill)
        {
            ResizableList<WingedEdge> keepList = wingededge.RepVEL[keep].EdgeList;
            ResizableList<WingedEdge> killList = wingededge.RepVEL[kill].EdgeList;
            int nKeepN = keepList.Count;
            int nKillN = killList.Count;
            int nShareVtxN = 0;

            if (IsPyramidShrink(wingededge, keep, kill))
            {
                return false;
            }

            Dictionary<int, int> hitCount = new Dictionary<int, int>();
            for (int i = 0; i < nKeepN; i++)
            {
                WingedEdge keepEdge = keepList[i];
                int v0 = keepEdge.Vertex0 == keep ? keepEdge.Vertex1 : keepEdge.Vertex0;

                if (v0 != kill)
                {
                    if (!hitCount.ContainsKey(v0))
                    {
                        hitCount.Add(v0, 1);
                    }
                    else
                    {
                        hitCount[v0]++;
                    }
                }
            }
            for (int j = 0; j < nKillN; j++)
            {
                WingedEdge killEdge = killList[j];
                int v1 = killEdge.Vertex1 == kill ? killEdge.Vertex0 : killEdge.Vertex1;

                if (v1 != keep)
                {
                    if (!hitCount.ContainsKey(v1))
                    {
                        hitCount.Add(v1, 1);
                    }
                    else
                    {
                        hitCount[v1]++;
                    }
                }
            }

            bool error = false;
            nShareVtxN = 0;
            foreach (var pair in hitCount)
            {
                if (pair.Value == 2)
                {
                    nShareVtxN++;
                }
                if (pair.Value > 2)
                {
                    error = true;
                }
            }

            if (edge.IsOpen)
            {
                if (nShareVtxN > 1 || hitCount.Count <= 2)
                {
                    error = true;
                }
            }
            else
            {
                // 最後の１フェースか、または非多様体
                if (nShareVtxN > 2 || nShareVtxN <= 1 || hitCount.Count <= 2)
                {
                    error = true;
                }
            }
            if (error)
            {
                return false;
            }

            // 殆ど上の条件に引っかかるので要らんかも
            // 閉じた開放端が３つのエッジでしか構成されていない場合
            //if (edgetype == QemEdgeType.HardEdge)
            if (edge.Face0 == int.MaxValue || edge.Face1 == int.MaxValue)
            {
                ResizableList<WingedEdge> checkList = new ResizableList<WingedEdge>();
                for (int i = 0; i < nKeepN; i++)
                {
                    WingedEdge keepEdge = keepList[i].Clone() as WingedEdge;
                    int vid0 = keepEdge.Vertex0;
                    int vid1 = keepEdge.Vertex1;

                    if (vid0 == kill)
                    {
                        vid0 = keep;
                    }
                    else if (vid1 == kill)
                    {
                        vid1 = keep;
                    }
                    keepEdge.Vertex0 = Math.Min(vid0, vid1);
                    keepEdge.Vertex1 = Math.Max(vid0, vid1);

                    checkList.Add(keepEdge);
                }

                for (int i = 0; i < nKillN; i++)
                {
                    WingedEdge killEdge = killList[i].Clone() as WingedEdge;

                    int vid0 = killEdge.Vertex0;
                    int vid1 = killEdge.Vertex1;

                    if (vid0 == kill)
                    {
                        vid0 = keep;
                    }
                    else if (vid1 == kill)
                    {
                        vid1 = keep;
                    }

                    killEdge.Vertex0 = Math.Min(vid0, vid1);
                    killEdge.Vertex1 = Math.Max(vid0, vid1);

                    checkList.Add(killEdge);
                }

                checkList.Sort();

                WingedEdge prev = null;
                foreach (var wingedEdge in checkList)
                {
                    if (wingedEdge != null && prev != null)
                    {
                        if (wingedEdge.IsVerticesEqual(prev))
                        {
                            if (prev.Face1 == int.MaxValue && wingedEdge.Face1 == int.MaxValue)
                            {
                                if (wingedEdge.Face0 != prev.Face0)
                                {
                                    return false;
                                }
                            }
                        }
                    }
                    prev = wingedEdge;
                }
            }
            return true;
        }

        /// <summary>
        /// 最後のサブメッシュかどうか？を取得
        /// </summary>
        /// <param name="submeshCounter"></param>
        /// <param name="wedge"></param>
        /// <returns> true : 最後のサブメッシュ＝リダクション不可 </returns>
        private bool IsLastSubMeshFace(ResizableList<int> sbcounter, WingedEdge wedge)
        {
            if (wedge == null) { return false; }
            int smid0 = int.MaxValue;
            int smid1 = int.MaxValue;

            if (wedge.Face0 != int.MaxValue)
            {
                smid0 = mMesh.mFaceSubMeshID[wedge.Face0];
            }
            if (wedge.Face1 != int.MaxValue)
            {
                smid1 = mMesh.mFaceSubMeshID[wedge.Face1];
            }

            if (smid0 == int.MaxValue && smid1 == int.MaxValue)
            {
                return true;
            }

            if (smid0 == smid1 && smid0 != int.MaxValue)
            {
                if (sbcounter[smid0] <= 2) { return true; }
            }

            if (smid1 != int.MaxValue)
            {
                if (smid1 >= sbcounter.Count)
                {
                    throw ExcepHandle.CreateException("submesh id is bigger than list count");
                }
                if (sbcounter[smid1] <= 2) { return true; }
            }
            if (smid0 != int.MaxValue)
            {
                if (smid0 >= sbcounter.Count)
                {
                    throw ExcepHandle.CreateException("submesh id is bigger than list count");
                }
                if (sbcounter[smid0] <= 2) { return true; }
            }
            return false;
        }

        /// <summary>
        /// マテリアルシェイプが簡略化可能な状態にあるかどうか？をチェックします
        /// </summary>
        /// <param name="submeshCounter"></param>
        /// <param name="wedge"></param>
        /// <returns> true : 最後のサブメッシュ＝リダクション不可 </returns>
        private bool CheckRateForMaterialShape(ResizableList<int> mscounter, WingedEdge wedge)
        {
            if (wedge == null)
            {
                return false;
            }

            var msid0 = int.MaxValue;
            var msid1 = int.MaxValue;
            if (wedge.Face0 != int.MaxValue)
            {
                msid0 = mMesh.FaceMaterialShapeId[wedge.Face0];
            }
            if (wedge.Face1 != int.MaxValue)
            {
                msid1 = mMesh.FaceMaterialShapeId[wedge.Face1];
            }

            if (msid0 == int.MaxValue && msid1 == int.MaxValue)
            {
                return true;
            }

            if (msid1 != int.MaxValue)
            {
                if (msid1 >= mscounter.Count || msid1 >= RatePerAllMhapes.Length)
                {
                    throw ExcepHandle.CreateException("material shape id is bigger than list count");
                }
                if (mscounter[ msid1] <= OriginalMaterialShapeFaceCount[ msid1] * RatePerAllMhapes[ msid1] ||
                    mscounter[ msid1] <= 2)
                {
                    return true;
                }
            }
            if (msid0 != int.MaxValue)
            {
                if (msid0 >= mscounter.Count || msid0 >= RatePerAllMhapes.Length)
                {
                    throw ExcepHandle.CreateException("material shape id is bigger than list count");
                }
                if (mscounter[ msid0] <= OriginalMaterialShapeFaceCount[ msid0] * RatePerAllMhapes[ msid0] ||
                    mscounter[ msid0] <= 2)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 法線ゼロのものがあるかどうか？を確認して、ゼロのものがあれば
        /// 再計算を行う様に修正した。
        /// </summary>
        /// <returns>
        /// </returns>
        private void CheckVtxNormalAndRecompute(int vid)
        {
            try
            {
                int keepID = vid;
                int sourceN = mMesh.SourceN;
                int[] vKeepBuff0 = new int[sourceN];
                int di = 0;
                foreach (var fid in VtxTriMesh.VTL[vid].TriList)
                {
                    if (fid == int.MaxValue) { continue; }

                    VtxTriMesh.GetVBuff(ref vKeepBuff0, fid, keepID);
                    for (int s = 0; s < sourceN; s++)
                    {
                        MeshSourceBase meshsrc = (MeshSourceBase)mMesh.Sources[s];
                        if (mPropuse[(int)meshsrc.Type] && meshsrc.Type == SourceType.Normal)
                        {
                            if (vKeepBuff0[s] != int.MaxValue)
                            {
                                Vector3 tnml = new Vector3();
                                MeshSourceDblCtrler.GetAsVec3(meshsrc, vKeepBuff0[s], ref tnml);
                                // 法線がゼロであれば、g3dchk でエラーになるようなので
                                if (tnml.Norm() < double.Epsilon)
                                {
                                    double sum = 0;
                                    foreach (var nfi in VtxTriMesh.VTL[vid].TriList)
                                    {
                                        tnml += mMesh.mFaceNormal[nfi] * mMesh.mFaceArea[nfi];
                                        sum += mMesh.mFaceArea[nfi];
                                    }
                                    if (sum > double.Epsilon)
                                    {
                                        tnml *= (1.0f / sum);
                                    }
                                }
                                else
                                {
                                    tnml.y = 1.0;
                                }
                                MeshSourceDblCtrler.SetAsVec3(meshsrc, vKeepBuff0[s], tnml);
                            }
                            di += meshsrc.Stride;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// リダクションを実行する
        /// </summary>
        public int Execute(out bool[] vertexMoved, out bool[] vertexDeleted)
        {
#if VERBOSE
            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine("Using New Quadric Error Method");
#endif

            int totalsum = 0;
            int retryCount = 0;
            vertexMoved = new bool[mMesh.VertexN];
            vertexDeleted = new bool[mMesh.VertexN];
            Array.Clear(vertexMoved, 0, mMesh.VertexN);
            Array.Clear(vertexDeleted, 0, mMesh.VertexN);

            for (int rci = 0; rci < retryCount + 1; rci++)
            {
                List<int[]> vertexborder = null;
                QemHeapData data = null;
                WingedEdge edge = null;

                try
                {
                    PreExecute(ref vertexborder);
                }
                catch (Exception ex)
                {
                    ex.Data.Add("rmesh", mMesh);
#if DEBUG
                    ex.Data.Add("cmesh", mCloneMesh);
#endif
                    throw ExcepHandle.CreateException("Execute::failed preproc", ex);
                }

                try
                {
                    ResizableList<WingedEdge> edges = WingEdedgeMesh.RepresentativeEdges;
                    ResizableList<QemHeapData> edgeHeapData = new ResizableList<QemHeapData>();
                    var currentEdgeNum = 0;
                    var sum = 0;
                    var count = 0;
                    var redfaces = 0;
                    var error = 0;
                    int vi0;
                    int vi1;
                    int kill;
                    int keep;
                    var availableFaceNum = 0;

                    edgeHeapData.Resize(edges.Count, null);
                    // 初期のヒープ(min-heap)を構築
                    BuildHeap(WingEdedgeMesh, vertexborder, ref edgeHeapData);

                    availableFaceNum = mMesh.FaceN;
                    var impracticableFaces = mMesh.FaceN - availableFaceNum;
                    //以下三行は、プロファイル機能を作成後に復活させる予定。備忘の為にコメントアウト
                    //var totalSumMeshFaceN = 0;
                    //nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_OnSumeshBorder_FaceNumber_Notify", totalSumMeshFaceN));
                    //nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Impracticable_FaceNumber_Notify", impracticableFaces + totalSumMeshFaceN));

                    var originalFaceNum = mMesh.FaceN;
                    var currentFaceNum = mMesh.FaceN;
                    var targetFaceNum = originalFaceNum - (int)((float)(currentFaceNum - ExcludeFaceNum - impracticableFaces) * (1.0 - TargetRate));
                    var figLen = (int)Math.Log10((double)originalFaceNum) + 1;
                    var figSetting = "D" + figLen;

                    if (ExcludeFaceNum > 0)
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Exclude_FaceNumber_Notify", ExcludeFaceNum.ToString(figSetting)));
                    }
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Target_FaceNumber_Notify", targetFaceNum.ToString(figSetting)));

                    // 使いまわす為の位置オブジェクトを作成
                    VectorN[] plist = new VectorN[3];
                    for (int v = 0; v < 3; v++)
                    {
                        plist[v] = new VectorN((uint)DataDimension);
                    }

                    //------------------------------------------------
                    // 処理開始
                    // メインループ
                    currentEdgeNum = edges.Count;
                    while (currentFaceNum > targetFaceNum)
                    {
                        data = Heap.Top;
                        if (data == null)
                        {
                            break;
                        }

                        edge = edges[data.EdgeID];
                        if (data.EdgeID != edge.ID)
                        {
                            throw ExcepHandle.CreateException("Execute:: Edge ID was different");
                        }
                        if (edge == null)
                        {
                            throw ExcepHandle.CreateException("Execute::Null Edge was accessed");
                        }

                        vi0 = edge.Vertex0;
                        vi1 = edge.Vertex1;

                        if (edge.Face0 == int.MaxValue && edge.Face1 != int.MaxValue)
                        {
                            throw ExcepHandle.CreateException("Execute::Winged Edge is not correct. Both face is empty");
                        }

                        // 既に削除済みのエッジはパス
                        if (!edge.UnMovable)
                        {
                            if (vi0 == int.MaxValue || vi1 == int.MaxValue)
                            {
                                throw ExcepHandle.CreateException("Execute::trying to Shrink illegal edge.");
                            }

                            bool shrink = false;

                            // 大小比較の為に、重みによって大きくされている場合は小さくする
                            var vCostBody = data.VWeight <= 1.0 ? data.VCost : data.VCost / data.VWeight;

                            // いずれかが除外頂点の場合は無視
                            // サブメッシュ個数チェック
                            // ノンマニチェック
                            // 境界チェック
                            // クオリティチェック
                            // フリップチェック
                            if (
                                !IsEdgeType(data.EdgeType, QemEdgeType.NonManifold) &&
                                !CheckRateForMaterialShape(mMesh._MaterialShapeFaceCounter, edge) &&
                                !IsLastSubMeshFace(mMesh.mSubmeshCounter, edge) &&
                                TestSafeEdgeCollapse(WingEdedgeMesh, edge, vi0, vi1) &&
                                Quality > double.Epsilon &&
                                LimitCost > (vCostBody - MinCost) &&
                                Math.Abs(data.VCost - double.MaxValue) > 1e-8 &&
                                (Math.Abs(data.TexCost - double.MaxValue) > 1e-8 || !IsUVEnabled))
                            {
                                // エッジの縮退を行う
                                kill = 0;
                                keep = 0;
                                if (data.KeepSide == 0)
                                {
                                    kill = vi1;
                                    keep = vi0;
                                }
                                else
                                {
                                    kill = vi0;
                                    keep = vi1;
                                }

                                bool testResult = !CheckFlip || VtxTriMesh.TestShrinkEdge(WedgeMesh.Wedges[vi0], WedgeMesh.Wedges[vi1], data.NewPosition0);
                                testResult &= WingEdedgeMesh.TestShrinkRepEdge(keep, kill);

                                // 非多様体になるかどうか？
                                if (testResult)
                                {
                                    // [Debug]消去頂点が、妥当かどうか？を確認する[Debug]
                                    CheckLegalKillVertex(vertexborder[0], kill);
                                    Dictionary<SourceKey, int2> keeps = null;
                                    Dictionary<SourceKey, int2> kills = null;

                                    if (WingEdedgeMesh.ShrinkEdgeAll(keep, kill, out keeps, out kills))
                                    {
                                        edgeHeapData[data.EdgeID] = null;
                                        Heap.UpdateBucketNumber(currentEdgeNum);
                                        Heap.Pop();

                                        // 位置の移動を設定
                                        vertexMoved[keep] = true;
                                        vertexDeleted[kill] = true;

                                        // 最初に座標の入れ替え
                                        // インデックス違い、同一座標への対応として
                                        // kill側頂点で同じ値のものをすべて書き換える

                                        var nmlList = mMesh.GetSource(SourceType.Normal, null);
                                        if (OptPosPolicy != QemOptPos.ZeroOne)
                                        {
                                            // まず頂点座標の更新
                                            SetVertices(edge.SpaceKey, keeps[edge.SpaceKey][0], data.NewPosition[0], -1);
                                            // 法線は補間ではなく、再計算方式に変更
                                            // UV の更新
                                            if (IsUVEnabled)
                                            {
                                                foreach (var cep in edge.ChildEdges)
                                                {
                                                    if (mPropuse[(int)cep.Key.type])
                                                    {
                                                        int uvi = 0;
                                                        foreach (var ce in cep.Value)
                                                        {
                                                            if (ce.Face0 == edge.Face0)
                                                            {
                                                                SetVertices(ce.SpaceKey, keeps[ce.SpaceKey][0], data.NewPosition[0], -1);
                                                            }
                                                            else
                                                            {
                                                                SetVertices(ce.SpaceKey, keeps[ce.SpaceKey][1], data.NewPosition[1], -1);
                                                            }
                                                            uvi++;
                                                        }
                                                    }
                                                }
                                            }
                                            else
                                            {
                                                foreach (var cep in edge.ChildEdges)
                                                {
                                                    var uv0 = new Vector2();
                                                    var uv1 = new Vector2();
                                                    var source = mMesh.GetSource(cep.Key.type, cep.Key.name);
                                                    foreach (var wingedEdge in cep.Value)
                                                    {
                                                        int id0 = data.KeepSide == 0 ? wingedEdge.Vertex0 : wingedEdge.Vertex1;
                                                        int id1 = data.KeepSide == 0 ? wingedEdge.Vertex1 : wingedEdge.Vertex0;
                                                        MeshSourceDblCtrler.GetAsVec2(source, id0, ref uv0);
                                                        MeshSourceDblCtrler.GetAsVec2(source, id1, ref uv1);
                                                        var newUvPos = ComputeUvNewPos(wingedEdge, uv0, uv1, data);

                                                        SetVertices(wingedEdge.SpaceKey, keeps[edge.SpaceKey][0], newUvPos.Item1.ToVectorN(), 0);
                                                    }
                                                }
                                            }
                                        }

                                        shrink = true;
                                        redfaces = VtxTriMesh.ShrinkEdge(keeps, kills);
                                        VtxTriMesh.DeleteDuplicateFaceAndEdges(WingEdedgeMesh, keep);
                                        WedgeMesh.Update(keep, kill, VtxTriMesh);

                                        WingEdedgeMesh.CheckChidRef(keep);
                                        // フラグの和を取る
                                        foreach (var vtxBorder in vertexborder)
                                        {
                                            vtxBorder[keep] |= vtxBorder[kill];
                                        }

                                        int eN = WingEdedgeMesh.RepVEL[keep].EdgeList.Count;
                                        // Shrink によってゴミが発生する場合への対処
                                        // 面積的に縮退済みのフェースを消し去る
                                        VtxTriMesh.CleanDegenerateFace(keep);
                                        for (int ei = 0; ei < eN; ei++)
                                        {
                                            WingedEdge le = WingEdedgeMesh.RepVEL[keep].EdgeList[ei];
                                            int vid = 0;
                                            if (le.Vertex0 == keep || le.Vertex1 == keep)
                                            {
                                                vid = le.Vertex0 == keep ? le.Vertex1 : le.Vertex0;
                                                VtxTriMesh.CleanDegenerateFace(vid);
                                            }
                                        }

                                        // 毎回、Quadricを計算することでスムージングするか否か
                                        if (Smoothing)
                                        {
                                            // keep 側を更新
                                            ComputeVertexQuadrics(VtxTriMesh, keep, ref plist);
                                            // kill 側を更新
                                            foreach (var le in WingEdedgeMesh.RepVEL[keep].EdgeList)
                                            {
                                                int vid = le.Vertex0 == keep ? le.Vertex1 : le.Vertex0;
                                                ComputeVertexQuadrics(VtxTriMesh, vid, ref plist);
                                            }
                                            foreach (var le in WingEdedgeMesh.RepVEL[keep].EdgeList)
                                            {
                                                ComputeVirtualPlaneQuadris(le, ref plist);
                                            }
                                        }
                                        else
                                        {
                                            mQuads[keep].Set(mQuads[keep] + mQuads[kill]);
                                        }

                                        // 自分自身の周辺の更新
                                        UpdateHeap(keep, vertexborder, edgeHeapData);

                                        //1近傍周辺のヒープを更新
                                        for (int ei = 0; ei < eN; ei++)
                                        {
                                            WingedEdge le = WingEdedgeMesh.RepVEL[keep].EdgeList[ei];
                                            int vid = 0;
                                            if (le.Vertex0 == keep || le.Vertex1 == keep)
                                            {
                                                vid = le.Vertex0 == keep ? le.Vertex1 : le.Vertex0;
                                                UpdateHeap(vid, vertexborder, edgeHeapData);
                                            }
                                        }

                                        // 更新処理
                                        // 面法線と面積の更新を行います
                                        VtxTriMesh.RecomputeFaceNormalsAndArea(keep);
                                        // 角度を計算
                                        ReComputeFaceAngles(VtxTriMesh, keep);

                                        // 更新後の頂点法線を再計算する（補間しない様にした）
                                        // 法線インデックスごとに近傍を取得して、齟齬を起こさない様に抑制
                                        VtxTriMesh.ComputeVtxNormals(keep, nmlList.Name);

                                        // フェースの数を更新
                                        currentFaceNum -= redfaces;
                                        currentEdgeNum--;
                                        count += redfaces;
                                        sum += redfaces;
                                        if (count > 100)
                                        {
                                            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Face_Number_Notify",
                                                currentFaceNum.ToString(figSetting),
                                                targetFaceNum.ToString(figSetting),
                                                LimitCost >= double.MaxValue ? "MaxValue" : LimitCost.ToString("F3")));
                                            count = 0;
                                            //break;
                                        }
                                        // エラーが無くなった
                                        error = 0;
                                    }
                                }
                            }

                            // 安全な縮退ではないので、コスト最大にして更新する
                            if (!shrink)
                            {
                                int eid = data.EdgeID;
                                double oldCost = data.VCost;
                                if (edgeHeapData[eid] != null)
                                {
                                    Heap.Pop();
                                    edgeHeapData[eid].EdgeType = (int)QemEdgeType.NonManifold;
                                    edgeHeapData[eid].VCost = double.MaxValue;
                                    edgeHeapData[eid].EdgeID = data.EdgeID;
                                    edgeHeapData[eid].KeepSide = 0;
                                    Heap.Push(edgeHeapData[eid]);
                                }

                                // 既に臨界に達していて、かつまた、ヒープの最小コストがエラー値なら終了
                                error++;
                                if (error >= originalFaceNum && oldCost == double.MaxValue)
                                {
#if VERBOSE
                                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r current {0,6:d} -> target {1,6:d}\n", currentFaceNum,
                                                        targetFaceNum);
                                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\rこれ以上の頂点は削れない為、処理を終了します.");
#endif
                                    break;
                                }
                            }
                        }
                        else
                        {
                            Heap.Pop();
                            edgeHeapData[data.EdgeID] = null;
                        }
                    }
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Finish_Notify",
                        currentFaceNum.ToString(figSetting),
                        originalFaceNum.ToString(figSetting),
                        (double)currentFaceNum / (double)originalFaceNum));
                    if (RemoveUnUsedVertex || rci > 0)
                    {
                        mMesh.RemoveInvalidFaces();
                    }
                    totalsum += sum;
                }
                catch (Exception ex)
                {
                    ErrorObjectOut();
                    throw ExcepHandle.CreateException(ex);
                }
            }

            // 実行後処理
            try
            {
                PostExecute();
            }
            catch (Exception ex)
            {
                ex.Data.Add("rmesh", mMesh);
#if DEBUG
                ex.Data.Add("cmesh", mCloneMesh);
#endif
                ErrorObjectOut();
                throw ExcepHandle.CreateException("Execute::failed postproc.", ex);
            }
            return totalsum;
        }
    }
}

// ReSharper restore RedundantCast
// ReSharper restore CompareOfFloatsByEqualityOperator
// ReSharper restore SuggestUseVarKeywordEvident
// ReSharper restore RedundantAssignment
// ReSharper restore TooWideLocalVariableScope
// ReSharper restore PossibleIntendedRethrow
