﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public partial class QemSimp
    {
        [Conditional("DEBUG")]
        private void CheckLegalKillVertex(int[] vertexborder, int kill)
        {
            // 非多様体頂点を落とそうとしている
            if (IsBorder(vertexborder[kill], QemVertexType.Nonmanifold))
            {
                throw new Exception("CheckLegalKillVertex :: tried to kill non-manifold vertex");
            }

            if (OptPosPolicy == QemOptPos.ZeroOne)
            {
                // 角っこを落とそうとしている場合
                if (IsBorder(vertexborder[kill], QemVertexType.Corner))
                {
                    throw new Exception("CheckLegalKillVertex :: tried to kill a vertex on corner");
                }
                // 交差点を落とそうとしている
                if (IsBorder(vertexborder[kill], QemVertexType.Junction))
                {
                    throw new Exception("CheckLegalKillVertex :: tried to kill a vertex on Junction");
                }
            }
        }

        /// <summary>
        /// 正規の縮退操作であるかどうかをチェックします
        /// </summary>
        [Conditional("DEBUG")]
        private void CheckLegalShrink(VtxTriMesh vtmesh, WingedEdgeMesh wingededge, int keep, int kill)
        {
            VtxTri tlist = vtmesh.VTL[keep];
            tlist.TriList.Sort();
            VtxWnEdge elist = wingededge.RepVEL[keep];

            int sn = mMesh.Sources.Count;
            int po = mMesh.GetSourceOfset(SourceType.Position, null);
            List<int> adjacentVtx = new List<int>();
            List<int> adjacentFace = new List<int>();

            // 完全に消えてしまったのにチェックは不要
            if (tlist.TriList.Count == 0) { return; }

            // 隣接関係が一致しているか？をチェック
            // 一致していない場合は問題がある
            foreach (var tri in tlist.TriList)
            {
                int vn = mMesh.mVnums[tri];
                int vo = mMesh.mVoffs[tri];
                bool hit = false;
                for (int vi = 0; vi < vn; vi++)
                {
                    int pid = mMesh.mVbuffs[(vo + vi) * sn + po];
                    if (pid == keep)
                    {
                        hit = true;
                    }
                    else if (!adjacentVtx.Contains(pid))
                    {
                        adjacentVtx.Add(pid);
                    }
                }

                if (!hit)
                {
                    throw new Exception("CheckLegalShrink::Adjacent information dosent match to another.");
                }
            }
            foreach (var edge in elist.EdgeList)
            {
                if (!adjacentFace.Contains(edge.Face0) && edge.Face0 != int.MaxValue)
                {
                    adjacentFace.Add(edge.Face0);
                }
                if (!adjacentFace.Contains(edge.Face1) && edge.Face1 != int.MaxValue)
                {
                    adjacentFace.Add(edge.Face1);
                }
            }
            adjacentFace.Sort();
            adjacentVtx.Sort();
            if (adjacentFace.Count != tlist.TriList.Count)
            {
                throw new Exception("CheckLegalShrink::Adjacent face Indices have some error");
            }

            if (adjacentVtx.Count != elist.EdgeList.Count)
            {
                throw new Exception("CheckLegalShrink::Adjacent face Indices have some error.");
            }

            foreach (var edge in elist.EdgeList)
            {
                int other = edge.Vertex0 == keep ? edge.Vertex1 : edge.Vertex0;

                if (!adjacentVtx.Contains(other))
                {
                    throw new Exception("CheckLegalShrink::Adjacent vertex information have some error");
                }
                if (!tlist.TriList.Contains(edge.Face0) && edge.Face0 != int.MaxValue)
                {
                    throw new Exception("CheckLegalShrink::isolated edge was detected.");
                }
                if (!tlist.TriList.Contains(edge.Face1) && edge.Face1 != int.MaxValue)
                {
                    throw new Exception("CheckLegalShrink::isolated edge was detected.");
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="vtmesh"></param>
        /// <param name="nv"></param>
        /// <param name="vid0"></param>
        /// <param name="vid1"></param>
        /// <param name="cost"></param>
        [Conditional("DEBUG")]
        private void CheckNewPosLegal(VtxTriMesh vtmesh, VectorN nv, int vid0, int vid1, double cost = 0.0f)
        {
            if (cost == double.MaxValue) { return; }

            int[] vid = new int[2]
                {
                    vid0,
                    vid1
                };

            int sn = mMesh.Sources.Count;
            int po = mMesh.GetSourceOfset(SourceType.Position, null);
            Vector3[] plist = new Vector3[3];
            VectorN[] vlist = new VectorN[3];
            Vector3 np = new Vector3();
            for (int i = 0; i < 3; i++)
            {
                plist[i] = new Vector3();
                vlist[i] = new VectorN((uint)DataDimension);
            }

            for (int i = 0; i < 3; i++)
            {
                np[i] = nv[po + i];
            }

            double chkCost = 0;
            for (int i = 0; i < 2; i++)
            {
                VtxTri vtri = vtmesh.VTL[vid[i]];
                foreach (int tid in vtri.TriList)
                {
                    Vector3 fp = new Vector3();
                    int vo = mMesh.mVoffs[tid];
                    mMesh.GetFaceVertex(tid, ref vlist);
                    // 位置座標のみを抜き出します
                    for (int vi = 0; vi < 3; vi++)
                    {
                        int vindex = mMesh.mVbuffs[(vo + vi) * sn + po];
                        for (int si = 0; si < 3; si++)
                        {
                            plist[vi][si] = vlist[vi][po + si];
                        }
                        if (vindex == vid[i])
                        {
                            fp = plist[vi];
                        }
                    }
                    Vector3 e0 = plist[1] - plist[0];
                    Vector3 e1 = plist[2] - plist[0];
                    Vector3 nml = e0.Cross(e1);
                    nml.Normalize();

                    double d0 = nml.Dot(fp);
                    double d1 = nml.Dot(np);

                    double d = d1 - d0;
                    chkCost += d * d;
                }
            }
        }

        /// <summary>
        /// 新しい位置座標が、妥当であるか？をざっくりとテストします
        /// </summary>
        [Conditional("DEBUG")]
        private void CheckNewPosLegal(VectorN nv, VectorN v0, VectorN v1, VectorN fv0, VectorN fv1)
        {
            // まず、必要頂点が、Fullと同じ構成要素か？
            foreach (var d0 in v0.Values)
            {
                bool hit = false;
                foreach (var fd0 in fv0.Values)
                {
                    if (d0 == fd0)
                    {
                        hit = true;
                        break;
                    }
                }
                if (!hit)
                {
                    throw new Exception("CheckNewPosLegal :: v0 could not find any correspond value.");
                }
            }

            // まず、必要頂点が、Fullと同じ構成要素か？
            foreach (var d1 in v1.Values)
            {
                bool hit = false;
                foreach (var fd1 in fv1.Values)
                {
                    if (d1 == fd1)
                    {
                        hit = true;
                        break;
                    }
                }
                if (!hit)
                {
                    throw new Exception("CheckNewPosLegal :: v1 could not find any correspond value.");
                }
            }

            // ZeroOne時には絶対に一致するデータがあるはず
            if (OptPosPolicy == QemOptPos.ZeroOne)
            {
                // まず、必要頂点が、Fullと同じ構成要素か？
                foreach (var d0 in nv.Values)
                {
                    bool hit = false;
                    foreach (var fd0 in fv0.Values)
                    {
                        if (d0 == fd0)
                        {
                            hit = true;
                            break;
                        }
                    }
                    foreach (var fd1 in fv1.Values)
                    {
                        if (d0 == fd1)
                        {
                            hit = true;
                            break;
                        }
                    }
                    if (!hit)
                    {
                        throw new Exception("CheckNewPosLegal :: nv could not find any correspond value.");
                    }
                }
            }
            else if (OptPosPolicy == QemOptPos.OptimizeEdge)
            {
#if VERBOSE
                uint size = nv.Dim;
                for (int i = 0; i < size; i++)
                {
                    if ( Math.Abs(nv[i]) <= double.Epsilon )
                    {
                        if (
                            Math.Abs(v0[i]) > double.Epsilon &&
                            Math.Abs(v1[i]) > double.Epsilon &&
                            Math.Abs(v0[i] + v1[i]) > double.Epsilon )
                        {
                            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine("CheckNewPosLegal :: nv could not find any correspond value.");
                            break;
                        }
                    }
                }
#endif
            }
        }

        [Conditional("DEBUG")]
        private void IsEdgeBreak(WingedEdge edge, int vi0, int vi1, bool save = false)
        {
            bool hit = false;
            if (edge.Vertex0 == vi0 && edge.Vertex1 == vi1)
            {
                hit = true;
                Debugger.Break();
            }
            if (edge.Vertex0 == vi1 && edge.Vertex1 == vi0)
            {
                hit = true;
                Debugger.Break();
            }

            if (save && hit)
            {
                ErrorObjectOut();
            }
        }

        [Conditional("DEBUG")]
        private void IsOnEdgeBreak(WingedEdge edge, int vi0, int vi1, bool save = false)
        {
            bool hit = false;
            if (edge.Vertex0 == vi0 || edge.Vertex1 == vi1)
            {
                hit = true;
                Debugger.Break();
            }
            if (edge.Vertex0 == vi1 || edge.Vertex1 == vi0)
            {
                hit = true;
                Debugger.Break();
            }
            if (save && hit)
            {
                ErrorObjectOut();
            }
        }

        [Conditional("DEBUG")]
        private void IndexCheck()
        {
#if DEBUG
            try
            {
                if (OptPosPolicy == QemOptPos.ZeroOne)
                {
                    MeshIndexTrack mit = new MeshIndexTrack();
                    mit.Run(mCloneMesh, mMesh, true);
                }
            }
            catch
            {
                throw;
            }
#endif
        }

        // 環境依存テスト/デバッグ出力
        [Conditional("LOCAL_TEST")]
        private void AssembleVertices(Mesh mesh, int[] rBoder, QemVertexType types, List<Vector3> points)
        {
            if (points == null)
            {
                points = new List<Vector3>();
            }
            MeshSourceBase posList = mesh.GetSource(SourceType.Position, null);
            for (int i = 0; i < mesh.VertexN; i++)
            {
                if ((rBoder[i] & (int)types) != 0)
                {
                    Vector3 tmp = null;
                    MeshSourceDblCtrler.GetAsVec3(posList, i, ref tmp);
                    points.Add(tmp);
                }
            }
            XYZImporter.SavePointsToXYZ(points, null, @"C:\depot\model\out\test.xyz");
        }

        [Conditional("LOCAL_TEST")]
        private void TestOut()
        {
            mMesh.DebugOutput(mMesh.MeshName + ".out.obj");
        }

        [Conditional("LOCAL_TEST")]
        private void ErrorObjectOut()
        {
            try
            {
                mCloneMesh.DebugOutput(mCloneMesh.MeshName + ".clone.obj");
                mMesh.DebugOutput(mMesh.MeshName + ".error.obj");
            }
            catch (Exception ex)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog(ex.Message);
            }
        }
    }
}
