﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    internal class QemHeapData
    {
        private double[] mCost   = new double[(int)SourceType.NONE];

        private int mHeapPos = 0;
        private int mHeapList = 0;

        private int mKeep = int.MaxValue;
        private int mEdgeID = 0;

        internal VectorN[] NewPosition { get; set; } = new VectorN[2];
        private int mEdgeType;

        private double mInterpolateParam = 0.0;
        /// <summary>
        /// エッジの種別によるヒープオーダー
        /// </summary>
        /// <param name="et"></param>
        /// <returns></returns>
        private int EdgeTypeOrder(int et)
        {
            int order = 1;
            if ((et & (int)QemEdgeType.NonManifold) != 0)
            {
                order = int.MaxValue;
            }
            if ((et & (int)QemEdgeType.HardEdgeOT) != 0)
            {
                order = Math.Max(order, int.MaxValue / 10);
            }
            if ((et & (int)QemEdgeType.LocalLimitOver) != 0)
            {
                order = Math.Max(order, 2);
            }
            return order;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        internal QemHeapData()
        {
            for (int i = 0; i < (int)SourceType.NONE; i++)
            {
                mCost[i] = double.MaxValue;
            }
            VWeight = 1.0;
        }

        /// <summary>
        /// 初期化
        /// </summary>
        internal void Clear()
        {
            mInterpolateParam = 0;
            mKeep = 0;
            NewPosition[0].SetZero();
            NewPosition[1].SetZero();
            mEdgeType = 0;
            VWeight = 1.0;
            for (int i = 0; i < (int)SourceType.NONE; i++)
            {
                mCost[i] = double.MaxValue;
            }
        }

        /// <summary>
        /// 体積変化コスト
        /// </summary>
        public double VCost
        {
            get { return mCost[(int)SourceType.Position]; }
            set { mCost[(int)SourceType.Position] = value; }
        }

        /// <summary>
        /// 体積コスト用ウェイト
        /// </summary>
        public double VWeight { get; set; } = 1.0;

        /// <summary>
        /// テクスチャ座標変化コスト
        /// </summary>
        public double TexCost
        {
            get { return mCost[(int)SourceType.Texture]; }
            set { mCost[(int)SourceType.Texture] = value; }
        }

        /// <summary>
        /// 指定のプロパティに関するキーを取得します
        /// </summary>
        public double GetCost(SourceType key = SourceType.Position)
        {
            return mCost[(int)key];
        }

        /// <summary>
        /// 指定のプロパティに関するコストを設定します
        /// </summary>
        /// <param name="v"></param>
        /// <param name="key"></param>
        /// <returns></returns>
        public double SetCost(double v, SourceType key = SourceType.Position)
        {
            return mCost[(int)key] = v;
        }

        /// <summary>
        /// 線形補間時のパラメーター
        /// </summary>
        public double InterpolateParam
        {
            get { return mInterpolateParam; }
            set { mInterpolateParam = value; }
        }

        /// <summary>
        /// エッジID
        /// </summary>
        public int EdgeID
        {
            get { return mEdgeID; }
            set { mEdgeID = value; }
        }

        /// <summary>
        /// ヒープ内位置
        /// </summary>
        public int HeapPos
        {
            get { return mHeapPos; }
            set { mHeapPos = value; }
        }

        /// <summary>
        /// ヒープ内位置
        /// </summary>
        public int HeapList
        {
            get { return mHeapList; }
            set { mHeapList = value; }
        }

        /// <summary>
        /// リダクションのパラメーター
        /// </summary>
        public int KeepSide
        {
            get { return mKeep; }
            set { mKeep = value; }
        }

        /// <summary>
        /// 新しい頂点座標
        /// </summary>
        public VectorN NewPosition0
        {
            get { return NewPosition[0]; }
            set { NewPosition[0] = value; }
        }
        /// <summary>
        /// 新しい頂点座標
        /// </summary>
        public VectorN NewPosition1
        {
            get { return NewPosition[1]; }
            set { NewPosition[1] = value; }
        }
        /// <summary>
        /// エッジの種別を取得設定
        /// </summary>
        public int EdgeType
        {
            get { return mEdgeType; }
            set { mEdgeType = value; }
        }

        /// <summary>
        /// エッジの種別と、コストによって優先順位を決定する
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public int CompareTo(object obj, SourceType key)
        {
            QemHeapData rhs = (QemHeapData)obj;

            if (EdgeTypeOrder(mEdgeType) == EdgeTypeOrder(rhs.mEdgeType))
            {
                if (mCost[(int)key] < rhs.mCost[(int)key])
                {
                    return -1;
                }
                else if (mCost[(int)key] > rhs.mCost[(int)key])
                {
                    return 1;
                }
                return 0;
            }
            else if (EdgeTypeOrder(mEdgeType) < EdgeTypeOrder(rhs.mEdgeType))
            {
                return -1;
            }
            return 1;
        }
    }
}
