﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.nw4f_3dif;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public enum SourceType
    {
        Position,
        Normal,
        Texture,
        Color,
        Binormal,
        Tangent,
        BlendWeight,
        EtceteraInt,
        EtceteraDouble,
        NONE
    }

    public class SourceKey : IComparable, IEquatable<SourceKey>
    {
        public SourceKey(SourceType _type, string _name)
        {
            type = _type;
            name = _name;
        }

        public SourceType type { get; set; } = SourceType.Position;
        public string name { get; set; } = null;

        public override int GetHashCode()
        {
            int tmp = type.GetHashCode();
            //if (name != null)
            //    tmp = tmp^name.GetHashCode();
            return tmp;
        }

        public bool Equals(SourceKey rhs)
        {
            return CompareTo(rhs) == 0;
        }

        public int CompareTo(SourceType rtype, string rname)
        {
            if (rtype == type)
            {
                if (name != null && rname != null)
                {
                    return name.CompareTo(rname);
                }
                return 0;
            }
            return type.CompareTo(rtype);
        }

        public override bool Equals(object obj)
        {
            if (obj == null)
            {
                return false;
            }
            return Equals(obj as SourceKey);
        }

        public int CompareTo(object obj)
        {
            if (obj == null)
            {
                return -1;
            }
            SourceKey right = obj as SourceKey;
            return CompareTo(right.type, right.name);
        }
    }

    public class SourceKeyComparer : IEqualityComparer<SourceKey>
    {
        public bool Equals(SourceKey x, SourceKey y)
        {
            return x.Equals(y);
        }
        public int GetHashCode(SourceKey obj)
        {
            return obj.GetHashCode();
        }
    }

    /// <summary>
    /// ソースリストのベース
    /// </summary>
    public abstract class MeshSourceBase
    {
        public static Type GetSourceType(SourceType type)
        {
            switch (type)
            {
                case SourceType.EtceteraInt:
                    return typeof(MeshSource<int>);
                case SourceType.EtceteraDouble:
                    return typeof(MeshSource<double>);
                default:
                    return typeof(MeshSource<double>);
            }
        }

        public static Type GetValueType(SourceType type)
        {
            switch (type)
            {
                case SourceType.EtceteraInt:
                    return typeof(int);
                case SourceType.EtceteraDouble:
                    return typeof(double);
                default:
                    return typeof(double);
            }
        }

        public static object GetDefaultValue(SourceType type)
        {
            switch (type)
            {
                case SourceType.EtceteraInt:
                    return 0;
                case SourceType.EtceteraDouble:
                    return 0.0;
                default:
                    return 0.0;
            }
        }

        protected ResizableList<object> _row_data { get; set; } = new ResizableList<object>();
        protected int _stride { get; set; } = 1;
        //protected string _name = string.Empty;
        //protected SourceType _type = SourceType.NONE;
        protected SourceKey _key { get; set; } = new SourceKey(SourceType.Position, null);
        protected Type _dataType { get; set; }

        public MeshSourceBase(string name, SourceType type, int stride)
        {
            _key.name = name;
            _key.type = type;
            _stride = stride;
        }

        public vtx_attrib_quantize_typeType QuantizeType { get; set; } = vtx_attrib_quantize_typeType.none;

        public SourceKey Key
        {
            get { return _key; }
        }

        /// <summary>
        /// リソースの名前を取得
        /// </summary>
        public string Name
        {
            get { return _key.name; }
        }

        /// <summary>
        /// リソースの種別を取得
        /// </summary>
        public SourceType Type
        {
            get { return _key.type; }
        }

        /// <summary>
        /// １データ幅を取得
        /// </summary>
        public int Stride
        {
            get { return _stride; }
        }

        /// <summary>
        /// 生の数値配列のサイズを取得
        /// </summary>
        public int RowCount
        {
            get { return _row_data.Count; }
        }

        /// <summary>
        /// データの要素数を取得
        /// </summary>
        public int Count
        {
            get { return _row_data.Count / _stride; }
        }

        /// <summary>
        /// データの要素数を取得
        /// </summary>
        internal int Capacity
        {
            get { return _row_data.Capacity / _stride; }
        }

        public IEnumerable<object> Data
        {
            get { return _row_data; }
        }

        public abstract MeshSourceBase Clone();
        public abstract void SetLength(int length, bool isrow);
        public abstract void SetRowValue(int index, object data);
        public abstract void AddRowValue(object data);

        /// <summary>
        /// 生の値を取得する
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public object GetRowValue(int index)
        {
            if (_row_data.Count <= index)
            {
                throw ExcepHandle.CreateException(string.Format("Index over flow.index {0} Capa {1}", index, Capacity));
            }
            return (_row_data[index]);
        }

        /// <summary>
        /// 二つのソースが等しいか？
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        public bool EqualityTest(MeshSourceBase source)
        {
            if (Count != source.Count)
            {
                return false;
            }
            if (Stride != source.Stride)
            {
                return false;
            }
            if (Name != source.Name)
            {
                return false;
            }
            if (Type != source.Type)
            {
                return false;
            }

            Type type = GetValueType(Type);
            for (int i = 0; i < RowCount; i++)
            {
                if (type == typeof(double))
                {
                    if ((double)_row_data[i] != (double)source._row_data[i])
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        /// <summary>
        /// 量子化型によってUVのBVを取得します
        /// </summary>
        /// <returns></returns>
        public BoundingSquare GetQuantizeLimit()
        {
            var result = new BoundingSquare();

            switch (QuantizeType)
            {
                case vtx_attrib_quantize_typeType.unorm_8_8:
                case vtx_attrib_quantize_typeType.unorm_16_16:
                case vtx_attrib_quantize_typeType.unorm_16:
                case vtx_attrib_quantize_typeType.unorm_8_8_8_8:
                case vtx_attrib_quantize_typeType.unorm_10_10_10_2:
                    result.Update(new Vector2(0, 0));
                    result.Update(new Vector2(1, 1));
                    break;

                case vtx_attrib_quantize_typeType.snorm_10_10_10_2:
                case vtx_attrib_quantize_typeType.snorm_8_8_8_8:
                case vtx_attrib_quantize_typeType.snorm_8_8:
                case vtx_attrib_quantize_typeType.snorm_16_16:
                    result.Update(new Vector2(-1, -1));
                    result.Update(new Vector2(1, 1));
                    break;
                default:
                    result.Reset();
                    result.Update(new Vector2(-float.MinValue, -float.MinValue));
                    result.Update(new Vector2( float.MinValue,  float.MinValue));
                    break;
            }
            return result;
        }
    }

    /// <summary>
    ///
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class MeshSource<T> : MeshSourceBase where T : struct
    {
        public MeshSource(string name, SourceType type, int stride)
        : base(name, type, stride)
        {
            _dataType = typeof(int);
        }

        /// <summary>
        /// インデクサ
        /// </summary>
        public T this[int index]
        {
            get { return (T)_row_data[index]; }
            set
            {
                SetRowValue(index, value);
            }
        }

        /// <summary>
        /// クローン作製
        /// </summary>
        /// <returns></returns>
        public override MeshSourceBase Clone()
        {
            try
            {
                MeshSource<T> clone = new MeshSource<T>(Name, Type, Stride);
                clone.QuantizeType = QuantizeType;
                clone.SetLength(RowCount, true);
                for (int i = 0; i < clone.RowCount; i++)
                {
                    clone._row_data[i] = _row_data[i];
                }
                return clone;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Failed to clone.", ex);
            }
        }

        /// <summary>
        /// リソースのサイズを設定する
        /// </summary>
        /// <param name="length">長さ</param>
        /// <param name="isrow">true : そのまま false: ストライドとかける</param>
        public override void SetLength(int length, bool isrow)
        {
            try
            {
                if (isrow)
                {
                    _row_data.Resize(length, 0.0);
                }
                else
                {
                    _row_data.Resize(length * _stride, 0.0);
                }
            }
            catch (Exception exp)
            {
                string message = string.Format("Failed to set length {0},{1} ", length, _row_data.Count);
                throw ExcepHandle.CreateException(message, exp);
            }
        }

        /// <summary>
        /// 値を追加
        /// </summary>
        /// <param name="data"></param>
        public override void AddRowValue(object data)
        {
            _row_data.Add(data);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="index"></param>
        /// <param name="data"></param>
        public override void SetRowValue(int index, object data)
        {
            if (RowCount <= index)
            {
                throw ExcepHandle.CreateException(string.Format("Index over flow.index {0} Capa {1}", index, Capacity));
            }
            _row_data[index] = data;
        }

        /// <summary>
        /// 指定のインデックスへ、生のデータ配列を格納する
        /// </summary>
        /// <param name="index">格納先インデックス</param>
        /// <param name="data">格納するデータ</param>
        public void SetRowValues(int index, T[] data)
        {
            if (RowCount <= index)
            {
                throw ExcepHandle.CreateException(string.Format("Index over flow.index {0} Capa {1}", index, Capacity));
            }
            for (int i = 0; i < _stride; i++)
            {
                _row_data[index + i] = data[i];
            }
        }

        /// <summary>
        /// 指定のインデックスへ、生のデータ配列を格納する
        /// </summary>
        /// <param name="index">格納先インデックス</param>
        /// <param name="data">格納するデータ</param>
        public void SetRowValue(int index, T data)
        {
            if (RowCount <= index)
            {
                throw ExcepHandle.CreateException(string.Format("Index over flow.index {0} Capa {1}", index, Capacity));
            }
            _row_data[index] = data;
        }

        /// <summary>
        /// 指定インデックスから、生の配列を取得する
        /// </summary>
        /// <param name="index">取得開始インデックス</param>
        /// <param name="data">データ格納先配列</param>
        public void GetRowValue(int index, ref T[] data)
        {
            if (RowCount <= index)
            {
                throw ExcepHandle.CreateException(string.Format("Index over flow.index {0} Capa {1}", index, Capacity));
            }

            data = new T[_stride];
            for (int i = 0; i < _stride; i++)
            {
                data[i] = ((T)_row_data[index + i]);
            }
        }

        /// <summary>
        /// 指定インデックスから、生の数値を取得する
        /// </summary>
        /// <param name="index">取得開始インデックス</param>
        public new T GetRowValue(int index)
        {
            if (RowCount <= index)
            {
                throw ExcepHandle.CreateException(string.Format("Index over flow.index {0} Capa {1}", index, Capacity));
            }
            return ((T)_row_data[index]);
        }

        public T CastData(object data)
        {
            return (T)data;
        }
    }

    /// <summary>
    /// インターフェース
    /// </summary>
    public class MeshSourceDblCtrler
    {
        /// <summary>
        /// 非数が含まれているか？をチェックします
        /// </summary>
        /// <param name="src"></param>
        public static void CheckNaN(MeshSourceBase src)
        {
            MeshSource<double> dsrc = src as MeshSource<double>;
            for (int i = 0; i < dsrc.RowCount; i++)
            {
                if (dsrc[i] != dsrc[i])
                {
                    throw ExcepHandle.CreateException("Nan");
                }
            }
        }

        /// <summary>
        /// Vector3 として設定する
        /// </summary>
        /// <param name="index">Vec3としてのインデックス</param>
        /// <param name="x">x</param>
        /// <param name="y">y</param>
        /// <param name="z">z</param>
        public static void SetAsVec3(MeshSourceBase src, int index, double x, double y, double z)
        {
            if (src.Capacity <= index)
            {
                throw ExcepHandle.CreateException("Index over flow.");
            }
            if (src.Stride < 3)
            {
                throw ExcepHandle.CreateException($"Strides are less than 3. (stride:{src.Stride})");
            }
            if (double.IsNaN(x) || double.IsNaN(y) || double.IsNaN(z))
            {
                throw ExcepHandle.CreateException("Nan value is found.");
            }

            src.SetRowValue(index * src.Stride + 0, x);
            src.SetRowValue(index * src.Stride + 1, y);
            src.SetRowValue(index * src.Stride + 2, z);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="src"></param>
        /// <param name="index"></param>
        /// <param name="v"></param>
        public static void SetAsVec3(MeshSourceBase src, int index, Vector3 v)
        {
            SetAsVec3(src, index, v[0], v[1], v[2]);
        }

        /// <summary>
        /// Vector3 として取得する
        /// </summary>
        /// <param name="src"></param>
        /// <param name="index"></param>
        /// <param name="val"></param>
        public static void GetAsVec3(MeshSourceBase src, int index, ref Vector3 val)
        {
            if (src.Capacity <= index)
            {
                throw ExcepHandle.CreateException("Index over flow.");
            }
            if (src.Stride < 3)
            {
                throw ExcepHandle.CreateException($"Strides are less than 3. (stride:{src.Stride})");
            }
            if (val == null)
            {
                val = new Vector3();
            }

            val[0] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 0));
            val[1] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 1));
            val[2] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 2));
        }

        /// <summary>
        /// Vector2として設定する
        /// </summary>
        /// <param name="index">vec2としてのインデックス</param>
        /// <param name="x">x</param>
        /// <param name="y">y</param>
        public static void SetAsVec2(MeshSourceBase src, int index, double x, double y)
        {
            try
            {
                if (src.Capacity <= index)
                {
                    throw ExcepHandle.CreateException("Index over flow.");
                }
                if (src.Stride != 2)
                {
                    throw ExcepHandle.CreateException("Strides are not same.");
                }

                src.SetRowValue(index * src.Stride + 0, x);
                src.SetRowValue(index * src.Stride + 1, y);
            }
            catch (Exception exp)
            {
                string message = string.Format("Fail to set vec2 {0},{1}/{2} ", index, x, y);
                throw ExcepHandle.CreateException(message, exp);
            }
        }

        /// <summary>
        /// Vector2 としてデータを取得
        /// </summary>
        /// <param name="index"></param>
        /// <param name="val"></param>
        public static void GetAsVec2(MeshSourceBase src, int index, ref Vector2 val)
        {
            try
            {
                if (src.Capacity <= index)
                {
                    throw ExcepHandle.CreateException("Index over flow.");
                }
                if (src.Stride != 2)
                {
                    throw ExcepHandle.CreateException("Stride is illegal.");
                }

                val[0] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 0));
                val[1] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 1));
            }
            catch (Exception exp)
            {
                string message = string.Format("Fail to get value as vec2 {0}", index);
                throw new Exception(message, exp);
            }
        }

        /// <summary>
        /// Vector4 として取得する
        /// </summary>
        /// <param name="src"></param>
        /// <param name="index"></param>
        /// <param name="val"></param>
        public static void GetAsVec4(MeshSourceBase src, int index, ref Vector4 val)
        {
            if (src.Capacity <= index)
            {
                throw ExcepHandle.CreateException("Index over flow.");
            }
            if (src.Stride < 4)
            {
                throw ExcepHandle.CreateException("sride is illegal");
            }
            if (val == null)
            {
                val = new Vector4();
            }

            val[0] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 0));
            val[1] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 1));
            val[2] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 2));
            val[3] = Convert.ToDouble(src.GetRowValue(index * src.Stride + 3));
        }

        /// <summary>
        /// Vector4として値を設定する
        /// </summary>
        /// <param name="index">Vector4としてのインデックス</param>
        /// <param name="x">x</param>
        /// <param name="y">y</param>
        /// <param name="z">z</param>
        /// <param name="w">w</param>
        public static void SetAsVec4(MeshSourceBase src, int index, double x, double y, double z, double w)
        {
            try
            {
                if (src.Capacity <= index)
                {
                    throw ExcepHandle.CreateException("Index over flow.");
                }
                if (src.Stride != 4)
                {
                    throw ExcepHandle.CreateException("Stride is illegal.");
                }
                if (double.IsNaN(x) || double.IsNaN(y) || double.IsNaN(z) || double.IsNaN(w))
                {
                    throw ExcepHandle.CreateException("Nan value is found.");
                }

                src.SetRowValue(index * src.Stride + 0, x);
                src.SetRowValue(index * src.Stride + 1, y);
                src.SetRowValue(index * src.Stride + 2, z);
                src.SetRowValue(index * src.Stride + 3, w);
            }
            catch (Exception exp)
            {
                string message = string.Format("fail to set vec4 {0},{1}/{2}/{3}/{4} ", index, x, y, z, w);
                throw ExcepHandle.CreateException(message, exp);
            }
        }

        /// <summary>
        /// Vector4として値を設定する
        /// </summary>
        /// <param name="index">Vector4としてのインデックス</param>
        /// <param name="x">x</param>
        /// <param name="y">y</param>
        /// <param name="z">z</param>
        public static void SetAsVec4(MeshSourceBase src, int index, double x, double y, double z)
        {
            try
            {
                if (src.Capacity <= index)
                {
                    throw ExcepHandle.CreateException("Index over flow.");
                }
                if (src.Stride != 4)
                {
                    throw ExcepHandle.CreateException("Stride is illegal");
                }
                if (double.IsNaN(x) || double.IsNaN(y) || double.IsNaN(z))
                {
                    throw ExcepHandle.CreateException("Nan value is found.");
                }

                src.SetRowValue(index * src.Stride + 0, x);
                src.SetRowValue(index * src.Stride + 1, y);
                src.SetRowValue(index * src.Stride + 2, z);
            }
            catch (Exception exp)
            {
                string message = string.Format("fail to set vec4 {0},{1}/{2}/{3} ", index, x, y, z);
                throw ExcepHandle.CreateException(message, exp);
            }
        }

        /// <summary>
        /// Vector3 として取得する
        /// </summary>
        /// <param name="src"></param>
        /// <param name="index"></param>
        /// <param name="val"></param>
        public static void GetAsVecN(MeshSourceBase src, int index, int stride, ref VectorN val)
        {
            if (src.Capacity <= index)
            {
                throw ExcepHandle.CreateException("Index over flow.");
            }
            if (src.Stride < stride)
            {
                throw ExcepHandle.CreateException("Stride is larger than source stride.");
            }
            if (val == null)
            {
                val = new VectorN((uint)stride);
            }

            for (int i = 0; i < stride; i++)
            {
                val[i] = Convert.ToDouble(src.GetRowValue(index * src.Stride + i));
            }
        }
    }
}
