﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.iflib;
using nw4f.tinymathlib;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    public class MeshIndexTrack
    {
        private Mesh mSMesh = null;
        private Mesh mDMesh = null;
        private int _table_size = 1000;

        private SpacialHashList<long> mSrcFIndexList;
        private UniformGrid<int> mPosToInt;

        private int sX = 50;
        private int sY = 50;
        private int sZ = 50;
        private double mPerc = ShapeMerger.Percentage;

        /// <summary>
        /// ソースになるメッシュ
        /// </summary>
        public Mesh SourceMesh
        {
            get { return mSMesh; }
            set { mSMesh = value; }
        }

        /// <summary>
        /// 編集されるメッシュ
        /// </summary>
        public Mesh DestMesh
        {
            get { return mDMesh; }
            set { mDMesh = value; }
        }

        public double Percentage
        {
            get { return mPerc; }
            set
            {
                mPerc = value;
                mPerc = Math.Max(0.0, mPerc);
                mPerc = Math.Min(1.0, mPerc);
            }
        }
        /// <summary>
        /// 一致する頂点とみなすための距離の閾値
        /// </summary>
        public double _threshold { get; set; } = 1e-5;
        public double Threshold
        {
            get { return _threshold; }
            set { _threshold = value; }
        }

        /// <summary>
        /// 頂点結合に使用する閾値を、メッシュから求める
        /// </summary>
        private void ComputeThreshold()
        {
            Threshold = mSMesh.ComputeAverageEdgeLength() * Percentage;
        }

        /// <summary>
        /// ハッシュテーブルを作成します
        /// </summary>
        private void CreateHashTable()
        {
            mSMesh.ComputeBounds();
            BoundingBox bb = mSMesh.GetTotalBoundingBox();
            bb.Stretch(0.05f);
            mPosToInt = new UniformGrid<int>();
            mPosToInt.Initialize(bb, sX, sY, sZ, false);

            mSrcFIndexList = new SpacialHashList<long>();
            mSrcFIndexList.Count = _table_size;
        }

        /// <summary>
        /// 事前準備として、ソースとなるメッシュのWedgeをハッシュテーブルに分ける
        /// </summary>
        private void Prepare()
        {
            try
            {
                CreateHashTable();

                Vector3 tv3 = new Vector3();
                int faceN = mSMesh.FaceN;
                int sn = mSMesh.SourceN;
                int po = mSMesh.GetSourceOfset(SourceType.Position, null);
                MeshSourceBase Positions = mSMesh.GetSource(SourceType.Position, null);

                for (int fi = 0; fi < faceN; fi++)
                {
                    int vo = mSMesh.mVoffs[fi];
                    int vn = mSMesh.mVnums[fi];
                    for (int vi = 0; vi < vn; vi++)
                    {
                        int vid = mSMesh.mVbuffs[(vo + vi) * sn + po];
                        MeshSourceDblCtrler.GetAsVec3(Positions, vid, ref tv3);
                        int x = 0, y = 0, z = 0;
                        mPosToInt.GetSpacePos(ref x, ref y, ref z, tv3);
                        for (int xi = Math.Max(x - 1, 0); xi < Math.Min(x + 1, sX); xi++)
                        {
                            for (int yi = Math.Max(y - 1, 0); yi < Math.Min(y + 1, sY); yi++)
                            {
                                for (int zi = Math.Max(z - 1, 0); zi < Math.Min(z + 1, sZ); zi++)
                                {
                                    mSrcFIndexList.Add(xi, yi, zi, fi);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Prepare()::failed to preparation.", ex);
            }
        }

        private Vector3 _tmpV3 = new Vector3();

        /// <summary>
        /// 指定された頂点座標と一致する頂点IDを元メッシュから取得します
        /// </summary>
        /// <param name="pos"></param>
        /// <returns></returns>
        private ResizableList<int> SearchCoincidentSrcFaces(Vector3 pos)
        {
            ResizableList<int> result = new ResizableList<int>();
            int x = 0;
            int y = 0;
            int z = 0;

            mPosToInt.GetSpacePos(ref x, ref y, ref z, pos);
            //mSrcFIndexList.Add(x, y, z, fi);

            for (int xi = Math.Max(x - 1, 0); xi < Math.Min(x + 1, sX); xi++)
            {
                for (int yi = Math.Max(y - 1, 0); yi < Math.Min(y + 1, sY); yi++)
                {
                    for (int zi = Math.Max(z - 1, 0); zi < Math.Min(z + 1, sZ); zi++)
                    {
                        List<long> hashList = mSrcFIndexList.GetList(xi, yi, zi);
                        foreach (int fid in hashList)
                        {
                            result.Add(fid);
                        }
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// 実行
        /// </summary>
        /// <param name="src"></param>
        /// <param name="dst"></param>
        /// <param name="onlyCheck"></param>
        /// <param name="hardCheck"></param>
        public void Run(Mesh src = null, Mesh dst = null, bool onlyCheck = false, bool hardCheck = false)
        {
            ComputeThreshold();
            //Hard は、元の面の頂点構成に完全一致
            //Soft は、ストリームの値のうち一つでも一致すればＯＫ
            if (hardCheck)
            {
                HardReplace(src, dst, onlyCheck);
            }
            else
            {
                SoftReplace(src, dst, onlyCheck);
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        private Dictionary<int, List<VectorN>> CreateHasedList(MeshSourceBase source)
        {
            Dictionary<int, List<VectorN>> result = new Dictionary<int, List<VectorN>>();

            return result;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="src"></param>
        /// <param name="dst"></param>
        /// <param name="onlyCheck"></param>
        public void SoftReplace(Mesh src = null, Mesh dst = null, bool onlyCheck = false)
        {
            if (src != null)
            {
                mSMesh = src;
            }
            if (dst != null)
            {
                mDMesh = dst;
            }
            if (mSMesh.GetSources(SourceType.Position).Count != 1)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Merge_Vertex", "IndexTracker"));
                return;
            }

            var dstFaceNum = mDMesh.FaceN;
            var dstSourceNum = mDMesh.SourceN;
            var tmpDestPoslist = new VectorN[3];

            var dstPropDim = mDMesh.GetPropDimension();
            for (var vertexIndex = 0; vertexIndex < 3; vertexIndex++)
            {
                tmpDestPoslist[vertexIndex] = new VectorN((uint)dstPropDim);
            }

            for (var faceIndex = 0; faceIndex < dstFaceNum; faceIndex++)
            {
                var vertxOffset = mDMesh.mVoffs[faceIndex];
                var vertexNum = mDMesh.mVnums[faceIndex];
                var sourceId = new int[vertexNum];

                for (var sourceIndex = 0; sourceIndex < dstSourceNum; sourceIndex++)
                {
                    MeshSourceBase dst_source = mDMesh.Sources[sourceIndex];
                    MeshSourceBase src_source = mSMesh.GetSource(dst_source.Key.type, dst_source.Key.name);
                    if (src_source == null)
                    {
                        throw new Exception("Source stream can not be found.");
                    }

                    mDMesh.GetFaceVtxIndices(faceIndex, ref sourceId, dst_source.Key.type, dst_source.Key.name);
                    var tmpVectorN = new VectorN((uint)dst_source.Stride);
                    var tmpVectorN2 = new VectorN((uint)src_source.Stride);
                    for (int vertexIndex = 0; vertexIndex < vertexNum; vertexIndex++)
                    {
                        MeshSourceDblCtrler.GetAsVecN(dst_source, sourceId[vertexIndex], dst_source.Stride, ref tmpVectorN);
                        MeshSourceDblCtrler.GetAsVecN(src_source, sourceId[vertexIndex], src_source.Stride, ref tmpVectorN2);
                        // 二つの値が既にほぼ同じである場合
                        if (!VectorN.IsNearEqual(tmpVectorN, tmpVectorN2))
                        {
                            var minIndex = -1;
                            var minLen = double.MaxValue;
                            //まず力づくでチェック
                            for (var positionIndex = 0; positionIndex < src_source.Count; positionIndex++)
                            {
                                MeshSourceDblCtrler.GetAsVecN(src_source, positionIndex, src_source.Stride, ref tmpVectorN2);
                                var d = tmpVectorN - tmpVectorN2;

                                if (d.Norm() < minLen)
                                {
                                    minLen = d.Norm();
                                    minIndex = positionIndex;
                                    if (minLen <= double.Epsilon)
                                    {
                                        break;
                                    }
                                }
                            }

                            // 一致したＩＤに書き換える
                            if (minLen > 1e-5 && minIndex == -1)
                            {
                                throw new Exception("Duplicate value cant be found in the original source stream.");
                            }
                            else
                            {
                                mDMesh.mVbuffs[dstSourceNum * (vertxOffset + vertexIndex) + sourceIndex] = minIndex;
                            }
                        }
                    }
                }
                if ((faceIndex % 10) == 0)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog(string.Format("\r Index Tracker [{0,7:d}]/[{1,7:d}]", faceIndex, dstFaceNum));
                }
            }
        }

        /// <summary>
        /// 編集対象となるメッシュの、各フェースの各位置、プロパティをオリジナルの
        /// メッシュの各値に置き換えて行きます。
        /// 面頂点の構成が完全に一致する必要があります。
        /// </summary>
        public void HardReplace(Mesh src = null, Mesh dst = null, bool onlyCheck = false)
        {
            try
            {
                if (src != null)
                {
                    mSMesh = src;
                }
                if (dst != null)
                {
                    mDMesh = dst;
                }

                if (mSMesh.GetSources(SourceType.Position).Count != 1)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Merge_Vertex", "IndexTracker"));
                    return;
                }

                Prepare();

                int dfaceN = mDMesh.FaceN;
                int sn = mDMesh.SourceN;
                int po = mDMesh.GetSourceOfset(SourceType.Position, null);

                Vector3 dpos = new Vector3();
                VectorN[] tmpDPlist = new VectorN[3];

                int[] srcIndex = new int[sn];
                int dstPromDim = mDMesh.GetPropDimension();
                for (int vi = 0; vi < 3; vi++)
                {
                    tmpDPlist[vi] = new VectorN((uint)dstPromDim);
                }

                for (int fi = 0; fi < dfaceN; fi++)
                {
                    int vo = mDMesh.mVoffs[fi];
                    int vn = mDMesh.mVnums[fi];
                    mDMesh.GetFaceVertex(fi, ref tmpDPlist);

                    for (int vi = 0; vi < vn; vi++)
                    {
                        int dvid = mDMesh.mVbuffs[(vo + vi) * sn + po];
                        VectorN dstVtx = tmpDPlist[vi];
                        // ソース側から位置座標が一致するWedgeを全取得してくる
                        dpos[0] = dstVtx[po + 0];
                        dpos[1] = dstVtx[po + 1];
                        dpos[2] = dstVtx[po + 2];

                        ResizableList<int> srcWVertices = SearchCoincidentSrcFaces(dpos);
                        bool find = false;
                        foreach (var sfaceIndex in srcWVertices)
                        {
                            // 一致する頂点を取得
                            int svo = mSMesh.mVoffs[sfaceIndex];
                            int svn = mSMesh.mVnums[sfaceIndex];
                            int ssn = mSMesh.SourceN;
                            for (int vj = 0; vj < svn && !find; vj++)
                            {
                                find = false;
                                // 初期化
                                for (int si = 0; si < sn; si++)
                                {
                                    srcIndex[si] = int.MaxValue;
                                }
                                Vector3 spos = new Vector3();
                                int vid = mSMesh.mVbuffs[(svo + vj) * ssn + po];
                                mSMesh.GetVertexPos(vid, ref spos);
                                Vector3 diff = spos - dpos;

                                if (Threshold >= diff.Norm())
                                {
                                    find = true;
                                    int dim = 0;
                                    for (int sj = 0; sj < sn && find; sj++)
                                    {
                                        bool error = false;
                                        int sid = mSMesh.mVbuffs[(svo + vj) * ssn + sj];
                                        int sindex = sid * mSMesh.Sources[sj].Stride;

                                        // 位置は二回チェックしなくていい
                                        if (mSMesh.Sources[sj].Type != SourceType.Position)
                                        {
                                            for (int sti = 0; sti < mSMesh.Sources[sj].Stride; sti++)
                                            {
                                                Type type = MeshSourceBase.GetSourceType(mSMesh.Sources[sj].Type);
                                                if (type == typeof(MeshSource<int>))
                                                {
                                                    int val = (int)mSMesh.Sources[sj].GetRowValue(sindex + sti);
                                                    if (Math.Abs(val - dstVtx[dim + sti]) > double.Epsilon)
                                                    {
                                                        error = true;
                                                    }
                                                }
                                                else
                                                {
                                                    double val = Convert.ToDouble(mSMesh.Sources[sj].GetRowValue(sindex + sti));
                                                    if (Math.Abs(val - dstVtx[dim + sti]) > 1e-7)
                                                    {
                                                        error = true;
                                                    }
                                                }
                                            }
                                        }

                                        // 座標が完全に一致した場合
                                        if (!error)
                                        {
                                            srcIndex[sj] = sid;
                                        }
                                        else
                                        {
                                            find = false;
                                        }
                                        dim += mSMesh.Sources[sj].Stride;
                                    }
                                }
                            }

                            // 完全に一致する面頂点を見つけた場合
                            if (find)
                            {
                                for (int si = 0; si < sn; si++)
                                {
                                    mDMesh.mVbuffs[(vo + vi) * sn + si] = srcIndex[si];
                                }
                                break;
                            }
                        }
                        if (!find)
                        {
                            string message = string.Format(" -> dfid {0} dvid {1}", fi, vi);
                            throw new Exception("MeshIndexTrack:: there are no coinsidence vertex." + message);
                        }
                    }
                }

                // 一致するインデックスがあるかどうかをチェックするだけなら不要
                if (!onlyCheck)
                {
                    // ソースを複製する
                    for (int si = 0; si < sn; si++)
                    {
                        mDMesh.Sources[si] = mSMesh.Sources[si].Clone();
                    }
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Failed to tarack index.", ex);
            }
        }
    }
}
