﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.iflib;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    /// <summary>
    /// 面毎の頂点インデックスの再配置を行うクラス
    /// Fmdbファイルの仕様上、全てのリソースの長さが同じであることを期待される
    /// その状態にするために、インデックスの再配置や必要であればリソースの作成まで含めて行うクラス
    /// （注）Fmdbへの出力に対応した処理で、メッシュデータの構成的には不要
    /// </summary>
    public class FaceAttributeRearrange
    {
        private Mesh mMesh = null;

        public FaceAttributeRearrange(Mesh mesh)
        {
            mMesh = mesh;
        }

        public void Run()
        {
            if (mMesh != null && mMesh.GetSources(SourceType.Position).Count != 1)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Merge_Vertex", "ReArrange"));
                return;
            }

            try
            {
                var wedgeSplitter = new WedgeSplitter(mMesh);
                wedgeSplitter.Run();
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Winged edge splitter stopped with failure.", ex);
            }

            try
            {
                Mesh origin = mMesh.Clone();
                MeshSourceBase poslist = origin.GetSource(SourceType.Position, null);
                if (poslist == null)
                {
                    var stack = new StackFrame(true);
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_InternalExceptionMsg", stack.GetMethod().ToString(), Path.GetFileName(stack.GetFileName())));
                    throw new Exception("No verteces are found.");
                }

                bool isSame = true;
                foreach (var source in mMesh.Sources)
                {
                    if (source.Type == SourceType.Position)
                    {
                        continue;
                    }
                    if (source.Count != poslist.Count)
                    {
                        isSame = false;
                    }
                }

                // ソースを作り直す
                if (!isSame)
                {
                    foreach (MeshSourceBase dsource in mMesh.Sources)
                    {
                        dsource.SetLength(origin.VertexN, false);
                    }
                }

                int sn = origin.SourceN;
                int po = origin.GetSourceOfset(SourceType.Position, null);
                int faceN = origin.FaceN;
                var dstPositions = mMesh.GetSource(SourceType.Position, null);

                // クローン（オリジナル）からWedgeを作成
                var mw = new MeshWedges(origin);
                mw.Create();

                foreach (var we in mw.Wedges)
                {
                    if (we == null) { continue; }

                    int prpN = we.Props.Count;
                    int maxN = 0;
                    int pi = 0;
                    for (pi = 0; pi < prpN; pi++)
                    {
                        maxN = Math.Max(maxN, we.Props[pi].Count);
                    }

                    if (maxN != 1)
                    {
                        throw ExcepHandle.CreateException("Winged edge split is not completed or has functional error.");
                    }

                    pi = 0;
                    foreach (MeshSourceBase source in origin.Sources)
                    {
                        int pid = we.Props[pi][0];
                        MeshSourceBase dsource = mMesh.GetSource(source.Type, source.Name);

                        if (dsource == dstPositions)
                        {
                            continue;
                        }

                        int stride = source.Stride;
                        for (int st = 0; st < stride; st++)
                        {
                            var value = source.GetRowValue(pid * stride + st);
                            // 一対一で対応する用に貼り直す
                            dsource.SetRowValue(we.Vertex * stride + st, value);
                        }
                        pi++;
                    }
                }

                for (int fi = 0; fi < faceN; fi++)
                {
                    int vo = mMesh.mVoffs[fi];
                    for (int vi = 0; vi < 3; vi++)
                    {
                        int vid = mMesh.mVbuffs[(vi + vo) * sn + po];
                        for (int si = 0; si < sn; si++)
                        {
                            mMesh.mVbuffs[(vi + vo) * sn + si] = vid;
                        }
                    }
                }

                // 最後にプロパティの全カウントが一致してるか？
                mMesh.PropsCountEqualityTest();
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Rearrange proc was stopped with failure." + ex.Message);
            }
        }
    }
}
