﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.iflib;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    /// <summary>
    /// 形状の重複検知クラス
    /// </summary>
    public class DupShapeDetector
    {
        public enum DuplicateResult
        {
            NotDuplicate,
            Perfect,
            Patritional
        }

        private class FaceInfo : IComparable, IEquatable<FaceInfo>
        {
            public int meshId { get; set; }   //　メッシュ本体へのインデックス
            public int faceId { get; set; }   //  フェースのインデックス
            public long faceHash { get; set; }   //  フェースのインデックス

            public int CompareTo(object obj)
            {
                var rhs = obj as FaceInfo;
                if (rhs == null)
                {
                    return -1;
                }
                return (int)(faceHash - rhs.faceHash);
            }

            public bool Equals(FaceInfo other)
            {
                if (other == null)
                {
                    return false;
                }

                return other.faceHash == faceHash;
            }

            public override bool Equals(object obj)
            {
                if (obj == null)
                {
                    return false;
                }

                FaceInfo faceInfo = obj as FaceInfo;
                if (faceInfo == null)
                {
                    return false;
                }
                else
                {
                    return Equals(faceInfo);
                }
            }

            public override int GetHashCode()
            {
                return base.GetHashCode();
            }
            public static bool operator ==(FaceInfo info1, FaceInfo info2)
            {
                if (((object)info1) == null || ((object)info2) == null)
                {
                    return object.Equals(info1, info2);
                }

                return info1.Equals(info2);
            }

            public static bool operator !=(FaceInfo info1, FaceInfo info2)
            {
                if (((object)info1) == null || ((object)info2) == null)
                {
                    return !object.Equals(info1, info2);
                }

                return !(info1.Equals(info2));
            }
        }

        private ResizableList<Mesh> mSourceMeshList = null;
        private ResizableList<ResizableList<FaceInfo>> mSourceMeshInfos { get; set; } = null;
        private ResizableList<ResizableList<long>> mSourceVtxHashes = null;

        private ResizableList<List<Tuple<int, DuplicateResult>>> mResultList = null;

        /// <summary>
        /// 重複検知をしているか？(Execの後に呼ぶ)
        /// </summary>
        public bool Duplicated
        {
            get
            {
                if (mResultList == null) { return false; }
                var sum = mResultList.Sum(o => o.Count);
                return sum > 0;
            }
        }

        /// <summary>
        /// 重複検知
        /// </summary>
        /// <param name="sourceList"></param>
        public DupShapeDetector(IEnumerable<Mesh> sourceList)
        {
            mSourceMeshList = new ResizableList<Mesh>();
            mSourceMeshInfos = new ResizableList<ResizableList<FaceInfo>>();
            mSourceVtxHashes = new ResizableList<ResizableList<long>>();
            mSourceMeshInfos.Resize(sourceList.Count(), null);
            mSourceVtxHashes.Resize(sourceList.Count(), null);
            foreach (var mesh in sourceList)
            {
                mSourceMeshList.Add(mesh);
            }
        }

        /// <summary>
        /// 各メッシュのポリゴン情報を生成
        /// </summary>
        /// <param name="mi"></param>
        private void CreateMeshPolyInfo(int mi)
        {
            var mesh = mSourceMeshList[mi];
            mSourceMeshInfos[mi] = new ResizableList<FaceInfo>();
            mSourceVtxHashes[mi] = new ResizableList<long>();

            var vertN = mesh.VertexN;
            var faceN = mesh.FaceN;
            var sn = mesh.SourceN;
            var po = mesh.GetSourceOfset(SourceType.Position, null);
            var vpos = new Vector3();

            mSourceMeshInfos[mi].Resize(faceN, null);
            mSourceVtxHashes[mi].Resize(vertN, 0);
            for (int fi = 0; fi < faceN; fi++)
            {
                var vn = mesh.VNums[fi];
                var vo = mesh.VOffset[fi];
                var center = new Vector3();
                center.SetZero();
                for (int vi = 0; vi < vn; vi++)
                {
                    var vid = mesh.VBuffer[(vo + vi) * sn + po];
                    mesh.GetVertexPos(vid, ref vpos);
                    // 頂点ハッシュを格納する
                    mSourceVtxHashes[mi][vid] = SpacialHash.hash_func(vpos);
                    center = center + vpos;
                }
                //重心を計算
                center = center / (double)vn;
                mSourceMeshInfos[mi][fi] = new FaceInfo();
                mSourceMeshInfos[mi][fi].meshId = mi;
                mSourceMeshInfos[mi][fi].faceId = fi;
                mSourceMeshInfos[mi][fi].faceHash = SpacialHash.hash_func(center);
            }
            // SequenceEqual をするために
            mSourceMeshInfos[mi].Sort();
            mSourceVtxHashes[mi].Sort();
        }

        /// <summary>
        /// 二つのメッシュの重複検知を行う
        /// </summary>
        /// <param name="mi"></param>
        /// <param name="mj"></param>
        /// <returns></returns>
        private DuplicateResult IsDuplicate(int mi, int mj, bool perfectMatchOnly = false)
        {
            // 完全一致の検知
            if (mSourceMeshInfos[mi].SequenceEqual(mSourceMeshInfos[mj]))
            {
                // 完全一致
                if (mSourceVtxHashes[mi].SequenceEqual(mSourceVtxHashes[mj]))
                {
                    return DuplicateResult.Perfect;
                }
            }

            if (!perfectMatchOnly)
            {
                //部分的に一致した場合
                var dupInfos = new List<Tuple<FaceInfo, FaceInfo>>();
                mSourceMeshInfos[mi].ForEach
                    (
                        o =>
                        {
                            var dupInfo = mSourceMeshInfos[mj].Find(oo => oo.faceHash == o.faceHash);
                            if (dupInfo != null)
                            {
                                dupInfos.Add(Tuple.Create(o, dupInfo));
                            }
                        });

                // ハッシュが重複している面の頂点が一致している？
                // 一ポリゴンでも重複があれば、検知する
                if (dupInfos.Any())
                {
                    foreach (var tuple in dupInfos)
                    {
                        var mesh1 = mSourceMeshList[tuple.Item1.meshId];
                        var mesh2 = mSourceMeshList[tuple.Item2.meshId];
                        var hashes1 = mSourceVtxHashes[tuple.Item1.meshId];
                        var hashes2 = mSourceVtxHashes[tuple.Item2.meshId];

                        var fid1 = tuple.Item1.faceId;
                        var fid2 = tuple.Item2.faceId;
                        int[] indices1 = null;
                        int[] indices2 = null;

                        mesh1.GetFaceVtxIndices(fid1, ref indices1);
                        mesh2.GetFaceVtxIndices(fid2, ref indices2);
                        bool duplicate = true;
                        foreach (var id1 in indices1)
                        {
                            bool hit = false;
                            var hash1 = hashes1[id1];
                            foreach (var id2 in indices2)
                            {
                                var hash2 = hashes2[id2];
                                if (hash1 == hash2)
                                {
                                    hit = true;
                                    break;
                                }
                            }
                            if (!hit)
                            {
                                duplicate = false;
                                break;
                            }
                        }
                        // 部分重複を検知
                        if (duplicate)
                        {
                            return DuplicateResult.Patritional;
                        }
                    }
                }
            }
            return DuplicateResult.NotDuplicate;
        }

        /// <summary>
        /// 重複前検知
        /// </summary>
        private void PreProc()
        {
            for (int mi = 0; mi < mSourceMeshList.Count; mi++)
            {
                CreateMeshPolyInfo(mi);
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_DupChecker_PreProc_Message", mi));
            }
        }

        /// <summary>
        /// 重複検知を実行する
        /// </summary>
        public bool Execute()
        {
            // 事前処理 -- 検知のための情報を作成するなど準備を行う
            PreProc();
            var meshN = mSourceMeshList.Count;
            mResultList = new ResizableList<List<Tuple<int, DuplicateResult>>>();

            mResultList.Resize(meshN, null);
            for (int i = 0; i < meshN; i++)
            {
                mResultList[i] = new List<Tuple<int, DuplicateResult>>();
            }
            for (int i = 0; i < meshN - 1; i++)
            {
                for (int j = i + 1; j < meshN; j++)
                {
                    var result = IsDuplicate(i, j);
                    // 重複タイプと同時に
                    if (result != DuplicateResult.NotDuplicate)
                    {
                        mResultList[i].Add(Tuple.Create(j, result));
                        mResultList[j].Add(Tuple.Create(i, result));
                    }
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_DupChecker_Proc_Message", i, j));
                }
            }

            return Duplicated;
        }

        public List<List<int>> UniqueDuplicateList
        {
            get
            {
                if (!Duplicated)
                {
                    return null;
                }
                var meshN = mSourceMeshList.Count;
                var vist = new ResizableList<int>();
                vist.Resize(meshN, -1);

                // ユニークなリストを作成するための準備
                int ulc = 0;
                for (int mi = 0; mi < meshN; mi++)
                {
                    if (mResultList[mi].Any() && vist[mi] == -1)
                    {
                        vist[mi] = ulc;
                        foreach (var r in mResultList[mi])
                        {
                            vist[r.Item1] = ulc;
                        }
                        ulc++;
                    }
                }
                // ユニークな重複オブジェクトリストを作成
                var result = new ResizableList<List<int>>();
                result.Resize(ulc, null);
                for (int i = 0; i < ulc; i++)
                {
                    result[i] = new List<int>();
                }
                for (int mi = 0; mi < meshN; mi++)
                {
                    if (vist[mi] >= 0)
                    {
                        result[vist[mi]].Add(mi);
                    }
                }
                for (int i = 0; i < ulc; i++)
                {
                    result[i].Distinct();
                }
                return result;
            }
        }
    }
}
