﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.iflib;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    public class ClosedVtxMarger
    {
        private Mesh mClone = null;
        private Mesh mMesh = null;
        private double mPerc = ShapeMerger.Percentage;
        private bool mbCountSubmeshFace = false;

#if MERGEOT
        private Octree<int> mOctree = new Octree<int>();
#else
        private uint[] mSpaceSize = new uint[3];
        private Vector3 mUnitLen = new Vector3();
        private ResizableList<ResizableList<int>> mIndices = new ResizableList<ResizableList<int>>();
#endif
        public bool CountSubmeshFace
        {
            get { return mbCountSubmeshFace; }
            set { mbCountSubmeshFace = value; }
        }

        public double Percentage
        {
            get { return mPerc; }
            set
            {
                mPerc = value;
                mPerc = Math.Max(0.0, mPerc);
                mPerc = Math.Min(1.0, mPerc);
            }
        }

        public ClosedVtxMarger(Mesh mesh)
        {
            mClone = mesh.Clone();
            mMesh = mesh;
        }

#if MERGEOT

        /// <summary>
        /// Dummy
        /// </summary>
        public void SetSpaceInfo(uint sx, uint sy, uint sz)
        {
            // Dummy
        }

        /// <summary>
        /// ８分木を初期化します
        /// </summary>
        private void InitializeOctree()
        {
            mMesh.ComputeBounds();
            BoundingBox bb = mMesh.GetTotalBoundingBox();
            mOctree.Create(1, bb.Min, bb.Max);

            try
            {
                MeshSourceBase source = mMesh.GetSource(SourceType.Position, null);
                int posN = mMesh.VertexN;

                BoundingBox lbox = new BoundingBox();
                Vector3 lunit = new Vector3(Threshold, Threshold, Threshold);
                for (int vi = 0; vi < posN; vi++)
                {
                    Vector3 tv3 = new Vector3();
                    MeshSourceDblCtrler.GetAsVec3(source, vi, ref tv3);
                    lbox.Reset();
                    lbox.Update(tv3 - lunit);
                    lbox.Update(tv3 + lunit);
                    mOctree.AddObject(vi, lbox);
                }
                ComputeThreshold();
            }
            catch (Exception ex)
            {
                //ex.Data.Add("smesh", mSourceMeshList[i]);
                string msg = string.Format(@"Fail to build octree for mesh {0}", mMesh.Name);
                throw new Exception(msg, ex);
            }
        }

        /// <summary>
        /// リスト内から位置が一致する頂点を抜き出してピックアップします
        /// </summary>
        private int PickSharedVerticesOctree(double threshold, ref ResizableList<int> map)
        {
            int vtxN = mMesh.VertexN;
            MeshSourceBase posList = mMesh.GetSource(SourceType.Position, null);

            map = new ResizableList<int>();
            map.Resize(vtxN, 0);
            for (int vi = 0; vi < vtxN; vi++)
            {
                map[vi] = vi;
            }

            Vector3 v0 = new Vector3();
            Vector3 v1 = new Vector3();
            double argmax = 0;

            WingedEdgeMesh test = new WingedEdgeMesh();
            test.Create(mMesh);
            test.CreateVEL();

            ResizableList<bool> border = new ResizableList<bool>();
            ResizableList<double> hsdrf = new ResizableList<double>();
            border.Resize(vtxN,false);
            hsdrf.Resize(vtxN,double.MaxValue);
            foreach (var wingedEdge in test.WingedEdges)
            {
                if (wingedEdge.Face0 == int.MaxValue || wingedEdge.Face1 == int.MaxValue)
                {
                    border[wingedEdge.Vertex0] = true;
                    border[wingedEdge.Vertex1] = true;
                }
            }

            for (int vi = 0; vi < vtxN; vi++)
            {
                MeshSourceDblCtrler.GetAsVec3(posList, vi, ref v0);
                ResizableList<OctreeNode<int>> nodes = new ResizableList<OctreeNode<int>>();
                mOctree.WindowQueryForBB(nodes, v0, null);

                double minLen = double.MaxValue;

                foreach (var octreeNode in nodes)
                {
                    int lvN = octreeNode.Objects.Count;
                    if (lvN > 1)
                    {
                        for (int vj = 0; vj < lvN; vj++)
                        {
                            int vid0 = octreeNode.Objects[vj];
                            if (vid0 <= vi) { continue; }
                            if (map[vid0] != vid0) { continue; }

                            MeshSourceDblCtrler.GetAsVec3(posList, vid0, ref v1);
                            Vector3 dist = v1 - v0;
                            if (dist.Norm() <= threshold)
                            {
                                map[vid0] = vi;
                            }

                            if (border[vid0])
                            {
                                if (minLen > dist.Norm())
                                {
                                    hsdrf[vid0] = dist.Norm();
                                    minLen = dist.Norm();
                                }
                            }
                        }
                    }
                }

                if (argmax < minLen && minLen != double.MaxValue)
                {
                    argmax = minLen;
                }
            }

            int sharedCount = 0;
            int rvtxN = 0;
            for (int i = 0; i < map.Count; i++)
            {
                if (map[i] == i)
                    rvtxN++;
                else
                    sharedCount++;
            }
#if VERBOSE
            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine("Vertex Number {0} - merged {1} ", rvtxN, sharedCount);
#endif
            return sharedCount;
        }
#else
        /// <summary>
        /// ユニフォームグリッドの分割数を設定
        /// </summary>
        public void SetSpaceInfo(uint sx, uint sy, uint sz)
        {
            if (sx == 0 || sy == 0 || sz == 0)
            {
                throw ExcepHandle.CreateException("Grid size is zero");
            }

            mSpaceSize[0] = sx;
            mSpaceSize[1] = sy;
            mSpaceSize[2] = sz;
        }

        /// <summary>
        /// 各グリッドの単位サイズを計算
        /// </summary>
        private void ComputeUnit()
        {
            mMesh.ComputeBounds();
            Vector3 Size = mMesh.GetTotalBoundingBox().Size();
            uint s = 1;
            for (int i = 0; i < 3; i++)
            {
                if (mSpaceSize[i] > 0)
                {
                    mUnitLen[i] = Size[i] / (double)mSpaceSize[i];
                    s *= mSpaceSize[i];
                }
            }

            mIndices.Resize((int)s, null);
            for (int i = 0; i < s; i++)
            {
                mIndices[i] = new ResizableList<int>();
            }
        }

        /// <summary>
        /// 与えられた頂点座標から、頂点が含まれる空間座標を取得
        /// </summary>
        private void GetSpaceIndex(Vector3 pos, ref int sx, ref int sy, ref int sz)
        {
            Vector3 len = pos - mMesh.GetTotalBoundingBox().Min;
            sx = (int)(len.x / mUnitLen.x);
            sy = (int)(len.y / mUnitLen.y);
            sz = (int)(len.z / mUnitLen.z);

            sx = Math.Max(0, sx);
            sy = Math.Max(0, sy);
            sz = Math.Max(0, sz);
            sx = Math.Min(sx, (int)mSpaceSize[0] - 1);
            sy = Math.Min(sy, (int)mSpaceSize[1] - 1);
            sz = Math.Min(sz, (int)mSpaceSize[2] - 1);
        }

        /// <summary>
        /// 空間座標から配列インデックスを取得します
        /// </summary>
        /// <param name="sx"></param>
        /// <param name="sy"></param>
        /// <param name="sz"></param>
        /// <returns></returns>
        private int GetArrayIndex(int sx, int sy, int sz)
        {
            int floorSize = (int)(mSpaceSize[0] * mSpaceSize[2]);
            return floorSize * sy + sz * (int)mSpaceSize[0] + sx;
        }

        /// <summary>
        /// 配列インデックスから空間インデックスを取得します
        /// </summary>
        /// <param name="arrayIndex"></param>
        /// <param name="sx"></param>
        /// <param name="sy"></param>
        /// <param name="sz"></param>
        private void GetSpaceIndex(int arrayIndex, ref int sx, ref int sy, ref int sz)
        {
            int tmp;
            int floorSize = (int)(mSpaceSize[0] * mSpaceSize[2]);
            sy = arrayIndex / floorSize;
            tmp = arrayIndex % floorSize;

            sz = tmp / (int)mSpaceSize[0];
            sx = tmp % (int)mSpaceSize[0];
        }

        /// <summary>
        /// 頂点結合に使用する閾値を、メッシュから求める
        /// </summary>
        private void ComputeThreshold()
        {
            Threshold = mMesh.ComputeAverageEdgeLength() * Percentage;
        }

        /// <summary>
        /// 空間座標内に
        /// </summary>
        private void InitializeUniformGrid()
        {
            ComputeUnit();

            int vtxN = mMesh.VertexN;
            MeshSourceBase posList = mMesh.GetSource(SourceType.Position, null);

            Vector3 tmpPos = new Vector3();
            int sx = 0;
            int sy = 0;
            int sz = 0;

            int tx = 0;
            int ty = 0;
            int tz = 0;

            int index = 0;
            for (int vi = 0; vi < vtxN; vi++)
            {
                MeshSourceDblCtrler.GetAsVec3(posList, vi, ref tmpPos);
                GetSpaceIndex(tmpPos, ref sx, ref sy, ref sz);
                index = GetArrayIndex(sx, sy, sz);
                GetSpaceIndex(index, ref tx, ref ty, ref tz);

                Nintendo.Foundation.Contracts.Assertion.Operation.True(sx == tx && sy == ty && sz == tz);
                mIndices[index].Add(vi);
            }

            ComputeThreshold();
        }

        private int GetSpaceArraySize()
        {
            return (int)(mSpaceSize[0] * mSpaceSize[1] * mSpaceSize[2]);
        }

        /// <summary>
        /// リスト内から位置が一致する頂点を抜き出してピックアップします
        /// </summary>
        private int PickSharedVerticesFromUniformedGrid(double threshold, ref ResizableList<int> map)
        {
            var vtxN = mMesh.VertexN;
            var posList = mMesh.GetSource(SourceType.Position, null);

            map = new ResizableList<int>();
            map.Resize(vtxN, 0);
            for (var vi = 0; vi < vtxN; vi++)
            {
                map[vi] = vi;
            }

            var v0 = new Vector3();
            var v1 = new Vector3();
            var sizeX = 0;
            var sizeY = 0;
            var sizeZ = 0;
            var cellId = 0;
            for (var vi = 0; vi < vtxN; vi++)
            {
                MeshSourceDblCtrler.GetAsVec3(posList, vi, ref v0);
                GetSpaceIndex(v0, ref sizeX, ref sizeY, ref sizeZ);
                cellId = GetArrayIndex(sizeX, sizeY, sizeZ);
                if (map[vi] != vi)
                {
                    continue;
                }

                var lvN = mIndices[cellId].Count;
                if (lvN > 1)
                {
                    for (var vj = 0; vj < lvN; vj++)
                    {
                        var vid0 = mIndices[cellId][vj];
                        if (vid0 <= vi)
                        {
                            continue;
                        }
                        if (map[vid0] != vid0)
                        {
                            continue;
                        }

                        MeshSourceDblCtrler.GetAsVec3(posList, vid0, ref v1);
                        var dist = v1 - v0;
                        if (dist.Norm() <= threshold)
                        {
                            map[vid0] = vi;
                        }
                    }
                }
            }

            int sharedCount = 0;
            int rvtxN = 0;
            for (int i = 0; i < map.Count; i++)
            {
                if (map[i] == i)
                {
                    rvtxN++;
                }
                else
                {
                    sharedCount++;
                }
            }
            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Merge_Vertex", vtxN, sharedCount));
            return sharedCount;
        }
#endif
        /// <summary>
        /// 頂点に再度インデックスを割り振ります。
        /// </summary>
        private void CleanDegenerateFaces(ResizableList<int> smcnt)
        {
            int faceN = mMesh.FaceN;
            for (int fi = 0; fi < faceN; fi++)
            {
                int smi = mMesh.mFaceSubMeshID[fi];
                if (mMesh.IsFaceDegenerate(fi) && (!mbCountSubmeshFace || smcnt[smi] > 1))
                {
                    mMesh.InvalidateFace(fi);
                    if (mbCountSubmeshFace)
                    {
                        smcnt[smi]--;
                    }
                }
            }
        }

        /// <summary>
        /// 頂点に再度インデックスを割り振ります。
        /// </summary>
        private void ReIndexVertices(ResizableList<int> map, ResizableList<int> smcnt)
        {
            int faceN = mMesh.FaceN;
            int vn = 0;
            int vo = 0;
            int sn = mMesh.SourceN;
            int pof = mMesh.GetSourceOfset(SourceType.Position, null);

            for (int fi = 0; fi < faceN; fi++)
            {
                vo = mMesh.mVoffs[fi];
                vn = mMesh.mVnums[fi];

                for (int vi = 0; vi < vn; vi++)
                {
                    int vid = mMesh.VBuffer[(vo + vi) * sn + pof];
                    mMesh.VBuffer[(vo + vi) * sn + pof] = map[vid];
                }
            }
            CleanDegenerateFaces(smcnt);
        }

        [Conditional("VERBOSE")]
        private void test_at_posrun()
        {
            try
            {
                MeshSourceBase src = mMesh.GetSource(SourceType.Texture, null);
                if (src == null)
                {
                    return;
                }
                //Mesh.CheckSharedPropRegularity(mMesh);
                MeshIndexTrack mit = new MeshIndexTrack();
                Mesh test = mMesh.Clone();
                mit.Run(mClone, test, true);
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// 非多様体エッジを元に戻す
        /// </summary>
        private void RevertNonmanifoldEdge()
        {
            WingedEdgeMesh wem = new WingedEdgeMesh();
            wem.CreateAll(mMesh);

            // 非多様体エッジが存在するので、それを元に戻す
            if (wem.RepresentativeNmfdEdges.Count > 0)
            {
                //mMesh = mClone.Clone();
                int sn = mMesh.SourceN;
                int po = mMesh.GetSourceOfset(SourceType.Position, null);
                VtxTriMesh vtm = new VtxTriMesh();
                vtm.Create(mMesh);

                foreach (var nme in wem.RepresentativeNmfdEdges)
                {
                    for (int i = 0; i < 2; i++)
                    {
                        int evid = (i == 0) ? nme.Vertex0 : nme.Vertex1;
                        foreach (var fi in vtm.VTL[evid].TriList)
                        {
                            int vn = mMesh.VNums[fi];
                            int vo = mMesh.VOffset[fi];
                            for (int vi = 0; vi < vn; vi++)
                            {
                                mMesh.mVbuffs[(vo + vi) * sn + po] = mClone.VBuffer[(vo + vi) * sn + po];
                            }
                            mMesh.ValidateFace(fi);
                        }
                    }
                }
            }
            //ボウタイチェック
            {
                int sn = mMesh.SourceN;
                int po = mMesh.GetSourceOfset(SourceType.Position, null);

                VtxTriMesh vtm = new VtxTriMesh();
                vtm.Create(mMesh);
                for (int vi = 0; vi < mMesh.VertexN; vi++)
                {
                    int nonmani = 0;
                    int opensum = 0;
                    foreach (var fi in vtm.VTL[vi].TriList)
                    {
                        int vn = mMesh.VNums[fi];
                        int vo = mMesh.VOffset[fi];

                        int error = 0;
                        for (int vj = 0; vj < vn; vj++)
                        {
                            int mvid = mMesh.mVbuffs[(vo + vj) * sn + po];
                            if (mvid != vi)
                            {
                                List<int> list = vtm.GetNeighborTrigs(fi, vi, mvid);
                                if (list.Count == 0)
                                {
                                    error++;
                                }
                            }
                        }
                        if (error >= 2)
                        {
                            nonmani++;
                        }

                        if (error >= 1)
                        {
                            opensum++;
                        }
                    }

                    // nonmani 一つの場合は角
                    // opensum 2の場合は角または、単にOpenが連続してるだけ
                    if (nonmani >= 2 || opensum >= 3)
                    {
                        foreach (var fi in vtm.VTL[vi].TriList)
                        {
                            int vn = mMesh.VNums[fi];
                            int vo = mMesh.VOffset[fi];
                            for (int vj = 0; vj < vn; vj++)
                            {
                                mMesh.mVbuffs[(vo + vj) * sn + po] = mClone.VBuffer[(vo + vj) * sn + po];
                            }
                            mMesh.ValidateFace(fi);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 一致する頂点とみなすための距離の閾値
        /// </summary>
        public double _threshold { get; set; } = 1e-5;
        public double Threshold
        {
            get { return _threshold; }
            set { _threshold = value; }
        }

        [Conditional("DEBUG")]
        private void TestSubmeshArea()
        {
            ResizableList<double> submeshArea = new ResizableList<double>();
            int faceN = mMesh.FaceN;
            mMesh.RecomputeFaceNormalsAndArea();

            submeshArea.Resize(mMesh.SubMeshInformationList.Count, 0.0);
            for (int fi = 0; fi < faceN; fi++)
            {
                double farea = mMesh.GetFaceArea(fi);
                int subMeshId = mMesh.mFaceSubMeshID[fi];
                if (subMeshId >= submeshArea.Count || subMeshId < 0)
                {
                    throw ExcepHandle.CreateException("SubmeshID is bigger than list length.");
                }

                submeshArea[subMeshId] += farea;
            }

            // サブメッシュの合計面積がゼロの場合は、そのサブメッシュは消滅することになる
            for (int i = 0; i < submeshArea.Count; i++)
            {
                if (submeshArea[i] <= double.Epsilon)
                {
                    var msg = string.Format(
                        "SubMesh[{0}] have only zero area faces. these faces will degenerate. and this subemsh will be disappear.(this is critical fault.)",
                        i);
                    throw ExcepHandle.CreateException(msg);
                }
            }
        }

        /// <summary>
        /// 実行
        /// </summary>
        public int Run()
        {
            try
            {
                if (mMesh == null)
                {
                    return 0;
                }
                if (mMesh != null && mMesh.GetSources(SourceType.Position).Count != 1)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Merge_Vertex", "Merge Vertex"));
                    return -1;
                }

                mMesh.GetSubMeshCount();
                // 問題をはっきりさせるためにテストを埋めた
                TestSubmeshArea();

                CleanDegenerateFaces(mMesh.mSubmeshCounter);

                int mergeCount = 0;
                {
                    ResizableList<int> map = new ResizableList<int>();
#if MERGEOT
                    InitializeOctree();
                    mergeCount = PickSharedVerticesOctree(Threshold, ref map);
#else
                    InitializeUniformGrid();
                    mergeCount = PickSharedVerticesFromUniformedGrid(Threshold, ref map);
#endif
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Start_Closed_Vertex", Threshold));

                    ReIndexVertices(map, mMesh.mSubmeshCounter);

                    mMesh.RemoveInvalidFaces();
                    if (!mMesh.ReComputeSubMeshArray())
                    {
                        return -1;
                    }
                    mMesh.RemoveUnusedVertex();

                    if (mMesh.FaceN == 0)
                    {
                        return -1;
                    }
                }

                mMesh.DebugOutput(mMesh.MeshName + ".am.obj");
                test_at_posrun();

                return mergeCount;
            }
            catch (Exception ex)
            {
                ExcepHandle.CreateException(ex);
                throw;
            }
        }
    }
}
