﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    internal partial class ClosedPropManager : PropMapBase
    {
        private Dictionary<int, int> mSurjectionMap = new Dictionary<int, int>();
        public Dictionary<int, int> SurjectionMap
        {
            get { return mSurjectionMap; }
        }

        /// <summary>
        /// 一致する頂点とみなすための距離の閾値
        /// </summary>
        private double _threshold = 1e-5;
        public double Threshold
        {
            get { return _threshold; }
            set { _threshold = value; }
        }

        [Conditional("DEBUG")]
        private void CheckDuplicateProp(Mesh mesh, SourceType type, string name)
        {
            MeshWedges tmpWedge = new MeshWedges(mesh);
            tmpWedge.Create();

            MeshSourceBase source = mesh.GetSource(type, name);
            if (source == null) { return; }

            //ウェッジ全体で検索
            foreach (var wdvtx in tmpWedge.Wedges)
            {
                if (wdvtx == null || wdvtx.Keys == null || wdvtx.Keys.Count == 0) { continue; }

                int propN = wdvtx.Keys.Count;

                // 指定のプロパティを検索する
                int prid = -1;
                for (int i = 0; i < propN; i++)
                {
                    if (wdvtx.Keys[i].type == type && (name == null || wdvtx.Keys[i].name == name))
                    {
                        prid = i;
                        break;
                    }
                }
                if (prid == -1) { continue; }

                int propValN = wdvtx.Props[prid].Count;

                for (int i = 0; i < propValN - 1; i++)
                {
                    for (int j = i + 1; j < propValN; j++)
                    {
                        int id0 = wdvtx.Props[prid][i];
                        int id1 = wdvtx.Props[prid][j];

                        if (id0 != id1)
                        {
                            double dist = 0.0;
                            for (int si = 0; si < source.Stride; si++)
                            {
                                double v0 = Convert.ToDouble(source.GetRowValue(id0 * source.Stride + si));
                                double v1 = Convert.ToDouble(source.GetRowValue(id1 * source.Stride + si));
                                dist += (v1 - v0) * (v1 - v0);
                            }

                            // 距離が近い場合は、その中で最少のIDになるように設定する
                            if (dist < Threshold)
                            {
                                string msg = string.Format("Prop merge error :: vtx{0} , {1} | {2}", wdvtx.Vertex, id0, id1);
                                throw new Exception(msg);
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 重複状態を記録する、各面-頂点毎の全射マップを作成
        /// 単射マップと同時に利用する事で、
        /// </summary>
        /// <param name="hit"></param>
        public void CreateSurjectionMap(int[] hit)
        {
            //-------------------------------------------------------------------
            //重複しているプロパティでID統合を行う

            int offs = Mesh.GetSourceOfset(SourceKey.type, SourceKey.name);
            int sn = Mesh.SourceN;
            for (int fi = 0; fi < Mesh.FaceN; fi++)
            {
                int vo = Mesh.VOffset[fi];
                int vn = Mesh.VNums[fi];
                for (int vi = 0; vi < vn; vi++)
                {
                    int pid = Mesh.VBuffer[(vo + vi) * sn + offs];
                    Mesh.VBuffer[(vo + vi) * sn + offs] = hit[pid];
                    mSurjectionMap[pid] = hit[pid];
                }
            }
        }

        /// <summary>
        /// 1-ring 近接で、値が共有できるものを纏める
        /// </summary>
        /// <param name="hit"></param>
        private void CreateSourceSurjectionMap(out int[] hit)
        {
            hit = null;

            // プロパティは数値でハッシュにしてしまうと、遠く離れた頂点の値も
            // 同じに扱ってしまう場合がでる
            MeshWedges wedge = new MeshWedges(Mesh);
            wedge.Create();
            MeshSourceBase source = Mesh.GetSource(SourceKey.type, SourceKey.name);
            if (source == null) { return; }

            hit = new int[source.Count];
            for (int i = 0; i < source.Count; i++)
            {
                hit[i] = i;
            }

            // ウェッジ全体で検索
            // 共有頂点で、一致しているプロパティを検索する
            foreach (var wdvtx in wedge.Wedges)
            {
                if (wdvtx == null || wdvtx.Keys == null || wdvtx.Keys.Count == 0) { continue; }

                int propN = wdvtx.Keys.Count;
                // 指定のプロパティを検索する
                int prid = -1;
                for (int i = 0; i < propN; i++)
                {
                    if (wdvtx.Keys[i].type == SourceKey.type && (SourceKey.name == null || wdvtx.Keys[i].name == SourceKey.name))
                    {
                        prid = i;
                        break;
                    }
                }
                if (prid == -1) { continue; }

                //--------------------------------------
                int propValN = wdvtx.Props[prid].Count;
                if (propValN > 1)
                {
                    for (int i = 0; i < propValN - 1; i++)
                    {
                        for (int j = i + 1; j < propValN; j++)
                        {
                            int id0 = wdvtx.Props[prid][i];
                            int id1 = wdvtx.Props[prid][j];

                            if (id0 != id1)
                            {
                                double dist = 0.0;
                                for (int si = 0; si < source.Stride; si++)
                                {
                                    double v0 = Convert.ToDouble(source.GetRowValue(id0 * source.Stride + si));
                                    double v1 = Convert.ToDouble(source.GetRowValue(id1 * source.Stride + si));
                                    dist += (v1 - v0) * (v1 - v0);
                                }

                                // 距離が近い場合は、その中で最少のIDになるように設定する
                                if (dist < Threshold)
                                {
                                    int minId = Math.Min(id0, id1);
                                    hit[id0] = Math.Min(hit[id0], minId);
                                    hit[id1] = Math.Min(hit[id1], minId);
                                }
                            }
                        }
                    }
                }
                else
                {
                    int id0 = wdvtx.Props[prid][0];
                    hit[id0] = id0;
                }
            }

            //-----------------------------------
            ResizableList<bool> usedList = new ResizableList<bool>();
            usedList.Resize(hit.Length, false);
            for (int id = 0; id < hit.Length; id++)
            {
                usedList[hit[id]] = true;
            }

            //-----------------------------------
            // ソース短縮
            ResizableList<int> tmpList = new ResizableList<int>();
            tmpList.Resize(hit.Length, int.MaxValue);
            int uniqueId = 0;
            int stride = source.Stride;
            for (int id = 0; id < hit.Length; id++)
            {
                if (usedList[id])
                {
                    for (int sti = 0; sti < stride; sti++)
                    {
                        object v = source.GetRowValue(id * stride + sti);
                        source.SetRowValue(uniqueId * stride + sti, v);
                    }
                    tmpList[id] = uniqueId++;
                }
            }
            source.SetLength(uniqueId, false);
            // replace
            for (int id = 0; id < hit.Length; id++)
            {
                int index = tmpList[hit[id]];
                hit[id] = index;
            }
        }

        /// <summary>
        /// 近傍を見て、近接しているプロパティを纏める処理を実行
        /// </summary>
        public void Run()
        {
            int[] hit = null;
            CreateSourceSurjectionMap(out hit);
            CreateSurjectionMap(hit);
            CheckDuplicateProp(Mesh, SourceKey.type, SourceKey.name);
        }

        /// <summary>
        /// 逆写像を取得します
        /// </summary>
        /// <returns></returns>
        public Dictionary<int, List<int>> GetReverseMap()
        {
            Dictionary<int, List<int>> revMap = new Dictionary<int, List<int>>();
            foreach (var item in mSurjectionMap)
            {
                int nv = item.Value;
                if (!revMap.ContainsKey(nv))
                {
                    revMap.Add(nv, new List<int>());
                }
                revMap[nv].Add(item.Key);
            }
            return revMap;
        }
    }
}
