﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

namespace nw4f.meshlib
{
    /// <summary>
    /// ヒープリスト
    /// 内部クラス
    /// </summary>
    internal class BucketHeapList<T> where T : QemHeapData
    {
        internal enum HeapType
        {
            Primary = 0,
            Secondly = 1,
            None = 2
        }

        internal double mPrimaryRate { get; set; } = 0.05;
        internal int mPrimaryHeapCount { get; set; } = 0;

        internal List<T> mWaitList { get; set; } = new List<T>();
        internal HeapList<T> mPrimaryHeap { get; set; } = new HeapList<T>();
        internal HeapList<T> mSecondlyHeap { get; set; } = new HeapList<T>();

        internal double PrimaryRate
        {
            get { return mPrimaryRate; }
            set { mPrimaryRate = value; }
        }

        /// <summary>
        /// 待機リストに追加します
        /// </summary>
        /// <param name="val">値</param>
        internal int AddToWaitList(T val)
        {
            //if (!mWaitList.Contains(val))
            {
                mWaitList.Add(val);
            }
            return mWaitList.Count;
        }

        internal void ConstructBucketHeaps()
        {
            mWaitList.Sort((l, r) => l.CompareTo(r, SourceType.Position));

            mPrimaryHeap.Key = SourceType.Texture;
            mSecondlyHeap.Key = SourceType.Position;

            int limitCount = (int)(mPrimaryRate * (double)mWaitList.Count);
            // メインヒープを構築
            if (limitCount > 0)
            {
                mPrimaryHeapCount = limitCount;
                for (int i = 0; i < limitCount; i++)
                {
                    mWaitList[i].HeapList = (int)HeapType.Primary;
                    mPrimaryHeap.Push(mWaitList[i]);
                }
            }
            // 待機ヒープを構築
            for (int i = limitCount; i < mWaitList.Count; i++)
            {
                mWaitList[i].HeapList = (int)HeapType.Secondly;
                mSecondlyHeap.Push(mWaitList[i]);
            }
            mWaitList.Clear();
        }

        /// <summary>
        /// バケットの数を更新
        /// </summary>
        /// <param name="maxNum"></param>
        internal void UpdateBucketNumber(int maxNum)
        {
            if (maxNum > 0)
            {
                mPrimaryHeapCount = (int)(mPrimaryRate * (double)maxNum);
            }
        }

        /// <summary>
        /// 先頭を取得します。
        /// </summary>
        internal T Top
        {
            get
            {
                if (mPrimaryHeap.List.Count > 0)
                {
                    return mPrimaryHeap.Top;
                }
                else
                {
                    return mSecondlyHeap.Top;
                }
            }
        }

        /// <summary>
        /// オブジェクトを直接削除する
        /// </summary>
        /// <param name="obj"></param>
        internal void Pop(T obj)
        {
            if (obj.HeapList == (int)HeapType.Primary)
            {
                obj.HeapList = (int)HeapType.None;
                mPrimaryHeap.Pop(obj);
            }
            else
            {
                obj.HeapList = (int)HeapType.None;
                mSecondlyHeap.Pop(obj);
            }
        }

        /// <summary>
        /// 先頭を削除して、優先ヒープを更新
        /// </summary>
        internal void Pop()
        {
            if (mPrimaryHeap.List.Count > 0)
            {
                T top = mPrimaryHeap.Top;
                top.HeapList = (int)HeapType.None;
                mPrimaryHeap.Pop();
            }
            else
            {
                T top = mSecondlyHeap.Top;
                top.HeapList = (int)HeapType.None;
                mSecondlyHeap.Pop();
            }
        }

        /// <summary>
        /// 先頭を削除して、優先ヒープを更新
        /// </summary>
        internal void Push(T obj)
        {
            obj.HeapList = (int)HeapType.Secondly;
            mSecondlyHeap.Push(obj);

            if (mPrimaryHeapCount > mPrimaryHeap.List.Count)
            {
                T bt = mSecondlyHeap.Top;
                mSecondlyHeap.Pop();

                bt.HeapList = (int)HeapType.Primary;
                mPrimaryHeap.Push(bt);
            }
        }
    }
}
