﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public class BoundingSquare
    {
        private Vector2 mMin = new Vector2();
        private Vector2 mMax = new Vector2();

        /// <summary>
        /// 最小座標
        /// </summary>
        public Vector2 Min
        {
            get { return mMin; }
            set { mMin = value; }
        }

        /// <summary>
        /// 最大座標
        /// </summary>
        public Vector2 Max
        {
            get { return mMax; }
            set { mMax = value; }
        }

        public BoundingSquare()
        {
            Reset();
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public void Reset()
        {
            mMin.x = mMin.y = float.MaxValue;
            mMax.x = mMax.y = -float.MaxValue;
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public void Round(int degits = 3)
        {
            mMin.x = Math.Round(mMin.x, degits, MidpointRounding.AwayFromZero);
            mMin.y = Math.Round(mMin.y, degits, MidpointRounding.AwayFromZero);
            mMax.x = Math.Round(mMax.x, degits, MidpointRounding.AwayFromZero);
            mMax.y = Math.Round(mMax.y, degits, MidpointRounding.AwayFromZero);
        }

        /// <summary>
        /// 中心を求める
        /// </summary>
        /// <returns></returns>
        public Vector2 Center()
        {
            return (mMin + mMax) * 0.5;
        }

        /// <summary>
        /// 対角線ベクトル
        /// </summary>
        /// <returns></returns>
        public Vector2 Size()
        {
            return (mMax - mMin);
        }

        /// <summary>
        /// 境界ボリュームを更新
        /// </summary>
        /// <param name="pos"></param>
        public void Update(Vector2 pos)
        {
            if (pos == null)
            {
                return;
            }
            mMin.x = Math.Min(pos.x, mMin.x);
            mMin.y = Math.Min(pos.y, mMin.y);
            mMax.x = Math.Max(pos.x, mMax.x);
            mMax.y = Math.Max(pos.y, mMax.y);
        }

        /// <summary>
        /// AABBが完全に中に入っているか？
        /// </summary>
        /// <param name="bBox"></param>
        /// <returns></returns>
        public bool Inside(BoundingSquare bBox)
        {
            return
            (
                mMax.x - bBox.mMax.x >= -1e-5 &&
                mMin.x - bBox.mMin.x <= 1e-5 &&
                mMax.y - bBox.mMax.y >= -1e-5 &&
                mMin.y - bBox.mMin.y <= 1e-5);
        }

        /// <summary>
        /// 指定された点が、BVの中に含まれているか？
        /// </summary>
        /// <param name="pt"></param>
        /// <returns></returns>
        public bool Inside(Vector2 pt)
        {
            return mMin.x <= pt.x &&
                   mMax.x >= pt.x &&
                   mMin.y <= pt.y &&
                   mMax.y >= pt.y;
        }

        /// <summary>
        /// 指定の頂点が、BVの中に含まれるように丸める
        /// </summary>
        /// <param name="pt"></param>
        /// <returns></returns>
        public Vector2 Round(Vector2 pt)
        {
            var result = new Vector2();
            result.x = Math.Max(Math.Min(pt.x, mMax.x), mMin.x);
            result.y = Math.Max(Math.Min(pt.y, mMax.y), mMin.y);
            return result;
        }
    }
}
