﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using nw4f.tinymathlib;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    public class BoundaryMetric
    {
        private int mDim;
        private MatrixMN mA;
        private VectorN mB;
        private double mC;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="d">次元数</param>
        public BoundaryMetric(int d)
        {
            mDim = d;
            mA = new MatrixMN(mDim, mDim);
            mB = new VectorN((uint)mDim);
            mC = 0.0f;
        }

        /// <summary>
        /// 値をクリアする
        /// </summary>
        public void Clear()
        {
            mA.SetZero();
            mB.SetZero();
            mC = 0;
        }

        /// <summary>
        /// 値をセットする
        /// </summary>
        /// <param name="rhs"></param>
        public void Set(BoundaryMetric rhs)
        {
            if (rhs.mDim == mDim)
            {
                mA.Set(rhs.mA);
                mB.Set(rhs.mB);
                mC = rhs.mC;
                mDim = rhs.mDim;
            }
            else
            {
                int dim = Math.Min(mDim, rhs.mDim);
                for (int di = 0; di < dim; di++)
                {
                    for (int dj = 0; dj < dim; dj++)
                    {
                        mA[di, dj] = rhs.mA[di, dj];
                    }
                    mB[di] = rhs.mB[di];
                }
                mC = rhs.mC;
            }
        }

        /// <summary>
        /// Nanであるかどうか？をチェックする
        /// </summary>
        /// <returns></returns>
        public bool IsNaN()
        {
            return mA.IsNaN() || mB.IsNaN() || double.IsNaN(mC);
        }

        /// <summary>
        /// A,B,Cの３つと、面積のみがたされる
        /// </summary>
        public static BoundaryMetric operator +(BoundaryMetric lhs, BoundaryMetric rhs)
        {
            if (lhs.mDim != rhs.mDim)
            {
                throw ExcepHandle.CreateException("rank of quadrics are different.");
            }

            BoundaryMetric result = new BoundaryMetric(lhs.mDim);
            result.mA = lhs.mA + rhs.mA;
            result.mB = lhs.mB + rhs.mB;
            result.mC = lhs.mC + rhs.mC;
            return result;
        }

        /// <summary>
        /// A,B,Cの３つと、面積のみがたされる
        /// </summary>
        public static BoundaryMetric operator *(BoundaryMetric lhs, double rhs)
        {
            BoundaryMetric result = new BoundaryMetric(lhs.mDim);
            result.mA = lhs.mA * rhs;
            result.mB = lhs.mB * rhs;
            result.mC = lhs.mC * rhs;
            return result;
        }

        /// <summary>
        /// 作用素の作成
        /// </summary>
        /// <param name="p1"></param>
        /// <param name="p2"></param>
        /// <param name="p3"></param>
        public void Compute(Vector3 ev0, Vector3 ev1)
        {
            Vector3 e1 = new Vector3();
            Vector3 e2 = new Vector3();
            MatrixMN m0 = new MatrixMN(3, 3);
            MatrixMN m1 = new MatrixMN(3, 3);

            e1 = ev1 - ev0;
            e2 = ev1.Cross(ev0);

            m0[0, 0] = 0.0;
            m0[0, 1] = -e1.z;
            m0[0, 2] = e1.y;
            m0[1, 0] = e1.z;
            m0[1, 1] = 0.0;
            m0[1, 2] = -e1.x;
            m0[2, 0] = -e1.y;
            m0[2, 1] = e1.x;
            m0[2, 2] = 0.0;

            m1 = m0.GetTranspose();
            Vector3 tv = e1.Cross(e2);

            mA = m1 * m0;
            mB[0] = tv[0];
            mB[1] = tv[1];
            mB[2] = tv[2];
            mC = e2.Dot(e2);
        }

        /// <summary>
        /// コストの計算を行う
        /// </summary>
        /// <param name="v"></param>
        /// <returns></returns>
        public double Evaluate(VectorN v)
        {
            var cost = 0.0;
            cost += v.Dot(mA * v) * 0.5;
            cost += mB.Dot(v);
            cost += mC * 0.5;
            //cost *= (edgeL);
            return cost;
        }
    }
}
