﻿using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;
using nw.g3d.iflib;
using System;
using System.Collections.Generic;
using System.Linq;

namespace _3dIntermediateFileMaterialGenerator
{
    public class MaterialGenerator
    {
        public MaterialGenerator()
        {
        }

        public Context Context { get; private set; }

        public void Execute(Context context)
        {
            this.Context = context;

            switch (this.Context.Subcommand)
            {
                case Context.SubcommandType.Create:
                    {
                        // 新規作成
                        var file = new IntermediateFile(IntermediateFileKind.Model);
                        Model model = file.GetRootEntity<Model>();
                        model.Skeleton.RootBone = new Bone() { Name = "root" };

                        AddEmptyMaterials(file, this.Context.MaterialNames);

                        IfWriteUtility.WriteIntermediateFile(file, this.Context.OutputFmdPath, Context.XsdBasePath);
                    }
                    break;
                case Context.SubcommandType.Add:
                    {
                        // 既存のファイルに追加
                        IntermediateFile file = IfReadUtility.ReadIntermediateFile(this.Context.InputFmdPath, Context.XsdBasePath);
                        AddEmptyMaterials(file, this.Context.MaterialNames);

                        if (!string.IsNullOrEmpty(this.Context.OutputFmdPath))
                        {
                            IfWriteUtility.WriteIntermediateFile(file, this.Context.OutputFmdPath, Context.XsdBasePath);
                        }
                        else
                        {
                            IfWriteUtility.WriteIntermediateFile(file, this.Context.InputFmdPath, Context.XsdBasePath);
                        }
                    }
                    break;
                case Context.SubcommandType.Remove:
                    {
                        IntermediateFile file = IfReadUtility.ReadIntermediateFile(this.Context.InputFmdPath, Context.XsdBasePath);
                        RemoveMaterials(file.GetRootEntity<Model>(), this.Context.MaterialNames);

                        if (!string.IsNullOrEmpty(this.Context.OutputFmdPath))
                        {
                            IfWriteUtility.WriteIntermediateFile(file, this.Context.OutputFmdPath, Context.XsdBasePath);
                        }
                        else
                        {
                            IfWriteUtility.WriteIntermediateFile(file, this.Context.InputFmdPath, Context.XsdBasePath);
                        }
                    }
                    break;
                case Context.SubcommandType.Rename:
                    {
                        IntermediateFile file = IfReadUtility.ReadIntermediateFile(this.Context.InputFmdPath, Context.XsdBasePath);
                        RenameMaterial(file.GetRootEntity<Model>(), this.Context.OldMaterialName, this.Context.NewMaterialName);
                        if (!string.IsNullOrEmpty(this.Context.OutputFmdPath))
                        {
                            IfWriteUtility.WriteIntermediateFile(file, this.Context.OutputFmdPath, Context.XsdBasePath);
                        }
                        else
                        {
                            IfWriteUtility.WriteIntermediateFile(file, this.Context.InputFmdPath, Context.XsdBasePath);
                        }
                    }
                    break;
                default:
                    throw new Exception($"Unknown subcommand type {this.Context.Subcommand}");
            }
        }

        private static void RemoveShapes(Model model, string materialName)
        {
            List<Shape> removeShapes = new List<Shape>();
            foreach (var shape in model.Shapes.TakeWhile(x => x.ShapeInfo.MatName == materialName))
            {
                removeShapes.Add(shape);
            }

            List<Stream> removeStreams = new List<Stream>();
            foreach (var shape in removeShapes)
            {
                Vertex vertex = shape.ShapeInfo.Vertex;
                foreach (Mesh mesh in shape.Meshes)
                {
                    removeStreams.Add(mesh.IndexStream);
                }

                foreach (VtxAttrib attr in vertex.VtxAttribs)
                {
                    removeStreams.Add(attr.Stream);
                }

                model.Shapes.Remove(shape);
                model.Vertexes.Remove(vertex);
            }

            // シェイプに関連するストリームの除去
            foreach (var stream in removeStreams.Distinct())
            {
                model.Streams.Remove(stream);
            }
        }

        private static void AddEmptyMaterials(IntermediateFile file, IEnumerable<string> materialNames)
        {
            Model model = file.GetRootEntity<Model>();
            if (model.Skeleton.CountBones() == 0)
            {
                throw new Exception("マテリアルを追加するにはボーンが必要ですが、モデルにボーンがひとつもありません");
            }

            Bone bone = model.Skeleton.RootBone;
            foreach (string materialName in materialNames)
            {
                if (model.Materials.Any(x => x.Name == materialName))
                {
                    throw new Exception($"{materialName} は既に存在するため追加することができません。");
                }

                Material material = model.CreateMaterial(materialName);
                Shape shape = model.CreateShape(material, bone, new int[] { 0, 0, 0 });

                // ダミーの頂点属性を追加(とりあえずよく使われる既定のものを追加)
                // TODO: ユーザー定義のものは未対応なので、オプションで追加できるようにするなど対応する
                shape.AddVertexAttributePosition(new float[] { 0.0f, 0.0f, 0.0f });
                shape.AddVertexAttributeColor(new float[] { 0.0f, 0.0f, 0.0f, 1.0f });
                shape.AddVertexAttributeNormal(new float[] { 0.0f, 1.0f, 0.0f });
                shape.AddVertexAttributeTangent(new float[] { 1.0f, 0.0f, 0.0f });
                shape.AddVertexAttributeBinormal(new float[] { 0.0f, 0.0f, 1.0f });
                shape.AddVertexAttributeUv(new float[] { 0.0f, 0.0f });
                shape.AddVertexAttributeUv(new float[] { 0.0f, 0.0f });
                shape.AddVertexAttributeUv(new float[] { 0.0f, 0.0f });
                shape.AddVertexAttributeBlendWeight(new float[] { 1.0f, 0.0f, 0.0f, 0.0f });
                shape.AddVertexAttributeBlendWeight(new float[] { 0.0f, 0.0f, 0.0f, 0.0f });
                shape.AddVertexAttributeBlendIndex(new int[] { 0, 0, 0, 0 });
                shape.AddVertexAttributeBlendIndex(new int[] { 0, 0, 0, 0 });
            }
        }

        private static string GetShapeName(string materialName)
        {
            return "Shape__" + materialName;
        }

        private static void RemoveMaterials(Model model, IEnumerable<string> materialNames)
        {
            foreach (string materialName in materialNames)
            {
                RemoveMaterial(model, materialName);
            }
        }

        private static void RemoveMaterial(Model model, string materialName)
        {
            Material material = model.Materials.FirstOrDefault(x => x.Name == materialName);
            if (material == null)
            {
                // 見つからなければ無視
                return;
            }

            model.Materials.Remove(material);

            RemoveShapes(model, materialName);
        }

        private static void RenameMaterial(Model model, string oldMatName, string newMatName)
        {
            Material material = model.Materials.FirstOrDefault(x => x.Name == oldMatName);
            if (material == null)
            {
                throw new Exception($"マテリアル \"{oldMatName}\" がみつかりませんでした。");
            }

            material.Name = newMatName;

            Shape shape = model.Shapes.FirstOrDefault(x => x.ShapeInfo.MatName == oldMatName);
            if (shape == null)
            {
                return;
            }

            shape.ShapeInfo.MatName = newMatName;
            shape.Name = GetShapeName(newMatName);
        }
    }
}
