﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace nw.g3d.iflib
{
    using System;

    /// <summary>
    /// 条件が満たされることを保証する際に利用するクラスです。
    /// 条件が満たされ無い場合には、例外を送出します。
    /// プログラムの構造上ありえないような条件の確認には Assert クラスを利用します。
    /// </summary>
    public static class Ensure
    {
        /// <summary>
        /// 引数に関する例外を送出します。
        /// </summary>
        public static class Argument
        {
            /// <summary>
            /// 引数が true であるか判定し、true でない場合は例外を送出します。
            /// </summary>
            /// <param name="status">引数です。</param>
            public static void True(bool status)
            {
                if (status == false)
                {
                    throw new ArgumentException();
                }
            }

            /// <summary>
            /// 引数がnullでは無いか判断し、不正な場合は例外を送出します。
            /// </summary>
            /// <param name="argument">引数です。</param>
            public static void NotNull(object argument)
            {
                if (argument == null)
                {
                    throw new ArgumentNullException();
                }
            }

            /// <summary>
            /// 文字列引数が妥当か判断し、不正な場合は例外を送出します。
            /// </summary>
            /// <param name="stringArg">文字列引数です。</param>
            public static void StringNotEmpty(string stringArg)
            {
                Ensure.Argument.NotNull(stringArg);
                Ensure.Argument.True(stringArg != string.Empty);
            }
        }

        /// <summary>
        /// 操作に関する例外を送出します。
        /// </summary>
        public static class Operation
        {
            /// <summary>
            /// 操作が true であるか判定し、true でない場合は例外を送出します。
            /// </summary>
            /// <param name="status">状態です。</param>
            public static void True(bool status)
            {
                True(status, string.Empty);
            }

            /// <summary>
            /// 操作が true であるか判定し、true でない場合は例外を送出します。
            /// </summary>
            /// <param name="status">状態です。</param>
            /// <param name="debugMessage">
            /// デバック用文字列です。
            /// ユーザーに表示するメッセージではないのでローカライズの必要はありません。
            /// </param>
            public static void True(bool status, string debugMessage)
            {
                if (status == false)
                {
                    throw new InvalidOperationException(debugMessage);
                }
            }

            /// <summary>
            /// オブジェクトがnullであるか判定し、nullでない場合は例外を送出します。
            /// </summary>
            /// <param name="referenceObject">オブジェクトです。</param>
            public static void ObjectNull(object referenceObject)
            {
                Ensure.Operation.True(referenceObject == null);
            }

            /// <summary>
            /// オブジェクトがnullで無いか判定し、nullの場合は例外を送出します。
            /// </summary>
            /// <param name="referenceObject">オブジェクトです。</param>
            public static void ObjectNotNull(object referenceObject)
            {
                Ensure.Operation.True(referenceObject != null);
            }

            /// <summary>
            /// 除算の分母が0.0fであるかを判定し、0.0fの場合は例外を送出します。
            /// </summary>
            /// <param name="divider">除算の分母です。</param>
            public static void DividerNotZero(float divider)
            {
                if (FloatUtility.NearlyEqual(divider, 0f))
                {
                    throw new DivideByZeroException();
                }
            }
        }
    }
}
