﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace nw.g3d.iflib
{
    using System.Diagnostics;

    /// <summary>
    /// 正方行列の演算関数です。
    /// </summary>
    /// <typeparam name="TMatrix">テンプレート行列型です。</typeparam>
    internal static class MatrixSquareFunction<TMatrix> where TMatrix : IMatrix, new()
    {
        /// <summary>
        /// 2 つの行列を乗算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static IMatrix Multiply(TMatrix lhs, TMatrix rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            DebugMatrixRowColumnCount(lhs);
            DebugMatrixRowColumnCount(rhs);

            int dimension = lhs.ColumnCount;
            IMatrix result = new TMatrix();
            for (int i = 0; i < lhs.RowCount; i++)
            {
                for (int j = 0; j < lhs.ColumnCount; j++)
                {
                    float sum = 0f;
                    for (int k = 0; k < dimension; k++)
                    {
                        sum += lhs[i, k] * rhs[k, j];
                    }

                    result[i, j] = sum;
                }
            }

            return result;
        }

        /// <summary>
        /// 転置行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        public static void Transpose(TMatrix mtx)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);
            for (int i = 0; i < mtx.RowCount; i++)
            {
                for (int j = 0; j < mtx.ColumnCount; j++)
                {
                    if (i == j)
                    {
                        break;
                    }

                    float temp = mtx[i, j];
                    mtx[i, j] = mtx[j, i];
                    mtx[j, i] = temp;
                }
            }
        }

        /// <summary>
        /// 行列の行数と列数が演算に対して適切であることを確認します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        [Conditional("DEBUG")]
        private static void DebugMatrixRowColumnCount(TMatrix mtx)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(mtx.RowCount == mtx.ColumnCount, "Unexpected case!");
        }
    }
}
