﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace nw.g3d.iflib
{
    using System.Diagnostics;
    using System.Text;

    /// <summary>
    /// 行列の演算関数です。
    /// </summary>
    /// <typeparam name="TMatrix">テンプレート行列オブジェクトです。</typeparam>
    internal static class MatrixFunction<TMatrix> where TMatrix : class, IMatrix, new()
    {
        //-----------------------------------------------------------------
        // 数学演算
        //-----------------------------------------------------------------

        /// <summary>
        /// 転置行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        public static void Transpose(TMatrix mtx)
        {
            Ensure.Argument.NotNull(mtx);

            int minCount = System.Math.Min(mtx.RowCount, mtx.ColumnCount);
            for (int i = 0; i < minCount; i++)
            {
                for (int j = 0; j < minCount; j++)
                {
                    if (i == j)
                    {
                        break;
                    }

                    float temp = mtx[i, j];
                    mtx[i, j] = mtx[j, i];
                    mtx[j, i] = temp;
                }
            }

            // 正方でない部分を 0 で埋める
            for (int i = minCount; i < mtx.RowCount; i++)
            {
                for (int j = 0; j < mtx.ColumnCount; j++)
                {
                    mtx[i, j] = 0f;
                }
            }

            // 正方でない部分を 0 で埋める
            for (int j = minCount; j < mtx.ColumnCount; j++)
            {
                for (int i = 0; i < mtx.RowCount; i++)
                {
                    mtx[i, j] = 0f;
                }
            }
        }

        /// <summary>
        /// 2 つの行列を加算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static IMatrix Add(TMatrix lhs, TMatrix rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            IMatrix result = new TMatrix();
            for (int i = 0; i < lhs.RowCount; i++)
            {
                for (int j = 0; j < lhs.ColumnCount; j++)
                {
                    result[i, j] = lhs[i, j] + rhs[i, j];
                }
            }

            return result;
        }

        /// <summary>
        /// 2 つの行列を減算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static IMatrix Subtract(TMatrix lhs, TMatrix rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            IMatrix result = new TMatrix();
            for (int i = 0; i < lhs.RowCount; i++)
            {
                for (int j = 0; j < lhs.ColumnCount; j++)
                {
                    result[i, j] = lhs[i, j] - rhs[i, j];
                }
            }

            return result;
        }

        /// <summary>
        /// 行列とスカラーを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static IMatrix Multiply(TMatrix mtx, float scalar)
        {
            Ensure.Argument.NotNull(mtx);

            IMatrix result = new TMatrix();
            for (int i = 0; i < mtx.RowCount; i++)
            {
                for (int j = 0; j < mtx.ColumnCount; j++)
                {
                    result[i, j] = mtx[i, j] * scalar;
                }
            }

            return result;
        }

        /// <summary>
        /// 行列とスカラーを除算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static IMatrix Divide(TMatrix mtx, float scalar)
        {
            Ensure.Argument.NotNull(mtx);
            Ensure.Operation.DividerNotZero(scalar);

            IMatrix result = new TMatrix();
            for (int i = 0; i < mtx.RowCount; i++)
            {
                for (int j = 0; j < mtx.ColumnCount; j++)
                {
                    result[i, j] = mtx[i, j] / scalar;
                }
            }

            return result;
        }

        /// <summary>
        /// 行列の成分を全て 0 に設定します。
        /// </summary>
        /// <param name="mtx">設定対象のテンプレート行列オブジェクトです。</param>
        public static void SetZero(TMatrix mtx)
        {
            Ensure.Argument.NotNull(mtx);

            for (int i = 0; i < mtx.RowCount; i++)
            {
                for (int j = 0; j < mtx.ColumnCount; j++)
                {
                    mtx[i, j] = 0f;
                }
            }
        }

        /// <summary>
        /// 単位行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        public static void SetIdentity(TMatrix mtx)
        {
            Ensure.Argument.NotNull(mtx);

            for (int i = 0; i < mtx.RowCount; i++)
            {
                for (int j = 0; j < mtx.ColumnCount; j++)
                {
                    if (i == j)
                    {
                        mtx[i, j] = 1f;
                    }
                    else
                    {
                        mtx[i, j] = 0f;
                    }
                }
            }
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">設定元の行列です。</param>
        /// <param name="destination">設定先の行列です。</param>
        public static void Set(TMatrix source, TMatrix destination)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.NotNull(destination);

            for (int i = 0; i < source.RowCount; i++)
            {
                for (int j = 0; j < source.ColumnCount; j++)
                {
                    destination[i, j] = source[i, j];
                }
            }
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <typeparam name="TSourceMatrix">設定元の型パラメーターです。</typeparam>
        /// <param name="source">設定元の行列です。</param>
        /// <param name="destination">設定先の行列です。</param>
        public static void Set<TSourceMatrix>(TSourceMatrix source, TMatrix destination)
            where TSourceMatrix : IMatrix
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.NotNull(destination);

            for (int i = 0; i < source.RowCount; i++)
            {
                if (i >= destination.RowCount)
                {
                    break;
                }

                for (int j = 0; j < source.ColumnCount; j++)
                {
                    if (j >= destination.ColumnCount)
                    {
                        break;
                    }

                    destination[i, j] = source[i, j];
                }
            }

            for (int i = source.RowCount; i < destination.RowCount; i++)
            {
                for (int j = source.ColumnCount; j < destination.ColumnCount; j++)
                {
                    if (i == j)
                    {
                        destination[i, j] = 1f;
                    }
                    else
                    {
                        destination[i, j] = 0f;
                    }
                }
            }
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 2 つの行列の値が等しいかを示します。
        /// </summary>
        /// <param name="lhs">比較する左辺値行列です。</param>
        /// <param name="rhs">比較する右辺値行列です。</param>
        /// <returns>等しい場合は true を返します。それ以外の場合は false を返します。</returns>
        public static bool Equals(TMatrix lhs, TMatrix rhs)
        {
            Ensure.Argument.NotNull(lhs);
            if (lhs == rhs)
            {
                return true;
            }

            if (rhs == null)
            {
                return false;
            }

            for (int i = 0; i < lhs.RowCount; i++)
            {
                for (int j = 0; j < lhs.ColumnCount; j++)
                {
                    if (lhs[i, j] != rhs[i, j])
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// 2 つの行列の値が等しいかを示します。
        /// </summary>
        /// <param name="lhs">比較する左辺値行列です。</param>
        /// <param name="rhs">比較する右辺値行列です。</param>
        /// <param name="tolerance">各成分毎の差の最大許容値です。</param>
        /// <returns>等しい場合は true を返します。それ以外の場合は false を返します。</returns>
        public static bool Equals(TMatrix lhs, TMatrix rhs, float tolerance)
        {
            Ensure.Argument.NotNull(lhs);
            if (lhs == rhs)
            {
                return true;
            }

            if (rhs == null)
            {
                return false;
            }

            for (int i = 0; i < lhs.RowCount; i++)
            {
                for (int j = 0; j < lhs.ColumnCount; j++)
                {
                    if (!FloatUtility.NearlyEqual(lhs[i, j], rhs[i, j], tolerance))
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <param name="mtx">ハッシュ値を取得する行列です。</param>
        /// <returns>ハッシュ値です。</returns>
        public static int GetHashCode(IMatrix mtx)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True((mtx.RowCount > 0) && (mtx.ColumnCount > 0), "Unexpected case!");
            int hashCode = 0;
            for (int i = 0; i < mtx.RowCount; i++)
            {
                for (int j = 0; j < mtx.ColumnCount; j++)
                {
                    hashCode ^= mtx[i, j].GetHashCode();
                }
            }

            return hashCode;
        }

        //-----------------------------------------------------------------
        // 文字列化
        //-----------------------------------------------------------------

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <param name="mtx">文字列化対照の行列インターフェースを持つ型です。</param>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public static string ToString(IMatrix mtx)
        {
            StringBuilder builder = new StringBuilder();
            builder.Append("(");
            for (int i = 0; i < mtx.RowCount; i++)
            {
                builder.Append("[");
                for (int j = 0; j < mtx.ColumnCount; j++)
                {
                    builder.AppendFormat("{0}", mtx[i, j]);
                    if (j == mtx.ColumnCount - 1)
                    {
                        builder.Append("]");
                    }

                    if (i != mtx.RowCount - 1 || j != mtx.ColumnCount - 1)
                    {
                        builder.Append(" ");
                    }
                }
            }

            builder.Append(")");
            return builder.ToString();
        }
    }
}
