﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace nw.g3d.iflib
{
    using System;
    using System.Diagnostics;

    /// <summary>
    /// 行列の３次元のスケールと回転に関する演算関数です。
    /// </summary>
    /// <typeparam name="TMatrix">テンプレート行列型です。</typeparam>
    internal static class Matrix3DScaleRotateFunction<TMatrix> where TMatrix : IMatrix, new()
    {
        /// <summary>
        /// x 軸、y 軸、z 軸に沿ってスケーリングする行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="x">x 軸に沿ったスケールです。</param>
        /// <param name="y">y 軸に沿ったスケールです。</param>
        /// <param name="z">z 軸に沿ったスケールです。</param>
        public static void SetScale(TMatrix mtx, float x, float y, float z)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            mtx.SetIdentity();
            mtx[0, 0] = x;
            mtx[1, 1] = y;
            mtx[2, 2] = z;
        }

        /// <summary>
        /// 任意の軸を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="axis">回転軸です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public static void SetRotateAxis(TMatrix mtx, Vector3 axis, float radian)
        {
            Ensure.Argument.NotNull(mtx);
            Ensure.Argument.NotNull(axis);

            DebugMatrixRowColumnCount(mtx);

            // RevolutionSDK 準拠 mtx.c C_MTXRotAxisRad
            float s = (float)Math.Sin(radian);
            float c = (float)Math.Cos(radian);
            float t = 1 - c;

            axis.Normalize();

            float x = axis.X;
            float y = axis.Y;
            float z = axis.Z;
            float xSq = x * x;
            float ySq = y * y;
            float zSq = z * z;

            mtx[0, 0] = (t * xSq) + c;
            mtx[0, 1] = (t * x * y) - (s * z);
            mtx[0, 2] = (t * x * z) + (s * y);

            mtx[1, 0] = (t * x * y) + (s * z);
            mtx[1, 1] = (t * ySq) + c;
            mtx[1, 2] = (t * y * z) - (s * x);

            mtx[2, 0] = (t * x * z) - (s * y);
            mtx[2, 1] = (t * y * z) + (s * x);
            mtx[2, 2] = (t * zSq) + c;
        }

        /// <summary>
        /// x 軸を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public static void SetRotateX(TMatrix mtx, float radian)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            // RevolutionSDK 準拠 mtx.c C_MTXRotTrig
            float s = (float)Math.Sin(radian);
            float c = (float)Math.Cos(radian);

            mtx.SetIdentity();
            mtx[1, 1] = c;
            mtx[1, 2] = -s;
            mtx[2, 1] = s;
            mtx[2, 2] = c;
        }

        /// <summary>
        /// y 軸を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public static void SetRotateY(TMatrix mtx, float radian)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            // RevolutionSDK 準拠 mtx.c C_MTXRotTrig
            float s = (float)Math.Sin(radian);
            float c = (float)Math.Cos(radian);

            mtx.SetIdentity();
            mtx[0, 0] = c;
            mtx[0, 2] = s;
            mtx[2, 0] = -s;
            mtx[2, 2] = c;
        }

        /// <summary>
        /// z 軸を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public static void SetRotateZ(TMatrix mtx, float radian)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            // RevolutionSDK 準拠 mtx.c C_MTXRotTrig
            float s = (float)Math.Sin(radian);
            float c = (float)Math.Cos(radian);

            mtx.SetIdentity();
            mtx[0, 0] = c;
            mtx[0, 1] = -s;
            mtx[1, 0] = s;
            mtx[1, 1] = c;
        }

        /// <summary>
        /// クォータニオンが表す回転行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="quat">クォータニオンです。</param>
        public static void SetRotate(TMatrix mtx, Quaternion quat)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            // RevolutionSDK 準拠 mtx.c C_MTXQuat
            float s, xs, ys, zs, wx, wy, wz, xx, xy, xz, yy, yz, zz;

            s = 2.0f / ((quat.X * quat.X) + (quat.Y * quat.Y) + (quat.Z * quat.Z) + (quat.W * quat.W));

            xs = quat.X * s;
            ys = quat.Y * s;
            zs = quat.Z * s;
            wx = quat.W * xs;
            wy = quat.W * ys;
            wz = quat.W * zs;
            xx = quat.X * xs;
            xy = quat.X * ys;
            xz = quat.X * zs;
            yy = quat.Y * ys;
            yz = quat.Y * zs;
            zz = quat.Z * zs;

            mtx.SetIdentity();

            mtx[0, 0] = 1.0f - (yy + zz);
            mtx[0, 1] = xy - wz;
            mtx[0, 2] = xz + wy;

            mtx[1, 0] = xy + wz;
            mtx[1, 1] = 1.0f - (xx + zz);
            mtx[1, 2] = yz - wx;

            mtx[2, 0] = xz - wy;
            mtx[2, 1] = yz + wx;
            mtx[2, 2] = 1.0f - (xx + yy);
        }

        /// <summary>
        /// オイラー角の表現する行列を取得します。
        /// オイラー角の表現法はxyzの順です。
        /// </summary>
        /// <param name="mtx">設定する回転行列です。</param>
        /// <param name="rotate">オイラー角（ラジアン単位)のベクトルです。</param>
        public static void SetRotateEuler(TMatrix mtx, Vector3 rotate)
        {
            Ensure.Argument.NotNull(mtx);

            // NW4RSystemLib 準拠 math_types.cpp MTX34RotXYZFIdx
            mtx.SetIdentity();

            double sinx = Math.Sin(rotate.X);
            double cosx = Math.Cos(rotate.X);
            double siny = Math.Sin(rotate.Y);
            double cosy = Math.Cos(rotate.Y);
            double sinz = Math.Sin(rotate.Z);
            double cosz = Math.Cos(rotate.Z);
            double f1, f2;

            mtx[2, 0] = (float)-siny;
            mtx[0, 0] = (float)(cosz * cosy);
            mtx[1, 0] = (float)(sinz * cosy);
            mtx[2, 1] = (float)(cosy * sinx);
            mtx[2, 2] = (float)(cosy * cosx);

            f1 = cosx * sinz;
            f2 = sinx * cosz;

            mtx[0, 1] = (float)((f2 * siny) - f1);
            mtx[1, 2] = (float)((f1 * siny) - f2);

            f1 = sinx * sinz;
            f2 = cosx * cosz;
            mtx[0, 2] = (float)((f2 * siny) + f1);
            mtx[1, 1] = (float)((f1 * siny) + f2);
        }

        /// <summary>
        /// 回転行列のオイラー角を取得します。
        /// オイラー角の表現法はxyzの順です。
        /// </summary>
        /// <param name="mtx">回転行列です。</param>
        /// <param name="rotate">設定するオイラー角（ラジアン単位)のベクトルです。</param>
        public static void GetRotateEuler(TMatrix mtx, Vector3 rotate)
        {
            Ensure.Argument.NotNull(mtx);

            double cosEpsilon = 0.00001;
            double syr = -mtx[2, 0];
            double cyr2 = 1.0 - (syr * syr);
            if (cyr2 < 0.0)
            {
                cyr2 = 0.0;
            }

            double cyr = Math.Sqrt(cyr2);
            if (cyr < cosEpsilon)
            {
                rotate.X = (float)Math.Atan2(-mtx[1, 2], mtx[1, 1]);
                rotate.Y = (syr > 0.0) ? (float)Math.PI / 2 : (float)-Math.PI / 2;
                rotate.Z = 0.0F;
            }
            else
            {
                rotate.X = (float)Math.Atan2(mtx[2, 1], mtx[2, 2]);
                rotate.Y = (float)Math.Atan2(syr, cyr);
                rotate.Z = (float)Math.Atan2(mtx[1, 0], mtx[0, 0]);
            }
        }

        /// <summary>
        /// 行列の行数と列数が演算に対して適切であることを確認します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        [Conditional("DEBUG")]
        private static void DebugMatrixRowColumnCount(TMatrix mtx)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(mtx.RowCount >= 3, "Unexpected case!");
            Nintendo.Foundation.Contracts.Assertion.Operation.True(mtx.ColumnCount >= 3, "Unexpected case!");
        }
    }
}
