﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace nw.g3d.iflib
{
    using System.Diagnostics;

    /// <summary>
    /// 3 × 3 行列の演算関数です。
    /// </summary>
    /// <typeparam name="TMatrix">テンプレート行列型です。</typeparam>
    internal static class Matrix33Function<TMatrix> where TMatrix : IMatrix, new()
    {
        /// <summary>
        /// 3 × 3 の部分行列の逆行列を取得します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        /// <returns>逆行列です。</returns>
        public static TMatrix Invert(TMatrix mtx)
        {
            // RevolutionSDK 準拠 mtx.c C_MTXInverse
            DebugMatrixRowColumnCount(mtx);

            float det = GetDeterminant(mtx);
            if (det == 0f)
            {
                return default(TMatrix);
            } // null を返す

            det = 1.0f / det;

            TMatrix inv = new TMatrix();

            inv[0, 0] = ((mtx[1, 1] * mtx[2, 2]) - (mtx[2, 1] * mtx[1, 2])) * det;
            inv[0, 1] = -((mtx[0, 1] * mtx[2, 2]) - (mtx[2, 1] * mtx[0, 2])) * det;
            inv[0, 2] = ((mtx[0, 1] * mtx[1, 2]) - (mtx[1, 1] * mtx[0, 2])) * det;

            inv[1, 0] = -((mtx[1, 0] * mtx[2, 2]) - (mtx[2, 0] * mtx[1, 2])) * det;
            inv[1, 1] = ((mtx[0, 0] * mtx[2, 2]) - (mtx[2, 0] * mtx[0, 2])) * det;
            inv[1, 2] = -((mtx[0, 0] * mtx[1, 2]) - (mtx[1, 0] * mtx[0, 2])) * det;

            inv[2, 0] = ((mtx[1, 0] * mtx[2, 1]) - (mtx[2, 0] * mtx[1, 1])) * det;
            inv[2, 1] = -((mtx[0, 0] * mtx[2, 1]) - (mtx[2, 0] * mtx[0, 1])) * det;
            inv[2, 2] = ((mtx[0, 0] * mtx[1, 1]) - (mtx[1, 0] * mtx[0, 1])) * det;

            return inv;
        }

        /// <summary>
        /// 3 × 3 の部分行列の行列式を取得します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        /// <returns>行列式です。</returns>
        public static float GetDeterminant(TMatrix mtx)
        {
            // RevolutionSDK 準拠 mtx.c C_MTXInverse
            DebugMatrixRowColumnCount(mtx);

            return
                (mtx[0, 0] * mtx[1, 1] * mtx[2, 2]) +
                (mtx[0, 1] * mtx[1, 2] * mtx[2, 0]) +
                (mtx[0, 2] * mtx[1, 0] * mtx[2, 1]) -
                (mtx[2, 0] * mtx[1, 1] * mtx[0, 2]) -
                (mtx[1, 0] * mtx[0, 1] * mtx[2, 2]) -
                (mtx[0, 0] * mtx[2, 1] * mtx[1, 2]);
        }

        /// <summary>
        /// 行列の行数と列数が演算に対して適切であることを確認します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        [Conditional("DEBUG")]
        private static void DebugMatrixRowColumnCount(TMatrix mtx)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(mtx.RowCount >= 3, "Unexpected case!");
            Nintendo.Foundation.Contracts.Assertion.Operation.True(mtx.ColumnCount >= 3, "Unexpected case!");
        }
    }
}
