﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace nw.g3d.iflib
{
    using System;

    /// <summary>
    /// 2 × 2 行列を表します。
    /// </summary>
    public sealed class Matrix22 : IMatrix, IEquatable<Matrix22>
    {
        //-----------------------------------------------------------------
        private const int MatrixRowCount = 2;
        private const int MatrixColumnCount = 2;
        private readonly float[,] m = new float[MatrixRowCount, MatrixColumnCount]
            {
                { 1f, 0f },
                { 0f, 1f },
            };

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Matrix22()
        {
        }

        /// <summary>
        /// 行列の成分を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="m00">[0, 0]成分です。</param>
        /// <param name="m01">[0, 1]成分です。</param>
        /// <param name="m10">[1, 0]成分です。</param>
        /// <param name="m11">[1, 1]成分です。</param>
        public Matrix22(
            float m00,
            float m01,
            float m10,
            float m11)
        {
            this.Set(m00, m01, m10, m11);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Matrix22(Matrix22 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Matrix22(Matrix23 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Matrix22(Matrix33 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Matrix22(Matrix34 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Matrix22(Matrix44 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="row0">第０行の成分を格納したベクトルです。</param>
        /// <param name="row1">第１行の成分を格納したベクトルです。</param>
        public Matrix22(Vector2 row0, Vector2 row1)
        {
            this.Set(row0, row1);
        }

        /// <summary>
        /// 成分が全て 0 の行列を作成します。
        /// </summary>
        /// <returns>成分が全て 0 の行列です。</returns>
        public static Matrix22 Zero
        {
            get
            {
                var result = new Matrix22();
                result.SetZero();
                return result;
            }
        }

        /// <summary>
        /// 単位行列を作成します。
        /// </summary>
        /// <returns>単位行列です。</returns>
        public static Matrix22 Identity
        {
            get
            {
                return new Matrix22();
            }
        }

        //-----------------------------------------------------------------
        // プロパティの取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// 列数を取得します。
        /// </summary>
        public int ColumnCount
        {
            get
            {
                return MatrixColumnCount;
            }
        }

        /// <summary>
        /// 行数を取得します。
        /// </summary>
        public int RowCount
        {
            get
            {
                return MatrixRowCount;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 行列式を取得します。
        /// </summary>
        /// <returns>行列式の値です。</returns>
        public float Determinant
        {
            get
            {
                return Matrix22Function<Matrix22>.GetDeterminant(this);
            }
        }

        //-----------------------------------------------------------------
        // 行列の成分の取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// 行成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Rows
        {
            get
            {
                IVector[] data =
                {
                     new Vector2(this[0, 0], this[0, 1]),
                     new Vector2(this[1, 0], this[1, 1])
                };

                return data;
            }
        }

        /// <summary>
        /// 列成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Columns
        {
            get
            {
                IVector[] data =
                {
                     new Vector2(this[0, 0], this[1, 0]),
                     new Vector2(this[0, 1], this[1, 1])
                };

                return data;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したインデックス位置の成分へのアクセスを行うインデクサです。
        /// </summary>
        /// <param name="row">アクセスする成分の行の 0 から始まるインデックスです。</param>
        /// <param name="column">アクセスする成分の列の 0 から始まるインデックスです。</param>
        /// <returns>指定した位置の成分です。</returns>
        public float this[int row, int column]
        {
            get
            {
                return this.m[row, column];
            }

            set
            {
                this.m[row, column] = value;
            }
        }

        //-----------------------------------------------------------------
        // オペレータ
        //-----------------------------------------------------------------

        /// <summary>
        /// 2 つの行列を加算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix22 operator +(Matrix22 lhs, Matrix22 rhs)
        {
            return (Matrix22)MatrixFunction<Matrix22>.Add(lhs, rhs);
        }

        /// <summary>
        /// 2 つの行列を減算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix22 operator -(Matrix22 lhs, Matrix22 rhs)
        {
            return (Matrix22)MatrixFunction<Matrix22>.Subtract(lhs, rhs);
        }

        /// <summary>
        /// 行列とスカラーを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix22 operator *(Matrix22 mtx, float scalar)
        {
            return (Matrix22)MatrixFunction<Matrix22>.Multiply(mtx, scalar);
        }

        /// <summary>
        /// 行列とスカラーを除算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix22 operator /(Matrix22 mtx, float scalar)
        {
            return (Matrix22)MatrixFunction<Matrix22>.Divide(mtx, scalar);
        }

        /// <summary>
        /// 2 つの行列を乗算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix22 operator *(Matrix22 lhs, Matrix22 rhs)
        {
            return (Matrix22)MatrixSquareFunction<Matrix22>.Multiply(lhs, rhs);
        }

        /// <summary>
        /// 行列とベクトルを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="vec">演算の右のベクトルです。</param>
        /// <returns>演算結果です。</returns>
        public static Vector2 operator *(Matrix22 mtx, Vector2 vec)
        {
            Ensure.Argument.NotNull(mtx);
            Ensure.Argument.NotNull(vec);

            return new Vector2(
                (mtx[0, 0] * vec.X) + (mtx[0, 1] * vec.Y),
                (mtx[1, 0] * vec.X) + (mtx[1, 1] * vec.Y));
        }

        /// <summary>
        /// 指定した行列の逆行列を作成します。
        /// 逆行列に設定できなかった場合は行列の成分を変更しません。
        /// </summary>
        /// <param name="mat">指定の行列です。</param>
        public static void Invert(Matrix22 mat)
        {
            Matrix22 result = InverseSolver(mat);

            Ensure.Operation.ObjectNotNull(result);

            mat.Set(result);
        }

        /// <summary>
        /// 逆行列に設定します。
        /// 逆行列が存在しない場合は指定した行列に設定します。
        /// </summary>
        /// <param name="mtx">元の行列です。</param>
        /// <param name="substitute">逆行列が存在しない場合に
        /// 代わりに設定する行列です。</param>
        /// <returns>逆行列が存在した場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool SafeInvert(Matrix22 mtx, Matrix22 substitute)
        {
            Ensure.Argument.NotNull(substitute);

            Matrix22 result = InverseSolver(mtx);
            if (result == null)
            {
                if (mtx != substitute)
                {
                    mtx.Set(substitute);
                }

                return false;
            }

            mtx.Set(result);
            return true;
        }

        /// <summary>
        /// 指定した行列の転置行列を取得します。
        /// </summary>
        /// <param name="mtx">指定の行列です。</param>
        /// <returns>転置行列です。</returns>
        public static Matrix22 Transpose(Matrix22 mtx)
        {
            Matrix22 result = new Matrix22(mtx);
            MatrixSquareFunction<Matrix22>.Transpose(result);
            return result;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new Matrix22(this);
        }

        /// <summary>
        /// 行列の成分を全て 0 に設定します。
        /// </summary>
        public void SetZero()
        {
            MatrixFunction<Matrix22>.SetZero(this);
        }

        /// <summary>
        /// 単位行列に設定します。
        /// </summary>
        public void SetIdentity()
        {
            MatrixFunction<Matrix22>.SetIdentity(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as Matrix22);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix22 source)
        {
            MatrixFunction<Matrix22>.Set(source, this);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix23 source)
        {
            MatrixFunction<Matrix22>.Set<Matrix23>(source, this);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix33 source)
        {
            MatrixFunction<Matrix22>.Set<Matrix33>(source, this);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix34 source)
        {
            MatrixFunction<Matrix22>.Set<Matrix34>(source, this);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix44 source)
        {
            MatrixFunction<Matrix22>.Set<Matrix44>(source, this);
        }

        /// <summary>
        /// 行列の成分を指定の値に設定します。
        /// </summary>
        /// <param name="m00">[0, 0]成分です。</param>
        /// <param name="m01">[0, 1]成分です。</param>
        /// <param name="m10">[1, 0]成分です。</param>
        /// <param name="m11">[1, 1]成分です。</param>
        public void Set(
            float m00,
            float m01,
            float m10,
            float m11)
        {
            this[0, 0] = m00;
            this[0, 1] = m01;

            this[1, 0] = m10;
            this[1, 1] = m11;
        }

        /// <summary>
        /// 行列の成分を指定の値に設定します。
        /// </summary>
        /// <param name="row0">第０行の成分を格納したベクトルです。</param>
        /// <param name="row1">第１行の成分を格納したベクトルです。</param>
        public void Set(Vector2 row0, Vector2 row1)
        {
            this[0, 0] = row0.X;
            this[0, 1] = row0.Y;
            this[1, 0] = row1.X;
            this[1, 1] = row1.Y;
        }

        //-----------------------------------------------------------------
        // 数学演算
        //-----------------------------------------------------------------

        /// <summary>
        /// 逆行列に設定します。
        /// 逆行列に設定できなかった場合は行列の成分を変更しません。
        /// </summary>
        /// <exception cref="T:System.InvalidOperationException">逆行列が求められません。</exception>
        public void Invert()
        {
            Invert(this);
        }

        /// <summary>
        /// 逆行列に設定します。
        /// 逆行列が存在しない場合は指定した行列に設定します。
        /// </summary>
        /// <param name="substitute">逆行列が存在しない場合に
        /// 代わりに設定する行列です。</param>
        /// <returns>逆行列が存在した場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public bool SafeInvert(Matrix22 substitute)
        {
            return SafeInvert(this, substitute);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 転置行列に設定します。
        /// </summary>
        public void Transpose()
        {
            MatrixSquareFunction<Matrix22>.Transpose(this);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// x 軸、y 軸に沿ってスケーリングする行列に設定します。
        /// </summary>
        /// <param name="scale">スケールです。</param>
        public void SetScale(Vector2 scale)
        {
            Ensure.Argument.NotNull(scale);

            this.SetScale(scale.X, scale.Y);
        }

        /// <summary>
        /// x 軸、y 軸に沿ってスケーリングする行列に設定します。
        /// </summary>
        /// <param name="x">x 軸に沿ったスケールです。</param>
        /// <param name="y">y 軸に沿ったスケールです。</param>
        public void SetScale(float x, float y)
        {
            Matrix2DScaleRotateFunction<Matrix22>.SetScale(this, x, y);
        }

        /// <summary>
        /// 行列を x 軸、y 軸に沿ってスケーリングします。
        /// </summary>
        /// <param name="scale">スケールです。</param>
        public void Scale(Vector2 scale)
        {
            Ensure.Argument.NotNull(scale);

            Matrix22 mtx = new Matrix22();
            mtx.SetScale(scale);
            this.Set(mtx * this);
        }

        /// <summary>
        /// 行列を x 軸、y 軸に沿ってスケーリングします。
        /// </summary>
        /// <param name="x">x 軸に沿ったスケールです。</param>
        /// <param name="y">y 軸に沿ったスケールです。</param>
        public void Scale(float x, float y)
        {
            Matrix22 mtx = new Matrix22();
            mtx.SetScale(x, y);
            this.Set(mtx * this);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 原点を中心に回転する行列に設定します。反時計方向が正の回転方向です。
        /// </summary>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void SetRotate(float radian)
        {
            Matrix2DScaleRotateFunction<Matrix22>.SetRotate(this, radian);
        }

        /// <summary>
        /// 原点を中心に行列を回転します。
        /// </summary>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void Rotate(float radian)
        {
            var mtx = new Matrix22();
            mtx.SetRotate(radian);
            this.Set(mtx * this);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 要素を格納した配列を返します。
        /// </summary>
        /// <returns>要素の配列です。</returns>
        public float[] ToArray()
        {
            return new float[]
            {
                this.m[0, 0],
                this.m[0, 1],
                this.m[1, 0],
                this.m[1, 1]
            };
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Matrix22);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Matrix22 other)
        {
            return MatrixFunction<Matrix22>.Equals(this, other);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <param name="tolerance">各成分毎の差の最大許容値です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Matrix22 other, float tolerance)
        {
            return MatrixFunction<Matrix22>.Equals(this, other, tolerance);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return MatrixFunction<Matrix22>.GetHashCode(this);
        }

        //-----------------------------------------------------------------
        // 文字列化
        //-----------------------------------------------------------------

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return MatrixFunction<Matrix22>.ToString(this);
        }

        private static Matrix22 InverseSolver(Matrix22 mtx)
        {
            return Matrix22Function<Matrix22>.Invert(mtx);
        }
    }
}
