﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデル更新ユーティリティ
    public static class IfModelUpdateUtility
    {
        /// <summary>
        /// 対象プラットフォームです。
        /// </summary>
        public enum TargetPlatform
        {
            Nx,
            Cafe,
        }

        // マテリアルの情報を更新する
        public static void UpdateModelInfo(
            modelType model,
            List<G3dStream> stream)
        {
            UpdateMaterial(model);
            UpdateBone(model);
            UpdateShape(model, stream);
        }

        // マテリアルの情報を更新する
        private static void UpdateMaterial(modelType model)
        {
            model_infoType modelInfo = model.model_info;
            if (model.material_array != null)
            {
                modelInfo.material_count = model.material_array.length;
            }
        }

        // ボーンの情報を更新する
        private static void UpdateBone(modelType model)
        {
            model_infoType modelInfo = model.model_info;

            int smoothSkinMtxCount = 0;
            int rigidSkinMtxCount = 0;
            if (model.skeleton.bone_array != null &&
                model.skeleton.bone_array.bone != null)
            {
                foreach (boneType bone in model.skeleton.bone_array.bone)
                {
                    // smooth skinning と rigid skinning で使用されるボーンを数える
                    if (bone.matrix_index[0] >= 0)
                    {
                        smoothSkinMtxCount++;
                    }
                    if (bone.matrix_index[1] >= 0)
                    {
                        rigidSkinMtxCount++;
                    }
                }
            }

            modelInfo.bone_count =
                (model.skeleton.bone_array != null) ? model.skeleton.bone_array.length : 0;
            modelInfo.smooth_skinning_matrix = smoothSkinMtxCount;
            modelInfo.rigid_skinning_matrix = rigidSkinMtxCount;
        }

        // シェイプの情報を更新する
        private static void UpdateShape(modelType model, List<G3dStream> stream)
        {
            model_infoType modelInfo = model.model_info;

            // shape_count
            modelInfo.shape_count =
                (model.shape_array != null) ? model.shape_array.length : 0;

            modelInfo.smooth_skinning_shape = 0;
            modelInfo.rigid_skinning_shape = 0;

            if (model.shape_array != null && model.shape_array.shape != null)
            {
                foreach (shapeType shape in model.shape_array.shape)
                {
                    // smooth_skinning_shape
                    // rigid_skinning_shape
                    int vtxSkinCount = shape.shape_info.vertex_skinning_count;
                    if (vtxSkinCount == 1)
                    {
                        modelInfo.rigid_skinning_shape++;
                    }
                    else if (vtxSkinCount >= 2)
                    {
                        modelInfo.smooth_skinning_shape++;
                    }
                }
            }
        }

        // シェイプの頂点数をカウントします
        public static int GetVertexCount(modelType model, shapeType shape)
        {
            if (model.vertex_array.vertex == null)
            {
                return 0;
            }

            vertexType vertex = model.vertex_array.vertex[shape.shape_info.vertex_index];
            if (vertex.vtx_attrib_array.vtx_attrib != null)
            {
                return vertex.vtx_attrib_array.vtx_attrib.Select(x => x.count).FirstOrDefault();
            }
            else
            {
                return 0;
            }
        }

        // シェイプのインデックス数をカウントします
        public static int GetIndexCount(shapeType shape)
        {
            var query =
                from mesh in shape.mesh_array.mesh
                from submesh in mesh.submesh_array.submesh
                select submesh.count;

            return query.Sum();
        }

        // シェイプのトライアングル数をカウントします
        public static int GetTriangleCount(shapeType shape)
        {
            var query =
                from mesh in shape.mesh_array.mesh
                from submesh in mesh.submesh_array.submesh
                select
                    (mesh.mode == mesh_modeType.triangles) ?
                        submesh.count / 3 :
                        submesh.count - 2;

            return query.Sum();
        }

        /// <summary>
        /// メッシュの処理頂点数をカウントします。
        /// </summary>
        /// <param name="mesh">対象メッシュです。</param>
        /// <param name="meshIndexStream">対象メッシュのインデックスストリームです。</param>
        /// <param name="targetPlatform">頂点処理数を計算するプラットフォームを指定します。</param>
        /// <returns></returns>
        public static int GetProcessVertexCount(meshType mesh, List<int> meshIndexStream, TargetPlatform targetPlatform)
        {
            switch (targetPlatform)
            {
                case TargetPlatform.Nx:
                    return GetProcessVertexCountForNx(mesh, meshIndexStream);
                case TargetPlatform.Cafe:
                    return GetProcessVertexCountForCafe(mesh, meshIndexStream);
                default:
                    throw new Exception($"Invalid TargetPlatform: {targetPlatform}");
            }
        }

        // シェイプの全メッシュ合算の処理頂点数をカウントします
        public static int GetProcessVertexCount(shapeType shape, List<G3dStream> stream, bool isModeNX)
        {
            int processVertexCount = 0;
            foreach (var mesh in shape.mesh_array.mesh)
            {
                processVertexCount += GetProcessVertexCount(mesh, stream[mesh.stream_index].IntData, isModeNX ? TargetPlatform.Nx : TargetPlatform.Cafe);
            }

            return processVertexCount;
        }

        private class VertexCache
        {
            public int ProcessCount { get; private set; }
            private const int CacheSize = 14;
            private LinkedList<int> cache = null;

            public void Reset()
            {
                cache = new LinkedList<int>();

                for (int i = 0; i != CacheSize; ++i)
                {
                    cache.AddLast(-1);
                }
            }

            public void Process(int index)
            {
                // キャッシュにある
                if (cache.Find(index) != null)
                {
                    // なにもしない
                }
                // キャッシュにない
                else
                {
                    // カウントする
                    ++ProcessCount;

                    // 末端に登録する
                    cache.AddFirst(index);

                    // キャッシュからあふれるものを削除する
                    cache.RemoveLast();
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(cache.Count == CacheSize);
            }
        }

        private static int GetProcessVertexCountForNx(meshType mesh, List<int> meshIndexStream)
        {
            int processVertexCount = 0;
            foreach (var submesh in mesh.submesh_array.submesh)
            {
                int offset = submesh.offset;
                int vertexCount = submesh.count;
                int[] subMeshIndex = new int[vertexCount];
                for (int index = 0; index < vertexCount; ++index)
                {
                    subMeshIndex[index] = meshIndexStream[offset + index];
                }
                processVertexCount += G3dProcessingVertexCounter.CountTrianglesNX(subMeshIndex);
            }

            return processVertexCount;
        }

        private static int GetProcessVertexCountForCafe(meshType mesh, List<int> meshIndexStream)
        {
            var vertexCache = new VertexCache();

            foreach (var submesh in mesh.submesh_array.submesh)
            {
                // サブメッシュ単位でキャッシュをリセットする
                vertexCache.Reset();
                int offset = submesh.offset;
                int vertexCount = submesh.count;

                switch (mesh.mode)
                {
                    case mesh_modeType.triangle_fan:
                        {
                            for (int i = 1; i < vertexCount - 1; ++i)
                            {
                                vertexCache.Process(meshIndexStream[offset]);
                                vertexCache.Process(meshIndexStream[i + offset]);
                                vertexCache.Process(meshIndexStream[i + offset + 1]);
                            }

                            break;
                        }

                    case mesh_modeType.triangle_strip:
                        {
                            for (int i = 0; i < vertexCount - 2; ++i)
                            {
                                vertexCache.Process(meshIndexStream[i + offset]);
                                vertexCache.Process(meshIndexStream[i + offset + 1]);
                                vertexCache.Process(meshIndexStream[i + offset + 2]);
                            }

                            break;
                        }

                    case mesh_modeType.triangles:
                        {
                            for (int i = 0; i < vertexCount; i += 3)
                            {
                                vertexCache.Process(meshIndexStream[i + offset]);
                                vertexCache.Process(meshIndexStream[i + offset + 1]);
                                vertexCache.Process(meshIndexStream[i + offset + 2]);
                            }

                            break;
                        }
                    default:
                        throw new Exception($"Invalid mesh_modeType: {mesh.mode}");
                }
            }

            return vertexCache.ProcessCount;
        }
    }
}
