﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // 中間ファイルソートユーティリティ
    public static class IfModelSortUtility
    {
        // ボーンがソートされているか
        // ソートされていれば -1 を返す
        // ソートされていなければ、ソートされていないボーンのインデックスを返す
        public static int IsBoneSorted(modelType model)
        {
            // ボーン名をキーにしたボーンテーブル
            // bone.parent_name からボーンを取得するのに使用する
            Dictionary<string, boneType> boneTable =
                new Dictionary<string, boneType>();
            // ボーンをキーにしたボーンの最後の子供のテーブル
            // 兄弟ボーンの並びが名前の文字コード順になっているのかを確認するのに使用する
            Dictionary<boneType, boneType> childTable =
                new Dictionary<boneType, boneType>();

            boneType[] bones = model.skeleton.bone_array.bone;
            for (int i = 0; i < bones.Length; i++)
            {
                boneType bone = bones[i];
                // 親ノードを検索するためのボーンテーブルに登録
                boneTable.Add(bone.name, bone);

                if (bone.parent_name == null || bone.parent_name.Length == 0)
                {
                    // ルートボーンはインデックス 0 でなければならない
                    if (i != 0) { return (i - 1); }
                }
                else
                {
                    boneType parent;
                    if (!boneTable.TryGetValue(bone.parent_name, out parent))
                    {
                        return (i - 1);
                    }

                    boneType sibling;
                    if (childTable.TryGetValue(parent, out sibling))
                    {
                        if (string.CompareOrdinal(sibling.name, bone.name) >= 0)
                        {
                            return (i - 1);
                        }

                        // 次の兄弟ボーンの並びを確認するためにテーブルを更新する
                        childTable[parent] = bone;
                    }
                    else
                    {
                        childTable.Add(parent, bone);
                    }
                }
            }

            return -1;
        }

        // マテリアルがソートされているか
        // ソートされていれば -1 を返す
        // ソートされていなければ、ソートされていないマテリアルのインデックスを返す
        public static int IsMaterialSorted(modelType model)
        {
            if (model.material_array == null ||
                model.material_array.material == null) { return -1; }

            materialType[] materials = model.material_array.material;
            for (int i = 1; i < materials.Length; i++)
            {
                if (string.CompareOrdinal(materials[i - 1].name, materials[i].name) >= 0)
                {
                    return (i - 1);
                }
            }
            return -1;
        }

        // シェイプがソートされているか
        // ソートされていれば -1 を返す
        // ソートされていなければ、ソートされていないシェイプのインデックスを返す
        public static int IsShapeSorted(modelType model)
        {
            if (model.shape_array == null)
            {
                return -1;
            }

            // ボーン名をキーにしたボーンの辞書を作成する。
            Dictionary<string, boneType> boneTable =
                new Dictionary<string, boneType>();
            foreach (boneType bone in model.skeleton.bone_array.bone)
            {
                boneTable.Add(bone.name, bone);
            }

            // shapeType は参照する bone の index 順でソートされている。
            // index が同じ場合は shape name の文字コード順でソートされている。
            shapeType[] shapes = model.shape_array.shape;
            for (int i = 1; i < shapes.Length; i++)
            {
                shapeType frontShape = shapes[i - 1];
                shapeType backShape = shapes[i];
                boneType frontBone = boneTable[frontShape.shape_info.bone_name];
                boneType backBone = boneTable[backShape.shape_info.bone_name];

                if (frontBone.index == backBone.index)
                {
                    if (string.CompareOrdinal(frontShape.name, backShape.name) >= 0)
                    {
                        return (i - 1);
                    }
                }
                else if (frontBone.index > backBone.index)
                {
                    return (i - 1);
                }
            }
            return -1;
        }
    }
}
