﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using nw.g3d.iflib.nw3de;
using nw.g3d.iflib.Resources;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public static class IfApplyBaseMaterialUtility
    {
        /// <summary>
        /// マテリアルが参照するシェーダー定義ファイル
        /// シェーダーアーカイブ名をキーにして検索
        /// </summary>
        private static readonly Dictionary<string, shader_definitionType> ShaderDefinitions = new Dictionary<string, shader_definitionType>();

        /// <summary>
        /// モデル
        /// パスをキーにして検索
        /// </summary>
        private static readonly Dictionary<string, NwifData> ModelCache = new Dictionary<string, NwifData>();
        private static string _shaderDefinitionFolder;
        private static string _xsdBasePath;

        public static void Initialize()
        {
            lock (ShaderDefinitions)
            {
                ShaderDefinitions.Clear();
            }
            lock (ModelCache)
            {
                ModelCache.Clear();
            }
        }

        private struct NwifData
        {
            public List<G3dStream> Streams;
            public nw4f_3difType Nwif;
        }

        private static NwifData GetNwifFromCache(string documentPath)
        {
            var fullPath = Path.GetFullPath(Environment.ExpandEnvironmentVariables(documentPath)).ToLower();
            if (!File.Exists(fullPath))
            {
                throw new Exception(string.Format(StringResource.AssignBaseMaterialUtility_Process_ErrorParentMaterialFileNotFound, fullPath));
            }
            if (!G3dPath.IsModelPath(fullPath) && !G3dPath.IsMaterialPath(fullPath))
            {
                throw new Exception(string.Format(StringResource.AssignBaseMaterialUtility_Process_ErrorParentMaterialFileIsNotModel, fullPath));
            }
            NwifData nwifData;
            lock (ModelCache)
            {
                if (!ModelCache.TryGetValue(fullPath, out nwifData))
                {
                    nwifData.Streams = new List<G3dStream>();
                    nwifData.Nwif = IfReadUtility.Read(nwifData.Streams, documentPath, _xsdBasePath);
                    var materialType = nwifData.Nwif.RootElement as materialType;
                    if (materialType != null)
                    {
                        // マテリアル中間ファイルではファイル名がマテリアル名。
                        // 中間ファイル内のマテリアル名とファイル名が一致していなければファイル名で上書きしておく。
                        var materialName = System.IO.Path.GetFileNameWithoutExtension(documentPath);
                        if (!string.Equals(materialType.name, materialName))
                        {
                            materialType.name = materialName;
                        }
                    }
                    ModelCache.Add(fullPath, nwifData);
                }
            }
            return nwifData;
        }

        private static IG3dRootElement GetNwifRootFromCache(string documentPath)
        {
            var nwifData = GetNwifFromCache(documentPath);
            return nwifData.Nwif?.RootElement;
        }

        public class ParentMaterialData
        {
            public string Fullpath { get; set; }
            public string MaterialName { get; set; }
            public modelType Model { get; set; }
            public materialType Material { get; set; }
            public nw3de_MaterialReference MaterialReference { get; set; }
            public ParentMaterialData Child { get; set; }
            public List<ParentMaterialData> Parents { get; set; } = new List<ParentMaterialData>();

            public T GetResolvedValue<T>(shading_modelType shadingModel, T shaderValue) where T : class
            {
                bool found;
                return GetResolvedValue(shadingModel, shaderValue, out found);
            }

            public T GetResolvedValue<T>(shading_modelType shadingModel, T shaderValue, out bool found) where T : class
            {
                MaterialReferenceBehaviorItem referenceBehaviorItem;
                var material = GetMaterial();
                string valueId;
                T value = null;
                if (shaderValue is attrib_assignType)
                {
                    valueId = (shaderValue as attrib_assignType).id;
                    value = material?.shader_assign?.attrib_assign_array?.attrib_assign?.FirstOrDefault(
                        x => x.id == valueId) as T;
                    referenceBehaviorItem = MaterialReference?.AttribAssignBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else if (shaderValue is render_infoType)
                {
                    valueId = (shaderValue as render_infoType).name;
                    value = material?.shader_assign?.render_info_array?.render_info?.FirstOrDefault(
                        x => x.name == valueId) as T;
                    referenceBehaviorItem = MaterialReference?.RenderInfoBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else if (shaderValue is shader_optionType)
                {
                    valueId = (shaderValue as shader_optionType).id;
                    value = material?.shader_assign?.shader_option_array?.shader_option?.FirstOrDefault(
                        x => x.id == valueId) as T;
                    referenceBehaviorItem = MaterialReference?.ShaderOptionBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else if (shaderValue is sampler_assignType)
                {
                    valueId = (shaderValue as sampler_assignType).id;
                    value = material?.shader_assign?.sampler_assign_array?.sampler_assign?.FirstOrDefault(
                        x => x.id == valueId) as T;
                    referenceBehaviorItem = MaterialReference?.SamplerAssignBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else if (shaderValue is shader_paramType)
                {
                    valueId = (shaderValue as shader_paramType).id;
                    value = material?.shader_assign?.shader_param_array?.shader_param?.FirstOrDefault(
                        x => x.id == valueId) as T;
                    referenceBehaviorItem = MaterialReference?.ShaderParamBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                    found = false;
                    return null;
                }

                ShaderItemValueState valueState;
                // 祖先の ChildRestriction によって valueState を変更
                var childRestrictionState = GetResolvedChildRestrictionState(shaderValue);
                switch (childRestrictionState)
                {
                    case ShaderItemRestrictionState.ForceRefer:
                        // 親マテリアルまたはその祖先のどれかがForceReferの場合はForceReferになっている親マテリアルを探し
                        // その値を採用する（その親マテリアル自身が持つ値ではなく、参照を解決した値を採用する）
                        Debug.Assert(Parents.Any(), "ForceRefer: ParentMaterial is not exist.");

                        // 後に指定されたベースマテリアルから検索
                        foreach (var parentMaterial in Enumerable.Reverse(Parents))
                        {
                            // この親マテリアルまたはその祖先がForceReferの場合
                            var resolvedRestriction = parentMaterial.GetResolvedChildRestrictionState(shaderValue, ignoreSelf: false);
                            if (resolvedRestriction == ShaderItemRestrictionState.ForceRefer)
                            {
                                // この親マテリアル自身がForceReferの場合、その値を採用する
                                var restriction = parentMaterial.GetChildRestrictionState(shaderValue);
                                if (restriction == ShaderItemRestrictionState.ForceRefer)
                                {
                                    var parentShaderParam = parentMaterial.GetResolvedValue(shadingModel, shaderValue, out found);
                                    found = true;
                                    return parentShaderParam;
                                }
                            }
                        }

                        valueState = ShaderItemValueState.Refer;
                        break;
                    case ShaderItemRestrictionState.ForceOverride:
                        valueState = ShaderItemValueState.Override;
                        break;
                    case ShaderItemRestrictionState.None:
                    default:
                        valueState = referenceBehaviorItem?.Value ?? ShaderItemValueState.Refer;
                        break;
                }

                switch (valueState)
                {
                    case ShaderItemValueState.Override:
                        found = true;
                        return value;
                    case ShaderItemValueState.Default:
                        found = true;
                        // デフォルト値を返す
                        return DefaultValue(shadingModel, shaderValue);
                    case ShaderItemValueState.Refer:
                    default:
                        T parentShaderParam = null;
                        if (Parents.Any())
                        {
                            // 後に指定されたベースマテリアルから検索
                            foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                            {
                                parentShaderParam = parentMaterialData.GetResolvedValue(shadingModel, shaderValue, out found);
                                if (found)
                                {
                                    return parentShaderParam;
                                }
                            }
                            // 値が確定しない場合(全てRefer) found は false で 最後の値を返す
                            found = false;
                            return parentShaderParam;
                        }
                        else
                        {
                            // 親マテリアルがない場合は found は false で 自分の値を返す
                            found = false;
                            return value;
                        }
                }
            }

            private object GetSamplerParamValue(string samplerName, string paramName)
            {
                var sampler = GetMaterial().sampler_array?.sampler?.FirstOrDefault(x => x.name == samplerName);
                if (sampler == null)
                {
                    return null;
                }
                switch (paramName)
                {
                    case "tex_name":
                        return sampler.tex_name;
                    case "wrap_u":
                        return sampler.wrap.u;
                    case "wrap_v":
                        return sampler.wrap.v;
                    case "wrap_w":
                        return sampler.wrap.w;
                    case "filter_mag":
                        return sampler.filter.mag;
                    case "filter_min":
                        return sampler.filter.min;
                    case "filter_mip":
                        return sampler.filter.mip;
                    case "filter_max_aniso":
                        return sampler.filter.max_aniso;
                    case "lod_min":
                        return sampler.lod.min;
                    case "lod_max":
                        return sampler.lod.max;
                    case "lod_bias":
                        return sampler.lod.bias;
                }
                return null;
            }

            public object GetResolvedSamplerParamValue(string samplerName, string paramName)
            {
                bool found;
                //Material foundMaterial;
                return GetResolvedSamplerParamValue(samplerName, paramName, out found/*, out foundMaterial*/);
            }
            public object GetResolvedSamplerParamValue(string samplerName, string paramName, out bool found/*, out Material foundMaterial*/)
            {
                //foundMaterial = this;

                var samplerParamBehavior = GetReferenceBehavioSamplerParam(samplerName, paramName);
                var childRestrictionState = GetResolvedSamplerParamChildRestrictionState(samplerName, paramName);

                ShaderItemValueState valueState;
                var hasParentSampler = childRestrictionState.HasValue;
                switch (childRestrictionState)
                {
                    case ShaderItemRestrictionState.ForceRefer:
                        // 親マテリアルまたはその祖先のどれかがForceReferの場合はForceReferになっている親マテリアルを探し
                        // その値を採用する（その親マテリアル自身が持つ値ではなく、参照を解決した値を採用する）
                        Debug.Assert(Parents.Any(), "ForceRefer: ParentMaterial is not exist.");

                        // 後に指定されたベースマテリアルから検索
                        foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                        {
                            // この親マテリアルまたはその祖先がForceReferの場合
                            var resolvedRestriction = parentMaterialData.GetResolvedSamplerParamChildRestrictionState(samplerName, paramName, ignoreSelf: false);
                            if (resolvedRestriction == ShaderItemRestrictionState.ForceRefer)
                            {
                                // この親マテリアル自身がForceReferの場合、その値を採用する
                                var restriction = parentMaterialData.GetSamplerParamChildRestrictionState(samplerName, paramName);
                                if (restriction == ShaderItemRestrictionState.ForceRefer)
                                {
                                    var parentShaderParam = parentMaterialData.GetResolvedSamplerParamValue(samplerName, paramName, out found);
                                    found = true;
                                    return parentShaderParam;
                                }
                            }
                        }
                        valueState = ShaderItemValueState.Refer;
                        break;
                    case ShaderItemRestrictionState.ForceOverride:
                        valueState = ShaderItemValueState.Override;
                        break;
                    case ShaderItemRestrictionState.None:
                    default:
                        valueState = samplerParamBehavior?.Value ?? ShaderItemValueState.Refer;
                        break;
                }

                var value = GetSamplerParamValue(samplerName, paramName);

                switch (valueState)
                {
                    case ShaderItemValueState.Override:
                        found = true;
                        return value;
                    case ShaderItemValueState.Default:
                        found = true;
                        return null;
                    case ShaderItemValueState.Refer:
                    default:
                        object parentShaderParam = null;
                        if (hasParentSampler && Parents.Any())
                        {
                            // 後に指定されたベースマテリアルから検索
                            foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                            {
                                parentShaderParam = parentMaterialData.GetResolvedSamplerParamValue(samplerName, paramName, out found);
                                if (found)
                                {
                                    return parentShaderParam;
                                }
                            }

                            // 値が確定しない場合(全てRefer) found は false で 最後の値を返す
                            found = false;
                            return parentShaderParam;
                        }
                        else
                        {
                            // 親マテリアルがない場合は found は false で 自分の値を返す
                            found = false;
                            return value;
                        }
                }
            }

            // 全てのパラメータを解決したサンプラー情報を返す
            public samplerType GetResolvedSampler(samplerType sampler)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.NotNull(sampler);
                var samplerName = sampler.name;

                if (!Parents.Any() || !IsReferenceSampler(samplerName))
                {
                    return sampler;
                }

                // デフォルト値を入れておく
                var newSampler = new samplerType
                {
                    name = samplerName,
                    hint = sampler?.hint ?? string.Empty,
                    tex_name = string.Empty,
                    wrap = new wrapType()
                    {
                        u = wrap_uvwType.clamp,
                        v = wrap_uvwType.clamp,
                        w = wrap_uvwType.clamp,
                    },
                    filter = new filterType()
                    {
                        mag = filter_mag_minType.linear,
                        min = filter_mag_minType.linear,
                        mip = filter_mipType.point,
                        max_aniso = filter_max_anisoType.aniso_1,
                    },
                    lod = new lodType()
                    {
                        min = 0,
                        max = 13,
                        bias = 0,
                    },
                };

                newSampler.tex_name = GetResolvedSamplerParamValue(samplerName, "tex_name") as string ?? newSampler.tex_name;
                newSampler.wrap.u = GetResolvedSamplerParamValue(samplerName, "wrap_u") as wrap_uvwType? ?? newSampler.wrap.u;
                newSampler.wrap.v = GetResolvedSamplerParamValue(samplerName, "wrap_v") as wrap_uvwType? ?? newSampler.wrap.v;
                newSampler.wrap.w = GetResolvedSamplerParamValue(samplerName, "wrap_w") as wrap_uvwType? ?? newSampler.wrap.w;
                newSampler.filter.mag = GetResolvedSamplerParamValue(samplerName, "filter_mag") as filter_mag_minType? ?? newSampler.filter.mag;
                newSampler.filter.min = GetResolvedSamplerParamValue(samplerName, "filter_min") as filter_mag_minType? ?? newSampler.filter.min;
                newSampler.filter.mip = GetResolvedSamplerParamValue(samplerName, "filter_mip") as filter_mipType? ?? newSampler.filter.mip;
                newSampler.filter.max_aniso = GetResolvedSamplerParamValue(samplerName, "filter_max_aniso") as filter_max_anisoType? ?? newSampler.filter.max_aniso;
                newSampler.lod.min = GetResolvedSamplerParamValue(samplerName, "lod_min") as float? ?? newSampler.lod.min;
                newSampler.lod.max = GetResolvedSamplerParamValue(samplerName, "lod_max") as float? ?? newSampler.lod.max;
                newSampler.lod.bias = GetResolvedSamplerParamValue(samplerName, "lod_bias") as float? ?? newSampler.lod.bias;

                return newSampler;
            }

            private materialType GetMaterial()
            {
                var material = (Model != null) ? Model.material_array?.material?.FirstOrDefault(x => x.name == MaterialName) : Material;
                Nintendo.Foundation.Contracts.Assertion.Operation.NotNull(material);
                return material;
            }

            public MaterialReferenceSamplerBehaviors GetReferenceBehaviorSamplers()
            {
                return MaterialReference?.SamplerBehaviors;
            }

            public SamplerBehaviorItem GetReferenceBehaviorSampler(string samplerName)
            {
                return GetReferenceBehaviorSamplers()?.SamplerBehaviorItems?.FirstOrDefault(x => x.Id == samplerName);
            }

            public MaterialReferenceBehaviorItem GetReferenceBehavioSamplerParam(string samplerName, string paramName)
            {
                return GetReferenceBehaviorSampler(samplerName)?.SamplerParamBehaviors?.FirstOrDefault(x => x.Id == paramName);
            }

            public ShaderItemRestrictionState? GetSamplerParamChildRestrictionState(string samplerName, string paramName)
            {
                if (Material.sampler_array?.sampler?.Any(x => x.name == samplerName) != true)
                {
                    return null;
                }
                return GetReferenceBehavioSamplerParam(samplerName, paramName)?.ChildRestriction ?? ShaderItemRestrictionState.None;
            }

            public ShaderItemRestrictionState? GetResolvedSamplerParamChildRestrictionState(string samplerName, string paramName, bool ignoreSelf = true)
            {
                ShaderItemRestrictionState? restriction = null;
                if (!ignoreSelf)
                {
                    restriction = GetSamplerParamChildRestrictionState(samplerName, paramName);
                }

                // 祖先を ChildRestriction == ForceRefer, ForceOverride が見つかるまで探す
                // 全てでChildRestriction == null が返された場合は 親サンプラが存在しない場合なので区別のためにnullを返す
                foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                {
                    var parentRestriction = parentMaterialData.GetResolvedSamplerParamChildRestrictionState(samplerName, paramName, false);
                    switch (parentRestriction)
                    {
                        case ShaderItemRestrictionState.None:
                            // Noneが帰ってきた場合は親サンプラが存在しているので返値に反映する
                            if (restriction == null)
                            {
                                restriction = parentRestriction;
                            }
                            break;
                        case ShaderItemRestrictionState.ForceRefer:
                        case ShaderItemRestrictionState.ForceOverride:
                            return parentRestriction;
                        default:
                            break;
                    }
                }

                return restriction;
            }

            private List<samplerType> GetParentRequiredForChildSamplers()
            {
                var samplers = new List<samplerType>();

                foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                {
                    var parentMaterial = parentMaterialData.GetMaterial();
                    samplers.AddRange(parentMaterialData.GetParentRequiredForChildSamplers().Where(x => samplers.All(y => x.name != y.name)));
                    if (parentMaterial.sampler_array != null)
                    {
                        var parentSamplers = parentMaterial.sampler_array.sampler;
                        foreach (var parentSampler in parentSamplers.Where(x => samplers.All(y => x.name != y.name)))
                        {
                            var samplerBehavior = parentMaterialData.GetReferenceBehaviorSampler(parentSampler.name);
                            if (samplerBehavior?.IsRequiredForChild == true)
                            {
                                if (samplers.All(x => x.name != parentSampler.name))
                                {
                                    samplers.Add(parentSampler);
                                }
                            }
                        }
                    }
                }

                return samplers;
            }

            // 親マテリアルから継承するサンプラが存在しない場合に追加する
            public samplerType[] UpdateSamplers()
            {
                var material = GetMaterial();
                var samplers = material?.sampler_array?.sampler ?? new samplerType[0];
                var newSamplers = new List<samplerType>(samplers);
                var addSamplers = GetParentRequiredForChildSamplers().Where(x => newSamplers.All(y => x.name != y.name)).ToList();

                newSamplers.AddRange(addSamplers);
                return newSamplers.ToArray();
            }

            public bool IsRequiredChildSampler(string samplerName)
            {
                foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                {
                    if (parentMaterialData.GetReferenceBehaviorSampler(samplerName)?.IsRequiredForChild == true || parentMaterialData.IsRequiredChildSampler(samplerName))
                    {
                        return true;
                    }
                }

                return false;
            }

            public bool IsReferenceSampler(string samplerName)
            {
                foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                {
                    if (parentMaterialData.GetMaterial().sampler_array?.sampler?.Any(x => x.name == samplerName) == true || parentMaterialData.IsReferenceSampler(samplerName))
                    {
                        return true;
                    }
                }

                return false;
            }

            private ShaderItemRestrictionState GetChildRestrictionState<T>(T shaderValue)
            {
                string valueId;
                MaterialReferenceBehaviorItem referenceBehaviorItem = null;

                if (shaderValue is attrib_assignType)
                {
                    valueId = (shaderValue as attrib_assignType).id;
                    referenceBehaviorItem = MaterialReference?.AttribAssignBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else if (shaderValue is render_infoType)
                {
                    valueId = (shaderValue as render_infoType).name;
                    referenceBehaviorItem = MaterialReference?.RenderInfoBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else if (shaderValue is shader_optionType)
                {
                    valueId = (shaderValue as shader_optionType).id;
                    referenceBehaviorItem = MaterialReference?.ShaderOptionBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else if (shaderValue is sampler_assignType)
                {
                    valueId = (shaderValue as sampler_assignType).id;
                    referenceBehaviorItem = MaterialReference?.SamplerAssignBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else if (shaderValue is shader_paramType)
                {
                    valueId = (shaderValue as shader_paramType).id;
                    referenceBehaviorItem = MaterialReference?.ShaderParamBehaviors?.FirstOrDefault(x => x.Id == valueId);
                }
                else
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                }

                return referenceBehaviorItem?.ChildRestriction ?? ShaderItemRestrictionState.None;
            }

            /// <summary>
            /// サンプラー構成の制限を持つ親マテリアルを取得します。
            /// </summary>
            /// <returns></returns>
            public ParentMaterialData FindChildSamplerStructureRestrictionMaterial()
            {
                if (Parents.Any())
                {
                    foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                    {
                        var structureRestrictionMaterial = parentMaterialData.FindChildSamplerStructureRestrictionMaterial();
                        if (structureRestrictionMaterial != null)
                        {
                            return structureRestrictionMaterial;
                        }

                        var parentRestriction = parentMaterialData.MaterialReference?.SamplerBehaviors?.ChildSamplerStructureRestriction;
                        if (parentRestriction != null && parentRestriction.Value != ChildSamplerStructureRestrictionState.None)
                        {
                            return parentMaterialData;
                        }
                    }
                }

                return null;
            }

            private ShaderItemRestrictionState GetResolvedChildRestrictionState<T>(T shaderValue, bool ignoreSelf = true) where T : class
            {
                var restriction = ShaderItemRestrictionState.None;

                if (!ignoreSelf)
                {
                    restriction = GetChildRestrictionState(shaderValue);
                }

                // 祖先を ChildRestriction != Noneが見つかるまで探す
                if (Parents.Any())
                {
                    foreach (var parentMaterialData in Enumerable.Reverse(Parents))
                    {
                        var parentRestriction = parentMaterialData.GetResolvedChildRestrictionState(shaderValue, false);
                        if (parentRestriction != ShaderItemRestrictionState.None)
                        {
                            return parentRestriction;
                        }
                    }
                }

                return restriction;
            }

            private T DefaultValue<T>(shading_modelType shadingModel, T shaderValue) where T : class
            {
                var material = GetMaterial();
                Nintendo.Foundation.Contracts.Assertion.Operation.NotNull(material);
                string valueId;
                if (shaderValue is attrib_assignType)
                {
                    valueId = (shaderValue as attrib_assignType).id;
                    var defaultAttrib = shadingModel.attrib_var_array?.attrib_var?.FirstOrDefault(x => x.id == valueId);
                    if (defaultAttrib == null)
                    {
                        throw new Exception(string.Format(StringResource.ApplyBaseMaterialUtility_Process_AttribAssignNotFound, MaterialName, shadingModel.name, valueId));
                    }
                    var attribAssign = new attrib_assignType
                    {
                        id = valueId,
                        attrib_name =
                            IfAttributeAssignUtility.GetAttribNameFromHint(this.Model, MaterialName, defaultAttrib) ??
                            string.Empty
                    };
                    return attribAssign as T;
                }
                else if (shaderValue is render_infoType)
                {
                    valueId = (shaderValue as render_infoType).name;
                    var defaultRenderInfo = shadingModel.render_info_slot_array?.render_info_slot?.FirstOrDefault(x => x.name == valueId);
                    if (defaultRenderInfo == null)
                    {
                        throw new Exception(string.Format(StringResource.ApplyBaseMaterialUtility_Process_RenderInfoNotFound, MaterialName, shadingModel.name, valueId));
                    }
                    var values = IfRenderInfoUtility.CreateRenderInfoValue(defaultRenderInfo, null);
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(values.All(x => !string.IsNullOrWhiteSpace(x)));
                    var arrayString = IfShaderAssignUtility.MakeArrayString(values);
                    var renderInfo = new render_infoType
                    {
                        name = valueId,
                        Value = arrayString != string.Empty ? arrayString : null,
                        type = defaultRenderInfo.Type(),
                        count = values.Count
                    };

                    return renderInfo as T;
                }
                else if (shaderValue is shader_optionType)
                {
                    valueId = (shaderValue as shader_optionType).id;
                    var defaultShaderOption = shadingModel.option_var_array?.option_var?.FirstOrDefault(x => x.id == valueId);
                    if (defaultShaderOption == null)
                    {
                        throw new Exception(string.Format(StringResource.ApplyBaseMaterialUtility_Process_ShaderOptionNotFound, MaterialName, shadingModel.name, valueId));
                    }
                    var shaderOption = new shader_optionType
                    {
                        id = valueId,
                        value = defaultShaderOption.@default
                    };
                    return shaderOption as T;
                }
                else if (shaderValue is sampler_assignType)
                {
                    valueId = (shaderValue as sampler_assignType).id;
                    var defaultSampler = shadingModel.sampler_var_array?.sampler_var?.FirstOrDefault(x => x.id == valueId);
                    if (defaultSampler == null)
                    {
                        throw new Exception(string.Format(StringResource.ApplyBaseMaterialUtility_Process_SamplerAssignNotFound, MaterialName, shadingModel.name, valueId));
                    }
                    var samplerAssign = new sampler_assignType
                    {
                        id = valueId,
                        sampler_name =
                            IfSamplerAssignUtility.GetSamplerNameFromHint(material?.sampler_array?.sampler,
                                defaultSampler) ?? string.Empty
                    };
                    return samplerAssign as T;
                }
                else if (shaderValue is shader_paramType)
                {
                    valueId = (shaderValue as shader_paramType).id;
                    var defaultShaderParam = shadingModel.MaterialUniforms().FirstOrDefault(x => x.id == valueId);
                    if (defaultShaderParam == null)
                    {
                        throw new Exception(string.Format(StringResource.ApplyBaseMaterialUtility_Process_ShaderParamNotFound, MaterialName, shadingModel.name, valueId));
                    }
                    var shaderParam = new shader_paramType
                    {
                        id = valueId,
                        type = defaultShaderParam.type,
                        Value = defaultShaderParam.Default(),
                        original_hint = string.Empty,
                        depend = defaultShaderParam.depend
                    };
                    return shaderParam as T;
                }
                else
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                }
                return null;
            }
        }

        public static nw4f_3difType ApplyBaseMaterial(string modelPath, string outputPath, string shaderPath, bool disableFileInfo, string xsdBasePath = null, string[] parentMaterialPaths = null)
        {
            _shaderDefinitionFolder = Environment.ExpandEnvironmentVariables(shaderPath);
            if (!Directory.Exists(_shaderDefinitionFolder))
            {
                throw new Exception(string.Format(StringResource.Error_Assign_ShaderDirectorNotFound, _shaderDefinitionFolder));
            }
            _xsdBasePath = xsdBasePath;

            Initialize();

            var nwifData = GetNwifFromCache(modelPath);

            materialType[] orgMaterials;
            switch (nwifData.Nwif.RootElement)
            {
                case modelType modelType:
                    orgMaterials = modelType.material_array?.material ?? new materialType[0];
                    break;
                case materialType materialType:
                    orgMaterials = new materialType[] { materialType };
                    break;
                default:
                    orgMaterials = null;
                    break;
            }
            Nintendo.Foundation.Contracts.Assertion.Operation.True(orgMaterials != null);

            // 編集前のモデルも必要
            var orgStreams = new List<G3dStream>();
            var orgNwif = IfReadUtility.Read(orgStreams, modelPath, _xsdBasePath);
            var orgModel = (modelType) orgNwif.Item;

            // マテリアル
            var materials = orgMaterials.ToList();

            // 親マテリアルのチェック
            if (materials.Select(IfToolData.GetMaterialReference).All(x => x == null || x.IsParentMaterialEmpty()))
            {
                // 適用しなくてもファイルは出力する
                if (!disableFileInfo)
                {
                    IfFileLogUtility.SetModify(nwifData.Nwif);
                }

                // 存在しないパラメータに対するマテリアル参照情報を整理する
                foreach (var material in materials)
                {
                    var shadingModel = GetShadingModel(material, material.shader_assign?.shader_archive, material.shader_assign?.shading_model);
                    var materialReference = IfToolData.GetMaterialReference(material);
                    if (materialReference != null)
                    {
                        IfApplyBaseMaterialUtility.UpdateMaterialReference(materialReference, shadingModel);
                        IfToolData.SetMaterialReference(material, materialReference);
                    }
                }

                IfWriteUtility.Write(nwifData.Nwif, nwifData.Streams, outputPath);
                throw new WarningException(string.Format(StringResource.Warning_ApplyBaseMaterialUtility_Process_NoBaseMaterial, modelPath));
            }

            // マテリアルを依存順にソートする(モデル内マテリアルの依存のみ)
            if (materials.Any())
            {
                Func<ParentMaterialData, List<ParentMaterialData>> collectBases = null;
                collectBases = data =>
                {
                    var bases = new List<ParentMaterialData>();
                    if (data.Parents.Any())
                    {
                        bases.AddRange(data.Parents);
                        bases.AddRange(data.Parents.SelectMany(x => collectBases(x)));
                    }
                    return bases;
                };

                var materialDatas = materials.Select(x => CollectBaseMaterialData(orgModel, x.name, modelPath, parentMaterialPaths)).ToList();

                materialDatas.Sort((lhsData, rhsData) =>
                {
                    var lhsBases = collectBases(lhsData).Where(x => string.Compare(modelPath, x.Fullpath, StringComparison.OrdinalIgnoreCase) == 0);
                    var rhsBases = collectBases(rhsData).Where(x => string.Compare(modelPath, x.Fullpath, StringComparison.OrdinalIgnoreCase) == 0);
                    if (lhsBases.Any(x => x.MaterialName == rhsData.MaterialName))
                    {
                        return 1;
                    }
                    else if (rhsBases.Any(x => x.MaterialName == lhsData.MaterialName))
                    {
                        return -1;
                    }
                    return 0;
                });

                materials = materialDatas.Select(x => orgMaterials.First(y => x.MaterialName == y.name)).ToList();
            }

            // マテリアルごとに処理
            foreach (var material in materials)
            {
                var materialName = material.name;
                if (string.IsNullOrEmpty(material.shader_assign?.shader_archive) || string.IsNullOrEmpty(material.shader_assign.shading_model))
                {
                    throw new Exception(string.Format(StringResource.AssignBaseMaterialUtility_Process_ErrorChildMaterialNotShaderAssigned,
                        materialName));
                }

                var materialReference = IfToolData.GetMaterialReference(material);
                if (materialReference == null)
                {
                    continue;
                }

                var shaderArchive = material.shader_assign.shader_archive;
                var shadingModelName = material.shader_assign.shading_model;

                // シェーダー定義とシェーディングモデルの取得
                shading_modelType shadingModel = GetShadingModel(material, shaderArchive, shadingModelName);

                var shaderAssign = material.shader_assign;
                var orgMaterial = orgModel.material_array.material.FirstOrDefault(x => x.name == materialName);
                Nintendo.Foundation.Contracts.Assertion.Operation.True(orgMaterial != null);

                // 存在しないパラメータに対するマテリアル参照情報を整理する
                IfApplyBaseMaterialUtility.UpdateMaterialReference(materialReference, shadingModel);
                IfToolData.SetMaterialReference(material, materialReference);

                if (materialReference.IsParentMaterialEmpty())
                {
                    continue;
                }

                // 再帰的に親マテリアル情報を集める
                var baseMaterialData = CollectBaseMaterialData(orgModel, materialName, modelPath, parentMaterialPaths);

                // MaterialReferenceBehavior の設定に従ってマテリアルの値を設定していく
                {
                    // サンプラー
                    var samplers = baseMaterialData.UpdateSamplers();

                    baseMaterialData.Material.sampler_array = samplers.ToG3dArrayElement<samplerType, sampler_arrayType>();
                    material.sampler_array = samplers.Select(sampler => baseMaterialData.GetResolvedSampler(sampler))
                        .ToG3dArrayElement<samplerType, sampler_arrayType>();
                }

                if (shadingModel.attrib_var_array?.attrib_var != null)
                {
                    var resolvedAttribAssigns = new List<attrib_assignType>();

                    foreach (var attribVar in shadingModel.attrib_var_array.attrib_var)
                    {
                        var attribAssign = shaderAssign.attrib_assign_array?.attrib_assign?.FirstOrDefault(x => x.id == attribVar.id) ??
                                           new attrib_assignType() {id = attribVar.id};
                        var baseAttribAssign = baseMaterialData.GetResolvedValue(shadingModel, attribAssign);
                        if (baseAttribAssign != null)
                        {
                            attribAssign.attrib_name = baseAttribAssign.attrib_name;
                            resolvedAttribAssigns.Add(attribAssign);
                        }
                        if (resolvedAttribAssigns.Any())
                        {
                            if (shaderAssign.attrib_assign_array == null)
                            {
                                shaderAssign.attrib_assign_array = new attrib_assign_arrayType() {attrib_assign = resolvedAttribAssigns.ToArray()};
                            }
                            else
                            {
                                shaderAssign.attrib_assign_array.attrib_assign = resolvedAttribAssigns.ToArray();
                            }
                        }
                        else
                        {
                            shaderAssign.attrib_assign_array = null;
                        }
                    }
                }

                if (shaderAssign.render_info_array?.render_info != null)
                {
                    foreach (var renderInfo in shaderAssign.render_info_array.render_info)
                    {
                        var baseRenderInfo = baseMaterialData.GetResolvedValue(shadingModel, renderInfo);
                        if (baseRenderInfo != null)
                        {
                            renderInfo.Value = baseRenderInfo.Value;
                            renderInfo.type = baseRenderInfo.type;
                            renderInfo.count = baseRenderInfo.count;
                        }
                    }
                }

                if (shadingModel.sampler_var_array?.sampler_var != null)
                {
                    var resolvedSamplerAssigns = new List<sampler_assignType>();
                    foreach (var samplerVar in shadingModel.sampler_var_array?.sampler_var)
                    {
                        var samplerAssign =
                            shaderAssign.sampler_assign_array?.sampler_assign?.FirstOrDefault(x => x.id == samplerVar.id) ??
                            new sampler_assignType() {id = samplerVar.id};
                        var baseSamplerAssign = baseMaterialData.GetResolvedValue(shadingModel, samplerAssign);
                        if (!string.IsNullOrEmpty(baseSamplerAssign?.sampler_name))
                        {
                            samplerAssign.sampler_name = baseSamplerAssign.sampler_name;
                            resolvedSamplerAssigns.Add(samplerAssign);
                        }
                    }
                    material.shader_assign.sampler_assign_array =
                        resolvedSamplerAssigns.ToG3dArrayElement<sampler_assignType, sampler_assign_arrayType>();
                }

                if (shaderAssign.shader_option_array?.shader_option != null)
                {
                    foreach (var shaderOption in shaderAssign.shader_option_array.shader_option)
                    {
                        var baseShaderOption = baseMaterialData.GetResolvedValue(shadingModel, shaderOption);
                        if (baseShaderOption != null)
                        {
                            shaderOption.value = baseShaderOption.value;
                        }
                    }
                }

                if (shaderAssign.shader_param_array?.shader_param != null)
                {
                    foreach (var shaderParam in shaderAssign.shader_param_array.shader_param)
                    {
                        var baseShaderParam = baseMaterialData.GetResolvedValue(shadingModel, shaderParam);
                        if (baseShaderParam != null)
                        {
                            shaderParam.type = baseShaderParam.type;
                            shaderParam.Value = baseShaderParam.Value;
                            shaderParam.original_hint = baseShaderParam.original_hint;
                            shaderParam.depend = baseShaderParam.depend;
                        }
                    }
                }

                // サンプラー構成の制限の適用
                ApplySamplerStructureRestriction(material, baseMaterialData);

                // 念のためサンプラー・サンプラー変数が空ならばnullにする
                if (material.sampler_array?.sampler?.Any() != true)
                {
                    material.sampler_array = null;
                }
                if (shaderAssign.sampler_assign_array?.sampler_assign?.Any() != true)
                {
                    shaderAssign.sampler_assign_array = null;
                }
            }

            // 中間ファイル出力
            if (!disableFileInfo)
            {
                IfFileLogUtility.SetModify(nwifData.Nwif);
            }
            IfWriteUtility.Write(nwifData.Nwif, nwifData.Streams, outputPath);
            return nwifData.Nwif;
        }

        public static void ApplySamplerStructureRestriction(materialType material, ParentMaterialData baseMaterialData)
        {
            if (material.sampler_array == null) return;

            var restrictionMaterial = baseMaterialData.FindChildSamplerStructureRestrictionMaterial();
            if (restrictionMaterial != null &&
                restrictionMaterial.MaterialReference?.SamplerBehaviors?.ChildSamplerStructureRestriction ==
                ChildSamplerStructureRestrictionState.DisallowAddOwnSampler)
            {
                // 構成の制限にないサンプラーを削除
                var restrictedSamplers = new List<samplerType>();
                var removedSamplers = new List<samplerType>();
                foreach (var sampler in material.sampler_array.sampler)
                {
                    if (restrictionMaterial.Material.sampler_array?.sampler?.Any(x => x.name == sampler.name) == true)
                    {
                        restrictedSamplers.Add(sampler);
                    }
                    else
                    {
                        removedSamplers.Add(sampler);
                    }
                }
                material.sampler_array = restrictedSamplers.ToG3dArrayElement<samplerType, sampler_arrayType>();

                // 削除されたサンプラーを持つサンプラー割り当てを削除
                if (material.shader_assign?.sampler_assign_array?.sampler_assign != null)
                {
                    var samplerAssigns = new List<sampler_assignType>();
                    foreach (var samplerAssign in material.shader_assign.sampler_assign_array.sampler_assign)
                    {
                        if (removedSamplers.All(x => x.name != samplerAssign.sampler_name))
                        {
                            samplerAssigns.Add(samplerAssign);
                        }
                    }
                    material.shader_assign.sampler_assign_array = samplerAssigns.ToG3dArrayElement<sampler_assignType, sampler_assign_arrayType>();
                }
            }
        }

        private static shading_modelType GetShadingModel(materialType material, string shaderArchive, string shadingModelName)
        {
            shader_definitionType shaderDefinition;
            var shadingModel = GetShadingModel(material, out shaderDefinition);

            // 見つからなかったときは中断
            if (shaderDefinition == null)
            {
                throw new Exception(string.Format(StringResource.ShaderDefinitionNotFound, material.name, shaderArchive));
            }
            else if (shadingModel == null)
            {
                throw new Exception(string.Format(StringResource.ShadingModelNotFound, material.name, shadingModelName));
            }
            else if (!shadingModel.material_shader)
            {
                // マテリアル用でなかった場合は中断
                throw new Exception(string.Format(StringResource.ShadingModelNotForMaterial, material.name, shadingModelName));
            }
            return shadingModel;
        }

        //private static BaseMaterialData CollectBaseMaterialData(modelType model, string materialName, string modelPath, string[] parentMaterialPaths, BaseMaterialData child = null)
        public static ParentMaterialData CollectBaseMaterialData(modelType model, string materialName, string modelPath, string[] parentMaterialPaths, ParentMaterialData child = null)
        {
            return CollectBaseMaterialData((IG3dRootElement)model, materialName, modelPath, parentMaterialPaths, child);
        }
        private static ParentMaterialData CollectBaseMaterialData(IG3dRootElement document, string materialName, string documentPath, string[] parentMaterialPaths, ParentMaterialData child = null)
        {
            materialType material;
            switch (document)
            {
                case modelType modelType:
                    material = modelType.material_array.material.FirstOrDefault(x => x.name == materialName);
                    break;
                case materialType materialType:
                    material = materialType;
                    break;
                default:
                    material = null;
                    break;
            }
            Nintendo.Foundation.Contracts.Assertion.Argument.NotNull(material);

            var target = new ParentMaterialData
            {
                Model = document as modelType,
                Material = material,
                MaterialName = materialName,
                Fullpath = documentPath,
                Child = child,
                MaterialReference = IfToolData.GetMaterialReference(material),
            };

            if (target.MaterialReference == null || target.MaterialReference.IsParentMaterialEmpty())
            {
                return target;
            }

            foreach (var baseMaterial in target.MaterialReference.ParentMaterials)
            {
                IG3dRootElement baseDocument;
                string baseDocumentFullPath;

                if (!string.IsNullOrEmpty(baseMaterial.Path))
                {
                    // 親ドキュメントファイル指定時はファイルをチェック
                    var baseDocumentPath = baseMaterial.Path;
                    baseDocumentFullPath = GetParentDocumentFullPath(baseDocumentPath, documentPath, parentMaterialPaths);
                    if (!File.Exists(baseDocumentFullPath))
                    {
                        throw new Exception(string.Format(StringResource.IfApplyBaseMaterialUtility_CollectBaseMaterialData_ErrorParentMaterialFileNotFound, baseDocumentFullPath ?? baseDocumentPath));
                    }

                    baseDocument = GetNwifRootFromCache(baseDocumentFullPath);
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(baseDocument != null);
                }
                else
                {
                    baseDocumentFullPath = documentPath;
                    baseDocument = document;
                }

                // 循環参照チェック
                {
                    var root = target;
                    while (root.Child != null)
                    {
                        root = root.Child;
                    }
                    Action<ParentMaterialData, List<ParentMaterialData>> action = null;
                    action = (data, list) =>
                    {
                        list.Add(data);
                        data.Parents.ForEach(x => action(x, list));
                    };
                    var datas = new List<ParentMaterialData>();
                    action(root, datas);

                    if (datas.Any(x => x.Fullpath == baseDocumentFullPath && x.MaterialName == baseMaterial.MaterialName))
                    {
                        throw new Exception(string.Format(StringResource.ApplyBaseMaterialUtility_CollectParentMaterialData_ErrorCircularReference, baseDocumentFullPath, baseMaterial.MaterialName));
                    }
                }

                var baseData = CollectBaseMaterialData(baseDocument, baseMaterial.MaterialName, baseDocumentFullPath, parentMaterialPaths, target);
                target.Parents.Add(baseData);
            }
            return target;
        }

        /// <summary>
        /// GetParentDocumentFullPath() に移行。
        /// 外部アプリから参照されている可能性があるため残しておく。
        /// </summary>
        public static string GetParentModelFullPath(string parentModelPath, string modelPath, string[] parentMaterialPaths = null)
        {
            return GetParentDocumentFullPath(parentModelPath, modelPath, parentMaterialPaths);
        }
        public static string GetParentDocumentFullPath(string parentDocumentPath, string documentPath, string[] parentMaterialPaths = null)
        {
            parentDocumentPath = Environment.ExpandEnvironmentVariables(parentDocumentPath);

            // 親マテリアルドキュメントのパスが絶対パスの場合は正規化だけして返す
            if (Path.IsPathRooted(parentDocumentPath))
            {
                return Path.GetFullPath(parentDocumentPath);
            }
            var baseDocumentFullPath = Path.GetFullPath(Path.Combine(Path.GetDirectoryName(documentPath), parentDocumentPath));

            // 子マテリアルモデルからの相対でファイルが見つかった場合はそのパスを返す
            if (File.Exists(baseDocumentFullPath))
            {
                return baseDocumentFullPath;
            }

            // 親マテリアル参照パスから再帰的に検索し、ファイルが見つかったらそれを返す
            if (parentMaterialPaths == null)
            {
                return null;
            }

            foreach (var parentMaterialPath in parentMaterialPaths)
            {
                if (!Directory.Exists(parentMaterialPath)) continue;
                var file = Directory.EnumerateFiles(parentMaterialPath, parentDocumentPath, SearchOption.AllDirectories).FirstOrDefault();
                if (!string.IsNullOrEmpty(file))
                {
                    return file;
                }
            }

            return null;
        }

        // シェーダー定義を取得する
        private static shading_modelType GetShadingModel(materialType material, out shader_definitionType shaderDefinition)
        {
            shaderDefinition = GetShaderDefinition(material.shader_assign.shader_archive);

            return shaderDefinition?.shading_model_array?.shading_model.FirstOrDefault(x => x.name == material.shader_assign.shading_model);
        }

        private static shader_definitionType GetShaderDefinition(string shaderArchive)
        {
            shader_definitionType shaderDefinition;
            lock (ShaderDefinitions)
            {
                if (ShaderDefinitions.ContainsKey(shaderArchive))
                {
                    // 既に探していれば再利用
                    shaderDefinition = ShaderDefinitions[shaderArchive];
                }
                else
                {
                    // シェーダー定義を探す
                    shaderDefinition = SearchShaderDefinition(_shaderDefinitionFolder, shaderArchive);
                    ShaderDefinitions.Add(shaderArchive, shaderDefinition);
                }
            }
            return shaderDefinition;
        }

        /// <summary>
        /// フォルダ以下から再帰的にシェーダー定義を探す
        /// </summary>
        private static shader_definitionType SearchShaderDefinition(string folder, string shaderArchive)
        {
            // MEMO: fsda は現在非対応がだ処理する。
            var fsds = "ba".Select(x => Path.Combine(folder, shaderArchive + ".fsd" + x));
            foreach (var fsd in fsds)
            {
                if (File.Exists(fsd))
                {
                    var streams = new List<G3dStream>();
                    var nwif = IfReadUtility.Read(streams, fsd, _xsdBasePath);
                    return (shader_definitionType)nwif.Item;
                }
            }

            foreach (var subFolder in Directory.GetDirectories(folder))
            {
                var shaderDefinition = SearchShaderDefinition(subFolder, shaderArchive);
                if (shaderDefinition != null)
                {
                    return shaderDefinition;
                }
            }

            return null;
        }

        public static IfShaderAssignUtility.CompareResult Compare(nw3de_MaterialReference materialReference, shading_modelType shadingModel)
        {
            if (materialReference == null || shadingModel == null)
            {
                return IfShaderAssignUtility.CompareResult.Ok;
            }

            if (materialReference.AttribAssignBehaviors?.Any() == true)
            {
                if (materialReference.AttribAssignBehaviors.Any(behaviorItem =>
                    shadingModel.attrib_var_array.attrib_var.All(x => x.id != behaviorItem.Id)))
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            if (materialReference.SamplerAssignBehaviors?.Any() == true)
            {
                if (materialReference.SamplerAssignBehaviors.Any(behaviorItem =>
                    shadingModel.sampler_var_array.sampler_var.All(x => x.id != behaviorItem.Id)))
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            if (materialReference.ShaderOptionBehaviors?.Any() == true)
            {
                if (materialReference.ShaderOptionBehaviors.Any(behaviorItem =>
                    shadingModel.option_var_array.option_var.All(x => x.id != behaviorItem.Id)))
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            if (materialReference.ShaderParamBehaviors?.Any() == true)
            {
                var uniforms = shadingModel.block_var_array?.block_var?.Where(x => x.type == block_var_typeType.material && x.uniform_var_array != null)
                    .SelectMany(x => x.uniform_var_array.uniform_var).ToArray();

                if (uniforms != null && materialReference.ShaderParamBehaviors.Any(behaviorItem =>
                    uniforms.All(x => x.id != behaviorItem.Id)))
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            if (materialReference.RenderInfoBehaviors?.Any() == true)
            {
                if (materialReference.RenderInfoBehaviors.Any(behaviorItem =>
                    shadingModel.render_info_slot_array.render_info_slot.All(x => x.name != behaviorItem.Id)))
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            return IfShaderAssignUtility.CompareResult.Ok;
        }

        public static void UpdateMaterialReference(nw3de_MaterialReference materialReference, shading_modelType shadingModel)
        {
            if (materialReference == null || shadingModel == null)
            {
                return;
            }
            if (materialReference.AttribAssignBehaviors?.Any() == true)
            {
                materialReference.AttribAssignBehaviors = materialReference.AttribAssignBehaviors
                    .Where(behaviorItem => shadingModel.attrib_var_array.attrib_var.Any(x => x.id == behaviorItem.Id)).ToList();
            }

            if (materialReference.SamplerAssignBehaviors?.Any() == true)
            {
                materialReference.SamplerAssignBehaviors = materialReference.SamplerAssignBehaviors.Where(behaviorItem =>
                    shadingModel.sampler_var_array.sampler_var.Any(x => x.id == behaviorItem.Id)).ToList();
            }

            if (materialReference.ShaderOptionBehaviors?.Any() == true)
            {
                materialReference.ShaderOptionBehaviors = materialReference.ShaderOptionBehaviors.Where(behaviorItem =>
                    shadingModel.option_var_array.option_var.Any(x => x.id == behaviorItem.Id)).ToList();
            }

            if (materialReference.ShaderParamBehaviors?.Any() == true)
            {
                var uniforms = shadingModel.block_var_array?.block_var?.Where(x => x.type == block_var_typeType.material && x.uniform_var_array != null)
                    .SelectMany(x => x.uniform_var_array.uniform_var).ToArray();
                if (uniforms != null)
                {
                    materialReference.ShaderParamBehaviors = materialReference.ShaderParamBehaviors.Where(behaviorItem =>
                        uniforms.Any(x => x.id == behaviorItem.Id)).ToList();
                }
            }

            if (materialReference.RenderInfoBehaviors?.Any() == true)
            {
                materialReference.RenderInfoBehaviors = materialReference.RenderInfoBehaviors.Where(behaviorItem =>
                    shadingModel.render_info_slot_array.render_info_slot.Any(x => x.name == behaviorItem.Id)).ToList();
            }
        }
    }
}
