﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public class SkeletalAnimCompressUtility
    {
        private static readonly float Tolerance = 1.0e-5f;

        /***
         * BoneAnimInfo をソートするためのデータ
         */
        private class BoneAnimInfoNode : IComparable
        {
            public BoneAnimInfo Anim { get; set; }
            public List<BoneAnimInfoNode> Children { get; set; } = new List<BoneAnimInfoNode>();

            public BoneAnimInfoNode(BoneAnimInfo animInfo)
            {
                Anim = animInfo;
            }

            public int CompareTo(object obj)
            {
                BoneAnimInfoNode other = (BoneAnimInfoNode)obj;
                return string.CompareOrdinal(Anim.BoneAnim.bone_name, other.Anim.BoneAnim.bone_name);
            }
            // 参照している BoneAnimInfo に新しいインデックスを設定します。
            public int SetNewIndex(int index)
            {
                // インデックスは深度優先で付ける。
                // 兄弟ボーン間ではアルファベット順に付けていく。
                // BoneAnimInfoNode.Child はこの時点でソート済み。
                Anim.NewIndex = index++;
                foreach (BoneAnimInfoNode child in Children)
                {
                    index = child.SetNewIndex(index);
                }
                return index;
            }

            // 削除されない BoneAnimInfo の新しい階層構造を取得します。
            // BoneAnimInfoNode の Children はボーン名のアルファベット順でソート済みです。
            public static BoneAnimInfoNode GetHierarchy(BoneAnimInfo rootBone)
            {
                if (rootBone == null)
                {
                    return null;
                }

                // rootBone には削除されないボーンが指定されていなければならない。
                Nintendo.Foundation.Contracts.Assertion.Operation.True(!rootBone.IsReservedToRemove);

                BoneAnimInfoNode boneNode = new BoneAnimInfoNode(rootBone);
                foreach (BoneAnimInfo child in rootBone.Children)
                {
                    GetHierarchySub(child, boneNode);
                }
                // 子供を列挙した後にアルファベット順にソートする
                boneNode.Children.Sort();
                return boneNode;
            }
            // 階層構造作成のサブルーチンです。
            // 子階層に向かって再帰的に呼び出されていきます。
            private static void GetHierarchySub(BoneAnimInfo bone, BoneAnimInfoNode parentNode)
            {
                BoneAnimInfoNode curNode;
                if (!bone.IsReservedToRemove)
                {
                    curNode = new BoneAnimInfoNode(bone);
                    parentNode.Children.Add(curNode);
                }
                else
                {
                    // 削除されるボーンの場合は新たに BoneAnimInfoNode を作成せずに
                    // 親のオブジェクトをそのまま使用する。
                    // そうすることによってこのボーンを削除された後の階層構造を再現できる。
                    curNode = parentNode;
                }

                foreach (BoneAnimInfo child in bone.Children)
                {
                    GetHierarchySub(child, curNode);
                }

                // 子供を列挙した後にアルファベット順にソートする
                if (curNode != parentNode)
                {
                    curNode.Children.Sort();
                }
            }
        }

        /***
         * 新しい BoneAnimInfo のインデックスを設定します。
         * rootBone を基点にして下位階層の削除されない BoneAnimInfo に新しいインデックスを設定します。
         * そのため、BoneAnimInfo.RemoveFlag がすでに設定されている必要があります。
         * ここで設定されるインデックスはソート済みの値です。
         * 中間ファイルに出力する際にはこのインデックスの順番通りにソートする必要があります。
         */
        public static void SetNewIndex(BoneAnimInfo rootBone)
        {
            // まずは削除されないボーンで出力用の階層構造を作成する
            BoneAnimInfoNode rootNode = BoneAnimInfoNode.GetHierarchy(rootBone);
            if (rootNode != null)
            {
                // その階層構造の順番に従って 0 から順にインデックスを割り振る
                rootNode.SetNewIndex(0);
            }
        }
        /***
         * ボーンのインデックスを比較します。
         * ボーン配列内を出力順にソートする時に使用します。
         */
        private static int CompareBoneIndex(bone_animType lhs, bone_animType rhs)
        {
            if (lhs.index == rhs.index)
            {
                return 0;
            }
            else if (lhs.index < rhs.index)
            {
                return -1;
            }
            else
            {
                return 1;
            }
        }
        /***
         * スケルタルアニメーションから不要なアニメーションを削除します。
         * BoneAnimInfo.RemoveFlag が true になっているアニメーションをスケルタルアニメーションの
         * アニメーション配列から削除します。
         * その時にアニメーションのインデックスとペアレントの情報も一緒に更新します。
         * この関数を実行した後は BoneAnimInfo と参照する bone_animType の階層構造の情報は一致しなくなります。
         */
        public static void RemoveBoneAnims(skeletal_animType anim, BoneAnimInfo[] animInfos)
        {
            // 新しいボーン配列の作成
            List<bone_animType> newBones = new List<bone_animType>();
            foreach (BoneAnimInfo boneInfo in animInfos)
            {
                if (!boneInfo.IsReservedToRemove)
                {
                    bone_animType bone = boneInfo.BoneAnim;

                    // 新しい配列内のインデックスを設定する
                    bone.index = boneInfo.NewIndex;
                    // 親が削除される場合は親の名前を変更する。
                    if (boneInfo.Parent != null && boneInfo.Parent.IsReservedToRemove)
                    {
                        BoneAnimInfo newParent = boneInfo.GetIncompressibleAncestorBone();
                        if (newParent != null)
                        {
                            bone.parent_name = newParent.BoneAnim.bone_name;
                        }
                        else
                        {
                            bone.parent_name = string.Empty;
                        }
                    }
                    newBones.Add(bone);
                }
            }
            // ボーンを出力順にソートする
            newBones.Sort(CompareBoneIndex);
            // 新しいボーン配列への差し替え
            anim.bone_anim_array.length = newBones.Count;
            anim.bone_anim_array.bone_anim = newBones.ToArray();
        }

        /**
         * 間引き後のノード間の参照関係を求めます。
         */
        public static void GetThinOutBoneRef(
            BoneAnimInfo[] boneInfos,
            out BoneAnimInfo[] parent)
        {
            int boneSize = boneInfos.Length;
            parent = new BoneAnimInfo[boneSize];
            for (int i = 0; i < boneSize; ++i)
            {
                if (boneInfos[i].IsReservedToRemove)
                {
                    continue;
                }
                //新しい親ノードIDを求める
                BoneAnimInfo bone = boneInfos[i].Parent;
                int count = 0;
                for (; bone != null && count < boneSize; ++count)
                {
                    if (!bone.IsReservedToRemove)
                    {
                        parent[i] = bone;
                        break;
                    }
                    else
                    {
                        bone = bone.Parent;
                    }
                }
                if (count == boneSize)
                {
                    throw new Exception("<bone> has cyclic indexing.");
                }
            }
        }

        /**
         * 子供のボーンリストを得ます。
         */
        public static void GetBoneDescendants(
            BoneAnimInfo boneInfo,
            List<BoneAnimInfo> dst)
        {
            foreach (BoneAnimInfo child in boneInfo.Children)
            {
                dst.Add(child);
                GetBoneDescendants(child, dst);
            }
        }

        /**
         * ancBoneで指す<bone>がboneで指す<bone>の先祖階層にあるか調べる。
         */
        private static bool IsAncestorBone(BoneAnimInfo bone, BoneAnimInfo ancBone)
        {
            BoneAnimInfo bi = bone;
            while (bi != null)
            {
                BoneAnimInfo parent = bi.Parent;
                if (parent == null)
                {
                    return false;
                }
                else if (parent == ancBone)
                {
                    return true;
                }
                bi = parent;
            }
            return false;
        }

        // fromBoneAnimInfo から toBoneAnimInfo までたどっていったボーンのリストを取得します。
        // 返されるリストは、先頭に fromBoneAnimInfo の親ボーンが設定されます。
        // リストには toBoneAnimInfo は含まれません。
        private static void GetBoneAncestors(
            BoneAnimInfo fromBoneAnimInfo,
            BoneAnimInfo toBoneAnimInfo,
            List<BoneAnimInfo> list)
        {
            BoneAnimInfo bone = fromBoneAnimInfo;
            while (bone != null)
            {
                // リストに toBoneAnimInfo を含めないようにするために、
                // 親が toBoneAnimInfo であれば探索を打ち切る。
                if (bone.Parent == null || bone.Parent == toBoneAnimInfo)
                {
                    break;
                }
                list.Add(bone.Parent);
                bone = bone.Parent;
            }
        }

        private static float CalcLerp(
            float frame0, float value0,
            float frame1, float value1,
            float frame)
        {
            if (frame0 == frame1)
            {
                return value0;
            }
            float t = (frame - frame0) / (frame1 - frame0);
            return (value0 * (1.0f - t)) + (value1 * t);
        }

        private static float CalcHermite(
            float frame0, float value0, float slope0,
            float frame1, float value1, float slope1,
            float frame)
        {
            float t1 = frame - frame0;
            float t2 = 1.0F / (frame1 - frame0);

            float t1t1t2 = t1 * t1 * t2;
            float t1t1t2t2 = t1t1t2 * t2;
            float t1t1t1t2t2 = t1 * t1t1t2t2;
            float t1t1t1t2t2t2 = t1t1t1t2t2 * t2;
            return
                value0 * (2.0F * t1t1t1t2t2t2 - 3.0F * t1t1t2t2 + 1.0F) +
                value1 * (-2.0F * t1t1t1t2t2t2 + 3.0F * t1t1t2t2) +
                slope0 * (t1t1t1t2t2 - 2.0F * t1t1t2 + t1) +
                slope1 * (t1t1t1t2t2 - t1t1t2);
        }

        private class Vec2Stream
        {
            private static readonly int Column = 2;
            private readonly List<float> FloatData;
            public Vec2Stream(G3dStream stream)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(stream.column == Column);
                FloatData = stream.FloatData;
            }
            public Vector2 this[int i]
            {
                get
                {
                    if ((i * Column) >= FloatData.Count)
                    {
                        throw new ArgumentOutOfRangeException();
                    }
                    int idx = i * Column;
                    return new Vector2(
                        FloatData[idx + 0],
                        FloatData[idx + 1]);
                }

                set
                {
                    if ((i * Column) >= FloatData.Count)
                    {
                        throw new ArgumentOutOfRangeException();
                    }
                    int idx = i * Column;
                    FloatData[idx + 0] = value.X;
                    FloatData[idx + 1] = value.Y;
                }
            }
            public int Length
            {
                get
                {
                    return FloatData.Count / Column;
                }
            }
        }

        private class Vec4Stream
        {
            private static readonly int Column = 4;
            private readonly List<float> FloatData;
            public Vec4Stream(G3dStream stream)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(stream.column == Column);
                FloatData = stream.FloatData;
            }
            public Vector4 this[int i]
            {
                get
                {
                    if ((i * Column) >= FloatData.Count)
                    {
                        throw new ArgumentOutOfRangeException();
                    }
                    int idx = i * Column;
                    return new Vector4(
                        FloatData[idx + 0],
                        FloatData[idx + 1],
                        FloatData[idx + 2],
                        FloatData[idx + 3]);
                }

                set
                {
                    if ((i * Column) >= FloatData.Count)
                    {
                        throw new ArgumentOutOfRangeException();
                    }
                    int idx = i * Column;
                    FloatData[idx + 0] = value.X;
                    FloatData[idx + 1] = value.Y;
                    FloatData[idx + 2] = value.Z;
                    FloatData[idx + 3] = value.W;
                }
            }
            public int Length
            {
                get
                {
                    return FloatData.Count / Column;
                }
            }
        }

        private static float GetValue_Step(
            step_curveType anim,
            List<G3dStream> streams,
            float frame)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(streams.Count > anim.stream_index && anim.stream_index >= 0);
            Vec2Stream keyData = new Vec2Stream(streams[anim.stream_index]);

            int keySize = keyData.Length;
            if (keySize == 0)
            {
                return 0;
            }

            var firstKey = keyData[0];
            var lastKey = keyData[keySize - 1];

            if (keySize == 1)
            {
                return firstKey[1];
            }
            int shiftCount;
            frame = CalcShiftFrame(
                frame,
                keyData[0][0],
                keyData[keySize - 1][0],
                anim.pre_wrap,
                anim.post_wrap,
                out shiftCount);

            if (frame <= firstKey[0])
            {
                return firstKey[1];
            }

            if (frame >= lastKey[0])
            {
                return lastKey[1];
            }

            int iKeyL = 0;
            int iKeyR = keySize - 1;
            Vector2 key;
            while (iKeyL + 1 < iKeyR)
            {
                int iKeyCenter = (iKeyL + iKeyR) / 2;
                key = keyData[iKeyCenter];
                if (frame < key[0])
                {
                    iKeyR = iKeyCenter;
                }
                else
                {
                    iKeyL = iKeyCenter;
                }
            }
            key = keyData[iKeyR];
            float value;
            if (FloatUtility.NearlyEqual(frame, key[0], Tolerance))
            {
                value = key[1];
            }
            else
            {
                value = keyData[iKeyL][1];
            }

            if ((shiftCount < 0 && anim.pre_wrap == curve_wrapType.relative_repeat) ||
                (shiftCount > 0 && anim.post_wrap == curve_wrapType.relative_repeat))
            {
                value += shiftCount * (lastKey[1] - firstKey[1]);
            }

            return value;
        }

        private static float GetValue_Linear(
            linear_curveType anim,
            List<G3dStream> streams,
            float frame)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(streams.Count > anim.stream_index && anim.stream_index >= 0);
            Vec2Stream keyData = new Vec2Stream(streams[anim.stream_index]);

            int keySize = keyData.Length;
            if (keySize == 0)
            {
                return 0;
            }

            var firstKey = keyData[0];
            var lastKey = keyData[keySize - 1];

            if (keySize == 1)
            {
                return firstKey[1];
            }

            int shiftCount;
            frame = CalcShiftFrame(
                frame,
                keyData[0][0],
                keyData[keySize - 1][0],
                anim.pre_wrap,
                anim.post_wrap,
                out shiftCount);

            if (frame <= firstKey[0])
            {
                return firstKey[1];
            }

            if (frame >= lastKey[0])
            {
                return lastKey[1];
            }

            int iKeyL = 0;
            int iKeyR = keySize - 1;

            while (iKeyL + 1 < iKeyR)
            {
                int iKeyCenter = (iKeyL + iKeyR) / 2;
                var key = keyData[iKeyCenter];
                if (frame <= key[0])
                {
                    iKeyR = iKeyCenter;
                }
                else
                {
                    iKeyL = iKeyCenter;
                }
            }

            Vector2 key0 = keyData[iKeyL];
            Vector2 key1 = keyData[iKeyR];
            float value;
            if (FloatUtility.NearlyEqual(frame, key1[0], Tolerance))
            {
                if (iKeyR < keySize - 1)
                {
                    var key = keyData[iKeyR + 1];
                    if (key1[0] == key[0])
                    {
                        value = key[1];
                    }
                    else
                    {
                        value = key1[1];
                    }
                }
                else
                {
                    value = key1[1];
                }
            }
            else
            {
                value = CalcLerp(key0[0], key0[1], key1[0], key1[1], frame);
            }

            if ((shiftCount < 0 && anim.pre_wrap == curve_wrapType.relative_repeat) ||
                (shiftCount > 0 && anim.post_wrap == curve_wrapType.relative_repeat))
            {
                value += shiftCount * (lastKey[1] - firstKey[1]);
            }
            return value;
        }

        private static float GetValue_Hermite(
            hermite_curveType anim,
            List<G3dStream> streams,
            float frame)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(streams.Count > anim.stream_index && anim.stream_index >= 0);
            Vec4Stream keyData = new Vec4Stream(streams[anim.stream_index]);

            int keySize = keyData.Length;
            if (keySize == 0)
            {
                return 0;
            }
            var firstKey = keyData[0];
            var lastKey = keyData[keySize - 1];

            if (keySize == 1)
            {
                return firstKey[1];
            }

            int shiftCount;
            // リピートを考慮したフレーム
            frame = CalcShiftFrame(
                frame,
                keyData[0][0],
                keyData[keySize - 1][0],
                anim.pre_wrap,
                anim.post_wrap,
                out shiftCount);

            if (frame <= firstKey[0])
            {
                return firstKey[1];
            }

            if (frame >= lastKey[0])
            {
                return lastKey[1];
            }

            int iKeyL = 0;
            int iKeyR = keySize - 1;

            while (iKeyL + 1 < iKeyR)
            {
                var iKeyCenter = (iKeyL + iKeyR) / 2;
                var key = keyData[iKeyCenter];
                if (frame <= key[0])
                {
                    iKeyR = iKeyCenter;
                }
                else
                {
                    iKeyL = iKeyCenter;
                }
            }

            Vector4 key0 = keyData[iKeyL];
            Vector4 key1 = keyData[iKeyR];
            float value;
            if (FloatUtility.NearlyEqual(frame, key1[0], Tolerance))
            {
                if (iKeyR < keySize - 1)
                {
                    var key = keyData[iKeyR + 1];
                    if (key1[0] == key[0])
                    {
                        value = key[1];
                    }
                    else
                    {
                        value = key1[1];
                    }
                }
                else
                {
                    value = key1[1];
                }
            }

            value = CalcHermite(key0[0], key0[1], key0[3], key1[0], key1[1], key1[2], frame);
            if ((shiftCount < 0 && anim.pre_wrap == curve_wrapType.relative_repeat) ||
                (shiftCount > 0 && anim.post_wrap == curve_wrapType.relative_repeat))
            {
                value += shiftCount * (lastKey[1] - firstKey[1]);
            }
            return value;
        }

        public static float CalcShiftFrame(float frame, float startFrame, float endFrame, curve_wrapType preWrap, curve_wrapType postWrap, out int shiftCount)
        {
            float targetFrame = frame;
            //float startFrame = keys[0].Frame;
            //float endFrame = keys[keys.Count - 1].Frame;
            float frameRange = endFrame - startFrame;

            if (frameRange == 0.0f)
            {
                // 同じフレームに複数キーが打たれている
                // どんなwrapモードでもフレームは同じ
                shiftCount = 0;
                return targetFrame;
            }

            float dist = targetFrame - startFrame;
            shiftCount = (int)Math.Floor(dist / frameRange);

            // キーフレームを打ってある範囲のフレームが指定されていない場合に、範囲内のフレームにずらす。
            if ((targetFrame < startFrame))
            {
                // 一番小さいキーフレームより小さいframeの場合、
                // preWrapに応じて処理する。
                switch (preWrap)
                {
                    case curve_wrapType.clamp:
                        targetFrame = startFrame;
                        break;
                    case curve_wrapType.repeat:
                    case curve_wrapType.relative_repeat:
                        {
                            targetFrame -= shiftCount * frameRange;
                            break;
                        }
                    case curve_wrapType.mirror:
                        {
                            targetFrame -= shiftCount * frameRange;
                            if (shiftCount % 2 != 0)
                            {
                                targetFrame = endFrame - (targetFrame - startFrame);
                            }
                            break;
                        }
                }
            }
            else if ((targetFrame > endFrame))
            {
                // 一番大きいキーフレームより大きいframe
                // postWrapに応じて処理する。
                switch (postWrap)
                {
                    case curve_wrapType.clamp:
                        targetFrame = endFrame;
                        break;
                    case curve_wrapType.repeat:
                    case curve_wrapType.relative_repeat:
                        {
                            targetFrame -= shiftCount * frameRange;
                            break;
                        }
                    case curve_wrapType.mirror:
                        {
                            targetFrame -= shiftCount * frameRange;
                            if (shiftCount % 2 != 0)
                            {
                                targetFrame = endFrame - (targetFrame - startFrame);
                            }
                            break;
                        }
                }
            }
            else
            {
                // 範囲内なのでずれはない
                shiftCount = 0;
            }

            return targetFrame;
        }

        private static float GetValue(
            bone_anim_targetType target,
            List<G3dStream> streams,
            float frame)
        {
            if (target != null)
            {
                object obj = target.Item;
                if (obj != null)
                {
                    if (obj.GetType() == typeof(hermite_curveType))
                    {
                        hermite_curveType curve = (hermite_curveType)obj;
                        return GetValue_Hermite(curve, streams, frame);
                    }
                    else if (obj.GetType() == typeof(linear_curveType))
                    {
                        linear_curveType curve = (linear_curveType)obj;
                        return GetValue_Linear(curve, streams, frame);
                    }
                    else if (obj.GetType() == typeof(step_curveType))
                    {
                        step_curveType curve = (step_curveType)obj;
                        return GetValue_Step(curve, streams, frame);
                    }
                }

                return target.base_value;
            }
            else
            {
                return 0.0f;
            }
        }

        private static float GetValue(
            List<AnimKey> curve,
            float frame)
        {
            int keySize = curve.Count;
            if (keySize == 0)
            {
                return 0;
            }
            if (keySize == 1 || frame <= curve[0].mFrame)
            {
                return curve[0].mValue;
            }
            if (frame >= curve[keySize - 1].mFrame)
            {
                return curve[keySize - 1].mValue;
            }

            int iKeyL = 0;
            int iKeyR = keySize - 1;

            while (iKeyL + 1 < iKeyR)
            {
                int iKeyCenter = (iKeyL + iKeyR) / 2;
                if (frame <= curve[iKeyCenter].mFrame)
                {
                    iKeyR = iKeyCenter;
                }
                else
                {
                    iKeyL = iKeyCenter;
                }
            }

            AnimKey key0 = curve[iKeyL];
            AnimKey key1 = curve[iKeyR];
            if (FloatUtility.NearlyEqual(frame, key1.mFrame, Tolerance))
            {
                if (iKeyR < keySize - 1 && key1.mFrame == curve[iKeyR + 1].mFrame)
                {
                    return curve[iKeyR + 1].mValue;
                }
                else
                {
                    return key1.mValue;
                }
            }

            return CalcHermite(key0.mFrame, key0.mValue, key0.mOutSlope, key1.mFrame, key1.mValue, key1.mInSlope, frame);
        }

        private static Vector3 GetScale(
            List<G3dStream> streams,
            bone_animType boneAnim,
            float frame)
        {
            return new Vector3(
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.scale_x), streams, frame),
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.scale_y), streams, frame),
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.scale_z), streams, frame));
        }

        private static Vector4 GetRotate(
            List<G3dStream> streams,
            bone_animType boneAnim,
            float frame)
        {
            return new Vector4(
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.rotate_x), streams, frame),
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.rotate_y), streams, frame),
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.rotate_z), streams, frame),
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.rotate_w), streams, frame));
        }

        private static Vector3 GetTranslate(
            List<G3dStream> streams,
            bone_animType boneAnim,
            float frame)
        {
            return new Vector3(
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.translate_x), streams, frame),
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.translate_y), streams, frame),
                GetValue(GetBoneAnimTarget(boneAnim, bone_anim_target_targetType.translate_z), streams, frame));
        }

        private static Matrix44 GetBoneTransformStandard(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo boneInfo,
            float frame)
        {
            // boneInfo からルートノードまでのボーンリストを作成する
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<BoneAnimInfo> bonePath = new List<BoneAnimInfo>();
            bonePath.Add(boneInfo);
            BoneAnimInfo parent = boneInfo.Parent;
            int boneSize = anim.bone_anim_array.length;
            int count = 0;
            for (; count < boneSize && parent != null; ++count)
            {
                bonePath.Add(parent);
                parent = parent.Parent;
            }
            // 親をたどっていった数とモデル中のボーン数が一致すると
            // 参照が循環しているということなのでエラーとする。
            if (count == boneSize)
            {
                throw new Exception("<bone> has cyclic indexing.");
            }

            // 変換はルートノードから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();

            // ルートノードから boneInfo までの変換を合成する。
            Matrix44 transform = Matrix44.Identity;
            Matrix44 s = new Matrix44();
            Matrix44 r = new Matrix44();
            Matrix44 t = new Matrix44();
            foreach (BoneAnimInfo bone in bonePath)
            {
                bone_animType b = bone.BoneAnim;
                Vector4 rot = GetRotate(streams, b, frame);
                if (bone.SkeletalAnimInfo.rotate_mode == skeletal_anim_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 euler = EulerXyz.ToRadian(
                        new Vector3(rot[0], rot[1], rot[2]));
                    EulerXyz.SetRotate(r, euler.X, euler.Y, euler.Z);
                }
                else
                {
                    r.SetRotate(new Quaternion(rot[0], rot[1], rot[2], rot[3]));
                }
                s.SetScale(GetScale(streams, b, frame));
                t.SetTranslate(GetTranslate(streams, b, frame));
                transform *= t * r * s;
            }

            return transform;
        }

        private static Matrix44 GetBoneTransform_Standard(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo fromBoneInfo,
            BoneAnimInfo toBoneInfo,
            float frame)
        {
            // fromBoneInfo から toBoneInfo までのボーンリストを作成する。
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<BoneAnimInfo> bonePath = new List<BoneAnimInfo>();
            bonePath.Add(fromBoneInfo);
            // fromBoneInfo から toBoneInfo までのボーンリストを取得する。
            // 得られた bonePath には toBoneInfo は含まれていない。
            GetBoneAncestors(fromBoneInfo, toBoneInfo, bonePath);

            // 変換は上位階層のボーンから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();

            // toBoneInfo の下位階層から boneInfo までの変換を合成する。
            Matrix44 transform = Matrix44.Identity;
            Matrix44 s = new Matrix44();
            Matrix44 r = new Matrix44();
            Matrix44 t = new Matrix44();
            foreach (BoneAnimInfo boneInfo in bonePath)
            {
                bone_animType bone = boneInfo.BoneAnim;
                Vector4 rot = GetRotate(streams, bone, frame);
                if (boneInfo.SkeletalAnimInfo.rotate_mode == skeletal_anim_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 euler = EulerXyz.ToRadian(
                        new Vector3(rot[0], rot[1], rot[2]));
                    EulerXyz.SetRotate(r, euler.X, euler.Y, euler.Z);
                }
                else
                {
                    r.SetRotate(new Quaternion(rot[0], rot[1], rot[2], rot[3]));
                }
                s.SetScale(GetScale(streams, bone, frame));
                t.SetTranslate(GetTranslate(streams, bone, frame));
                transform *= t * r * s;
            }

            return transform;
        }

        private static Matrix44 GetBoneTransformMaya(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo boneInfo,
            float frame)
        {
            // boneInfo からルートノードまでのボーンリストを作成する
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<BoneAnimInfo> bonePath = new List<BoneAnimInfo>();
            bonePath.Add(boneInfo);
            BoneAnimInfo parent = boneInfo.Parent;
            int boneSize = anim.bone_anim_array.length;
            int count = 0;
            for (; count < boneSize && parent != null; ++count)
            {
                bonePath.Add(parent);
                parent = parent.Parent;
            }
            // 親をたどっていった数とモデル中のボーン数が一致すると
            // 参照が循環しているということなのでエラーとする。
            if (count == boneSize)
            {
                throw new Exception("<bone> has cyclic indexing.");
            }

            // 変換はルートノードから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();

            // ルートノードから boneInfo までの変換を合成する。
            Matrix44 tr = Matrix44.Identity;
            Matrix44 psm = new Matrix44(); // 親のスケール行列
            Matrix44 rm = new Matrix44();
            Matrix44 tm = new Matrix44();
            Vector3 s = new Vector3(1.0f, 1.0f, 1.0f);
            foreach (BoneAnimInfo bone in bonePath)
            {
                bone_animType b = bone.BoneAnim;

                // 回転行列を求める。
                Vector4 rot = GetRotate(streams, b, frame);
                if (boneInfo.SkeletalAnimInfo.rotate_mode == skeletal_anim_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 euler = EulerXyz.ToRadian(
                        new Vector3(rot[0], rot[1], rot[2]));
                    EulerXyz.SetRotate(rm, euler.X, euler.Y, euler.Z);
                }
                else
                {
                    rm.SetRotate(new Quaternion(rot[0], rot[1], rot[2], rot[3]));
                }
                tm.SetTranslate(GetTranslate(streams, b, frame));

                // scale_compensate の設定に従って変換を求める。
                if (b.scale_compensate)
                {
                    // scale_compensate が ON の場合は親のスケールを translate にだけ適用する
                    tm[0, 3] *= s.X;
                    tm[1, 3] *= s.Y;
                    tm[2, 3] *= s.Z;
                    tr *= tm * rm;
                }
                else
                {
                    psm.SetScale(s);
                    tr *= psm * tm * rm;
                }
                s = GetScale(streams, b, frame);
            }
            // この時点で s には boneInfo.BoneAnim.scale が設定されている。
            // 最後に boneInfo のスケールを適用する。
            Matrix44 sm = new Matrix44();
            sm.SetScale(s);
            return tr * sm;
        }

        private static Matrix44 GetBoneTransform_Maya(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo fromBoneInfo,
            BoneAnimInfo toBoneInfo,
            float frame)
        {
            // fromBoneInfo から toBoneInfo までのボーンリストを作成する。
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<BoneAnimInfo> bonePath = new List<BoneAnimInfo>();
            bonePath.Add(fromBoneInfo);
            // fromBoneInfo から toBoneInfo までのボーンリストを取得する。
            // 得られた bonePath には toBoneInfo は含まれていない。
            GetBoneAncestors(fromBoneInfo, toBoneInfo, bonePath);

            // 変換は上位階層のボーンから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();

            // ルートノードから boneInfo までの変換を合成する。
            Matrix44 tr = Matrix44.Identity;
            Matrix44 psm = new Matrix44(); // 親のスケール行列
            Matrix44 rm = new Matrix44();
            Matrix44 tm = new Matrix44();
            bone_animType toBone = toBoneInfo.BoneAnim;
            Vector3 s = GetScale(streams, toBone, frame);
            s[0] = 1.0f / s[0];
            s[1] = 1.0f / s[1];
            s[2] = 1.0f / s[2];

            // toBoneInfo の子供の変換を求める
            {
                BoneAnimInfo bone = bonePath[0];
                bone_animType b = bone.BoneAnim;

                // 回転行列を求める。
                Vector4 rot = GetRotate(streams, b, frame);
                if (bone.SkeletalAnimInfo.rotate_mode == skeletal_anim_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 eulerBi = EulerXyz.ToRadian(
                        new Vector3(rot[0], rot[1], rot[2]));
                    EulerXyz.SetRotate(rm, eulerBi.X, eulerBi.Y, eulerBi.Z);
                }
                else
                {
                    rm.SetRotate(new Quaternion(rot[0], rot[1], rot[2], rot[3]));
                }
                tm.SetTranslate(GetTranslate(streams, b, frame));

                if (b.scale_compensate)
                {
                    psm.SetScale(s);
                    tr = tm * psm * rm;
                }
                else
                {
                    tr = tm * rm;
                }
                s = GetScale(streams, b, frame);

                // 最初のボーンの変換は既に求めたので、ボーンリストからそのボーンを削除しておく。
                bonePath.RemoveAt(0);
            }
            foreach (BoneAnimInfo bone in bonePath)
            {
                bone_animType b = bone.BoneAnim;
                // 回転行列を求める。
                Vector4 rot = GetRotate(streams, b, frame);
                if (bone.SkeletalAnimInfo.rotate_mode == skeletal_anim_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 euler = EulerXyz.ToRadian(
                        new Vector3(rot[0], rot[1], rot[2]));
                    EulerXyz.SetRotate(rm, euler.X, euler.Y, euler.Z);
                }
                else
                {
                    rm.SetRotate(new Quaternion(rot[0], rot[1], rot[2], rot[3]));
                }
                tm.SetTranslate(GetTranslate(streams, b, frame));

                // scale_compensate の設定に従って変換を求める。
                if (b.scale_compensate)
                {
                    // scale_compensate が ON の場合は親のスケールを translate にだけ適用する
                    tm[0, 3] *= s.X;
                    tm[1, 3] *= s.Y;
                    tm[2, 3] *= s.Z;
                    tr *= tm * rm;
                }
                else
                {
                    psm.SetScale(s);
                    tr *= psm * tm * rm;
                }
                s = GetScale(streams, b, frame);
            }
            // この時点で s には boneInfo.BoneAnim.scale が設定されている。
            // 最後に boneInfo のスケールを適用する。
            Matrix44 sm = new Matrix44();
            sm.SetScale(s);
            return tr * sm;
        }

        private static Matrix44 GetBoneTransformSoftimage(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo boneInfo,
            float frame)
        {
            return Matrix44.Identity;
        }

        private static Matrix44 GetBoneTransform_Softimage(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo fromBoneInfo,
            BoneAnimInfo toBoneInfo,
            float frame)
        {
            return Matrix44.Identity;
        }

        private static Matrix44 GetBoneTransform(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo boneAnimInfo,
            float frame)
        {
            if (boneAnimInfo == null)
            {
                return Matrix44.Identity;
            }
            switch (boneAnimInfo.SkeletalAnimInfo.scale_mode)
            {
                case skeletal_anim_info_scale_modeType.standard:
                    return GetBoneTransformStandard(anim, streams, boneAnimInfo, frame);
                case skeletal_anim_info_scale_modeType.maya:
                    return GetBoneTransformMaya(anim, streams, boneAnimInfo, frame);
                case skeletal_anim_info_scale_modeType.softimage:
                    return GetBoneTransformSoftimage(anim, streams, boneAnimInfo, frame);
                default:
                    throw new Exception("<skeletal_anim_info> has unexpected value of 'scale_mode'.");
            }
        }

        private static Matrix44 GetBoneTransform_Auto(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo fromBoneAnimInfo,
            BoneAnimInfo toBoneAnimInfo,
            float frame)
        {
            switch (toBoneAnimInfo.SkeletalAnimInfo.scale_mode)
            {
                case skeletal_anim_info_scale_modeType.standard:
                    return GetBoneTransform_Standard(anim, streams, fromBoneAnimInfo, toBoneAnimInfo, frame);
                case skeletal_anim_info_scale_modeType.maya:
                    return GetBoneTransform_Maya(anim, streams, fromBoneAnimInfo, toBoneAnimInfo, frame);
                case skeletal_anim_info_scale_modeType.softimage:
                    return GetBoneTransform_Softimage(anim, streams, fromBoneAnimInfo, toBoneAnimInfo, frame);
                default:
                    throw new Exception("<skeletal_anim_info> has unexpected value of 'scale_mode'.");
            }
        }

        /**
         * @brief fromBoneAnimInfo が指す <bone_anim> 座標系からtoBoneAnimInfoが指す <bone_anim> 座標系への変換行列を求めます。
         */
        private static Matrix44 GetBoneTransform(
            skeletal_animType anim,
            List<G3dStream> streams,
            BoneAnimInfo fromBoneAnimInfo,
            BoneAnimInfo toBoneAnimInfo,
            float frame)
        {
            if (fromBoneAnimInfo == toBoneAnimInfo)
            {
                return Matrix44.Identity;
            }
            if (fromBoneAnimInfo != null && toBoneAnimInfo != null)
            {
                if (IsAncestorBone(fromBoneAnimInfo, toBoneAnimInfo))
                {
                    return GetBoneTransform_Auto(anim, streams, fromBoneAnimInfo, toBoneAnimInfo, frame);
                }
                if (IsAncestorBone(toBoneAnimInfo, fromBoneAnimInfo))
                {
                    Matrix44 tr = GetBoneTransform_Auto(anim, streams, toBoneAnimInfo, fromBoneAnimInfo, frame);
                    tr.Invert();
                    return tr;
                }
            }
            Matrix44 toBoneTransform = GetBoneTransform(anim, streams, toBoneAnimInfo, frame);
            Matrix44 fromBoneTransform = GetBoneTransform(anim, streams, fromBoneAnimInfo, frame);
            toBoneTransform.Invert();
            return toBoneTransform * fromBoneTransform;
        }

        public class AnimKey
        {
            public float mFrame { get; set; }
            public float mValue { get; set; }
            public float mInSlope { get; set; }
            public float mOutSlope { get; set; }

            public AnimKey()
            {
            }
            public AnimKey(AnimKey other)
            {
                mFrame = other.mFrame;
                mValue = other.mValue;
                mInSlope = other.mInSlope;
                mOutSlope = other.mOutSlope;
            }
            public AnimKey(float frame, float value, float inslope, float outslope)
            {
                mFrame = frame;
                mValue = value;
                mInSlope = inslope;
                mOutSlope = outslope;
            }

            public void RoundToZero()
            {
                mFrame = FloatUtility.RoundToZero(mFrame, Tolerance);
                mValue = FloatUtility.RoundToZero(mValue, Tolerance);
                mInSlope = FloatUtility.RoundToZero(mInSlope, Tolerance);
                mOutSlope = FloatUtility.RoundToZero(mOutSlope, Tolerance);
            }
        }

        /// <summary>
        /// フレーム間のオイラー角の差を小さくします。
        /// </summary>
        /// <param name="rxArray">x 軸周りの回転のカーブです。</param>
        /// <param name="ryArray">y 軸周りの回転のカーブです。</param>
        /// <param name="rzArray">z 軸周りの回転のカーブです。</param>
        private static bool MakeContinuousXyzAngle(
            List<AnimKey> rxArray, List<AnimKey> ryArray, List<AnimKey> rzArray)
        {
            //-----------------------------------------------------------------------------
            // check frame size
            int frameSize = rxArray.Count;
            if (frameSize < 2)
            {
                return false;
            }

            if (frameSize != ryArray.Count || frameSize != rzArray.Count)
            {
                return false;
            }

            //-----------------------------------------------------------------------------
            // check constant
            var isConstantX = IsConstant(rxArray);
            var isConstantY = IsConstant(ryArray);
            var isConstantZ = IsConstant(rzArray);
            if (isConstantX && isConstantY && isConstantZ)
            {
                return false;
            }
            else if (isConstantY && isConstantZ)
            {
                return MakeContinuousAngleArray(rxArray);
            }
            else if (isConstantZ && isConstantX)
            {
                return MakeContinuousAngleArray(ryArray);
            }
            else if (isConstantX && isConstantY)
            {
                return MakeContinuousAngleArray(rzArray);
            }

            rxArray[0].mValue = NearestAngleMod360Degrees(0, rxArray[0].mValue);
            ryArray[0].mValue = NearestAngleMod360Degrees(0, ryArray[0].mValue);
            rzArray[0].mValue = NearestAngleMod360Degrees(0, rzArray[0].mValue);

            for (int i = 1; i < rxArray.Count; i++)
            {
                rxArray[i].mValue = NearestAngleMod360Degrees(rxArray[i - 1].mValue, rxArray[i].mValue);
                ryArray[i].mValue = NearestAngleMod360Degrees(ryArray[i - 1].mValue, ryArray[i].mValue);
                rzArray[i].mValue = NearestAngleMod360Degrees(rzArray[i - 1].mValue, rzArray[i].mValue);
                float diff = Math.Abs(rxArray[i].mValue - rxArray[i - 1].mValue)
                    + Math.Abs(ryArray[i].mValue - ryArray[i - 1].mValue)
                    + Math.Abs(rzArray[i].mValue - rzArray[i - 1].mValue);

                float x = NearestAngleMod360Degrees(rxArray[i - 1].mValue, rxArray[i].mValue + 180);
                float y = NearestAngleMod360Degrees(ryArray[i - 1].mValue, -ryArray[i].mValue + 180);
                float z = NearestAngleMod360Degrees(rzArray[i - 1].mValue, rzArray[i].mValue + 180);
                float diff2 = Math.Abs(x - rxArray[i - 1].mValue)
                    + Math.Abs(y - ryArray[i - 1].mValue)
                    + Math.Abs(z - rzArray[i - 1].mValue);

                if (diff2 < diff)
                {
                    rxArray[i].mValue = x;
                    ryArray[i].mValue = y;
                    rzArray[i].mValue = z;
                }
            }

            return true;
        }

        /// <summary>
        /// 基準からの角の差を360度の範囲に補正します。
        /// </summary>
        /// <param name="baseAngle">基準の角です。</param>
        /// <param name="angle">補正対象の角です。</param>
        /// <returns>補正結果を返します。</returns>
        private static float NearestAngleMod360Degrees(double baseAngle, double angle)
        {
            var dif = (angle - baseAngle) % 360;
            angle = dif + baseAngle;

            if (angle - baseAngle > 180)
            {
                angle -= 2 * 180;
            }
            else if (angle - baseAngle < -180)
            {
                angle += 2 * 180;
            }

            return (float)angle;
        }

        //-----------------------------------------------------------------------------
        //! @brief 角度の配列が連続的な値になるように調整します。角度の単位は degree です。
        //!
        //! @param[in,out] values 角度の配列です。
        //!
        //! @return 角度の配列を変更した場合は true を返します。
        //-----------------------------------------------------------------------------
        private static bool MakeContinuousAngleArray(List<AnimKey> values)
        {
            //-----------------------------------------------------------------------------
            // check frame count
            var subFrameCount = values.Count;
            if (subFrameCount < 2)
            {
                return false;
            }

            //-----------------------------------------------------------------------------
            // check constant
            if (IsConstant(values))
            {
                return false;
            }

            //-----------------------------------------------------------------------------
            // adjust
            bool changedFlag = false;
            for (int iFrame = 1; iFrame < subFrameCount; ++iFrame)
            {
                var v = values[iFrame].mValue;
                var lv = values[iFrame - 1].mValue;
                var diff = v - lv;
                var diffAbs = Math.Abs(diff);

                // 差を 360 未満にします。
                while (diffAbs >= 360.0f)
                {
                    if (diff > 0.0f)
                    {
                        v -= 360.0f;
                    }
                    else
                    {
                        v += 360.0f;
                    }
                    diffAbs -= 360.0f;
                }

                // ±360 の値をテスト
                diff = v - lv;
                if (diffAbs > 180.0f)
                {
                    if (diff > 0.0f)
                    {
                        var tv = v - 360.0f;
                        if (Math.Abs(tv - lv) < diffAbs)
                        {
                            v = tv;
                        }
                    }
                    else
                    {
                        var tv = v + 360.0f;
                        if (Math.Abs(tv - lv) < diffAbs)
                        {
                            v = tv;
                        }
                    }
                }

                if (v != values[iFrame].mValue)
                {
                    changedFlag = true;
                    values[iFrame].mValue = v;
                }
            }

            return changedFlag;
        }

        private static void OptimizeAnimKey(
            List<AnimKey> curve,
            List<AnimKey> fullCurve,
            float tolerance)
        {
            int keySize = curve.Count;
            if (keySize <= 2)
            {
                return;
            }

            int subFrameSize = fullCurve.Count;
            for (int iKeyCut = 1; iKeyCut + 1 < curve.Count;)
            {
                List<AnimKey> testCurve = new List<AnimKey>(curve.ToArray());
                testCurve.RemoveAt(iKeyCut);

                float checkStartFrame = curve[iKeyCut - 1].mFrame;
                float checkEndFrame = curve[iKeyCut + 1].mFrame;

                float errorMax = 0.0f;
                for (int iFrame = 0; iFrame < subFrameSize; ++iFrame)
                {
                    float frame = fullCurve[iFrame].mFrame;
                    if (frame <= checkStartFrame)
                    {
                        continue;
                    }
                    if (checkEndFrame <= frame)
                    {
                        break;
                    }
                    float value = GetValue(testCurve, frame);
                    float err = System.Math.Abs(fullCurve[iFrame].mValue - value);
                    if (err > errorMax)
                    {
                        errorMax = err;
                        if (errorMax >= tolerance)
                        {
                            break;
                        }
                    }
                }
                if (errorMax < tolerance)
                {
                    curve.RemoveAt(iKeyCut);
                }
                else
                {
                    ++iKeyCut;
                }
            }
        }

        /*
         * 参考値(fullValues)を元に、ファンクションカーブを生成する。
         *
         * 制限事項:
         * fullValueでは同一フレームに複数のキーを許さない。
         */
        private static void MakeAnimKey(
            List<AnimKey> curve,
            List<AnimKey> fullValues,
            float tolerance,
            bool isLoop,
            bool isAngle)
        {
            // すべてのキーをコピーする。
            // curve に変更を加えた後に fullValues を参照するので、
            // ディープコピーを行う必要がある。
            curve.Clear();
            foreach (AnimKey k in fullValues)
            {
                curve.Add(new AnimKey(k));
            }
            int keySize = curve.Count;
            if (keySize == 0)
            {
                return;
            }
            else if (keySize == 1)
            {
                curve[0].mInSlope = 0.0f;
                curve[0].mOutSlope = 0.0f;
                return;
            }
            // 中間区間のスロープを計算する。
            for (int iKey = 1; iKey + 1 < keySize; ++iKey)
            {
                AnimKey key = curve[iKey];
                AnimKey key0 = curve[iKey - 1];
                AnimKey key1 = curve[iKey + 1];
                float slope = key1.mValue - key0.mValue;
                float frameWidth = key1.mFrame - key0.mFrame;
                // curve は fullValues をコピーしている。
                // fullValues は最適化コンバータ内でベークされた配列なので、
                // キーが同じ時間に重なっている事はないはず。
                Nintendo.Foundation.Contracts.Assertion.Operation.True(frameWidth != 0.0f);
                slope /= frameWidth;
                key.mInSlope = key.mOutSlope = slope;
                key.RoundToZero();
            }
            // 最初のキーのスロープ値を計算する。
            {
                AnimKey key = curve[0];
                if (isAngle)
                {
                    // 回転角は連続性補正のため、最初と最後のフレーム値が同じとは限らない。
                    // そのため、前後の値を平均しても意味をなさない。
                    AnimKey key1 = curve[1];
                    float slope = key1.mValue - key.mValue;
                    float frameWidth = key1.mFrame - key.mFrame;
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(frameWidth != 0.0f);
                    slope /= frameWidth;
                    key.mInSlope = key.mOutSlope = slope;
                }
                else if (isLoop)
                {
                    // 最後のキーは最初のキーと同一なので、その一つ前の値を使う。
                    AnimKey key0 = curve[keySize - 2];
                    AnimKey key1 = curve[1];
                    float slope = key1.mValue - key0.mValue;
                    float frameWidth = key1.mFrame - curve[0].mFrame + curve[keySize - 1].mFrame - key0.mFrame;
                    if (frameWidth > 0.0f)
                    {
                        slope /= frameWidth;
                    }
                    key.mInSlope = key.mOutSlope = slope;
                }
                else
                {
                    AnimKey key1 = curve[1];
                    if (key.mFrame < key1.mFrame)
                    {
                        float slope = 2.0f * (key1.mValue - key.mValue) / (key1.mFrame - key.mFrame) - key1.mInSlope;
                        key.mInSlope = key.mOutSlope = slope;
                    }
                    else
                    {
                        Console.WriteLine("same frame in bake keys.");
                    }
                }
                key.RoundToZero();
            }
            // 最後のキーのスロープ値を計算する。
            {
                AnimKey key = curve[curve.Count - 1];
                if (isAngle)
                {
                    // 回転角は連続性補正のため、最初と最後のフレーム値が同じとは限らない。
                    // そのため、前後の値を平均しても意味をなさない。
                    AnimKey key0 = curve[keySize - 2];
                    if (key0.mFrame < key.mFrame)
                    {
                        float slope = (key.mValue - key0.mValue) / (key.mFrame - key0.mFrame);
                        key.mInSlope = key.mOutSlope = slope;
                    }
                    else
                    {
                        Console.WriteLine("same frame in bake keys.");
                    }
                }
                else if (isLoop)
                {
                    // 最後のキーは最初のキーと同じ。
                    float slope = curve[0].mInSlope;
                    key.mInSlope = key.mOutSlope = slope;
                }
                else
                {
                    AnimKey key0 = curve[keySize - 2];
                    if (key0.mFrame < key.mFrame)
                    {
                        float slope = 2.0f * (key.mValue - key0.mValue) / (key.mFrame - key0.mFrame) - key0.mOutSlope;
                        key.mInSlope = key.mOutSlope = slope;
                    }
                    else
                    {
                        Console.WriteLine("same frame in bake keys.");
                    }
                }
                key.RoundToZero();
            }
            // 要らないキーを間引く。
            if (true || !isAngle)
            {
                OptimizeAnimKey(curve, fullValues, tolerance);
            }
        }

        private static bool IsConstant(
            List<AnimKey> curve)
        {
            if (curve.Count == 0)
            {
                return false;
            }
            float baseValue = curve[0].mValue;
            for (int i = 1; i < curve.Count; i++)
            {
                AnimKey key = curve[i];
                if (key.mValue != baseValue)
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// fullValuesからカーブを生成する
        /// 並列実行対応のため、G3DStreamを返す
        /// </summary>
        /// <param name="target"></param>
        /// <param name="fullValues"></param>
        /// <param name="tolerance"></param>
        /// <param name="isLoop"></param>
        /// <param name="isAngle"></param>
        /// <returns></returns>
        public static G3dStream SetAnimation(
            bone_anim_targetType target,
            List<AnimKey> fullValues,
            float tolerance,
            bool isLoop,
            bool isAngle)
        {
            if (fullValues == null || !fullValues.Any())
            {
                return null;
            }

            var curve = new List<AnimKey>();

            // MakeAnimKeyは重いので、fullValuesが定数カーブでない場合だけ実行する
            if (IsConstant(fullValues))
            {
                // 定数カーブでもキーは一つ必要なため追加する
                var key = fullValues.FirstOrDefault();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(key != null, "key != null");
                key.RoundToZero();
                curve.Add(key);
            }
            else
            {
                MakeAnimKey(curve, fullValues, tolerance, isLoop, isAngle);
            }

            // 基底値を設定する
            target.base_value = curve[0].mValue;
            G3dStream newStream = null;
            // 定数でない場合はアニメーションカーブを作成する
            if (!IsConstant(curve))
            {
                // ストリームを新しく作成する
                newStream = new G3dStream
                {
                    type = stream_typeType.@float,
                    column = 4
                };

                var floatData = newStream.FloatData;
                foreach (var key in curve)
                {
                    floatData.Add(key.mFrame);
                    floatData.Add(key.mValue);
                    floatData.Add(key.mInSlope);
                    floatData.Add(key.mOutSlope);
                }

                // ターゲットにカーブの情報を登録する
                var hermiteCurve = new hermite_curveType
                {
                    count = curve.Count,
                    frame_type = curve_frame_typeType.none,
                    key_type = curve_key_typeType.none,
                    scale = 1.0f,
                    offset = 0.0f,
                    pre_wrap = curve_wrapType.clamp,
                    post_wrap = curve_wrapType.clamp,
                    baked = true,
                    // stream_indexは後で解決する
                    // stream_index = stream_index
                };
                target.Item = hermiteCurve;
            }
            return newStream;
        }

        private static bone_anim_targetType GetBoneAnimTarget(
            bone_animType anim,
            bone_anim_target_targetType targetTarget)
        {
            return anim.bone_anim_target.FirstOrDefault(target => target.target == targetTarget);
        }

        /**
         * アニメーションをマージします。
         * boneInfo から parentInfo までのボーンアニメーションをマージし、
         * parentInfo の座標系のアニメーションに変換します。
         */
        public class SetAnimationItem
        {
            public bone_anim_targetType Target { get; set; }
            public List<AnimKey> FullValues { get; set; }
            public float CurveTolerance { get; set; }
            public bool IsAngle { get; set; }
            public bool IsLoop { get; set; }
            public G3dStream Stream { get; set; }
            public SetAnimationItem(
                bone_anim_targetType target,
                List<AnimKey> fullValues,
                float curveTolerance,
                bool isLoop,
                bool isAngle)
            {
                Target = target;
                FullValues = fullValues;
                CurveTolerance = curveTolerance;
                IsAngle = isLoop;
                IsAngle = isAngle;
            }
        }
        public static List<SetAnimationItem> MergeAnimation(
            skeletal_animType skeletalAnim,
            List<G3dStream> streams,
            BoneAnimInfo parentInfo,
            BoneAnimInfo boneInfo)
        {
            skeletal_anim_infoType animInfo = boneInfo.SkeletalAnimInfo;
            int subFrame = animInfo.frame_resolution;
            bool isLoop = animInfo.loop;
            int frameSize = 1 + animInfo.frame_count * subFrame;
            var bone = boneInfo.BoneAnim;

            var tx = new List<AnimKey>(frameSize);
            var ty = new List<AnimKey>(frameSize);
            var tz = new List<AnimKey>(frameSize);
            var rx = new List<AnimKey>(frameSize);
            var ry = new List<AnimKey>(frameSize);
            var rz = new List<AnimKey>(frameSize);
            var sx = new List<AnimKey>(frameSize);
            var sy = new List<AnimKey>(frameSize);
            var sz = new List<AnimKey>(frameSize);
            for (int iFrame = 0; iFrame < frameSize; ++iFrame)
            {
                float frame = (float)(iFrame) / subFrame;
                Matrix44 nodeTransform =
                    GetBoneTransform(
                        skeletalAnim,
                        streams,
                        boneInfo,
                        parentInfo,
                        frame);
                var trans = new Transform();
                nodeTransform.GetTransform(trans);
                // 回転角度を Radian から Degree に変換する。
                trans.Rotate = EulerXyz.ToDegree(trans.Rotate);
                tx.Add(new AnimKey(frame, trans.Translate.X, 0, 0));
                ty.Add(new AnimKey(frame, trans.Translate.Y, 0, 0));
                tz.Add(new AnimKey(frame, trans.Translate.Z, 0, 0));
                rx.Add(new AnimKey(frame, trans.Rotate.X, 0, 0));
                ry.Add(new AnimKey(frame, trans.Rotate.Y, 0, 0));
                rz.Add(new AnimKey(frame, trans.Rotate.Z, 0, 0));
                sx.Add(new AnimKey(frame, trans.Scale.X, 0, 0));
                sy.Add(new AnimKey(frame, trans.Scale.Y, 0, 0));
                sz.Add(new AnimKey(frame, trans.Scale.Z, 0, 0));
            }

            var animationItems = new List<SetAnimationItem>();

            // Translate
            {
                float tolerance = animInfo.bake_tolerance_translate * animInfo.dcc_magnify;
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.translate_x);
                    animationItems.Add(new SetAnimationItem(target, tx, tolerance, isLoop, false));
                }
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.translate_y);
                    animationItems.Add(new SetAnimationItem(target, ty, tolerance, isLoop, false));
                }
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.translate_z);
                    animationItems.Add(new SetAnimationItem(target, tz, tolerance, isLoop, false));
                }
            }

            // Rotate
            {
                // 回転値が連続的になるようにXYZ３軸同時に補正
                if (MakeContinuousXyzAngle(rx, ry, rz))
                {
                    Console.WriteLine("rotation for node [{0}] is fixed as continuous.", bone.bone_name);
                }
                float tolerance = animInfo.bake_tolerance_rotate;
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.rotate_x);
                    animationItems.Add(new SetAnimationItem(target, rx, tolerance, isLoop, true));
                }
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.rotate_y);
                    animationItems.Add(new SetAnimationItem(target, ry, tolerance, isLoop, true));
                }
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.rotate_z);
                    animationItems.Add(new SetAnimationItem(target, rz, tolerance, isLoop, true));
                }
            }

            // Scale
            {
                float tolerance = animInfo.bake_tolerance_scale;
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.scale_x);
                    animationItems.Add(new SetAnimationItem(target, sx, tolerance, isLoop, false));
                }
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.scale_y);
                    animationItems.Add(new SetAnimationItem(target, sy, tolerance, isLoop, false));
                }
                {
                    bone_anim_targetType target =
                        GetBoneAnimTarget(bone, bone_anim_target_targetType.scale_z);
                    animationItems.Add(new SetAnimationItem(target, sz, tolerance, isLoop, false));
                }
            }
            return animationItems;
        }

        // アニメーションが設定されているか
        private static bool HasAnimation(bone_animType boneAnim)
        {
            return boneAnim.bone_anim_target.Any(target => target.Curve != null);
        }

        // バイナライズされるアニメーションが付いていてるボーンが削除されるか確認する
        public static void CheckBinalizeAnimationRemoved(BoneAnimInfo[] boneInfos)
        {
            foreach (var bi in boneInfos)
            {
                if (bi.IsReservedToRemove)
                {
                    var boneAnim = bi.BoneAnim;
                    if ((boneAnim.binarize_rotate || boneAnim.binarize_scale || boneAnim.binarize_translate)
                        && HasAnimation(boneAnim))
                    {
                        Console.Error.WriteLine(
                            IfStrings.Get("SkeletalAnimCompress_Warning_RemoveAnim", boneAnim.name));
                    }
                }
            }
        }

        // アニメーションが付いているボーンが削除されるか確認する
        public static void CheckAnimationRemoved(BoneAnimInfo[] boneInfos)
        {
            foreach (BoneAnimInfo bi in boneInfos)
            {
                if (bi.IsReservedToRemove)
                {
                    CheckAnimationRemoved(bi.BoneAnim);
                }
            }
        }

        // アニメーションが付いているボーンが削除されるか確認する
        public static void CheckAnimationRemoved(bone_animType boneAnim)
        {
            if (HasAnimation(boneAnim))
            {
                Console.Error.WriteLine(
                    IfStrings.Get("SkeletalAnimCompress_Warning_RemoveAnim", boneAnim.name));
            }
        }
    }

    // ボーン圧縮のためのユーティリティー
    public class BoneAnimInfo
    {
        public BoneAnimInfo(bone_animType boneAnim, skeletal_anim_infoType skeletalAnimInfo)
        {
            this.SkeletalAnimInfo = skeletalAnimInfo;
            this.BoneAnim = boneAnim;
        }
        public static void Setup(BoneAnimInfo[] boneInfos)
        {
            // ボーン検索のためのテーブル
            Dictionary<string, BoneAnimInfo> boneInfoTable =
                new Dictionary<string, BoneAnimInfo>();
            foreach (BoneAnimInfo bi in boneInfos)
            {
                boneInfoTable.Add(bi.BoneAnim.name, bi);
            }

            // 階層構造のセットアップ
            foreach (BoneAnimInfo bi in boneInfos)
            {
                string parentName = bi.BoneAnim.parent_name;
                if (parentName != string.Empty)
                {
                    BoneAnimInfo parentInfo;
                    if (!boneInfoTable.TryGetValue(parentName, out parentInfo))
                    {
                        IfStrings.Throw("SkeletalAnimCompressUtility_Error_ParentNotFound",
                            bi.BoneAnim.name, parentName);
                    }
                    parentInfo.Children.Add(bi);
                    bi.Parent = parentInfo;
                }
            }

            // 兄弟ボーンを文字コード順にソートする
            foreach (BoneAnimInfo bi in boneInfos)
            {
                bi.Children.Sort(
                    delegate (BoneAnimInfo lhs, BoneAnimInfo rhs)
                    {
                        return string.CompareOrdinal(lhs.BoneAnim.name, rhs.BoneAnim.name);
                    });
            }
        }

        // 削除されない子がいるか
        public bool HasChild()
        {
            foreach (BoneAnimInfo bone_animInfo in this.Children)
            {
                if (!bone_animInfo.IsReservedToRemove) { return true; }
            }
            return false;
        }

        // 文字列変換
        public override string ToString()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendFormat("{0} {1} {2}→",
                !this.IsReservedToRemove ? "○" : "×",
                this.RenderMatrix ? "○" : "×",
                this.BoneAnim.bone_name);
            foreach (BoneAnimInfo child in this.Children)
            {
                builder.AppendFormat("{0}{1},",
                    child.BoneAnim.bone_name,
                    child.RenderMatrix ? "○" : "×");
            }
            return builder.ToString();
        }

        public BoneAnimInfo GetIncompressibleAncestorBone()
        {
            BoneAnimInfo ancestorBone = this.Parent;
            while (ancestorBone != null)
            {
                if (!ancestorBone.IsReservedToRemove)
                {
                    break;
                }
                ancestorBone = ancestorBone.Parent;
            }
            return ancestorBone;
        }
        public int Index
        {
            get
            {
                return this.BoneAnim.index;
            }
        }

        public bool RenderMatrix
        {
            get
            {
                return this.BoneAnim.rigid_body ||
                    (this.BoneAnim.matrix_index[0] != -1) ||
                    (this.BoneAnim.matrix_index[1] != -1);
            }
            set
            {
                throw new NotImplementedException();
            }
        }

        public readonly skeletal_anim_infoType SkeletalAnimInfo;
        public readonly bone_animType BoneAnim;
        public readonly List<BoneAnimInfo> Children = new List<BoneAnimInfo>();
        public BoneAnimInfo Parent { get; set; }
        public bool IsReservedToRemove { get; set; }
        public int NewIndex { get; set; } = -1;
    }
}
