﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public class ModelCompressUtility
    {
        /***
         * BoneInfo をソートするためのデータ
         */
        private class BoneInfoNode : IComparable
        {
            public BoneInfo Bone { get; set; }
            public List<BoneInfoNode> Children { get; set; } = new List<BoneInfoNode>();

            public BoneInfoNode(BoneInfo bone)
            {
                Bone = bone;
            }

            public int CompareTo(object obj)
            {
                BoneInfoNode other = (BoneInfoNode)obj;
                return string.CompareOrdinal(Bone.Bone.name, other.Bone.Bone.name);
            }
            // 参照している BoneInfo に新しいインデックスを設定します。
            public int SetNewIndex(int index)
            {
                // インデックスは深度優先で付ける。
                // 兄弟ボーン間ではアルファベット順に付けていく。
                // BoneInfoNode.Child はこの時点でソート済み。
                Bone.NewIndex = index++;
                foreach (BoneInfoNode child in Children)
                {
                    index = child.SetNewIndex(index);
                }
                return index;
            }

            // 削除されない BoneInfo の新しい階層構造を取得します。
            // BoneInfoNode の Children はボーン名のアルファベット順でソート済みです。
            public static BoneInfoNode GetHierarchy(BoneInfo rootBone)
            {
                if (rootBone == null)
                {
                    return null;
                }

                // rootBone には削除されないボーンが指定されていなければならない。
                Nintendo.Foundation.Contracts.Assertion.Operation.True(!rootBone.IsReservedToRemove);

                BoneInfoNode boneNode = new BoneInfoNode(rootBone);
                foreach (BoneInfo child in rootBone.Children)
                {
                    GetHierarchySub(child, boneNode);
                }
                // 子供を列挙した後にアルファベット順にソートする
                boneNode.Children.Sort();
                return boneNode;
            }
            // 階層構造作成のサブルーチンです。
            // 子階層に向かって再帰的に呼び出されていきます。
            private static void GetHierarchySub(BoneInfo bone, BoneInfoNode parentNode)
            {
                BoneInfoNode curNode;
                if (!bone.IsReservedToRemove)
                {
                    curNode = new BoneInfoNode(bone);
                    parentNode.Children.Add(curNode);
                }
                else
                {
                    // 削除されるボーンの場合は新たに BoneInfoNode を作成せずに
                    // 親のオブジェクトをそのまま使用する。
                    // そうすることによってこのボーンを削除された後の階層構造を再現できる。
                    curNode = parentNode;
                }

                foreach (BoneInfo child in bone.Children)
                {
                    GetHierarchySub(child, curNode);
                }

                // 子供を列挙した後にアルファベット順にソートする
                if (curNode != parentNode)
                {
                    curNode.Children.Sort();
                }
            }
        }

        /***
         * 新しい BoneInfo のインデックスを設定します。
         * rootBone を基点にして下位階層の削除されない BoneInfo に新しいインデックスを設定します。
         * そのため、BoneInfo.RemoveFlag がすでに設定されている必要があります。
         * ここで設定されるインデックスはソート済みの値です。
         * 中間ファイルに出力する際にはこのインデックスの順番通りにソートする必要があります。
         */
        public static void SetNewIndex(BoneInfo rootBone)
        {
            // まずは削除されないボーンで出力用の階層構造を作成する
            BoneInfoNode rootNode = BoneInfoNode.GetHierarchy(rootBone);
            if (rootNode != null)
            {
                // その階層構造の順番に従って 0 から順にインデックスを割り振る
                rootNode.SetNewIndex(0);
            }
        }
        /***
         * ボーンのインデックスを比較します。
         * ボーン配列内を出力順にソートする時に使用します。
         */
        private static int CompareBoneIndex(boneType lhs, boneType rhs)
        {
            if (lhs.index == rhs.index)
            {
                return 0;
            }
            else if (lhs.index < rhs.index)
            {
                return -1;
            }
            else
            {
                return 1;
            }
        }
        /***
         * モデルから不要なボーンを削除します。
         * BoneInfo.RemoveFlag が true になっているボーンをモデルのボーン配列から削除します。
         * その時にボーンのインデックスとペアレントの情報も一緒に更新します。
         * この関数を実行した後は BoneInfo と参照する boneType の階層構造の情報は一致しなくなります。
         */
        public static void RemoveBones(modelType model, BoneInfo[] boneInfos)
        {
            // 新しいボーン配列の作成
            List<boneType> newBones = new List<boneType>();
            foreach (BoneInfo boneInfo in boneInfos)
            {
                if (!boneInfo.IsReservedToRemove)
                {
                    boneType bone = boneInfo.Bone;

                    // 新しい配列内のインデックスを設定する
                    bone.index = boneInfo.NewIndex;
                    // 親が削除される場合は親の名前を変更する。
                    if (boneInfo.Parent != null && boneInfo.Parent.IsReservedToRemove)
                    {
                        BoneInfo newParent = boneInfo.GetIncompressibleAncestorBone();
                        if (newParent != null)
                        {
                            bone.parent_name = newParent.Bone.name;
                        }
                        else
                        {
                            bone.parent_name = string.Empty;
                        }
                    }
                    newBones.Add(bone);
                }
            }
            // ボーンを出力順にソートする
            newBones.Sort(CompareBoneIndex);
            // 新しいボーン配列への差し替え
            model.skeleton.bone_array.length = newBones.Count;
            model.skeleton.bone_array.bone = newBones.ToArray();
        }

        /***
         * ボーンパレットを作成します。
         * boneType の matrix_index 属性の値が設定されているボーンを収集し
         * BonePalette に参照を追加します。
         * BonePalette 内の配列は頂点属性のマトリクスインデックスに格納されている
         * インデックスにて参照されます。
         */
        public static BonePalette CreateBonePalette(BoneInfo[] boneInfos, modelType model)
        {
            var bonePltt = new BonePalette(model.model_info.smooth_skinning_matrix + model.model_info.rigid_skinning_matrix);
            foreach (var boneInfo in boneInfos)
            {
                boneType bone = boneInfo.Bone;
                // matrix_index[0] に -1 以外が設定されていると
                // スムーススキンのボーンとして使用されている。
                int smoothSkinIdx = bone.matrix_index[0];
                if (smoothSkinIdx != -1)
                {
                    bonePltt.SmoothSkinPalette[smoothSkinIdx] = boneInfo;
                }
                // matrix_index[1] に -1 以外が設定されていると
                // スムーススキンのボーンとして使用されている。
                int rigidSkinIdx = bone.matrix_index[1];
                if (rigidSkinIdx != -1)
                {
                    bonePltt.RigidSkinPalette[rigidSkinIdx] = boneInfo;
                }
            }
            return bonePltt;
        }

        /**
         * 子供のボーンリストを得ます。
         */
        public static void GetBoneDescendants(
            BoneInfo boneInfo,
            List<BoneInfo> dst)
        {
            foreach (BoneInfo child in boneInfo.Children)
            {
                dst.Add(child);
                GetBoneDescendants(child, dst);
            }
        }

        /**
         * 間引き後のノード間の参照関係を求めます。
         */
        public static void GetThinOutBoneRef(
            BoneInfo[] boneInfos,
            out BoneInfo[] parent)
        {
            int boneSize = boneInfos.Length;
            parent = new BoneInfo[boneSize];
            for (int i = 0; i < boneSize; ++i)
            {
                if (boneInfos[i].IsReservedToRemove)
                {
                    continue;
                }
                //新しい親ノードIDを求める
                BoneInfo bone = boneInfos[i].Parent;
                int count = 0;
                for (; bone != null && count < boneSize; ++count)
                {
                    if (!bone.IsReservedToRemove)
                    {
                        parent[i] = bone;
                        break;
                    }
                    else
                    {
                        bone = bone.Parent;
                    }
                }
                if (count == boneSize)
                {
                    throw new Exception("<bone> has cyclic indexing.");
                }
            }
        }

        /**
         * ancBoneで指す<bone>がboneで指す<bone>の先祖階層にあるか調べる。
         */
        private static bool IsAncestorBone(BoneInfo bone, BoneInfo ancBone)
        {
            BoneInfo bi = bone;
            while (bi != null)
            {
                BoneInfo parent = bi.Parent;
                if (parent == null)
                {
                    return false;
                }
                else if (parent == ancBone)
                {
                    return true;
                }
                bi = parent;
            }
            return false;
        }

        // fromBoneInfo から toBoneInfo までたどっていったボーンのリストを取得します。
        // 返されるリストは、先頭に fromBoneInfo の親ボーンが設定されます。
        // リストには toBoneInfo は含まれません。
        private static void GetBoneAncestors(
            BoneInfo fromBoneInfo,
            BoneInfo toBoneInfo,
            List<BoneInfo> list)
        {
            BoneInfo bone = fromBoneInfo;
            while (bone != null)
            {
                // リストに toBoneInfo を含めないようにするために、
                // 親が toBoneInfo であれば探索を打ち切る。
                if (bone.Parent == null || bone.Parent == toBoneInfo)
                {
                    break;
                }
                list.Add(bone.Parent);
                bone = bone.Parent;
            }
        }

        private static Matrix44 GetBoneTransformStandard(modelType model, BoneInfo boneInfo)
        {
            // boneInfo からルートノードまでのボーンリストを作成する
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<BoneInfo> bonePath = new List<BoneInfo>();
            bonePath.Add(boneInfo);
            BoneInfo parent = boneInfo.Parent;
            int boneSize = model.skeleton.bone_array.bone.Length;
            int count = 0;
            for (; count < boneSize && parent != null; ++count)
            {
                bonePath.Add(parent);
                parent = parent.Parent;
            }
            // 親をたどっていった数とモデル中のボーン数が一致すると
            // 参照が循環しているということなのでエラーとする。
            if (count == boneSize)
            {
                throw new Exception("<bone> has cyclic indexing.");
            }

            // 変換はルートノードから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();

            // ルートノードから boneInfo までの変換を合成する。
            Matrix44 transform = Matrix44.Identity;
            Matrix44 s = new Matrix44();
            Matrix44 r = new Matrix44();
            Matrix44 t = new Matrix44();
            foreach (BoneInfo bone in bonePath)
            {
                boneType b = bone.Bone;
                if (bone.SkeletonInfo.rotate_mode == skeleton_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 euler = EulerXyz.ToRadian(new Vector3(
                        b.rotate[0],
                        b.rotate[1],
                        b.rotate[2]));
                    EulerXyz.SetRotate(r, euler.X, euler.Y, euler.Z);
                }
                else
                {
                    r.SetRotate(new Quaternion(b.rotate[0], b.rotate[1], b.rotate[2], b.rotate[3]));
                }
                s.SetScale(new Vector3(b.scale));
                t.SetTranslate(new Vector3(b.translate));
                transform *= t * r * s;
            }

            return transform;
        }

        private static Matrix44 GetBoneTransform_Standard(
            modelType model,
            BoneInfo fromBoneInfo,
            BoneInfo toBoneInfo)
        {
            // fromBoneInfo から toBoneInfo までのボーンリストを作成する。
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<BoneInfo> bonePath = new List<BoneInfo>();
            bonePath.Add(fromBoneInfo);
            // fromBoneInfo から toBoneInfo までのボーンリストを取得する。
            // 得られた bonePath には toBoneInfo は含まれていない。
            GetBoneAncestors(fromBoneInfo, toBoneInfo, bonePath);

            // 変換は上位階層のボーンから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();

            // toBoneInfo の下位階層から boneInfo までの変換を合成する。
            Matrix44 transform = Matrix44.Identity;
            Matrix44 s = new Matrix44();
            Matrix44 r = new Matrix44();
            Matrix44 t = new Matrix44();
            foreach (BoneInfo boneInfo in bonePath)
            {
                boneType bone = boneInfo.Bone;
                if (boneInfo.SkeletonInfo.rotate_mode == skeleton_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 euler = EulerXyz.ToRadian(new Vector3(
                        bone.rotate[0],
                        bone.rotate[1],
                        bone.rotate[2]));
                    EulerXyz.SetRotate(r, euler.X, euler.Y, euler.Z);
                }
                else
                {
                    r.SetRotate(new Quaternion(bone.rotate[0], bone.rotate[1], bone.rotate[2], bone.rotate[3]));
                }
                s.SetScale(new Vector3(bone.scale));
                t.SetTranslate(new Vector3(bone.translate));
                transform *= t * r * s;
            }

            return transform;
        }

        private static Matrix44 GetBoneTransformMaya(
            modelType model,
            BoneInfo boneInfo)
        {
            // boneInfo からルートノードまでのボーンリストを作成する
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<BoneInfo> bonePath = new List<BoneInfo>();
            bonePath.Add(boneInfo);
            BoneInfo parent = boneInfo.Parent;
            int boneSize = model.skeleton.bone_array.bone.Length;
            int count = 0;
            for (; count < boneSize && parent != null; ++count)
            {
                bonePath.Add(parent);
                parent = parent.Parent;
            }
            // 親をたどっていった数とモデル中のボーン数が一致すると
            // 参照が循環しているということなのでエラーとする。
            if (count == boneSize)
            {
                throw new Exception("<bone> has cyclic indexing.");
            }

            // 変換はルートノードから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();

            // ルートノードから boneInfo までの変換を合成する。
            Matrix44 tr = Matrix44.Identity;
            Matrix44 psm = new Matrix44(); // 親のスケール行列
            Matrix44 rm = new Matrix44();
            Matrix44 tm = new Matrix44();
            Vector3 s = new Vector3(1.0f, 1.0f, 1.0f);
            foreach (BoneInfo bone in bonePath)
            {
                boneType b = bone.Bone;

                // 回転行列を求める。
                if (boneInfo.SkeletonInfo.rotate_mode == skeleton_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 euler = EulerXyz.ToRadian(new Vector3(
                        b.rotate[0],
                        b.rotate[1],
                        b.rotate[2]));
                    EulerXyz.SetRotate(rm, euler.X, euler.Y, euler.Z);
                }
                else
                {
                    rm.SetRotate(new Quaternion(b.rotate[0], b.rotate[1], b.rotate[2], b.rotate[3]));
                }
                tm.SetTranslate(new Vector3(b.translate));

                // scale_compensate の設定に従って変換を求める。
                if (b.scale_compensate)
                {
                    // scale_compensate が ON の場合は親のスケールを translate にだけ適用する
                    tm[0, 3] *= s.X;
                    tm[1, 3] *= s.Y;
                    tm[2, 3] *= s.Z;
                    tr *= tm * rm;
                }
                else
                {
                    psm.SetScale(s);
                    tr *= psm * tm * rm;
                }
                s.Set(b.scale);
            }
            // この時点で s には boneInfo.Bone.scale が設定されている。
            // 最後に boneInfo のスケールを適用する。
            Matrix44 sm = new Matrix44();
            sm.SetScale(s);
            return tr * sm;
        }

        private static Matrix44 GetBoneTransform_Maya(
            modelType model,
            BoneInfo fromBoneInfo,
            BoneInfo toBoneInfo)
        {
            // fromBoneInfo から toBoneInfo までのボーンリストを作成する。
            // ボーンリストの先頭には引数で渡されたボーンを登録しておく。
            List<BoneInfo> bonePath = new List<BoneInfo>();
            bonePath.Add(fromBoneInfo);
            // fromBoneInfo から toBoneInfo までのボーンリストを取得する。
            // 得られた bonePath には toBoneInfo は含まれていない。
            GetBoneAncestors(fromBoneInfo, toBoneInfo, bonePath);

            // 変換は上位階層のボーンから順番に計算していくので、ボーンリストの順番を反転させる。
            bonePath.Reverse();

            // ルートノードから boneInfo までの変換を合成する。
            Matrix44 tr = Matrix44.Identity;
            Matrix44 psm = new Matrix44(); // 親のスケール行列
            Matrix44 rm = new Matrix44();
            Matrix44 tm = new Matrix44();
            boneType toBone = toBoneInfo.Bone;
            Vector3 s = new Vector3(
                1.0f / toBone.scale[0],
                1.0f / toBone.scale[1],
                1.0f / toBone.scale[2]);

            // toBoneInfo の子供の変換を求める
            {
                BoneInfo bone = bonePath[0];
                boneType b = bone.Bone;

                // 回転行列を求める。
                if (bone.SkeletonInfo.rotate_mode == skeleton_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 eulerBi = EulerXyz.ToRadian(new Vector3(
                        b.rotate[0],
                        b.rotate[1],
                        b.rotate[2]));
                    EulerXyz.SetRotate(rm, eulerBi.X, eulerBi.Y, eulerBi.Z);
                }
                else
                {
                    rm.SetRotate(new Quaternion(b.rotate[0], b.rotate[1], b.rotate[2], b.rotate[3]));
                }
                tm.SetTranslate(new Vector3(b.translate));

                if (b.scale_compensate)
                {
                    psm.SetScale(s);
                    tr = tm * psm * rm;
                }
                else
                {
                    tr = tm * rm;
                }
                s.Set(b.scale);

                // 最初のボーンの変換は既に求めたので、ボーンリストからそのボーンを削除しておく。
                bonePath.RemoveAt(0);
            }
            foreach (BoneInfo bone in bonePath)
            {
                boneType b = bone.Bone;
                // 回転行列を求める。
                if (bone.SkeletonInfo.rotate_mode == skeleton_info_rotate_modeType.euler_xyz)
                {
                    // CAUTION!!
                    // bone.rotate は４要素の配列のため、Vector3 のコンストラクタに配列を渡すと
                    // 例外が発生するので注意。
                    Vector3 euler = EulerXyz.ToRadian(new Vector3(
                        b.rotate[0],
                        b.rotate[1],
                        b.rotate[2]));
                    EulerXyz.SetRotate(rm, euler.X, euler.Y, euler.Z);
                }
                else
                {
                    rm.SetRotate(new Quaternion(b.rotate[0], b.rotate[1], b.rotate[2], b.rotate[3]));
                }
                tm.SetTranslate(new Vector3(b.translate));

                // scale_compensate の設定に従って変換を求める。
                if (b.scale_compensate)
                {
                    // scale_compensate が ON の場合は親のスケールを translate にだけ適用する
                    tm[0, 3] *= s.X;
                    tm[1, 3] *= s.Y;
                    tm[2, 3] *= s.Z;
                    tr *= tm * rm;
                }
                else
                {
                    psm.SetScale(s);
                    tr *= psm * tm * rm;
                }
                s.Set(b.scale);
            }
            // この時点で s には boneInfo.Bone.scale が設定されている。
            // 最後に boneInfo のスケールを適用する。
            Matrix44 sm = new Matrix44();
            sm.SetScale(s);
            return tr * sm;
        }

        private static Matrix44 GetBoneTransformSoftimage(
            modelType model,
            BoneInfo boneInfo)
        {
            return Matrix44.Identity;
        }

        private static Matrix44 GetBoneTransform_Softimage(
            modelType model,
            BoneInfo fromBoneInfo,
            BoneInfo toBoneInfo)
        {
            return Matrix44.Identity;
        }

        public static Matrix44 GetBoneTransform(modelType model, BoneInfo boneInfo)
        {
            if (boneInfo == null)
            {
                return Matrix44.Identity;
            }
            switch (model.skeleton.skeleton_info.scale_mode)
            {
                case skeleton_info_scale_modeType.standard:
                    return GetBoneTransformStandard(model, boneInfo);
                case skeleton_info_scale_modeType.maya:
                    return GetBoneTransformMaya(model, boneInfo);
                case skeleton_info_scale_modeType.softimage:
                    return GetBoneTransformSoftimage(model, boneInfo);
                default:
                    throw new Exception("<skeleton_info> has unexpected value of 'scale_mode'.");
            }
        }

        private static Matrix44 GetBoneTransform_Auto(
            modelType model,
            BoneInfo fromBoneInfo,
            BoneInfo toBoneInfo)
        {
            switch (model.skeleton.skeleton_info.scale_mode)
            {
                case skeleton_info_scale_modeType.standard:
                    return GetBoneTransform_Standard(model, fromBoneInfo, toBoneInfo);
                case skeleton_info_scale_modeType.maya:
                    return GetBoneTransform_Maya(model, fromBoneInfo, toBoneInfo);
                case skeleton_info_scale_modeType.softimage:
                    return GetBoneTransform_Softimage(model, fromBoneInfo, toBoneInfo);
                default:
                    throw new Exception("<skeleton_info> has unexpected value of 'scale_mode'.");
            }
        }

        public static Matrix44 GetBoneTransform(
            modelType model,
            BoneInfo fromBoneInfo,
            BoneInfo toBoneInfo)
        {
            if (fromBoneInfo != null && toBoneInfo != null)
            {
                if (IsAncestorBone(fromBoneInfo, toBoneInfo))
                {
                    return GetBoneTransform_Auto(model, fromBoneInfo, toBoneInfo);
                }
                if (IsAncestorBone(toBoneInfo, fromBoneInfo))
                {
                    Matrix44 tr = GetBoneTransform_Auto(model, toBoneInfo, fromBoneInfo);
                    tr.Invert();
                    return tr;
                }
            }
            if (fromBoneInfo != toBoneInfo)
            {
                Matrix44 toBoneTransform = GetBoneTransform(model, toBoneInfo);
                Matrix44 fromBoneTransform = GetBoneTransform(model, fromBoneInfo);
                toBoneTransform.Invert();
                return toBoneTransform * fromBoneTransform;
            }
            else
            {
                return Matrix44.Identity;
            }
        }

        // ポジションの変換から法線の変換を求める
        public static Matrix44 GetNormalTransform(Matrix44 posTransform)
        {
            Matrix44 nrmTransform = new Matrix44(posTransform);
            nrmTransform.Invert();
            nrmTransform.Transpose();
            nrmTransform[0, 3] = 0.0f;
            nrmTransform[1, 3] = 0.0f;
            nrmTransform[2, 3] = 0.0f;
            return nrmTransform;
        }

        // ポジションの変換から接線、従法線の変換を求める
        public static Matrix44 GetTangentBinormalTransform(Matrix44 posTransform)
        {
            Matrix44 tangentTransform = new Matrix44(posTransform);
            tangentTransform[0, 3] = 0.0f;
            tangentTransform[1, 3] = 0.0f;
            tangentTransform[2, 3] = 0.0f;
            return tangentTransform;
        }

        // モデル内の stream_index を新しい値に置きかえる
        public static void ReplaceStreamIndexInModel(modelType model, int[] newStreamIndex)
        {
            // fmd には以下のデータにstream_index が含まれているので、
            // それらのインデックスを新しい値に置き換えていく。
            // - modelType.vertex_array.vertex.vtx_attrib_array.vtx_attrib
            // - modelType.shape_array.shape.mesh

            // vtx_attrib の stream_index を置きかえる
            if (model.vertex_array != null &&
                model.vertex_array.vertex != null)
            {
                foreach (vertexType vertex in model.vertex_array.vertex)
                {
                    foreach (vtx_attribType vtxAttrib in vertex.vtx_attrib_array.vtx_attrib)
                    {
                        int newIndex = newStreamIndex[vtxAttrib.stream_index];
                        // stream は共有されないので必ずインデックスが -1 以外になっているはず
                        Nintendo.Foundation.Contracts.Assertion.Operation.True(newIndex != -1);
                        vtxAttrib.stream_index = newIndex;
                    }
                }
            }

            // mesh の stream_index を置きかえる
            if (model.shape_array != null &&
                model.shape_array.shape != null)
            {
                foreach (shapeType shape in model.shape_array.shape)
                {
                    meshType mesh = shape.mesh_array.mesh[0];
                    int newIndex = newStreamIndex[mesh.stream_index];
                    // stream は共有されないので必ずインデックスが -1 以外になっているはず
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(newIndex != -1);
                    mesh.stream_index = newIndex;
                }
            }
        }

        // シェープを変換する
        // - エンベロープをリジッドボディに変換
        public static void ConvertShape(
            modelType model,
            List<G3dStream> streams,
            BoneInfo toBone,
            ShapeInfo shapeInfo)
        {
            if (model.vertex_array == null ||
                model.vertex_array.vertex == null) { return; }

            // リジッドボディ以外のシェープではマトリクスインデックスと頂点ウェイトを削除する
            if (shapeInfo.SkinningMode == ShapeInfo.ShapeSkinningMode.RigidBody) { return; }

            shapeType shape = shapeInfo.Shape;
            vertexType vertex =
                model.vertex_array.vertex[shape.shape_info.vertex_index];
            // ストリームの削除フラグ。true が設定されるとストリームが削除される。
            bool[] removeStreamFlag = new bool[streams.Count];

            // シェープが参照する <vertex> からマトリクスインデックスと頂点ウェイトの
            // <vtx_attrib>, <input> を削除する。

            // newVtxAttribIdx には新しい <vtx_attrib> のインデクスが設定される。
            // 削除された要素のインデックスは -1 が設定される。
            int[] newVtxAttribIdx = new int[vertex.vtx_attrib_array.length];

            vtx_attribType[] vtxAttribs = vertex.vtx_attrib_array.vtx_attrib;
            int vtxAttribCount = 0;
            for (int i = 0; i < vtxAttribs.Length; i++)
            {
                vtx_attribType vtxAttrib = vtxAttribs[i];
                switch (vtxAttrib.name.Substring(0, 2))
                {
                    case "_i":
                    case "_w":
                        removeStreamFlag[vtxAttrib.stream_index] = true;
                        newVtxAttribIdx[i] = -1;
                        break;
                    default:
                        newVtxAttribIdx[i] = vtxAttribCount++;
                        break;
                }
            }

            // <vtx_attrib> を削除する
            vtx_attribType[] newVtxAttribs = new vtx_attribType[vtxAttribCount];
            foreach (vtx_attribType vtxAttrib in vtxAttribs)
            {
                int newIndex = newVtxAttribIdx[vtxAttrib.attrib_index];
                if (newIndex != -1)
                {
                    vtxAttrib.attrib_index = newIndex;
                    newVtxAttribs[newIndex] = vtxAttrib;
                }
            }
            vertex.vtx_attrib_array.length = vtxAttribCount;
            vertex.vtx_attrib_array.vtx_attrib = newVtxAttribs;

            // <input> を削除する
            if (vertex.vtx_buffer_array != null &&
                vertex.vtx_buffer_array.vtx_buffer != null)
            {
                foreach (vtx_bufferType vtxBuffer in vertex.vtx_buffer_array.vtx_buffer)
                {
                    // 残る input を数える
                    inputType[] inputs = vtxBuffer.input_array.input;
                    inputType[] newInputs = new inputType[vtxBuffer.input_array.length];
                    int inputCount = 0;
                    foreach (inputType input in vtxBuffer.input_array.input)
                    {
                        int newIndex = newVtxAttribIdx[input.attrib_index];
                        if (newIndex != -1)
                        {
                            input.index = inputCount;
                            input.attrib_index = newIndex;
                            newInputs[inputCount] = input;
                            inputCount++;
                        }
                    }
                    // input_array を置き換える
                    Array.Resize(ref newInputs, inputCount);
                    vtxBuffer.input_array.length = inputCount;
                    vtxBuffer.input_array.input = newInputs;
                }
            }

            // マトリクスインデックスと頂点ウェイトのストリームを削除する
            int[] newStreamIndex = new int[streams.Count];
            int streamCount = 0;
            int streamSize = streams.Count;
            for (int iStream = 0; iStream < streamSize; iStream++)
            {
                if (!removeStreamFlag[iStream])
                {
                    // モデル全体のストリームのインデックスを変更するために新しいインデックスを記録しておく
                    newStreamIndex[iStream] = streamCount++;
                }
                else
                {
                    newStreamIndex[iStream] = -1;
                    // ストリームを削除する
                    streams.RemoveAt(streamCount);
                }
            }

            // モデル全体のストリームのインデックスを新しい値に置き換える
            ModelCompressUtility.ReplaceStreamIndexInModel(model, newStreamIndex);

            // リジッドボディに変換する
            shape.shape_info.vertex_skinning_count = 0;
        }

        // キーシェイプが使用する頂点情報を削除する
        private static void RemoveKeyShapeVertex(shapeType shape, vertexType vertex)
        {
            if (shape.key_shape_array == null || shape.key_shape_array.length == 0)
            {
                return;
            }

            // key_shape が使用している頂点要素を削除する
            // 最初の key_shape をベースシェイプとし、その key_shape が参照している頂点要素のみ残す。
            int vtxAttribLen = vertex.vtx_attrib_array.length;
            bool[] removeVertex = new bool[vtxAttribLen];
            for (int i = 0; i < vtxAttribLen; i++)
            {
                removeVertex[i] = false;
            }

            // 先頭の key_shape はベースシェイプとみなすので、
            // 先頭以外が参照する頂点要素にフラグを設定する
            for (int i = 1; i < shape.key_shape_array.length; i++)
            {
                key_shapeType keyShape = shape.key_shape_array.key_shape[i];
                foreach (target_attribType target in keyShape.target_attrib_array.target_attrib)
                {
                    int attribIndex = Array.FindIndex(
                        vertex.vtx_attrib_array.vtx_attrib,
                        delegate(vtx_attribType attr)
                        {
                            return (string.Compare(attr.name, target.attrib_name) == 0);
                        });
                    if (attribIndex >= 0)
                    {
                        removeVertex[attribIndex] = true;
                    }
                }
            }

            // 新しい vtx_attrib_array を作成する
            int[] newAttribIndex = new int[vtxAttribLen];
            for (int i = 0; i < vtxAttribLen; i++)
            {
                newAttribIndex[i] = -1;
            }

            List<vtx_attribType> newVtxAttrib = new List<vtx_attribType>();
            int vtxCount = 0;
            for (int i = 0; i < vtxAttribLen; i++)
            {
                if (!removeVertex[i])
                {
                    newAttribIndex[i] = vtxCount;
                    vertex.vtx_attrib_array.vtx_attrib[i].attrib_index = vtxCount;
                    newVtxAttrib.Add(vertex.vtx_attrib_array.vtx_attrib[i]);
                    vtxCount++;
                }
            }
            vertex.vtx_attrib_array.length = newVtxAttrib.Count;
            vertex.vtx_attrib_array.vtx_attrib = newVtxAttrib.ToArray();

            // 新しい vtx_buffer を作成する
            foreach (vtx_bufferType vtxBuffer in vertex.vtx_buffer_array.vtx_buffer)
            {
                List<inputType> newInput = new List<inputType>();
                foreach (inputType input in vtxBuffer.input_array.input)
                {
                    if (!removeVertex[input.attrib_index])
                    {
                        input.attrib_index = newAttribIndex[input.attrib_index];
                        newInput.Add(input);
                    }
                }
                vtxBuffer.input_array.input = newInput.ToArray();
                vtxBuffer.input_array.length = newInput.Count;
            }
        }

        // キーシェイプを削除する
        public static void RemoveKeyShape(modelType model)
        {
            if (model.shape_array == null ||
                model.shape_array.shape == null) { return; }
            if (model.vertex_array == null ||
                model.vertex_array.vertex == null) { return; }

            foreach (shapeType shape in model.shape_array.shape)
            {
                // key_shape が参照する vtx_attrib と vtx_buffer を削除する
                vertexType vertex =
                    model.vertex_array.vertex[shape.shape_info.vertex_index];
                RemoveKeyShapeVertex(shape, vertex);

                // key_shape を削除する
                shape.key_shape_array = null;
            }
        }

        // シェイプをソートする
        public static void SortShape(modelType model)
        {
            if (model.shape_array == null ||
                model.shape_array.shape == null) { return; }
            if (model.vertex_array == null ||
                model.vertex_array.vertex == null) { return; }

            // ボーン名をキーにしたボーンの辞書を作成する。
            Dictionary<string, boneType> boneTable =
                new Dictionary<string, boneType>();
            foreach (boneType bone in model.skeleton.bone_array.bone)
            {
                boneTable.Add(bone.name, bone);
            }

            // シェイプをソートする
            Array.Sort(model.shape_array.shape, delegate(shapeType lhs, shapeType rhs)
                {
                    if (lhs.shape_info.bone_name == rhs.shape_info.bone_name)
                    {
                        return string.CompareOrdinal(lhs.name, rhs.name);
                    }
                    else
                    {
                        boneType lbone = boneTable[lhs.shape_info.bone_name];
                        boneType rbone = boneTable[rhs.shape_info.bone_name];
                        if (lbone.index < rbone.index)
                        {
                            return -1;
                        }
                        else if (lbone.index == rbone.index)
                        {
                            return 0;
                        }
                        else
                        {
                            return 1;
                        }
                    }
                });

            // シェイプのストリームインデックスの更新と頂点データをソートする
            List<vertexType> newVertex = new List<vertexType>();
            vertexType[] oldVertex = model.vertex_array.vertex;
            int shapeCount = 0;
            foreach (shapeType shape in model.shape_array.shape)
            {
                // インデックスを更新する
                shape.index = shapeCount;

                // 新しい頂点データ配列を作成する
                int vertexIndex = shape.shape_info.vertex_index;
                shape.shape_info.vertex_index = newVertex.Count;

                // 頂点データを更新する
                vertexType vertex = oldVertex[vertexIndex];
                vertex.vertex_index = newVertex.Count;
                newVertex.Add(oldVertex[vertexIndex]);

                shapeCount++;
            }
            model.vertex_array.vertex = newVertex.ToArray();
            model.vertex_array.length = newVertex.Count;
        }
    }

    // ボーン圧縮のためのユーティリティー
    public class BoneInfo
    {
        public BoneInfo(boneType bone, skeletonInfoType skeletonInfo)
        {
            this.SkeletonInfo = skeletonInfo;
            this.Bone = bone;
        }

        public static BoneInfo[] CreateAndSetup(modelType model)
        {
            BoneInfo[] boneInfos = new BoneInfo[model.skeleton.bone_array.length];
            for (int boneIndex = 0; boneIndex < boneInfos.Length; ++boneIndex)
            {
                boneInfos[boneIndex] = new BoneInfo(model.skeleton.bone_array.bone[boneIndex], model.skeleton.skeleton_info);
            }
            BoneInfo.Setup(boneInfos);
            return boneInfos;
        }

        public static void Setup(BoneInfo[] boneInfos)
        {
            // ボーン検索のためのテーブル
            Dictionary<string, BoneInfo> boneInfoTable =
                new Dictionary<string, BoneInfo>();
            foreach (BoneInfo bi in boneInfos)
            {
                boneInfoTable.Add(bi.Bone.name, bi);
            }

            // 階層構造のセットアップ
            foreach (BoneInfo bi in boneInfos)
            {
                string parentName = bi.Bone.parent_name;
                if (!string.IsNullOrEmpty(parentName))
                {
                    BoneInfo parentInfo;
                    if (!boneInfoTable.TryGetValue(parentName, out parentInfo))
                    {
                        IfStrings.Throw("ModelCompressUtility_Error_ParentNotFound",
                            bi.Bone.name, parentName);
                    }
                    parentInfo.Children.Add(bi);
                    bi.Parent = parentInfo;
                }
            }

            // 兄弟ボーンを文字コード順にソートする
            foreach (BoneInfo bi in boneInfos)
            {
                bi.Children.Sort(
                    delegate (BoneInfo lhs, BoneInfo rhs)
                    {
                        return string.CompareOrdinal(lhs.Bone.name, rhs.Bone.name);
                    });
            }
        }

        // 削除されない子がいるか
        public bool HasChild()
        {
            foreach (BoneInfo boneInfo in this.Children)
            {
                if (!boneInfo.IsReservedToRemove) { return true; }
            }
            return false;
        }

        // 文字列変換
        public override string ToString()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendFormat("{0} {1} {2}→",
                !this.IsReservedToRemove ? "○" : "×",
                this.RenderMatrix ? "○" : "×",
                this.Bone.name);
            foreach (BoneInfo child in this.Children)
            {
                builder.AppendFormat("{0}{1},",
                    child.Bone.name,
                    child.RenderMatrix ? "○" : "×");
            }
            return builder.ToString();
        }

        public BoneInfo GetIncompressibleAncestorBone()
        {
            BoneInfo ancestorBone = this.Parent;
            while (ancestorBone != null)
            {
                if (!ancestorBone.IsReservedToRemove)
                {
                    break;
                }
                ancestorBone = ancestorBone.Parent;
            }
            return ancestorBone;
        }
        public int Index
        {
            get
            {
                return this.Bone.index;
            }
        }
        public bool RenderMatrix
        {
            get
            {
                if (this.Bone != null)
                {
                    return this.Bone.rigid_body ||
                        (this.Bone.matrix_index[0] != -1) ||
                        (this.Bone.matrix_index[1] != -1);
                }
                else
                {
                    return false;
                }
            }
        }

        public readonly skeletonInfoType SkeletonInfo;
        public readonly boneType Bone;
        public readonly List<BoneInfo> Children = new List<BoneInfo>();
        public BoneInfo Parent { get; set; }
        public bool IsReservedToRemove { get; set; }
        public int NewIndex { get; set; } = -1;
    }

    public class ShapeInfo
    {
        public enum ShapeSkinningMode
        {
            RigidBody,    // リジッドボディ
            RigidSkin,    // リジッドスキン
            SmoothSkin,    // スムーズスキン
        }

        public ShapeInfo(shapeType shape)
        {
            this.Shape = shape;
        }
        public void Setup(BoneInfo[] boneInfos)
        {
            foreach (BoneInfo bone in boneInfos)
            {
                if (bone.Bone.name == this.Shape.shape_info.bone_name)
                {
                    this.Bone = bone;
                    break;
                }
            }
        }
        public ShapeSkinningMode SkinningMode
        {
            get
            {
                int vtx_skin_count = Shape.shape_info.vertex_skinning_count;
                if (vtx_skin_count == 1)
                {
                    return ShapeSkinningMode.RigidSkin;
                }
                else if (vtx_skin_count >= 2)
                {
                    return ShapeSkinningMode.SmoothSkin;
                }
                else
                {
                    return ShapeSkinningMode.RigidBody;
                }
            }
        }

        public readonly shapeType Shape;
        // 参照先のボーン
        public BoneInfo Bone { get; set; }
    }

    public class BonePalette
    {
        private BoneInfo[] mSmoothSkinPalette;
        private BoneInfo[] mRigidSkinPalette;

        public BonePalette(int nbPalette)
        {
            mSmoothSkinPalette = new BoneInfo[nbPalette];
            mRigidSkinPalette = new BoneInfo[nbPalette];
        }

        public BoneInfo[] SmoothSkinPalette
        {
            get { return mSmoothSkinPalette; }
        }
        public BoneInfo[] RigidSkinPalette
        {
            get { return mRigidSkinPalette; }
        }
        public int Count
        {
            get { return mSmoothSkinPalette.Length; }
        }
    }

    public class VertexTransform
    {
        public Matrix44 mTransform { get; set; } = new Matrix44();
        public int mNbInfluencedVertex { get; set; } = 0;
        public int[] mNewIndex { get; set; }

        public VertexTransform()
        {
            mNewIndex = new int[0];
        }

        public VertexTransform(int nbVertex)
        {
            mNewIndex = new int[nbVertex];
            for (uint i = 0; i < nbVertex; i++)
            {
                mNewIndex[i] = -1;
            }
        }
    }

    public class TransformRequest
    {
        public int mSrcStreamIdx { get; set; } = 0;
        public List<ShapeInfo> mShapes { get; set; } = new List<ShapeInfo>();
        public List<VertexTransform> mVertexTransform { get; set; } = new List<VertexTransform>();
        public int mNbTotalVertex { get; set; } = 0;

        // 頂点座標の変換要求を作成する
        public static TransformRequest CreateRequest(
            modelType model,
            List<G3dStream> streams,
            BonePalette bonePltt,
            BoneInfo toBone,
            ShapeInfo shapeInfo,
            vtx_attribType vtxAttrib)
        {
            shapeType shape = shapeInfo.Shape;
            vertexType vertex = model.vertex_array.vertex[shape.shape_info.vertex_index];
            // 法線、接線、従法線の変換を行うかどうかを取得する。
            bool needNrmTrans = NeedNormalTransform(vtxAttrib);

            TransformRequest req = new TransformRequest();
            req.mSrcStreamIdx = vtxAttrib.stream_index;
            req.mShapes.Add(shapeInfo);

            G3dStream indexStream = streams[shape.mesh_array.mesh[0].stream_index];
            G3dStream vtxStream = streams[vtxAttrib.stream_index];
            int nbVertex = vtxStream.FloatData.Count / vtxStream.column;
            switch (shapeInfo.SkinningMode)
            {
                case ShapeInfo.ShapeSkinningMode.RigidBody:
                    // 単純メッシュなので、頂点は shape.shape_info.bone_index が
                    // 参照するボーンのローカル座標系で定義されている。
                    // そのため、頂点配列全体に同じ座標変換を適用する。
                    VertexTransform vtxTrans = new VertexTransform(nbVertex);
                    vtxTrans.mTransform =
                            ModelCompressUtility.GetBoneTransform(
                                model,
                                shapeInfo.Bone,
                                toBone);
                    if (needNrmTrans)
                    {
                        vtxTrans.mTransform =
                            ModelCompressUtility.GetNormalTransform(vtxTrans.mTransform);
                    }
                    // 参照している頂点にマークを付ける
                    foreach (int idx in indexStream.IntData)
                    {
                        vtxTrans.mNewIndex[idx] = 1;
                    }

                    req.mVertexTransform.Add(vtxTrans);
                    break;
                case ShapeInfo.ShapeSkinningMode.RigidSkin:
                    // リジッドスキンなので、頂点座標は各頂点が参照するボーンの
                    // ローカル座標系で定義されている。
                    vtx_attribType blendIdxAttrib = GetVertexAttribute(vertex, "_i");
                    G3dStream blendIdxStream = streams[blendIdxAttrib.stream_index];
                    // リジッドスキンならブレンドインデックスが存在しているはず。
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(blendIdxAttrib != null);

                    // このシェープで使用される変換の配列。
                    // この配列は BonePalette.RigidSkinPalette の配列と一対一に対応し、
                    // blendIdxStream の各要素はこの配列内の変換を指す。
                    VertexTransform[] vtxTransArray = new VertexTransform[bonePltt.Count];
                    foreach (int vtxIdx in indexStream.IntData)
                    {
                        VertexTransform rigidVtxTrans;
                        int blendIdx = blendIdxStream.IntData[vtxIdx];
                        if (vtxTransArray[blendIdx] == null)
                        {
                            rigidVtxTrans = new VertexTransform(nbVertex);
                            // BonePalette のボーンから toBone への変換を登録する
                            rigidVtxTrans.mTransform =
                                ModelCompressUtility.GetBoneTransform(
                                    model,
                                    bonePltt.RigidSkinPalette[blendIdx],
                                    toBone);
                            if (needNrmTrans)
                            {
                                rigidVtxTrans.mTransform =
                                    ModelCompressUtility.GetNormalTransform(rigidVtxTrans.mTransform);
                            }

                            vtxTransArray[blendIdx] = rigidVtxTrans;
                        }
                        else
                        {
                            rigidVtxTrans = vtxTransArray[blendIdx];
                        }

                        // rigidVtxTrans の変換を適用する頂点にマークを付ける
                        rigidVtxTrans.mNewIndex[vtxIdx] = 1;
                    }
                    // TransformRequest に作成した変換を登録する
                    foreach (VertexTransform tr in vtxTransArray)
                    {
                        if (tr != null)
                        {
                            req.mVertexTransform.Add(tr);
                        }
                    }
                    break;
                case ShapeInfo.ShapeSkinningMode.SmoothSkin:
                    // スムーススキンなので、頂点座標はモデル座標系(ワールド座標系)
                    // で定義されている。そのため、移動するボーンのローカル座標系に変換する。
                    VertexTransform skinVtxTrans = new VertexTransform(nbVertex);
                    skinVtxTrans.mTransform =
                            ModelCompressUtility.GetBoneTransform(model, toBone);
                    skinVtxTrans.mTransform.Invert();
                    if (needNrmTrans)
                    {
                        skinVtxTrans.mTransform =
                            ModelCompressUtility.GetNormalTransform(skinVtxTrans.mTransform);
                    }
                    // 参照している頂点にマークを付ける
                    foreach (int idx in indexStream.IntData)
                    {
                        skinVtxTrans.mNewIndex[idx] = 1;
                    }

                    req.mVertexTransform.Add(skinVtxTrans);
                    break;
            }

            // 変換する頂点を数える
            // VertexTransform.mNbInfluencedVertex にはその変換が適用される頂点数が、
            // TransformRequest.mNbTotalVertex には変換要求が持つ変換が適用される
            // 延べ頂点数が設定される。
            req.mNbTotalVertex = 0;
            foreach (VertexTransform vtxTrans in req.mVertexTransform)
            {
                foreach (int i in vtxTrans.mNewIndex)
                {
                    if (i > 0)
                    {
                        vtxTrans.mNbInfluencedVertex++;
                    }
                }
                req.mNbTotalVertex += vtxTrans.mNbInfluencedVertex;
            }
            // TODO: ストリームの複製を実装したら削除すること
            // 現在はストリームの複製を実装していないので、変換する頂点数と
            // ソースのストリームの頂点数は一致するはず。
            Nintendo.Foundation.Contracts.Assertion.Operation.True(req.mNbTotalVertex == nbVertex);

            return req;
        }

        // 頂点を変換する
        public void ApplyTransform(
            modelType model,
            List<G3dStream> streams,
            bool normalize)
        {
            G3dStream stream = streams[mSrcStreamIdx];
            // CAUTION
            // 頂点のストリームは接線と従法線の場合に float4 になる事があるので
            // 要素数を 3 と仮定せず、必ずストリームから column を取得すること。
            int column = stream.column;
            Nintendo.Foundation.Contracts.Assertion.Operation.True((mNbTotalVertex * column) == stream.FloatData.Count);
            float[] buffer = new float[mNbTotalVertex * column];
            List<float> vtxSrc = stream.FloatData;
            Vector3 errNrm = new Vector3(0, 1, 0);

            foreach (VertexTransform trans in mVertexTransform)
            {
                // なるべく値が変化しないようにするため、変換が単位行列ならば、そのまま値をコピーする。
                if (trans.mTransform.Equals(Matrix44.Identity))
                {
                    for (int i = 0; i < trans.mNewIndex.Length; i++)
                    {
                        int newIdx = trans.mNewIndex[i];
                        if (newIdx >= 0)
                        {
                            // TODO: ストリームの複製を実装したら変更すること
                            // まだストリームの複製を行わないので buffer の添字には i を使用する。
                            // 複製を行うようになったら出力先には newIdx を添字にするように変更する。
                            int ofs = i * column;
                            buffer[ofs + 0] = vtxSrc[ofs + 0];
                            buffer[ofs + 1] = vtxSrc[ofs + 1];
                            buffer[ofs + 2] = vtxSrc[ofs + 2];
                            // float4 の場合は４要素目をそのままコピーする
                            if (column == 4)
                            {
                                buffer[ofs + 3] = vtxSrc[ofs + 3];
                            }
                        }
                    }
                }
                else
                {
                    Vector3 vtx = new Vector3();
                    for (int i = 0; i < trans.mNewIndex.Length; i++)
                    {
                        int newIdx = trans.mNewIndex[i];
                        if (newIdx >= 0)
                        {
                            // TODO: ストリームの複製を実装したら変更すること
                            // まだストリームの複製を行わないので buffer の添字には i を使用する。
                            // 複製を行うようになったら出力先には newIdx を添字にするように変更する。
                            int ofs = i * column;
                            vtx.Set(vtxSrc[ofs + 0], vtxSrc[ofs + 1], vtxSrc[ofs + 2]);
                            vtx = trans.mTransform * vtx;
                            if (normalize)
                            {
                                vtx.SafeNormalize(errNrm);
                            }
                            buffer[ofs + 0] = vtx.X;
                            buffer[ofs + 1] = vtx.Y;
                            buffer[ofs + 2] = vtx.Z;
                            // float4 の場合は４要素目をそのままコピーする
                            if (column == 4)
                            {
                                buffer[ofs + 3] = vtxSrc[ofs + 3];
                            }
                        }
                    }
                }
            }

            // ストリームの float 配列を置きかえる
            stream.FloatData.Clear();
            stream.FloatData.AddRange(buffer);
        }

        // vertexType から name で指定されたタイプの頂点属性を取得する。
        private static vtx_attribType GetVertexAttribute(vertexType vertex, string name)
        {
            foreach (vtx_attribType attrib in vertex.vtx_attrib_array.vtx_attrib)
            {
                if (attrib.name.Substring(0, 2) == name)
                {
                    return attrib;
                }
            }
            return null;
        }

        // 法線用の変換が必要か取得する。
        private static bool NeedNormalTransform(vtx_attribType vtxAttrib)
        {
            switch (vtxAttrib.name.Substring(0, 2))
            {
                case "_n":
                case "_t":
                case "_b":
                    return true;
                default:
                    return false;
            }
        }
    }

    /***
     * TransformRequest を保持するクラスです。
     */
    public class TransformRequestHolder
    {
        public readonly modelType Model;
        public readonly List<G3dStream> Streams;
        public readonly BonePalette BonePltt;
        public readonly List<TransformRequest> PosRequests = new List<TransformRequest>();
        public readonly List<TransformRequest> NrmRequests = new List<TransformRequest>();
        public readonly List<TransformRequest> TngtRequests = new List<TransformRequest>();
        public readonly List<TransformRequest> BnrmRequests = new List<TransformRequest>();

        public TransformRequestHolder(
            modelType model,
            List<G3dStream> streams,
            BonePalette bonePltt)
        {
            Model = model;
            Streams = streams;
            BonePltt = bonePltt;
        }

        // 変換要求を追加する
        public void InsertTransformRequest(
            BoneInfo toBone,
            ShapeInfo shapeInfo)
        {
            if (Model.vertex_array == null ||
                Model.vertex_array.vertex == null) { return; }

            // MEMO:
            // NW4R ではここで同じ頂点ストリームに対する変換要求をマージし、同じ頂点ストリームに対して
            // 異なる変換を行う場合は頂点ストリームを伸長して１つの頂点ストリームに複数の座標系の
            // 頂点データを混在させるようにしていた。
            // 現状の NW4F では複数の shape が１つの vertex を参照する事はないため、マージ処理を行わない。
            // 将来的に複数の shape が１つの vertex を参照するようになったら同様のマージ処理を実装する。

            vertexType vertex =
                Model.vertex_array.vertex[shapeInfo.Shape.shape_info.vertex_index];
            foreach (vtx_attribType vtxAttrib in vertex.vtx_attrib_array.vtx_attrib)
            {
                switch (vtxAttrib.name.Substring(0, 2))
                {
                    case "_p":
                        TransformRequest posReq =
                            TransformRequest.CreateRequest(
                                Model,
                                Streams,
                                BonePltt,
                                toBone,
                                shapeInfo,
                                vtxAttrib);
                        PosRequests.Add(posReq);
                        break;
                    case "_n":
                        TransformRequest nrmReq =
                            TransformRequest.CreateRequest(
                                Model,
                                Streams,
                                BonePltt,
                                toBone,
                                shapeInfo,
                                vtxAttrib);
                        NrmRequests.Add(nrmReq);
                        break;
                    case "_t":
                        TransformRequest tgtReq =
                            TransformRequest.CreateRequest(
                                Model,
                                Streams,
                                BonePltt,
                                toBone,
                                shapeInfo,
                                vtxAttrib);
                        TngtRequests.Add(tgtReq);
                        break;
                    case "_b":
                        TransformRequest binrmReq =
                            TransformRequest.CreateRequest(
                                Model,
                                Streams,
                                BonePltt,
                                toBone,
                                shapeInfo,
                                vtxAttrib);
                        BnrmRequests.Add(binrmReq);
                        break;
                }
            }
        }

        public void ApplyTransform()
        {
            // 頂点座標を変換する
            G3dParallel.ForEach(PosRequests, delegate(TransformRequest req)
                {
                    req.ApplyTransform(Model, Streams, false);
                });

            // 頂点法線を変換する
            G3dParallel.ForEach(NrmRequests, delegate(TransformRequest req)
                {
                    req.ApplyTransform(Model, Streams, true);
                });

            // 接線を変換する
            G3dParallel.ForEach(TngtRequests, delegate(TransformRequest req)
                {
                    req.ApplyTransform(Model, Streams, true);
                });

            // 従法線を変換する
            G3dParallel.ForEach(BnrmRequests, delegate(TransformRequest req)
                {
                    req.ApplyTransform(Model, Streams, true);
                });
        }
    }
}
