﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    internal class BoneVisibilityAnimCompressUtility
    {
        /***
         * BoneVisAnimInfo をソートするためのデータ
         */
        private class BoneVisAnimInfoNode : IComparable
        {
            public BoneVisAnimInfo Anim { get; set; }
            public List<BoneVisAnimInfoNode> Children { get; set; } = new List<BoneVisAnimInfoNode>();

            public BoneVisAnimInfoNode(BoneVisAnimInfo animInfo)
            {
                Anim = animInfo;
            }

            public int CompareTo(object obj)
            {
                BoneVisAnimInfoNode other = (BoneVisAnimInfoNode)obj;
                return string.CompareOrdinal(Anim.BoneAnim.bone_name, other.Anim.BoneAnim.bone_name);
            }
            // 参照している BoneVisAnimInfo に新しいインデックスを設定します。
            public int SetNewIndex(int index)
            {
                // インデックスは深度優先で付ける。
                // 兄弟ボーン間ではアルファベット順に付けていく。
                // BoneVisAnimInfoNode.Child はこの時点でソート済み。
                Anim.NewIndex = index++;
                foreach (BoneVisAnimInfoNode child in Children)
                {
                    index = child.SetNewIndex(index);
                }
                return index;
            }

            // 削除されない BoneVisAnimInfo の新しい階層構造を取得します。
            // BoneVisAnimInfoNode の Children はボーン名のアルファベット順でソート済みです。
            public static BoneVisAnimInfoNode GetHierarchy(BoneVisAnimInfo rootBone)
            {
                if (rootBone == null)
                {
                    return null;
                }

                // rootBone には削除されないボーンが指定されていなければならない。
                Nintendo.Foundation.Contracts.Assertion.Operation.True(!rootBone.RemoveFlag);

                BoneVisAnimInfoNode boneNode = new BoneVisAnimInfoNode(rootBone);
                foreach (BoneVisAnimInfo child in rootBone.Children)
                {
                    GetHierarchySub(child, boneNode);
                }
                // 子供を列挙した後にアルファベット順にソートする
                boneNode.Children.Sort();
                return boneNode;
            }
            // 階層構造作成のサブルーチンです。
            // 子階層に向かって再帰的に呼び出されていきます。
            private static void GetHierarchySub(BoneVisAnimInfo bone, BoneVisAnimInfoNode parentNode)
            {
                BoneVisAnimInfoNode curNode;
                if (!bone.RemoveFlag)
                {
                    curNode = new BoneVisAnimInfoNode(bone);
                    parentNode.Children.Add(curNode);
                }
                else
                {
                    // 削除されるボーンの場合は新たに BoneVisAnimInfoNode を作成せずに
                    // 親のオブジェクトをそのまま使用する。
                    // そうすることによってこのボーンを削除された後の階層構造を再現できる。
                    curNode = parentNode;
                }

                foreach (BoneVisAnimInfo child in bone.Children)
                {
                    GetHierarchySub(child, curNode);
                }

                // 子供を列挙した後にアルファベット順にソートする
                if (curNode != parentNode)
                {
                    curNode.Children.Sort();
                }
            }
        }

        /***
         * 新しい BoneVisAnimInfo のインデックスを設定します。
         * rootBone を基点にして下位階層の削除されない BoneVisAnimInfo に新しいインデックスを設定します。
         * そのため、BoneVisAnimInfo.RemoveFlag がすでに設定されている必要があります。
         * ここで設定されるインデックスはソート済みの値です。
         * 中間ファイルに出力する際にはこのインデックスの順番通りにソートする必要があります。
         */
        public static void SetNewIndex(BoneVisAnimInfo rootBone)
        {
            // まずは削除されないボーンで出力用の階層構造を作成する
            BoneVisAnimInfoNode rootNode = BoneVisAnimInfoNode.GetHierarchy(rootBone);
            if (rootNode != null)
            {
                // その階層構造の順番に従って 0 から順にインデックスを割り振る
                rootNode.SetNewIndex(0);
            }
        }
        /***
         * ボーンのインデックスを比較します。
         * ボーン配列内を出力順にソートする時に使用します。
         */
        private static int CompareBoneIndex(bone_vis_bone_animType lhs, bone_vis_bone_animType rhs)
        {
            if (lhs.index == rhs.index)
            {
                return 0;
            }
            else if (lhs.index < rhs.index)
            {
                return -1;
            }
            else
            {
                return 1;
            }
        }
        /***
         * スケルタルアニメーションから不要なアニメーションを削除します。
         * BoneVisAnimInfo.RemoveFlag が true になっているアニメーションをスケルタルアニメーションの
         * アニメーション配列から削除します。
         * その時にアニメーションのインデックスとペアレントの情報も一緒に更新します。
         * この関数を実行した後は BoneVisAnimInfo と参照する bone_vis_bone_animType の階層構造の情報は一致しなくなります。
         */
        public static void RemoveBoneAnims(bone_visibility_animType anim, BoneVisAnimInfo[] animInfos)
        {
            // 新しいボーン配列の作成
            List<bone_vis_bone_animType> newBones = new List<bone_vis_bone_animType>();
            foreach (BoneVisAnimInfo boneInfo in animInfos)
            {
                if (!boneInfo.RemoveFlag)
                {
                    bone_vis_bone_animType bone = boneInfo.BoneAnim;

                    // 新しい配列内のインデックスを設定する
                    bone.index = boneInfo.NewIndex;
                    // 親が削除される場合は親の名前を変更する。
                    if (boneInfo.Parent != null && boneInfo.Parent.RemoveFlag)
                    {
                        BoneVisAnimInfo newParent = boneInfo.GetIncompressibleAncestorBone();
                        if (newParent != null)
                        {
                            bone.parent_name = newParent.BoneAnim.bone_name;
                        }
                        else
                        {
                            bone.parent_name = string.Empty;
                        }
                    }
                    newBones.Add(bone);
                }
            }
            // ボーンを出力順にソートする
            newBones.Sort(CompareBoneIndex);
            // 新しいボーン配列への差し替え
            anim.bone_vis_bone_anim_array.length = newBones.Count;
            anim.bone_vis_bone_anim_array.bone_vis_bone_anim = newBones.ToArray();
        }

        /**
         * 間引き後のノード間の参照関係を求めます。
         */
        public static void GetThinOutBoneRef(
            BoneVisAnimInfo[] boneInfos,
            out BoneVisAnimInfo[] parent)
        {
            int boneSize = boneInfos.Length;
            parent = new BoneVisAnimInfo[boneSize];
            for (int i = 0; i < boneSize; ++i)
            {
                if (boneInfos[i].RemoveFlag)
                {
                    continue;
                }
                //新しい親ノードIDを求める
                BoneVisAnimInfo bone = boneInfos[i].Parent;
                int count = 0;
                for (; bone != null && count < boneSize; ++count)
                {
                    if (!bone.RemoveFlag)
                    {
                        parent[i] = bone;
                        break;
                    }
                    else
                    {
                        bone = bone.Parent;
                    }
                }
                if (count == boneSize)
                {
                    throw new Exception("<bone_vis_bone_anim> has cyclic indexing.");
                }
            }
        }

        /**
         * 子供のボーンリストを得ます。
         */
        public static void GetBoneDescendants(
            BoneVisAnimInfo boneInfo,
            List<BoneVisAnimInfo> dst)
        {
            foreach (BoneVisAnimInfo child in boneInfo.Children)
            {
                dst.Add(child);
                GetBoneDescendants(child, dst);
            }
        }

        // 無用なストリームを削除する
        public static void RemoveUselessStream(
            bone_visibility_animType visibilityAnim,
            List<G3dStream> streams)
        {
            int[] newStreamIndex = new int[streams.Count];
            for (int i = 0; i < streams.Count; i++)
            {
                newStreamIndex[i] = -1;
            }

            // 使用されているストリームのフラグを false に設定する
            // アニメーションカーブがストリームを共有することはないはず？(要確認)
            // なので newStreamIndex に直接インデックスを代入しても大丈夫。
            int streamCount = 0;
            foreach (bone_vis_bone_animType boneAnim in visibilityAnim.bone_vis_bone_anim_array.bone_vis_bone_anim)
            {
                step_curveType stepCurve = boneAnim.step_curve;
                if (stepCurve != null)
                {
                    newStreamIndex[stepCurve.stream_index] = streamCount++;
                }
            }

            // 残すストリームに新しいインデックスを割り当てる
            G3dStream[] remainStreams = new G3dStream[streamCount];
            int streamSize = streams.Count;
            for (int iStream = 0; iStream < streamSize; iStream++)
            {
                if (newStreamIndex[iStream] != -1)
                {
                    remainStreams[newStreamIndex[iStream]] = streams[iStream];
                }
            }
            streams.Clear();
            streams.AddRange(remainStreams);

            // アニメーションが参照しているストリームのインデックスを新しい値に置きかえる
            foreach (bone_vis_bone_animType boneAnim in visibilityAnim.bone_vis_bone_anim_array.bone_vis_bone_anim)
            {
                step_curveType stepCurve = boneAnim.step_curve;
                if (stepCurve != null)
                {
                    int newIndex = newStreamIndex[stepCurve.stream_index];
                    stepCurve.stream_index = newIndex;
                }
            }
        }

        // アニメーションが設定されているか
        private static bool HasAnimation(bone_vis_bone_animType boneAnim)
        {
            return (boneAnim.Curve != null);
        }

        // アニメーションが付いているボーンが削除されるか確認する
        public static void CheckAnimationRemoved(BoneVisAnimInfo[] boneInfos)
        {
            foreach (BoneVisAnimInfo bi in boneInfos)
            {
                if (bi.RemoveFlag)
                {
                    CheckAnimationRemoved(bi.BoneAnim);
                }
            }
        }

        // アニメーションが付いているボーンが削除されるか確認する
        public static void CheckAnimationRemoved(bone_vis_bone_animType boneAnim)
        {
            if (HasAnimation(boneAnim))
            {
                Console.Error.WriteLine(
                    IfStrings.Get("SkeletalAnimCompress_Warning_RemoveAnim", boneAnim.name));
            }
        }
    }

    // ボーンアニメーション情報
    public class BoneVisAnimInfo
    {
        // コンストラクタ
        public BoneVisAnimInfo(bone_vis_bone_animType bone_anim)
        {
            this.BoneAnim = bone_anim;
        }

        // セットアップ
        public void Setup(BoneVisAnimInfo[] boneAnimInfos)
        {
            string parentName = this.BoneAnim.parent_name;
            if (parentName != string.Empty)
            {
                BoneVisAnimInfo parentInfo = Array.Find<BoneVisAnimInfo>(boneAnimInfos,
                    delegate(BoneVisAnimInfo match)
                    { return (match.BoneAnim.bone_name == parentName); });
                if (parentInfo == null)
                {
                    IfStrings.Throw("IfBoneVisibilityAnimBoneCullCompressor_Error_ParentNotFound",
                        this.BoneAnim.bone_name, parentName);
                }
                parentInfo.Children.Add(this);
                this.Parent = parentInfo;
            }
        }

        // 削除されない子がいるか
        public bool HasChild()
        {
            foreach (BoneVisAnimInfo bone_animInfo in this.Children)
            {
                if (!bone_animInfo.RemoveFlag) { return true; }
            }
            return false;
        }

        // 文字列変換
        public override string ToString()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendFormat("{0} {1} {2}→",
                !this.RemoveFlag ? "○" : "×",
                this.RenderMatrix ? "○" : "×",
                this.BoneAnim.bone_name);
            foreach (BoneVisAnimInfo child in this.Children)
            {
                builder.AppendFormat("{0}{1},",
                    child.BoneAnim.bone_name,
                    child.RenderMatrix ? "○" : "×");
            }
            return builder.ToString();
        }

        public BoneVisAnimInfo GetIncompressibleAncestorBone()
        {
            BoneVisAnimInfo ancestorBone = this.Parent;
            while (ancestorBone != null)
            {
                if (!ancestorBone.RemoveFlag)
                {
                    break;
                }
                ancestorBone = ancestorBone.Parent;
            }
            return ancestorBone;
        }

        public bool RenderMatrix
        {
            get
            {
                return this.BoneAnim.rigid_body ||
                    (this.BoneAnim.matrix_index[0] != -1) ||
                    (this.BoneAnim.matrix_index[1] != -1);
            }
            set
            {
                throw new NotImplementedException();
            }
        }

        public readonly bone_vis_bone_animType BoneAnim;
        public readonly List<BoneVisAnimInfo> Children = new List<BoneVisAnimInfo>();
        public BoneVisAnimInfo Parent { get; set; }
        public bool RemoveFlag { get; set; }
        public int NewIndex { get; set; } = -1;
    }
}
