﻿using Nintendo.G3dTool.Entities;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

namespace nw.g3d.iflib.nw3de
{
    [Serializable]
    public class nw3de_MaterialReference : ToolDataObject
    {
        [XmlElement("ParentMaterial")]
        public List<ParentMaterial> ParentMaterials { get; set; }

        [XmlArrayItem("ShaderParamBehavior")]
        public List<MaterialReferenceBehaviorItem> ShaderParamBehaviors { get; set; }

        [XmlArrayItem("RenderInfoBehavior")]
        public List<MaterialReferenceBehaviorItem> RenderInfoBehaviors { get; set; }

        [XmlArrayItem("SamplerAssignBehavior")]
        public List<MaterialReferenceBehaviorItem> SamplerAssignBehaviors { get; set; }

        [XmlArrayItem("ShaderOptionBehavior")]
        public List<MaterialReferenceBehaviorItem> ShaderOptionBehaviors { get; set; }

        [XmlArrayItem("AttribAssignBehavior")]
        public List<MaterialReferenceBehaviorItem> AttribAssignBehaviors { get; set; }

        public MaterialReferenceSamplerBehaviors SamplerBehaviors { get; set; }

        public nw3de_MaterialReference()
        {
            ParentMaterials = new List<ParentMaterial>();
            ShaderParamBehaviors = new List<MaterialReferenceBehaviorItem>();
            RenderInfoBehaviors = new List<MaterialReferenceBehaviorItem>();
            SamplerAssignBehaviors = new List<MaterialReferenceBehaviorItem>();
            ShaderOptionBehaviors = new List<MaterialReferenceBehaviorItem>();
            AttribAssignBehaviors = new List<MaterialReferenceBehaviorItem>();
            SamplerBehaviors = new MaterialReferenceSamplerBehaviors();
        }

        public void ClearEmptyItems()
        {
            ParentMaterials = ParentMaterials?.Any() == true ? ParentMaterials : null;
            ShaderParamBehaviors = ShaderParamBehaviors?.Any() == true ? ShaderParamBehaviors : null;
            RenderInfoBehaviors = RenderInfoBehaviors?.Any() == true ? RenderInfoBehaviors : null;
            SamplerAssignBehaviors = SamplerAssignBehaviors?.Any() == true ? SamplerAssignBehaviors : null;
            ShaderOptionBehaviors = ShaderOptionBehaviors?.Any() == true ? ShaderOptionBehaviors : null;
            AttribAssignBehaviors = AttribAssignBehaviors?.Any() == true ? AttribAssignBehaviors : null;

            SamplerBehaviors?.ClearEmptyItems();
            SamplerBehaviors = SamplerBehaviors?.Empty() == false ? SamplerBehaviors : null;
        }

        public bool IsEmpty()
        {
            return IsParentMaterialEmpty() && IsParamBehaviorEmpty();
        }

        public bool IsParentMaterialEmpty()
        {
            return ParentMaterials?.Any() != true;
        }
        public bool IsParamBehaviorEmpty()
        {
            return (ShaderParamBehaviors?.Any() != true) &&
                   (RenderInfoBehaviors?.Any() != true) &&
                   (SamplerAssignBehaviors?.Any() != true) &&
                   (ShaderOptionBehaviors?.Any() != true) &&
                   (AttribAssignBehaviors?.Any() != true) &&
                   (SamplerBehaviors?.Empty() != false);
        }

        public bool Updated
        {
            get
            {
                return ParentMaterials?.Any(x => x.Updated) == true ||
                       ShaderParamBehaviors?.Any(x => x.Updated) == true ||
                       RenderInfoBehaviors?.Any(x => x.Updated) == true ||
                       SamplerAssignBehaviors?.Any(x => x.Updated) == true ||
                       ShaderOptionBehaviors?.Any(x => x.Updated) == true ||
                       AttribAssignBehaviors?.Any(x => x.Updated) == true ||
                       SamplerBehaviors?.SamplerBehaviorItems?.Any(x => x.Updated) == true;
            }
        }

        public override object CreateSerializableData()
        {
            if (this.IsEmpty())
            {
                return null;
            }

            return this;
        }
    }
    [Serializable]
    public class ParentMaterial : IXmlSerializable
    {
        // 親マテリアル名
        [XmlAttribute]
        public string MaterialName { get; set; }

        // 親マテリアルを持つモデル中間ファイルパス
        [XmlAttribute]
        public string Path { get; set; }
        // アップデートされたかどうか
        [XmlIgnore]
        public bool Updated { get; set; } = false;

        public XmlSchema GetSchema()
        {
            return null;
        }

        public void ReadXml(XmlReader reader)
        {
            reader.MoveToContent();
            MaterialName = reader.GetAttribute("MaterialName");

            Path = reader.GetAttribute("Path");
            if (string.IsNullOrEmpty(Path))
            {
                Path = reader.GetAttribute("Model");
                Updated = !string.IsNullOrEmpty(Path);
            }
            reader.Read();
        }

        public void WriteXml(XmlWriter writer)
        {
            writer.WriteAttributeString("MaterialName", MaterialName);
            if (!string.IsNullOrEmpty(Path))
            {
                writer.WriteAttributeString("Path", Path);
            }
        }
    }
    [Serializable]
    public class MaterialReferenceBehaviorItem : IXmlSerializable
    {
        // 対象のID
        [XmlAttribute]
        public string Id { get; set; } = string.Empty;

        // 値
        [XmlAttribute]
        public ShaderItemValueState Value { get; set; } = ShaderItemValueState.Refer;

        // 子マテリアルの制限
        [XmlAttribute]
        public ShaderItemRestrictionState ChildRestriction { get; set; } = ShaderItemRestrictionState.None;

        // アップデートされたかどうか
        [XmlIgnore]
        public bool Updated { get; set; } = false;

        public XmlSchema GetSchema()
        {
            return null;
        }

        public void ReadXml(XmlReader reader)
        {
            reader.MoveToContent();
            Id = reader.GetAttribute("Id");

            ShaderItemState editable;
            ShaderItemState visible;
            ShaderItemValueState value;
            ShaderItemRestrictionState restriction;
            Updated = false;

            if (!Enum.TryParse(reader.GetAttribute("Value"), true, out value))
            {
                value = ShaderItemValueState.Refer;
            }
            if (!Enum.TryParse(reader.GetAttribute("ChildRestriction"), true, out restriction))
            {
                Updated = true;
                if (Enum.TryParse(reader.GetAttribute("Editable"), true, out editable))
                {
                    switch (editable)
                    {
                        case ShaderItemState.False:
                            restriction = ShaderItemRestrictionState.ForceRefer;
                            break;
                        case ShaderItemState.True:
                        case ShaderItemState.Refer:
                        default:
                            restriction = ShaderItemRestrictionState.None;
                            break;
                    }
                }
                else
                {
                    restriction = ShaderItemRestrictionState.None;
                }
            }
            if (!Enum.TryParse(reader.GetAttribute("Visible"), true, out visible))
            {
                visible = ShaderItemState.Refer;
            }

            ChildRestriction = restriction;
            Value = value;
            reader.Read();
        }

        public void WriteXml(XmlWriter writer)
        {
            writer.WriteAttributeString("Id", Id);
            writer.WriteAttributeString("Value", Value.ToString());
            writer.WriteAttributeString("ChildRestriction", ChildRestriction.ToString());
        }

        // 互換性のため
        private enum ShaderItemState
        {
            Refer,
            False,
            True,
        }
    }

    [Serializable]
    public class MaterialReferenceSamplerBehaviors
    {
        [XmlAttribute]
        public ChildSamplerStructureRestrictionState ChildSamplerStructureRestriction { get; set; } = ChildSamplerStructureRestrictionState.None;

        [XmlElement("SamplerBehavior")]
        public List<SamplerBehaviorItem> SamplerBehaviorItems { get; set; } = new List<SamplerBehaviorItem>();

        public bool Empty()
        {
            if (ChildSamplerStructureRestriction != ChildSamplerStructureRestrictionState.None)
            {
                return false;
            }
            return SamplerBehaviorItems == null || !SamplerBehaviorItems.Any() ||
                   SamplerBehaviorItems.All(x => x.Empty());
        }

        public void ClearEmptyItems()
        {
            if (SamplerBehaviorItems != null && SamplerBehaviorItems.All(x => x.Empty()))
            {
                SamplerBehaviorItems = null;
            }
        }
    }

    [Serializable]
    public class SamplerBehaviorItem
    {
        // 対象のID
        [XmlAttribute]
        public string Id { get; set; } = string.Empty;

        // 子マテリアルで必須とするか
        [XmlAttribute]
        public bool IsRequiredForChild { get; set; } = false;

        // アップデートされたかどうか
        [XmlIgnore]
        public bool Updated { get; } = false;

        [XmlElement("SamplerParamBehavior")]
        public List<MaterialReferenceBehaviorItem> SamplerParamBehaviors { get; set; } = new List<MaterialReferenceBehaviorItem>();
        public bool Empty()
        {
            if (IsRequiredForChild == true)
            {
                return false;
            }
            return SamplerParamBehaviors == null || !SamplerParamBehaviors.Any();
        }
    }

    [Serializable]
    public enum ChildSamplerStructureRestrictionState
    {
        None,
        DisallowAddOwnSampler,
    }

    [Serializable]
    public enum ShaderItemValueState
    {
        Override,
        Refer,
        Default,
    }

    [Serializable]
    public enum ShaderItemRestrictionState
    {
        None,
        ForceRefer,
        ForceOverride,
    }
}
