﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;
using System.Xml.XPath;
using System.IO;
using System.Diagnostics;
using Microsoft.Scripting.Utils;
using nw.g3d.iflib.nw3de;
using nw.g3d.nw4f_3dif;
using Nintendo.G3dTool.Entities;

namespace nw.g3d.iflib
{
    public class IfToolData
    {
        /// <summary>
        /// T をXmlElement に変換します。level のタブインデントをXmlElement 内部に設定します。
        /// </summary>
        public static XmlElement ConvertToXmlElement<T>(T obj, int level)
        {
            // インデントのレベルは1以上
            Nintendo.Foundation.Contracts.Assertion.Operation.True(level > 0);

            // タワーに変換
            var root = new TTower<T>();
            var leaf = root;
            for (int i = 1; i < level; i++)
            {
                leaf.Tower = new TTower<T>();
                leaf = leaf.Tower;
            }
            leaf.Data = obj;

            // シリアライズ
            var stream = new MemoryStream();
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            settings.IndentChars = "\t";
            using (XmlWriter writer = XmlWriter.Create(stream, settings))
            {
                var serializer = new XmlSerializer(typeof(TTower<T>));
                serializer.Serialize(writer, root);
            }
            stream.Seek(0, SeekOrigin.Begin);

            // 変換
            var doc = XDocument.Load(stream, LoadOptions.PreserveWhitespace);
            doc.Root.Name = typeof(XmlElementTower).Name;
            var element = doc.XPathSelectElement("//Data");
            element.Name = typeof(T).Name;

            // デシリアライズ
            XmlElementTower node;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = new XmlSerializer(typeof(XmlElementTower));
                node = deserializer.Deserialize(reader) as XmlElementTower;
            }

            // タワーから要素を取り出します。
            while (node.Tower != null)
            {
                node = node.Tower;
            }
            return node.Elements[0];
        }

        /// <summary>
        /// XmlElement をT に変換します。
        /// </summary>
        public static T ConvertToEditData<T>(XmlElement elem) where T : class
        {
            // シリアライズ
            var doc = new XDocument();
            using (XmlWriter writer = doc.CreateWriter())
            {
                var serializer = new XmlSerializer(typeof(XmlElement));
                serializer.Serialize(writer, elem);
            }

            // 変換
            doc.Root.Name = typeof(T).Name;

            // デシリアライズ
            T t;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = new XmlSerializer(typeof(T));
                t = deserializer.Deserialize(reader) as T;
            }

            return t;
        }

        /// <summary>
        /// 変換用ヘルパクラス
        /// XmlSerializer で使うため private にはできない
        /// </summary>
        public class TTower<T>
        {
            [XmlElement]
            public TTower<T> Tower { get; set; }

            public T Data { get; set; }
        }

        /// <summary>
        /// 変換用ヘルパクラス
        /// XmlSerializer で使うため private にはできない
        /// </summary>
        public class XmlElementTower
        {
            [XmlElement]
            public XmlElementTower Tower { get; set; }

            [XmlAnyElement]
            public XmlElement[] Elements { get; set; }
        }

        public static T GetToolDataInfo<T>(IG3dRootElement rootElement) where T : class
        {
            if (rootElement.tool_data?.Any != null)
            {
                var index = rootElement.tool_data.Any.FindIndex(x => x.Name == typeof(T).Name);
                if (index >= 0)
                {
                    return IfToolData.ConvertToEditData<T>(rootElement.tool_data.Any[index]);
                }
            }
            return null;
        }

        public static void SetToolDataInfo<T>(IG3dRootElement rootElement, T info)
        {
            var elements = new List<XmlElement>();
            var index = -1;

            if (rootElement.tool_data != null && rootElement.tool_data.Any != null)
            {
                elements.AddRange(rootElement.tool_data.Any);
                index = elements.FindIndex(x => x.Name == typeof(T).Name);
            }

            if (info != null)
            {
                var element = ConvertToXmlElement(info, 1);
                if (index == -1)
                {
                    elements.Add(element);
                }
                else
                {
                    elements[index] = element;
                }
            }
            else if (index != -1)
            {
                // 既存の nw3de_XXXXInfo を削除
                elements.RemoveAt(index);
            }

            if (elements.Any())
            {
                rootElement.tool_data = new tool_dataType()
                {
                    Any = elements.ToArray(),
                };
            }
            else
            {
                rootElement.tool_data = null;
            }
        }

        public static bool Update(nw4f_3difType nwif)
        {
            var item = nwif?.Item;
            var model = item as modelType;
            if (model != null)
            {
                return UpdateMaterialReference(model);
            }
            return false;
        }

        #region マテリアル参照
        public static nw3de_MaterialReference GetMaterialReference(materialType material)
        {
            return GetToolDataInfo<nw3de_MaterialReference>(material) ?? new nw3de_MaterialReference();
        }

        public static void SetMaterialReference(materialType material, nw3de_MaterialReference materialReference)
        {
            materialReference?.ClearEmptyItems();
            var info = materialReference?.IsEmpty() == false ? materialReference : null;
            SetToolDataInfo(material, info);
        }

        // 旧仕様からの移行用にprivateで残しておく
        private static nw3de_ParentMaterialInfo GetParentMaterialInfoInModel(modelType model)
        {
            return GetToolDataInfo<nw3de_ParentMaterialInfo>(model) ?? new nw3de_ParentMaterialInfo();
        }

        // 旧仕様からの移行用にprivateで残しておく
        private static void SetParentMaterialInfoInModel(modelType model, nw3de_ParentMaterialInfo parentMaterialInfo)
        {
            var info = parentMaterialInfo?.ParentMaterialInfos?.Any() == true ? parentMaterialInfo : null;
            SetToolDataInfo(model, info);
        }

        // 旧仕様からの移行用にprivateで残しておく
        private static nw3de_MaterialReferenceBehavior GetMaterialReferenceBehaviorInModel(modelType model)
        {
            return GetToolDataInfo<nw3de_MaterialReferenceBehavior>(model) ?? new nw3de_MaterialReferenceBehavior();
        }

        // 旧仕様からの移行用にprivateで残しておく
        private static void SetMaterialReferenceBehaviorInModel(modelType model, nw3de_MaterialReferenceBehavior materialReferenceBehavior)
        {
            var info = materialReferenceBehavior?.MaterialReferenceBehaviors?.Any() == true
                ? materialReferenceBehavior
                : null;
            SetToolDataInfo(model, info);
        }

        private static bool UpdateMaterialReference(modelType model)
        {
            var update = false;
            // 旧式のモデルのToolDataに有る場合、マテリアルに移動する
            var materialReferenceBehaviorInModel = GetMaterialReferenceBehaviorInModel(model);
            var parentMaterialInfoInModel = GetParentMaterialInfoInModel(model);

            var materialNames = new List<string>();
            if (materialReferenceBehaviorInModel?.MaterialReferenceBehaviors?.Any() == true)
            {
                materialNames.AddRange(materialReferenceBehaviorInModel.MaterialReferenceBehaviors.Select(x => x.MaterialName));
            }
            if (parentMaterialInfoInModel?.ParentMaterialInfos?.Any() == true)
            {
                materialNames.AddRange(parentMaterialInfoInModel.ParentMaterialInfos.Select(x => x.MaterialName));
            }
            materialNames = materialNames.Distinct().ToList();

            if (materialNames.Any())
            {
                update = true;
                foreach (var materialName in materialNames)
                {
                    var material = model.material_array.material.FirstOrDefault(x => x.name == materialName);
                    // 存在しないマテリアルに対する
                    if (material == null)
                    {
                        continue;
                    }

                    var materialReference = GetToolDataInfo<nw3de_MaterialReference>(material);
                    // すでに新形式のデータが有る場合はコピーしない
                    if (materialReference != null)
                    {
                        continue;
                    }
                    materialReference = new nw3de_MaterialReference();

                    var parentMaterialInfo = parentMaterialInfoInModel?.ParentMaterialInfos?.FirstOrDefault(x => x.MaterialName == materialName);
                    var materialReferenceBehavior =
                        materialReferenceBehaviorInModel?.MaterialReferenceBehaviors?.FirstOrDefault(x => x.MaterialName == materialName);

                    if (parentMaterialInfo?.ParentMaterials != null)
                    {
                        materialReference.ParentMaterials = parentMaterialInfo.ParentMaterials;
                    }

                    if (materialReferenceBehavior?.AttribAssignBehaviors != null)
                    {
                        materialReference.AttribAssignBehaviors = materialReferenceBehavior.AttribAssignBehaviors;
                    }
                    if (materialReferenceBehavior?.RenderInfoBehaviors != null)
                    {
                        materialReference.RenderInfoBehaviors = materialReferenceBehavior.RenderInfoBehaviors;
                    }
                    if (materialReferenceBehavior?.SamplerAssignBehaviors != null)
                    {
                        materialReference.SamplerAssignBehaviors = materialReferenceBehavior.SamplerAssignBehaviors;
                    }
                    if (materialReferenceBehavior?.ShaderOptionBehaviors != null)
                    {
                        materialReference.ShaderOptionBehaviors = materialReferenceBehavior.ShaderOptionBehaviors;
                    }
                    if (materialReferenceBehavior?.ShaderParamBehaviors != null)
                    {
                        materialReference.ShaderParamBehaviors = materialReferenceBehavior.ShaderParamBehaviors;
                    }
                    if (materialReferenceBehavior?.SamplerBehaviors != null)
                    {
                        materialReference.SamplerBehaviors = materialReferenceBehavior.SamplerBehaviors;
                    }

                    SetMaterialReference(material, materialReference);
                }

                // 旧式のモデルのToolDataは削除
                SetMaterialReferenceBehaviorInModel(model, null);
                SetParentMaterialInfoInModel(model, null);
            }

            return update;
        }

        #endregion //マテリアル参照
    }
}
