﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public static class IfShaderParameterUtility
    {
        /// <summary>
        /// 比較
        /// </summary>
        public static IfShaderAssignUtility.CompareResult Compare(IEnumerable<shader_paramType> shader_params, shading_modelType shadingModel)
        {
            IEnumerable<uniform_varType> uniform_vars = shadingModel.MaterialUniforms();

            if (uniform_vars.Count() != shader_params.Count())
            {
                return IfShaderAssignUtility.CompareResult.Conflict;
            }

            if (shader_params.Zip(uniform_vars, (x, y) => x.id == y.id && x.type == y.type && x.depend == y.depend).Any(x => !x))
            {
                if (shader_params.OrderBy(x => x.id).Zip(uniform_vars.OrderBy(x => x.id), (x, y) => x.id == y.id && x.type == y.type && x.depend == y.depend).All(x => x))
                {
                    return IfShaderAssignUtility.CompareResult.NotSameOrder;
                }
                else
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            return IfShaderAssignUtility.CompareResult.Ok;
        }

        /// <summary>
        /// シェーダーパラメーターを作成する
        /// </summary>
        /// <remarks>
        /// 新規割り当て時は oldParams は空で渡す
        /// </remarks>
        internal static IEnumerable<shader_paramType> CreateParameters(
            shading_modelType shadingModel,
            IEnumerable<shader_paramType> oldParams,
            original_materialType originalMaterial)
        {
            foreach (var uniform_var in shadingModel.MaterialUniforms())
            {
                // 作成
                var param = new shader_paramType()
                {
                    id = uniform_var.id,
                    type = uniform_var.type,
                    Value = uniform_var.Default(),
                    original_hint = string.Empty,
                    depend = uniform_var.depend, // depend は uniform_var にあわせる
                };

                bool done = false;
                // 元の値を引き継ぐ
                var old = oldParams.FirstOrDefault(
                    x => x.id == uniform_var.id);
                if (old != null)
                {
                    if (old.type == uniform_var.type)
                    {
                        param.Value = old.Value;
                        param.original_hint = old.original_hint;
                        done = true;
                    }
                    else
                    {
                        // 変換する
                        int[] convertTable;
                        IfAnimationAssignUtility.GetConvertTable(old.type, uniform_var.type, out convertTable);
                        if (convertTable != null)
                        {
                            param.Value = GetConvertedValue(old, convertTable, uniform_var);
                            param.original_hint = old.original_hint;
                            done = true;
                        }
                    }
                }

                // オリジナルから割り当て
                if (!done && originalMaterial != null)
                {
                    string original_hint;
                    var value = GetOriginalMaterialValue(uniform_var, originalMaterial, out original_hint);
                    if (value != null)
                    {
                        param.Value = value;
                        param.original_hint = original_hint;
                    }
                }

                yield return param;
            }
        }

        /// <summary>
        /// シェーダーパラメーター変更メッセージの取得
        /// </summary>
        internal static IEnumerable<string> GetUpdateParameterMessage(
            IEnumerable<shader_paramType> shaderParams,
            original_materialType originalMaterial,
            shading_modelType shadingModel,
            bool ignoreOrder,
            IfShaderAssignUtility.InconsistentMessageType type)
        {
            // シェーダー定義側
            IEnumerable<uniform_varType> uniform_vars = shadingModel.MaterialUniforms();

            // 過剰
            var over = shaderParams.Select(x => x.id).Except(uniform_vars.Select(x => x.id)).ToArray();
            foreach (var item in over)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Over_Uniform, type), item);
            }

            // 不足
            var lack = uniform_vars.Where(x => !shaderParams.Any(y => x.id == y.id)).ToArray();
            foreach (var item in lack)
            {
                var builder = new StringBuilder();
                builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Lack_Uniform, type), item.id);
                bool original = false;
                if (item.hint != string.Empty && originalMaterial != null)
                {
                    string original_hint;
                    var value = GetOriginalMaterialValue(item, originalMaterial, out original_hint);
                    if (value != null)
                    {
                        original = true;
                        builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Original, type), value.Replace('\t', ' ').Replace('\n', ' '));
                    }
                }

                // オリジナルを使わない場合はデフォルト
                if (!original)
                {
                    builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Default, type), item.Default().Replace('\t', ' ').Replace('\n', ' '));
                }

                yield return builder.ToString();
            }

            // 型チェック
            var typeErrors = from var in uniform_vars
                             from param in shaderParams
                             where var.id == param.id && var.type != param.type
                             select new { var, param };
            foreach (var item in typeErrors)
            {
                var builder = new StringBuilder();
                builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Type_Uniform, type), item.param.id, item.param.type.ToString(), item.var.type.ToString());

                int[] convertTable;
                IfAnimationAssignUtility.GetConvertTable(item.param.type, item.var.type, out convertTable);

                bool done = false;
                // 変換
                if (convertTable != null)
                {
                    string value = GetConvertedValue(item.param, convertTable, item.var);
                    done = true;
                    builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Convert, type), value.Replace('\t', ' '));
                }

                // オリジナルから
                if (!done && item.var.hint != string.Empty && originalMaterial != null)
                {
                    string original_hint;
                    var value = GetOriginalMaterialValue(item.var, originalMaterial, out original_hint);
                    if (value != null)
                    {
                        done = true;
                        builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Original, type), value.Replace('\t', ' ').Replace('\n', ' '));
                    }
                }

                if (!done)
                {
                    builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Default, type), item.var.Default().Replace('\t', ' ').Replace('\n', ' '));
                }
                yield return builder.ToString();
            }

            // depend のチェック
            var dependErrors = from var in uniform_vars
                               from param in shaderParams
                               where var.id == param.id && var.type == param.type && var.depend != param.depend
                               select new { var, param };
            foreach (var item in dependErrors)
            {
                var builder = new StringBuilder();
                string message;
                if (string.IsNullOrEmpty(item.param.depend))
                {
                    message = IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Lack_Depend, type);
                }
                else if (string.IsNullOrEmpty(item.var.depend))
                {
                    message = IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Over_Depend, type);
                }
                else
                {
                    message = IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Another_Depend, type);
                }

                builder.AppendFormat(message, item.param.id, item.param.depend, item.var.id, item.var.depend);
                yield return builder.ToString();
            }

            // 順番のチェック
            if (!ignoreOrder &&
                over.Length == 0 &&
                lack.Length == 0 &&
                shaderParams.Zip(uniform_vars, (x, y) => x.id == y.id).Any(x => !x))
            {
                yield return IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Disorder_Uniform, type);
            }
        }

        /// <summary>
        /// マテリアル関係のユニフォームブロックに含まれるユニフォーム
        /// </summary>
        internal static IEnumerable<uniform_varType> MaterialUniforms(this shading_modelType shadingModel)
        {
            if (shadingModel == null || shadingModel.block_var_array == null)
            {
                return Enumerable.Empty<uniform_varType>();
            }

            return shadingModel.block_var_array.block_var.Where(x => x.type == block_var_typeType.material && x.uniform_var_array != null)
                .SelectMany(x => x.uniform_var_array.uniform_var);
        }

        /// <summary>
        /// item の取得
        /// </summary>
        /// <param name="uniform"></param>
        /// <returns></returns>
        private static ui_item_valueType Item(this uniform_varType uniform)
        {
            return uniform.ui_item == null ? ui_item_valueType.auto : uniform.ui_item.value;
        }

        /// <summary>
        /// デフォルト値の取得
        /// 無い場合は作る
        /// </summary>
        internal static string Default(this uniform_varType uniform)
        {
            // 3DEditor で生成した場合はnull もあり
            int count = Dimension(uniform.type);

            var array = new float[count];
            for (int i = 0; i != count; i++)
            {
                if (uniform.@default != null && i < uniform.@default.Length)
                {
                    array[i] = uniform.@default[i];
                }
                else
                {
                    // default がない場合はでっちあげる
                    if (uniform.Item() == ui_item_valueType.color)
                    {
                        array[i] = 1;
                    }
                    else
                    {
                        switch (uniform.type)
                        {
                            case shader_param_typeType.srt2d:
                                array[i] = i < 2 ? 1 : 0;
                                break;
                            case shader_param_typeType.srt3d:
                                array[i] = i < 3 ? 1 : 0;
                                break;
                            case shader_param_typeType.texsrt:
                                array[i] = (i == 1 || i == 2) ? 1 : 0;
                                break;
                            default:
                                array[i] = 0;
                                break;
                        }
                    }
                }
            }

            return IfShaderAssignUtility.MakeArrayString(array);
        }

        /// <summary>
        /// 次元
        /// </summary>
        internal static int Dimension(shader_param_typeType type)
        {
            int count = 0;
            switch (type)
            {
                case shader_param_typeType.@bool:
                    count = 1;
                    break;
                case shader_param_typeType.bool2:
                    count = 2;
                    break;
                case shader_param_typeType.bool3:
                    count = 3;
                    break;
                case shader_param_typeType.bool4:
                    count = 4;
                    break;
                case shader_param_typeType.@int:
                    count = 1;
                    break;
                case shader_param_typeType.int2:
                    count = 2;
                    break;
                case shader_param_typeType.int3:
                    count = 3;
                    break;
                case shader_param_typeType.int4:
                    count = 4;
                    break;
                case shader_param_typeType.@uint:
                    count = 1;
                    break;
                case shader_param_typeType.uint2:
                    count = 2;
                    break;
                case shader_param_typeType.uint3:
                    count = 3;
                    break;
                case shader_param_typeType.uint4:
                    count = 4;
                    break;
                case shader_param_typeType.@float:
                    count = 1;
                    break;
                case shader_param_typeType.float2:
                    count = 2;
                    break;
                case shader_param_typeType.float3:
                    count = 3;
                    break;
                case shader_param_typeType.float4:
                    count = 4;
                    break;
                case shader_param_typeType.float2x2:
                    count = 2 * 2;
                    break;
                case shader_param_typeType.float2x3:
                    count = 2 * 3;
                    break;
                case shader_param_typeType.float2x4:
                    count = 2 * 4;
                    break;
                case shader_param_typeType.float3x2:
                    count = 3 * 2;
                    break;
                case shader_param_typeType.float3x3:
                    count = 3 * 3;
                    break;
                case shader_param_typeType.float3x4:
                    count = 3 * 4;
                    break;
                case shader_param_typeType.float4x2:
                    count = 4 * 2;
                    break;
                case shader_param_typeType.float4x3:
                    count = 4 * 3;
                    break;
                case shader_param_typeType.float4x4:
                    count = 4 * 4;
                    break;
                case shader_param_typeType.srt2d:
                    count = 5;
                    break;
                case shader_param_typeType.texsrt:
                    count = 6;
                    break;
                case shader_param_typeType.srt3d:
                    count = 9;
                    break;
                default:
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                    break;
            }
            return count;
        }

        /// <summary>
        /// カラーのオリジナル適用判定
        /// </summary>
        public static Func<string, string, string, string, string, bool> IsOriginalColorApplicable { get; set; } = IsOriginalColorApplicableDefault;

        /// <summary>
        /// カラーのオリジナル適用判定のデフォルト
        /// </summary>
        private static bool IsOriginalColorApplicableDefault(
            string name,
            string item,
            string param_type,
            string hint,
            string original_hint)
        {
            return hint == original_hint;
        }

        /// <summary>
        /// テクスチャSRTのオリジナル適用判定
        /// </summary>
        public static Func<string, string, string, string, string, bool> IsOriginalSrtApplicable { get; set; } = IsOriginalSrtApplicableDefault;
        private static bool IsOriginalSrtApplicableDefault(
            string name,
            string param_type,
            string hint,
            string original_hint,
            string mode)
        {
            return hint == original_hint;
        }

        /// <summary>
        /// オリジナルの値の取得
        /// </summary>
        internal static string GetOriginalMaterialValue(uniform_varType uniform_var, original_materialType original, out string hint)
        {
            switch (uniform_var.type)
            {
                case shader_param_typeType.float3:
                case shader_param_typeType.float4:
                    if (original.original_color_array != null)
                    {
                        foreach (var color in original.original_color_array.original_color)
                        {
                            if (IsOriginalColorApplicable(uniform_var.id, uniform_var.Item().ToString(), uniform_var.type.ToString(), uniform_var.hint, color.hint))
                            {
                                if (uniform_var.type == shader_param_typeType.float3)
                                {
                                    hint = color.hint;
                                    return IfShaderAssignUtility.MakeArrayString(color.color);
                                }
                                else
                                {
                                    hint = color.hint;
                                    return IfShaderAssignUtility.MakeArrayString(color.color.Concat(Enumerable.Repeat<float>(1.0f, 1)));
                                }
                            }
                        }
                    }
                    break;
                case shader_param_typeType.srt2d:
                    if (original.original_texsrt_array != null)
                    {
                        foreach (var texsrt in original.original_texsrt_array.original_texsrt)
                        {
                            if (IsOriginalSrtApplicable(uniform_var.id, uniform_var.type.ToString(), uniform_var.hint, texsrt.hint, texsrt.mode.ToString()))
                            {
                                hint = texsrt.hint;
                                return IfShaderAssignUtility.MakeArrayString(
                                    new float[] {
                                        texsrt.scale[0], texsrt.scale[1],
                                        texsrt.rotate,
                                        texsrt.translate[0], texsrt.translate[1]
                                    });
                            }
                        }
                    }
                    break;
                case shader_param_typeType.texsrt:
                    if (original.original_texsrt_array != null)
                    {
                        foreach (var texsrt in original.original_texsrt_array.original_texsrt)
                        {
                            if (IsOriginalSrtApplicable(uniform_var.id, uniform_var.type.ToString(), uniform_var.hint, texsrt.hint, texsrt.mode.ToString()))
                            {
                                hint = texsrt.hint;
                                return MakeTexsrtValueFromOriginalTexsrt(texsrt);
                            }
                        }
                    }
                    break;
            }

            hint = null;
            return null;
        }

        public static string MakeTexsrtValueFromOriginalTexsrt(original_texsrtType texsrt)
        {
            return IfShaderAssignUtility.MakeArrayString(
                new float[] {
                    (texsrt.mode == original_texsrt_modeType.maya) ? 0.0f :
                    (texsrt.mode == original_texsrt_modeType.Item3dsmax) ? 1.0f :
                    (texsrt.mode == original_texsrt_modeType.softimage) ? 2.0f : 0.0f,
                    texsrt.scale[0], texsrt.scale[1],
                    texsrt.rotate,
                    texsrt.translate[0], texsrt.translate[1]
                });
        }

        /// <summary>
        /// 現在の値と変換テーブルを使って新しい値を作る
        /// </summary>
        private static string GetConvertedValue(shader_paramType param, int[] convertTable, uniform_varType uniform)
        {
            var values = G3dDataParser.Tokenize(param.Value);
            var defaults = G3dDataParser.Tokenize(uniform.Default());
            var converted = convertTable.Zip(defaults, (i, def) =>
            {
                if (0 <= i && i < values.Length)
                {
                    return values[i];
                }
                else
                {
                    return def;
                }
            });

            return IfShaderAssignUtility.MakeArrayString(converted);
        }
    }
}
