﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;

namespace nw.g3d.iflib
{
    public static class IfAnimationAssignUtility
    {
        #region 割り当て
        /// <summary>
        /// param_anim_targetType をオリジナルから作成
        /// </summary>
        public static param_anim_targetType CreateParamAnimTargetFromOriginalAnim(
            IG3dCurveContainer originalTarget,
            int componentIndex,
            shader_param_typeType paramType,
            IG3dShaderParamAnimInfo info,
            List<G3dStream> streams)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(originalTarget is original_color_anim_targetType || originalTarget is original_texsrt_anim_targetType);

            var paramAnim = new param_anim_targetType()
            {
                base_value = originalTarget.base_value,
                component_index = componentIndex,
            };

            IG3dQuantizedCurve curve = null;
            if (originalTarget.Curve is original_hermiteType)
            {
                curve = new hermite_curveType()
                {
                    baked = ((original_hermiteType)originalTarget.Curve).baked,
                };
            }
            else if (originalTarget.Curve is original_linearType)
            {
                curve = new linear_curveType()
                {
                    baked = ((original_linearType)originalTarget.Curve).baked,
                };
            }
            else if (originalTarget.Curve is original_stepType)
            {
                curve = new step_curveType()
                {
                    baked = ((original_stepType)originalTarget.Curve).baked,
                };
            }

            if (curve != null)
            {
                curve.count = originalTarget.Curve.count;
                curve.post_wrap = originalTarget.Curve.post_wrap;
                curve.pre_wrap = originalTarget.Curve.pre_wrap;
                curve.scale = 1;

                // 量子化
                Quantize(curve, componentIndex, paramType, info, streams[originalTarget.Curve.stream_index]);

                // オリジナルと同じにする。そのまま出力してはいけない
                curve.stream_index = originalTarget.Curve.stream_index;

                paramAnim.Curve = curve;
            }

            return paramAnim;
        }

        /// <summary>
        /// param_anim_targetType をオリジナルから作成
        /// </summary>
        public static param_anim_targetType CreateParamAnimTargetFromOriginalAnim(
            IG3dCurveContainer originalTarget,
            int componentIndex,
            shader_param_typeType paramType,
            material_anim_infoType info,
            List<G3dStream> streams)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(originalTarget is original_color_anim_targetType || originalTarget is original_texsrt_anim_targetType);

            var paramAnim = new param_anim_targetType()
            {
                base_value = originalTarget.base_value,
                component_index = componentIndex,
            };

            IG3dQuantizedCurve curve = null;
            if (originalTarget.Curve is original_hermiteType)
            {
                curve = new hermite_curveType()
                {
                    baked = ((original_hermiteType)originalTarget.Curve).baked,
                };
            }
            else if (originalTarget.Curve is original_linearType)
            {
                curve = new linear_curveType()
                {
                    baked = ((original_linearType)originalTarget.Curve).baked,
                };
            }
            else if (originalTarget.Curve is original_stepType)
            {
                curve = new step_curveType()
                {
                    baked = ((original_stepType)originalTarget.Curve).baked,
                };
            }

            if (curve != null)
            {
                curve.count = originalTarget.Curve.count;
                curve.post_wrap = originalTarget.Curve.post_wrap;
                curve.pre_wrap = originalTarget.Curve.pre_wrap;
                curve.scale = 1;

                // 量子化
                Quantize(curve, componentIndex, paramType, info, streams[originalTarget.Curve.stream_index]);

                // オリジナルと同じにする。そのまま出力してはいけない
                curve.stream_index = originalTarget.Curve.stream_index;

                paramAnim.Curve = curve;
            }

            return paramAnim;
        }

        /// <summary>
        /// texsrt, texsrt_ex のカーブのモード部分を生成
        /// </summary>
        public static param_anim_targetType CreateTexSrtModeParamAnimTargetFromOriginalAnim(
            original_texsrt_anim_modeType mode)
        {
            float baseValue = 0;
            switch (mode)
            {
                case original_texsrt_anim_modeType.maya:
                    baseValue = 0;
                    break;
                case original_texsrt_anim_modeType.Item3dsmax:
                    baseValue = 1;
                    break;
                case original_texsrt_anim_modeType.softimage:
                    baseValue = 2;
                    break;
            }

            return new param_anim_targetType()
            {
                base_value = baseValue,
                component_index = 0,
            };
        }

        /// <summary>
        /// 量子化
        /// </summary>
        public static void Quantize(
            IG3dQuantizedCurve curve,
            int componentIndex,
            shader_param_typeType paramType,
            IG3dShaderParamAnimInfo info,
            G3dStream stream)
        {
            if (curve == null)
            {
                return;
            }

            var IfAnimCurve = IfShaderParamAnimQuantizationAnalysisOptimizer.BuildIfAnimCurve(curve, paramType, componentIndex, stream);
            var result = IfShaderParamAnimQuantizationAnalysis.Analyse(info, paramType, componentIndex, IfAnimCurve, true);
            curve.frame_type = result.frameType;
            curve.key_type = result.keyType;
            curve.scale = result.scale;
            curve.offset = result.offset;
        }

        /// <summary>
        /// 量子化
        /// </summary>
        public static void Quantize(
            IG3dQuantizedCurve curve,
            int componentIndex,
            shader_param_typeType paramType,
            material_anim_infoType info,
            G3dStream stream)
        {
            if (curve == null)
            {
                return;
            }

            var IfAnimCurve = IfMaterialAnimQuantizationAnalysisOptimizer.BuildIfAnimCurve(curve, paramType, componentIndex, stream);
            var result = IfMaterialAnimQuantizationAnalysis.AnalyseShaderParameterCurve(info, paramType, componentIndex, IfAnimCurve, true);
            curve.frame_type = result.frameType;
            curve.key_type = result.keyType;
            curve.scale = result.scale;
            curve.offset = result.offset;
        }

        /// <summary>
        /// ヒントからカラーアニメーションのidを取得
        /// </summary>
        public static Func<string, string> GetIdFromOriginalColorAnim { get; set; } = GetIdFromOriginalColorAnimDefault;

        /// <summary>
        /// ヒントからテクスチャSRTアニメーションのidを取得
        /// </summary>
        public static Func<string, string> GetIdFromOriginalTexSRTAnim { get; set; } = GetIdFromOriginalTexSRTDefault;

        /// <summary>
        /// ヒントからカラーアニメーションのidを取得のデフォルト
        /// </summary>
        private static string GetIdFromOriginalColorAnimDefault(string hint)
        {
            switch (hint)
            {
                case "diffuse":
                    return "diffuse";
                case "opacity":
                    return "opacity";
                case "ambient":
                    return "ambient";
                case "emission":
                    return "emission";
                case "specular":
                    return "specular";
            }

            return null;
        }

        /// <summary>
        /// ヒントからテクスチャSRTアニメーションのidを取得のデフォルト
        /// </summary>
        private static string GetIdFromOriginalTexSRTDefault(string hint)
        {
            switch (hint)
            {
                case "albedo0":
                    return "tex_mtx0";
            }

            return null;
        }

        /// <summary>
        /// オリジナルに含まれるカラーアニメーションを取得する
        /// </summary>
        public static original_color_animType GetOriginalColorAnim(original_material_animType original, string paramAnimId, string targetHint)
        {
            foreach (var oriAnim in original.original_color_anim_array.GetItems())
            {
                if (!string.IsNullOrEmpty(targetHint) &&
                    oriAnim.hint == targetHint)
                {
                    return oriAnim;
                }
                else
                {
                    string paramId = GetIdFromOriginalColorAnim(oriAnim.hint);
                    if (!string.IsNullOrEmpty(paramId) &&
                        (paramId == paramAnimId))
                    {
                        return oriAnim;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// オリジナルに含まれるテクスチャSRTアニメーションを取得する
        /// </summary>
        public static original_texsrt_animType GetOriginalTexSRTAnim(original_material_animType original, string paramAnimId, string targetHint)
        {
            foreach (original_texsrt_animType oriAnim in original.original_texsrt_anim_array.GetItems())
            {
                if (!string.IsNullOrEmpty(targetHint) &&
                    oriAnim.hint == targetHint)
                {
                    return oriAnim;
                }
                else
                {
                    string paramId = GetIdFromOriginalTexSRTAnim(oriAnim.hint);
                    if (!string.IsNullOrEmpty(paramId) &&
                        (paramId == paramAnimId))
                    {
                        return oriAnim;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// オリジナルがあるかどうか
        /// </summary>
        public static bool HasOriginalParamAnim(
            original_material_animType original,
            string paramAnimId,
            shader_param_typeType type,
            string hint)
        {
            if (original == null)
            {
                return false;
            }

            switch (type)
            {
                case shader_param_typeType.float3:
                case shader_param_typeType.float4:
                    return GetOriginalColorAnim(original, paramAnimId, hint) != null;
                case shader_param_typeType.texsrt:
                case shader_param_typeType.srt2d:
                    return GetOriginalTexSRTAnim(original, paramAnimId, hint) != null;
            }

            return false;
        }

        public static param_animType CreateParamAnimFromOriginalColorAnim(
            original_color_animType originalColorAnim,
            string paramAnimId,
            shader_param_typeType type,
            List<G3dStream> streams,
            IG3dShaderParamAnimInfo info,
            bool addStream)
        {
            if (type != shader_param_typeType.float3 &&
                type != shader_param_typeType.float4)
            {
                return null;
            }

            List<param_anim_targetType> param_anim_targets = new List<param_anim_targetType>();
            foreach (var original_target in originalColorAnim.original_color_anim_target)
            {
                int componentIndex = ConvertColorTargetToComponentIndex(original_target.target);
                param_anim_targetType target = CreateParamAnimTargetFromOriginalAnim(original_target, componentIndex, type, info, streams);
                if (target.Curve != null)
                {
                    // ストリームを追加
                    if (addStream)
                    {
                        // 参照コピー
                        streams.Add(streams[target.Curve.stream_index]);
                        target.Curve.stream_index = streams.Count - 1;
                    }
                }

                param_anim_targets.Add(target);
            }

            return new param_animType()
            {
                id = paramAnimId,
                type = type,
                original_hint = originalColorAnim.hint,
                param_anim_target = param_anim_targets.ToArray(),
            };
        }

        public static param_animType CreateParamAnimFromOriginalSrtAnim(
            original_texsrt_animType originalTexsrtAnim,
            string paramAnimId,
            shader_param_typeType type,
            List<G3dStream> streams,
            IG3dShaderParamAnimInfo info,
            bool addStream)
        {
            List<param_anim_targetType> param_anim_targets = new List<param_anim_targetType>();

            // モードの設定
            if (type == shader_param_typeType.texsrt)
            {
                param_anim_targets.Add(CreateTexSrtModeParamAnimTargetFromOriginalAnim(originalTexsrtAnim.mode));
            }
            else if (type != shader_param_typeType.srt2d)
            {
                return null;
            }

            foreach (var original_target in originalTexsrtAnim.original_texsrt_anim_target)
            {
                int componentIndex = ConvertTexSRTTargetToComponentIndex(original_target.target, type);
                param_anim_targetType target = CreateParamAnimTargetFromOriginalAnim(original_target, componentIndex, type, info, streams);
                if (target.Curve != null)
                {
                    // ストリームを追加
                    if (addStream)
                    {
                        // 参照コピー
                        streams.Add(streams[target.Curve.stream_index]);
                        target.Curve.stream_index = streams.Count - 1;
                    }
                }

                param_anim_targets.Add(target);
            }

            return new param_animType()
            {
                id = paramAnimId,
                type = type,
                original_hint = originalTexsrtAnim.hint,
                param_anim_target = param_anim_targets.ToArray(),
            };
        }

        public static param_animType CreateParamAnimFromOriginalSrtAnim(
            original_texsrt_animType originalTexsrtAnim,
            string paramAnimId,
            shader_param_typeType type,
            List<G3dStream> streams,
            material_anim_infoType info,
            bool addStream)
        {
            List<param_anim_targetType> param_anim_targets = new List<param_anim_targetType>();

            // モードの設定
            if (type == shader_param_typeType.texsrt)
            {
                param_anim_targets.Add(CreateTexSrtModeParamAnimTargetFromOriginalAnim(originalTexsrtAnim.mode));
            }
            else if (type != shader_param_typeType.srt2d)
            {
                return null;
            }

            foreach (var original_target in originalTexsrtAnim.original_texsrt_anim_target)
            {
                int componentIndex = ConvertTexSRTTargetToComponentIndex(original_target.target, type);
                param_anim_targetType target = CreateParamAnimTargetFromOriginalAnim(original_target, componentIndex, type, info, streams);
                if (target.Curve != null)
                {
                    // ストリームを追加
                    if (addStream)
                    {
                        // 参照コピー
                        streams.Add(streams[target.Curve.stream_index]);
                        target.Curve.stream_index = streams.Count - 1;
                    }
                }

                param_anim_targets.Add(target);
            }

            return new param_animType()
            {
                id = paramAnimId,
                type = type,
                original_hint = originalTexsrtAnim.hint,
                param_anim_target = param_anim_targets.ToArray(),
            };
        }

        /// <summary>
        /// param_anim を生成
        /// </summary>
        public static param_animType CreateParamAnimFromOriginal(
            original_material_animType original,
            string paramAnimId,
            shader_param_typeType type,
            string hint,
            List<G3dStream> streams,
            IG3dShaderParamAnimInfo info,
            bool addStream)
        {
            if (original == null)
            {
                return null;
            }

            switch (type)
            {
                case shader_param_typeType.float3:
                case shader_param_typeType.float4:
                    // オリジナルを取得
                    original_color_animType originalColorAnim = GetOriginalColorAnim(original, paramAnimId, hint);
                    if (originalColorAnim != null)
                    {
                        return CreateParamAnimFromOriginalColorAnim(originalColorAnim, paramAnimId, type, streams, info, addStream);
                    }
                    break;
                case shader_param_typeType.texsrt:
                case shader_param_typeType.srt2d:
                    // オリジナルを取得
                    original_texsrt_animType originalTexsrtAnim = GetOriginalTexSRTAnim(original, paramAnimId, hint);
                    if (originalTexsrtAnim != null)
                    {
                        return CreateParamAnimFromOriginalSrtAnim(originalTexsrtAnim, paramAnimId, type, streams, info, addStream);
                    }
                    break;
            }

            return null;
        }

        /// <summary>
        /// コンポーネントインデックスを取得
        /// </summary>
        private static int ConvertColorTargetToComponentIndex(original_color_anim_target_targetType type)
        {
            switch (type)
            {
                case original_color_anim_target_targetType.color_r:
                    return 0;
                case original_color_anim_target_targetType.color_g:
                    return 1;
                case original_color_anim_target_targetType.color_b:
                    return 2;
            }

            return -1;
        }

        /// <summary>
        /// コンポーネントインデックスを取得
        /// </summary>
        private static int ConvertTexSRTTargetToComponentIndex(original_texsrt_anim_target_targetType type, shader_param_typeType paramType)
        {
            int offset = 0;
            if (paramType == shader_param_typeType.texsrt)
            {
                offset = 1;
            }

            switch (type)
            {
                case original_texsrt_anim_target_targetType.scale_x:
                    return 0 + offset;
                case original_texsrt_anim_target_targetType.scale_y:
                    return 1 + offset;
                case original_texsrt_anim_target_targetType.rotate:
                    return 2 + offset;
                case original_texsrt_anim_target_targetType.translate_x:
                    return 3 + offset;
                case original_texsrt_anim_target_targetType.translate_y:
                    return 4 + offset;
            }

            return -1;
        }

        /// <summary>
        /// オリジナルのアニメーションを適用
        /// </summary>
        public static void ApplyOriginalAnimation(shader_param_animType animation, shading_modelType shadingModel, List<G3dStream> streams, bool force)
        {
            if (animation.original_material_anim_array == null)
            {
                // オリジナルがなければ何もしない
                return;
            }

            if (!force && animation.shader_param_mat_anim_array != null)
            {
                // 既に割り当て済みなら何もしない
                return;
            }

            List<shader_param_mat_animType> shader_param_mat_anims = new List<shader_param_mat_animType>();

            // オリジナルから作成
            foreach (var original in animation.original_material_anim_array.GetItems())
            {
                var shader_param_mat_anim = CreateShaderParamMatAnimFromOriginal(
                    original,
                    shadingModel,
                    animation.shader_param_anim_info,
                    streams);
                if (shader_param_mat_anim != null)
                {
                    shader_param_mat_anims.Add(shader_param_mat_anim);
                }
            }

            // 置き換え
            animation.shader_param_mat_anim_array = shader_param_mat_anims.OrderBy(x => x.mat_name).ToG3dArrayElement<shader_param_mat_animType, shader_param_mat_anim_arrayType>();

            // 無駄なストリームを削除
            StreamUtility.SortStream(animation, streams);
        }

        /// <summary>
        /// オリジナルのアニメーションを適用
        /// </summary>
        public static void ApplyOriginalAnimation(material_animType animation, shading_modelType shadingModel, List<G3dStream> streams, bool force)
        {
            if (animation.original_per_material_anim_array == null)
            {
                // オリジナルがなければ何もしない
                return;
            }

            if (!force && animation.per_material_anim_array.GetItems().Any(x => x.shader_param_anim_array != null))
            {
                // 既に割り当て済みなら何もしない
                return;
            }

            var new_per_material_anims = new List<per_material_animType>();
            var updated = new HashSet<per_material_animType>();
            foreach (var original in animation.original_per_material_anim_array.GetItems())
            {
                var per_material_anim = animation.per_material_anim_array.GetItems().FirstOrDefault(x => x.mat_name == original.mat_name);

                // オリジナルから作成
                var assigned = CreateShaderParamMatAnimFromOriginal(
                    original,
                    shadingModel,
                    animation.material_anim_info,
                    streams);

                if (per_material_anim != null)
                {
                    per_material_anim.shader_param_anim_array = assigned != null ? assigned.shader_param_anim_array : null;
                    updated.Add(per_material_anim);
                }
                else if (assigned != null)
                {
                    new_per_material_anims.Add(assigned);
                }
            }

            foreach (var per_material_anim in animation.per_material_anim_array.GetItems())
            {
                if (!updated.Contains(per_material_anim))
                {
                    per_material_anim.shader_param_anim_array = null;
                }
            }

            // 置き換え
            animation.per_material_anim_array = animation.per_material_anim_array.GetItems().Concat(new_per_material_anims)
                .Where(x => x.shader_param_anim_array != null || x.tex_pattern_anim_array != null || x.material_visibility_anim != null)
                .OrderBy(x => x.mat_name).ToG3dArrayElement<per_material_animType, per_material_anim_arrayType>();

            // 無駄なストリームを削除
            StreamUtility.SortStream(animation, streams);
        }

        /// <summary>
        /// shader_param_mat_anim をオリジナルから作成
        /// </summary>
        private static shader_param_mat_animType CreateShaderParamMatAnimFromOriginal(
            original_material_animType original,
            shading_modelType shadingModel,
            shader_param_anim_infoType info,
            List<G3dStream> streams)
        {
            // param_anim をオリジナルから生成
            List<param_animType> param_anims = new List<param_animType>();
            foreach (var uniform in shadingModel.MaterialUniforms())
            {
                param_animType param_anim = CreateParamAnimFromOriginal(original, uniform.id, uniform.type, uniform.hint, streams, info, true);
                if (param_anim != null)
                {
                    param_anims.Add(param_anim);
                }
            }

            // shader_param_mat_anim を生成
            if (param_anims.Any())
            {
                return new shader_param_mat_animType()
                {
                    mat_name = original.mat_name,
                    param_anim_array = param_anims.ToG3dArrayElement<param_animType, param_anim_arrayType>(),
                };
            }

            return null;
        }

        /// <summary>
        /// per_material_anim をオリジナルから作成
        /// </summary>
        private static per_material_animType CreateShaderParamMatAnimFromOriginal(
            original_material_animType original,
            shading_modelType shadingModel,
            material_anim_infoType info,
            List<G3dStream> streams)
        {
            // param_anim をオリジナルから生成
            List<param_animType> param_anims = new List<param_animType>();
            foreach (var uniform in shadingModel.MaterialUniforms())
            {
                param_animType param_anim = CreateParamAnimFromOriginal(original, uniform.id, uniform.type, uniform.hint, streams, info, true);
                if (param_anim != null)
                {
                    param_anims.Add(param_anim);
                }
            }

            // shader_param_mat_anim を生成
            if (param_anims.Any())
            {
                return new per_material_animType()
                {
                    mat_name = original.mat_name,
                    shader_param_anim_array = param_anims.ToG3dArrayElement<param_animType, shader_param_anim_arrayType>(),
                };
            }

            return null;
        }
        #endregion

        #region アップデート
        /// <summary>
        /// アニメーションのエラーメッセージ、対応するユニフォーム、変換テーブルの取得
        /// </summary>
        public static IEnumerable<Tuple<string, param_animType, uniform_varType, int[]>> IsConsistentWithShadingModel(
            shader_param_mat_animType shader_param_mat_anim,
            shading_modelType shading_model,
            IfShaderAssignUtility.InconsistentMessageType type = IfShaderAssignUtility.InconsistentMessageType.Plan)
        {
            foreach (var param_anim in shader_param_mat_anim.param_anim_array.GetItems())
            {
                uniform_varType uniform;
                string message;
                int[] table;
                if (!IfAnimationAssignUtility.IsConsistentWithShadingModel(param_anim, shading_model, out uniform, out table, out message, type))
                {
                    yield return new Tuple<string, param_animType, uniform_varType, int[]>(
                        message,
                        param_anim,
                        uniform,
                        table);
                }
            }
        }

        /// <summary>
        /// アニメーションのエラーメッセージ、対応するユニフォーム、変換テーブルの取得
        /// </summary>
        public static IEnumerable<Tuple<string, param_animType, uniform_varType, int[]>> IsConsistentWithShadingModel(
            per_material_animType shader_param_mat_anim,
            shading_modelType shading_model,
            IfShaderAssignUtility.InconsistentMessageType type = IfShaderAssignUtility.InconsistentMessageType.Plan)
        {
            foreach (var param_anim in shader_param_mat_anim.shader_param_anim_array.GetItems())
            {
                uniform_varType uniform;
                string message;
                int[] table;
                if (!IfAnimationAssignUtility.IsConsistentWithShadingModel(param_anim, shading_model, out uniform, out table, out message, type))
                {
                    yield return new Tuple<string, param_animType, uniform_varType, int[]>(
                        message,
                        param_anim,
                        uniform,
                        table);
                }
            }
        }

        /// <summary>
        /// 変換
        /// </summary>
        public static void Convert(param_animType anim, uniform_varType uniform, int[] table)
        {
            if (uniform != null && table != null)
            {
                anim.type = uniform.type;
                anim.param_anim_target = (from i in Enumerable.Range(0, table.Length)
                                          let old = anim.param_anim_target.FirstOrDefault(x => x.component_index == table[i])
                                          where old != null
                                          select new param_anim_targetType()
                                          {
                                              component_index = i,
                                              Item = old.Item,
                                              base_value = old.base_value,
                                          }).ToArray();
            }
        }

        /// <summary>
        /// エラーがあるかどうか
        /// </summary>
        /// <param name="param_anim">カーブ</param>
        /// <param name="shading_model">シェーディングモデル</param>
        /// <param name="uniform_var">対応する uniform_var, 無い場合は null</param>
        /// <param name="convertTable">修正時の変換テーブル、修正後の component_index から修正前の component_index への対応、変換不可の場合は null</param>
        /// <param name="message">エラーメッセージ, エラーがない場合は null</param>
        public static bool IsConsistentWithShadingModel(
            param_animType param_anim,
            shading_modelType shading_model,
            out uniform_varType uniform_var,
            out int[] convertTable,
            out string message,
            IfShaderAssignUtility.InconsistentMessageType type)
        {
            convertTable = null;
            uniform_var = shading_model.MaterialUniforms().FirstOrDefault(x => x.id == param_anim.id);

            // ユニフォームがない場合
            if (uniform_var == null)
            {
                message = string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_NoUniform, type), param_anim.id);
                return false;
            }

            // 型に相違がある場合
            if (uniform_var.type != param_anim.type)
            {
                var builder = new StringBuilder();
                builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_InvalidCurveType, type),
                    param_anim.id, param_anim.type.ToString(), uniform_var.id, uniform_var.type);

                GetConvertTable(param_anim.type, uniform_var.type, out convertTable);

                // 自動修正可能な場合
                if (convertTable != null)
                {
                    builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Convert_Curve_Type, type), param_anim.id, uniform_var.type);

                    // カーブが失われる場合
                    int[] table = convertTable;
                    if (param_anim.param_anim_target.Any(x => !table.Contains(x.component_index)))
                    {
                        builder.Append(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Convert_Curve_Type_Lost, type));
                    }
                }

                message = builder.ToString();
                return false;
            }

            message = null;
            return true;
        }

        /// <summary>
        /// 変換テーブルの取得 (修正後の component_index から修正前の component_index へのテーブル)
        /// </summary>
        internal static void GetConvertTable(shader_param_typeType source, shader_param_typeType target, out int[] convertTable)
        {
            var repSource = Representative(source);
            var repTarget = Representative(target);

            if (repSource != repTarget)
            {
                // 変換しない
                convertTable = null;
                return;
            }

            if (repSource == shader_param_typeType.float2x2)
            {
                // 行列のインデックスの変換
                int rowSource;
                int columnSource;
                GetMatrixDimension(source, out rowSource, out columnSource);
                int rowTarget;
                int columnTarget;
                GetMatrixDimension(target, out rowTarget, out columnTarget);
                convertTable = new int[rowTarget * columnTarget];
                for (int i = 0; i < rowTarget; i++)
                {
                    for (int j = 0; j < columnTarget; j++)
                    {
                        int k = i * columnTarget + j;
                        if (i < rowSource && j < columnSource)
                        {
                            convertTable[k] = i * columnSource + j;
                        }
                        else
                        {
                            convertTable[k] = -1;
                        }
                    }
                }
            }
            else if (repSource == shader_param_typeType.srt2d)
            {
                if (source == target)
                {
                    convertTable = Enumerable.Range(0, IfShaderParameterUtility.Dimension(target)).ToArray();
                }
                else if (target == shader_param_typeType.srt3d)
                {
                    convertTable = new int[9]
                    {
                        0, 1, -1,
                        2, -1, -1,
                        3, 4, -1,
                    };
                }
                else
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(target == shader_param_typeType.srt2d);
                    convertTable = new int[5]
                    {
                        0, 1, //2,
                        3, //4, 5,
                        6, 7, //8,
                    };
                }
            }
            else
            {
                // ベクトルの変換
                var dimSource = IfShaderParameterUtility.Dimension(source);
                var dimTarget = IfShaderParameterUtility.Dimension(target);
                convertTable = Enumerable.Range(0, dimSource).Concat(Enumerable.Repeat(-1, dimTarget)).Take(dimTarget).ToArray();
            }
        }

        /// <summary>
        /// 変換可能な型の代表
        /// </summary>
        internal static shader_param_typeType Representative(shader_param_typeType type)
        {
            switch (type)
            {
                case shader_param_typeType.@bool:
                case shader_param_typeType.bool2:
                case shader_param_typeType.bool3:
                case shader_param_typeType.bool4:
                    return shader_param_typeType.@bool;
                case shader_param_typeType.@float:
                case shader_param_typeType.float2:
                case shader_param_typeType.float3:
                case shader_param_typeType.float4:
                    return shader_param_typeType.@float;
                case shader_param_typeType.@int:
                case shader_param_typeType.int2:
                case shader_param_typeType.int3:
                case shader_param_typeType.int4:
                    return shader_param_typeType.@int;
                case shader_param_typeType.@uint:
                case shader_param_typeType.uint2:
                case shader_param_typeType.uint3:
                case shader_param_typeType.uint4:
                    return shader_param_typeType.@uint;
                case shader_param_typeType.float2x2:
                case shader_param_typeType.float2x3:
                case shader_param_typeType.float2x4:
                case shader_param_typeType.float3x2:
                case shader_param_typeType.float3x3:
                case shader_param_typeType.float3x4:
                case shader_param_typeType.float4x2:
                case shader_param_typeType.float4x3:
                case shader_param_typeType.float4x4:
                    return shader_param_typeType.@float2x2;
                case shader_param_typeType.texsrt:
                    return shader_param_typeType.texsrt;
                case shader_param_typeType.srt2d:
                case shader_param_typeType.srt3d:
                    return shader_param_typeType.srt2d;
                default:
                    return type;
            }
        }

        /// <summary>
        /// 次元の取得
        /// </summary>
        private static void GetMatrixDimension(shader_param_typeType type, out int row, out int column)
        {
            switch (type)
            {
                case shader_param_typeType.float2x2:
                    row = 2;
                    column = 2;
                    break;
                case shader_param_typeType.float2x3:
                    row = 2;
                    column = 3;
                    break;
                case shader_param_typeType.float2x4:
                    row = 2;
                    column = 4;
                    break;
                case shader_param_typeType.float3x2:
                    row = 3;
                    column = 2;
                    break;
                case shader_param_typeType.float3x3:
                    row = 3;
                    column = 3;
                    break;
                case shader_param_typeType.float3x4:
                    row = 3;
                    column = 4;
                    break;
                case shader_param_typeType.float4x2:
                    row = 4;
                    column = 2;
                    break;
                case shader_param_typeType.float4x3:
                    row = 4;
                    column = 3;
                    break;
                case shader_param_typeType.float4x4:
                    row = 4;
                    column = 4;
                    break;
                default:
                    row = 0;
                    column = 0;
                    break;
            }
        }
        #endregion
    }
}
