﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // シェーダーパラメーターアニメーションの量子化分析
    public class IfShaderParamAnimQuantizationAnalysisOptimizer
        : IfShaderParamAnimOptimizer
    {
        // コンストラクタ
        public IfShaderParamAnimQuantizationAnalysisOptimizer() :
            base("IfShaderParamAnimQuantizationAnalysisOptimizer_Log") { }

        // プロセス
        public override string Process
        {
            get { return "quantization_analysis"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Empty;
        }

        private int FrameSize;
        private int QuantizedFrameSize;

        private class ParamTarget
        {
            public shader_param_typeType Type { get; set; }
            public param_anim_targetType Target { get; set; } = null;
        }

        // 最適化
        protected override void Optimize()
        {
            if (this.Target.shader_param_mat_anim_array == null) { return; }

            shader_param_animType shaderAnim = this.Target;
            shader_param_anim_infoType shaderAnimInfo = shaderAnim.shader_param_anim_info;
            IDictionary<ParamTarget, IfAnimCurve> curveDic =
                new Dictionary<ParamTarget, IfAnimCurve>();

            // カーブデータを構築する
            foreach (shader_param_mat_animType matAnim in shaderAnim.shader_param_mat_anim_array.shader_param_mat_anim)
            {
                if (matAnim.param_anim_array == null) { continue; }

                foreach (param_animType param in matAnim.param_anim_array.param_anim)
                {
                    if (param.param_anim_target == null) { continue; }

                    foreach (param_anim_targetType target in param.param_anim_target)
                    {
                        if (target.Item != null)
                        {
                            IfAnimCurve animCurve = new IfAnimCurve();
                            if (param.type == shader_param_typeType.@bool ||
                                param.type == shader_param_typeType.bool2 ||
                                param.type == shader_param_typeType.bool3 ||
                                param.type == shader_param_typeType.bool4)
                            {
                                animCurve.curveType = IfAnimCurve.CurveType.Bool;
                            }
                            else if (
                                param.type == shader_param_typeType.@int ||
                                param.type == shader_param_typeType.int2 ||
                                param.type == shader_param_typeType.int4 ||
                                param.type == shader_param_typeType.int3 ||
                                param.type == shader_param_typeType.@uint ||
                                param.type == shader_param_typeType.uint2 ||
                                param.type == shader_param_typeType.uint3 ||
                                param.type == shader_param_typeType.uint4)
                            {
                                animCurve.curveType = IfAnimCurve.CurveType.Int;
                            }
                            else
                            {
                                animCurve.curveType = IfAnimCurve.CurveType.Float;
                            }

                            // シェーダーパラメータータイプが srt2d と srt3d と texsrt の場合はランタイムの動作に
                            // 合わせて Radian に変換してから量子化解析を行う。
                            bool isDegree = false;
                            if (param.type == shader_param_typeType.srt2d)
                            {
                                if (target.component_index == (int)IfShaderParamAnimQuantizationAnalysis.Srt2dIndex.Rotate)
                                {
                                    isDegree = true;
                                }
                            }
                            else if (param.type == shader_param_typeType.srt3d)
                            {
                                if (target.component_index == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateX ||
                                    target.component_index == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateY ||
                                    target.component_index == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateZ)
                                {
                                    isDegree = true;
                                }
                            }
                            else if (param.type == shader_param_typeType.texsrt)
                            {
                                if (target.component_index == (int)IfShaderParamAnimQuantizationAnalysis.TexsrtIndex.Rotate)
                                {
                                    isDegree = true;
                                }
                            }

                            animCurve.BuildIfAnimCurve(target.Item, isDegree, this.Streams);

                            ParamTarget paramTarget = new ParamTarget();
                            paramTarget.Type = param.type;
                            paramTarget.Target = target;
                            curveDic.Add(paramTarget, animCurve);
                        }
                    }
                }
            }

            foreach (var curve in curveDic)
            {
                // カーブの量子化を評価して中間ファイルに適用する
                IfQuantizationAnalysisResult result =
                    IfShaderParamAnimQuantizationAnalysis.Analyse(
                        shaderAnimInfo,
                        curve.Key.Type,
                        curve.Key.Target.component_index,
                        curve.Value,
                        false);

                this.EnableProcessLog |= !result.skipped;

                ApplyQuantize(curve.Key, result);
            }
        }

        /// <summary>
        /// IfAnimCurve を生成します。
        /// </summary>
        public static IfAnimCurve BuildIfAnimCurve(IG3dQuantizedCurve curve, shader_param_typeType paramType, int componentIndex, G3dStream stream)
        {
            IfAnimCurve animCurve = new IfAnimCurve();
            switch (paramType)
            {
                case shader_param_typeType.@bool:
                case shader_param_typeType.bool2:
                case shader_param_typeType.bool3:
                case shader_param_typeType.bool4:
                    animCurve.curveType = IfAnimCurve.CurveType.Bool;
                    break;
                case shader_param_typeType.@int:
                case shader_param_typeType.int2:
                case shader_param_typeType.int4:
                case shader_param_typeType.int3:
                case shader_param_typeType.@uint:
                case shader_param_typeType.uint2:
                case shader_param_typeType.uint3:
                case shader_param_typeType.uint4:
                    animCurve.curveType = IfAnimCurve.CurveType.Int;
                    break;
                default:
                    animCurve.curveType = IfAnimCurve.CurveType.Float;
                    break;
            }

            // シェーダーパラメータータイプが srt2d と srt3d の場合はランタイムの動作に
            // 合わせて Radian に変換してから量子化解析を行う。
            bool isDegree = false;
            if (paramType == shader_param_typeType.srt2d)
            {
                if (componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.Srt2dIndex.Rotate)
                {
                    isDegree = true;
                }
            }
            else if (paramType == shader_param_typeType.srt3d)
            {
                if (componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateX ||
                    componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateY ||
                    componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateZ)
                {
                    isDegree = true;
                }
            }
            else if (paramType == shader_param_typeType.texsrt)
            {
                if (componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.TexsrtIndex.Rotate)
                {
                    isDegree = true;
                }
            }

            animCurve.BuildIfAnimCurve(curve, isDegree, stream);

            return animCurve;
        }

        /// <summary>
        /// 量子化タイプを適応します。
        /// </summary>
        private void ApplyQuantize(ParamTarget paramTarget, IfQuantizationAnalysisResult analysisResult)
        {
            curve_frame_typeType frameType = curve_frame_typeType.none;
            curve_key_typeType keyType = curve_key_typeType.none;
            int frameCount;

            if (paramTarget.Target.Item is hermite_curveType)
            {
                hermite_curveType hermiteCurve = paramTarget.Target.Item as hermite_curveType;
                hermiteCurve.frame_type = analysisResult.frameType;
                hermiteCurve.key_type = analysisResult.keyType;
                hermiteCurve.scale = analysisResult.scale;
                hermiteCurve.offset = analysisResult.offset;

                frameType = hermiteCurve.frame_type;
                keyType = hermiteCurve.key_type;
                frameCount = hermiteCurve.count;
            }
            else if (paramTarget.Target.Item is linear_curveType)
            {
                linear_curveType linearCurve = paramTarget.Target.Item as linear_curveType;
                linearCurve.frame_type = analysisResult.frameType;
                linearCurve.key_type = analysisResult.keyType;
                linearCurve.scale = analysisResult.scale;
                linearCurve.offset = analysisResult.offset;

                frameType = linearCurve.frame_type;
                keyType = linearCurve.key_type;
                frameCount = linearCurve.count;
            }
            else if (paramTarget.Target.Item is step_curveType)
            {
                step_curveType stepCurve = paramTarget.Target.Item as step_curveType;
                stepCurve.frame_type = analysisResult.frameType;
                stepCurve.key_type = analysisResult.keyType;
                stepCurve.scale = analysisResult.scale;
                stepCurve.offset = analysisResult.offset;

                frameType = stepCurve.frame_type;
                keyType = stepCurve.key_type;
                frameCount = stepCurve.count;
            }
            else
            {
                throw new NotImplementedException();
            }

            this.FrameSize += frameCount * sizeof(float);
            switch (frameType)
            {
                case curve_frame_typeType.none:
                case curve_frame_typeType.frame32:
                    this.QuantizedFrameSize += frameCount * sizeof(float);
                    break;
                case curve_frame_typeType.frame16:
                    this.QuantizedFrameSize += frameCount * sizeof(ushort);
                    break;
                case curve_frame_typeType.frame8:
                    this.QuantizedFrameSize += frameCount * sizeof(byte);
                    break;
            }
        }
    }
}
