﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    /// <summary>
    /// 最適化の結果を管理するためのクラスです。
    /// </summary>
    public class IfOptimizedResult
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public IfOptimizedResult()
        {
            this.Result = new Dictionary<IfPolygonMesh, List<IfPolygonPrimitive>>();
        }

        /// <summary>
        /// 最適結果です。
        /// </summary>
        public IDictionary<IfPolygonMesh, List<IfPolygonPrimitive>> Result { get; private set; }

        /// <summary>
        /// 描画単位となるポリゴンの集合(プリミティブ)です。
        /// </summary>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns></returns>
        public List<IfPolygonPrimitive> this[IfPolygonMesh polygonMesh]
        {
            get
            {
                return this.Result[polygonMesh];
            }

            set
            {
                this.Result[polygonMesh] = value;
            }
        }
    }

    public static class IfPolygonAnalyzer
    {
        /// <summary>
        /// モデルを解析し最適化対象の情報を抽出します。
        /// </summary>
        /// <param name="polygonMeshs">ポリゴンメッシュです。</param>
        /// <param name="model">モデルです。</param>
        public static void Analyze(IList<IfPolygonMesh> polygonMeshs, modelType model, List<G3dStream> streams)
        {
            if (model.shape_array != null)
            {
                foreach (var shape in model.shape_array.shape)
                {
                    meshType mesh = shape.mesh_array.mesh[0];
                    IfPolygonMesh pm = new IfPolygonMesh();
                    pm.Data = mesh;
                    if (AnalyzePolygon(pm, mesh, streams))
                    {
                        polygonMeshs.Add(pm);
                    }
                }
            }
        }

        /// <summary>
        /// 最適化結果をモデルに適用します。
        /// </summary>
        /// <param name="optimizedResult">最適化結果です。</param>
        /// <param name="model">対象モデルです。</param>
        public static void Apply(IfOptimizedResult optimizedResult, modelType model, List<G3dStream> streams)
        {
            foreach (var pair in optimizedResult.Result)
            {
                IfPolygonMesh pm = pair.Key;
                IList<IfPolygonPrimitive> primitives = pair.Value;

                Apply(pm, primitives, model, streams);
            }
        }

        /// <summary>
        /// 最適化結果をモデルに適用します。
        /// </summary>
        /// <param name="optimizedResult">最適化結果です。</param>
        /// <param name="model">対象モデルです。</param>
        public static void Apply(IfPolygonMesh pm, IList<IfPolygonPrimitive> primitives, modelType model, List<G3dStream> streams)
        {
            meshType mesh = pm.Data as meshType;

            int numTriangles = primitives.Sum((prim) => prim.Triangles.Count);
            Ensure.Operation.True(numTriangles * IfTriangle.VertexCount == mesh.count);

            mesh.mode = mesh_modeType.triangles;

            G3dStream stream = streams[mesh.stream_index];
            int dataIndex = 0;
            foreach (IfPolygonPrimitive primitive in primitives)
            {
                foreach (IfTriangle triangle in primitive.Triangles)
                {
                    foreach (int p in triangle.PositionIds)
                    {
                        stream.IntData[dataIndex] = p;
                        ++dataIndex;
                    }
                }
            }
        }

        /// <summary>
        /// メッシュを分析して最適化情報を抽出します。
        /// </summary>
        /// <returns>抽出に成功した場合には true を返します。</returns>
        private static bool AnalyzePolygon(IfPolygonMesh polygonMesh, meshType mesh, List<G3dStream> streams)
        {
            if (mesh.stream_index < streams.Count &&
                mesh.stream_index >= 0 &&
                streams[mesh.stream_index].IntData.Count != 0)
            {
                switch (mesh.mode)
                {
                    case mesh_modeType.triangles:
                        return BuildTriangles(polygonMesh, mesh, streams[mesh.stream_index].IntData);
#if false // FanおよびStripをTrianglesの変換には今のところ未対応。
                case IndexStreamPrimitiveModeCtr.TriangleFan:
                    return this.BuildTriangleFan(polygonMesh, indexStream);
                case IndexStreamPrimitiveModeCtr.TriangleStrip:
                    return this.BuildTriangleStrip(polygonMesh, indexStream);
#endif
                    default:
                        throw new NotImplementedException();
                }
            }

            return false;
        }
        /// <summary>
        /// トライアングルリストデータからリストを構築します。
        /// </summary>
        /// <returns>構築に成功したら true を返します。</returns>
        private static bool BuildTriangles(IfPolygonMesh polygonMesh, meshType mesh, List<int> stream)
        {
            if (mesh.count < 3)
            {
                return false;
            }

            if ((mesh.count % IfTriangle.VertexCount) != 0)
            {
                return false;
            }

            int triangleSize = mesh.count / IfTriangle.VertexCount;

            IfPolygonPrimitive polygonPrimitive = new IfPolygonPrimitive();
            for (int i = 0; i < triangleSize; ++i)
            {
                IfAnalyzedTriangle at = new IfAnalyzedTriangle();
                at.Triangle.PositionIds[0] = stream[(i * IfTriangle.VertexCount) + 0];
                at.Triangle.PositionIds[1] = stream[(i * IfTriangle.VertexCount) + 1];
                at.Triangle.PositionIds[2] = stream[(i * IfTriangle.VertexCount) + 2];
                polygonMesh.AnalyzedTriangles.Add(at);
                polygonPrimitive.Triangles.Add(at.Triangle);
            }

            polygonMesh.PolygonPrimitives.Add(polygonPrimitive);
            return true;
        }

#if false
        /// <summary>
        /// トライアングルファンデータからリストを構築します。
        /// </summary>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <param name="indexStream">頂点インデックスストリームです。</param>
        /// <returns>構築に成功したら true を返します。</returns>
        private bool BuildTriangleFan(IfPolygonMesh polygonMesh, IIndexStreamCtr indexStream)
        {
            if (indexStream.Count < 3)
            {
                return false;
            }

            int offset = Triangle.VertexCount - 1;
            int triangleSize = indexStream.Count;

            IfPolygonPrimitive polygonPrimitive = new IfPolygonPrimitive();
            for (int i = offset; i < triangleSize; ++i)
            {
                IfAnalyzedTriangle at = new IfAnalyzedTriangle();
                //// TODO: 型の不統一
                at.Triangle.PositionIds[0] = (int)indexStream[0];
                at.Triangle.PositionIds[1] = (int)indexStream[i - 1];
                at.Triangle.PositionIds[2] = (int)indexStream[i];
                polygonMesh.AnalyzedTriangles.Add(at);
                polygonPrimitive.Triangles.Add(at.Triangle);
            }

            polygonMesh.PolygonPrimitives.Add(polygonPrimitive);
            return true;
        }

        /// <summary>
        /// トライアングルストリップデータからリストを構築します。
        /// </summary>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <param name="indexStream">頂点インデックスストリームです。</param>
        /// <returns>構築に成功したら true を返します。</returns>
        private bool BuildTriangleStrip(IfPolygonMesh polygonMesh, IIndexStreamCtr indexStream)
        {
            if (indexStream.Count < 3)
            {
                return false;
            }

            int offset = IfTriangle.VertexCount - 1;
            int triangleSize = indexStream.Count;

            IfPolygonPrimitive polygonPrimitive = new IfPolygonPrimitive();
            for (int i = offset; i < triangleSize; ++i)
            {
                IfAnalyzedTriangle at = new IfAnalyzedTriangle();
                //// TODO: 型の不統一
                at.Triangle.PositionIds[0] = (int)indexStream[i - 2];
                at.Triangle.PositionIds[1] = (int)indexStream[i - 1];
                at.Triangle.PositionIds[2] = (int)indexStream[i];
                polygonMesh.AnalyzedTriangles.Add(at);
                polygonPrimitive.Triangles.Add(at.Triangle);
            }

            polygonMesh.PolygonPrimitives.Add(polygonPrimitive);
            return true;
        }
#endif
    }
}
