﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;

namespace nw.g3d.iflib
{
    /// <summary>
    /// シンプルなポリゴン最適化クラスです。
    /// </summary>
    public class IfOptimizeAlgorithmSimple : IfOptimizeAlgorithm
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public IfOptimizeAlgorithmSimple()
            : this(
                new FindTriangleDelegate(FindOptimizeTriangleDefault),
                new FindTriangleDelegate(FindNextTriangleDefault))
        {
        }

        /// <summary>
        /// 最適化のアルゴリズムを指定可能なコンストラクタです。
        /// </summary>
        /// <param name="findOptimizeTraiangle">最適なトライアングルを探すデリゲートです。</param>
        /// <param name="findNextTraiangle">次の起点となるトライアングルを探すデリゲートです。</param>
        public IfOptimizeAlgorithmSimple(
            FindTriangleDelegate findOptimizeTraiangle,
            FindTriangleDelegate findNextTraiangle)
        {
            this.FindOptimizeTriangle = findOptimizeTraiangle;
            this.FindNextTriangle = findNextTraiangle;
        }

        /// <summary>
        /// トライアングルを検索するデリゲートです。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>トライアングルを返します。</returns>
        public delegate IfAnalyzedTriangle FindTriangleDelegate(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh,
            int startIndex);

        /// <summary>
        /// 最適なトライアングルを探すデリゲートです。
        /// </summary>
        public FindTriangleDelegate FindOptimizeTriangle { get; private set; }

        /// <summary>
        /// 次の起点となるトライアングルを探すデリゲートです。
        /// </summary>
        public FindTriangleDelegate FindNextTriangle { get; private set; }

        //-----------------------------------------------------
        // 最適化用切替可能メソッド
        //-----------------------------------------------------
        // TODO キャッシュを考慮した探索基準も設ける

        /// <summary>
        /// 現在のトライアングルに対して最適なトライアングルを探す処理（標準版）です。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>
        /// 未使用の隣接トライアングルのうち最初のものを返します。
        /// 2頂点を共有する隣接トライアングルを優先します。
        /// </returns>
        public static IfAnalyzedTriangle FindOptimizeTriangleDefault(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh,
            int startIndex)
        {
            IfAnalyzedTriangle found = IfAnalyzedTriangle.FindUnused(current.SharedVertex2Triangles, startIndex);

            if (found == null)
            {
                found = IfAnalyzedTriangle.FindUnused(current.SharedVertex1Triangles, startIndex);
            }

            return found;
        }

        /// <summary>
        /// 現在のトライアングルに対して最適なトライアングルを探す処理（ランダム版）です。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>
        /// ランダムなトライアングルを返します。
        /// 2頂点を共有する隣接トライアングルを優先します。
        /// </returns>
        public static IfAnalyzedTriangle FindOptimizeTriangleRandom(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh)
        {
            IfAnalyzedTriangle found = IfAnalyzedTriangle.DrawLot(current.SharedVertex2Triangles);

            if (found == null)
            {
                found = IfAnalyzedTriangle.DrawLot(current.SharedVertex1Triangles);
            }

            return found;
        }

        /// <summary>
        /// 現在のトライアングルに対して最適なトライアングルを探す処理（隅を探す版）です。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>
        /// 隣接トライアングル数が最小のトライアングルからランダムに返します。
        /// 2頂点を共有する隣接トライアングルを優先します。
        /// </returns>
        public static IfAnalyzedTriangle FindOptimizeTriangleNook(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh)
        {
            IfAnalyzedTriangle found = null;

            if (0 < current.SharedVertex2Triangles.Count)
            {
                int count = current.SharedVertex2Triangles.Min(
                    (triangle) => triangle.IsUsed ? int.MaxValue : triangle.CountUnusedAdjacent());
                found = IfAnalyzedTriangle.DrawLotByAdjacentCount(current.SharedVertex2Triangles, count);
            }

            if (found == null && 0 < current.SharedVertex1Triangles.Count)
            {
                int count = current.SharedVertex1Triangles.Min(
                    (triangle) => triangle.IsUsed ? int.MaxValue : triangle.CountUnusedAdjacent());
                found = IfAnalyzedTriangle.DrawLotByAdjacentCount(current.SharedVertex1Triangles, count);
            }

            return found;
        }

        /// <summary>
        /// 現在のトライアングルに対して最適なトライアングルを探す処理（密集を探す版）です。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>
        /// 隣接トライアングル数が最大のトライアングルからランダムに返します。
        /// 2頂点を共有する隣接トライアングルを優先します。
        /// </returns>
        public static IfAnalyzedTriangle FindOptimizeTriangleCrowd(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh)
        {
            IfAnalyzedTriangle found = null;

            if (0 < current.SharedVertex2Triangles.Count)
            {
                int count = current.SharedVertex2Triangles.Max(
                    (triangle) => triangle.IsUsed ? -1 : triangle.CountUnusedAdjacent());
                found = IfAnalyzedTriangle.DrawLotByAdjacentCount(current.SharedVertex2Triangles, count);
            }

            if (found == null && 0 < current.SharedVertex1Triangles.Count)
            {
                int count = current.SharedVertex1Triangles.Max(
                    (triangle) => triangle.IsUsed ? -1 : triangle.CountUnusedAdjacent());
                found = IfAnalyzedTriangle.DrawLotByAdjacentCount(current.SharedVertex1Triangles, count);
            }

            return found;
        }

        /// <summary>
        /// 次の起点となるトライアングルを探す処理（標準版）です。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>未使用の最初のトライアングルを返します。</returns>
        public static IfAnalyzedTriangle FindNextTriangleDefault(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh,
            int startIndex)
        {
            IfAnalyzedTriangle found = IfAnalyzedTriangle.FindUnused(polygonMesh.AnalyzedTriangles, startIndex);
            return found;
        }

        /// <summary>
        /// 次の起点となるトライアングルを探す処理（ランダム版）です。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>未使用のトライアングルの１つをランダムに返します。</returns>
        public static IfAnalyzedTriangle FindNextTriangleRandom(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh,
            int startIndex)
        {
            IfAnalyzedTriangle found = IfAnalyzedTriangle.DrawLot(polygonMesh.AnalyzedTriangles);
            return found;
        }

        /// <summary>
        /// 次の起点となるトライアングルを探す処理（隅を探す版）です。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>
        /// 隣接トライアングル数が最小のトライアングルからランダムに返します。
        /// </returns>
        public static IfAnalyzedTriangle FindNextTriangleNook(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh,
            int startIndex)
        {
            int count = polygonMesh.AnalyzedTriangles.Min(
                (triangle) => triangle.IsUsed ? int.MaxValue : triangle.CountUnusedAdjacent());
            IfAnalyzedTriangle found =
                IfAnalyzedTriangle.DrawLotByAdjacentCount(polygonMesh.AnalyzedTriangles, count);

            return found;
        }

        /// <summary>
        /// 次の起点となるトライアングルを探す処理（密集を探す版）です。
        /// </summary>
        /// <param name="current">現在のトライアングルです。</param>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <returns>
        /// 隣接トライアングル数が最大のトライアングルからランダムに返します。
        /// </returns>
        public static IfAnalyzedTriangle FindNextTriangleCrowd(
            IfAnalyzedTriangle current,
            IfPolygonMesh polygonMesh)
        {
            int count = polygonMesh.AnalyzedTriangles.Max(
                (triangle) => triangle.IsUsed ? -1 : triangle.CountUnusedAdjacent());
            IfAnalyzedTriangle found =
                IfAnalyzedTriangle.DrawLotByAdjacentCount(polygonMesh.AnalyzedTriangles, count);

            return found;
        }

        //-----------------------------------------------------
        // IOptimizeAlgorithmメソッド
        //-----------------------------------------------------

        /// <summary>
        /// トライアングルリストの情報を事前に解析します。
        /// </summary>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        public void Analyze(IfPolygonMesh polygonMesh)
        {
            polygonMesh.AnalyzeTriangles();
        }

        /// <summary>
        /// トライアングルリストの頂点キャッシュ最適化を行います。
        /// </summary>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <param name="optimizedPolygonPrimitive">結果を格納します。</param>
        public void Optimize(IfPolygonMesh polygonMesh, IList<IfPolygonPrimitive> optimizedPolygonPrimitive, int startIndex)
        {
            IfPolygonPrimitive polygonPrimitive = new IfPolygonPrimitive();

            // 先頭のトライアングルを探す処理
            IfAnalyzedTriangle current = this.FindNextTriangle(null, polygonMesh, polygonMesh.Adjacencys[startIndex][0]);
            if (current == null)
            {
                throw new System.Exception("Not found triangle.");
            }

            current.IsUsed = true;
            polygonPrimitive.Triangles.Add(current.Triangle);

            int optimizeTriangleNum = 1;
            while (optimizeTriangleNum < polygonMesh.AnalyzedTriangles.Count)
            {
                IfAnalyzedTriangle found = null;

                // 最適なトライアングルを探す
                if (this.FindOptimizeTriangle != null)
                {
                    found = this.FindOptimizeTriangle(current, polygonMesh, 0);
                }

                // 最適なトライアングルが見つからなかった時に、
                // 次の起点となるトライアングルを探す
                if (found == null && this.FindNextTriangle != null)
                {
                    optimizedPolygonPrimitive.Add(polygonPrimitive);
                    found = this.FindNextTriangle(current, polygonMesh, 0);
                    polygonPrimitive = new IfPolygonPrimitive();
                }

                // 見つかったトライアングルをoptimizedTrianglesリストに追加する
                if (found == null)
                {
                    throw new System.Exception("Not found triangle.");
                }
                else
                {
                    found.IsUsed = true;
                    polygonPrimitive.Triangles.Add(found.Triangle);
                    current = found;
                    ++optimizeTriangleNum;
                }
            }

            optimizedPolygonPrimitive.Add(polygonPrimitive);
        }
    }
}
