﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルのプリミティブ最適化
    public class IfModelPrimitiveOptimizerOld : IfModelOptimizer
    {
        private string argument = null;
        private float preACMR = 0.0f;
        private float postACMR = 0.0f;
        private int preCacheMissCount = 0;
        private int postCacheMissCount = 0;
        private int totalVertexCount = 0;
        private int totalTriangleCount = 0;

        // コンストラクタ
        public IfModelPrimitiveOptimizerOld(string argument) :
            base("IfModelPrimitiveOptimizer_Log")
        {
            if (!string.IsNullOrEmpty(argument))
            {
                this.argument = argument;
            }
        }

        public override string ToString()
        {
            return string.Format("{0} transformed_vertices_per_triangles[{1:f3}->{2:f3}]", base.ToString(), preACMR, postACMR);
        }

        // プロセス
        public override string Process
        {
            get { return "optimize_primitive_old"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendFormat(
                "transformed_vertices[{0}({1:f0}%)/{2}({3:f0}%)/{4}]",
                postCacheMissCount, 100f * postCacheMissCount / totalVertexCount,
                preCacheMissCount, 100f * preCacheMissCount / totalVertexCount,
                totalVertexCount);
            builder.AppendFormat(
                " transformed_vertices_per_triangles[{0:f3}/{1:f3}]",
                postACMR, preACMR);
            return builder.ToString();
        }

        // 最適化
        protected override void Optimize()
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            foreach (shapeType shape in this.Target.shape_array.shape)
            {
                if (shape.mesh_array.mesh.Length > 1)
                {
                    IfStrings.Throw("IfModelOptimizer_Error_UnitedModelUnsupported");
                }
            }

            List<IfPolygonMesh> polygonMesh = new List<IfPolygonMesh>();
            IfPolygonAnalyzer.Analyze(polygonMesh, this.Target, this.Streams);

            // 最適化するべきポリゴンが存在しない
            if (polygonMesh.Count == 0)
            {
                return;
            }

            IList<IfPseudoVertexShader> preVertexShaders
                = new List<IfPseudoVertexShader>(polygonMesh.Count);

            preACMR = 0.0f;
            preCacheMissCount = 0;
            totalVertexCount = 0;
            totalTriangleCount = 0;

            // 事前に解析する。
            //IfParallel.ForEach(polygonMesh, delegate(IfPolygonMesh pm)
            foreach (var pm in polygonMesh)
            {
                IfPseudoVertexShader vs =
                    new IfPseudoVertexShader(new IfPseudoVertexCacheFIFO());

                foreach (var primitive in pm.PolygonPrimitives)
                {
                    vs.TransformVertices(primitive.Triangles);
                }

                preACMR += vs.AverageCacheMissRatio;
                preCacheMissCount += vs.CacheMissCount;
                totalVertexCount += vs.TotalVertexCount;
                totalTriangleCount += vs.TotalTriangleCount;
                preVertexShaders.Add(vs);
            }

            // 平均では正しくない
            preACMR /= preVertexShaders.Count;

            Debug.WriteLine("プリミティブ最適化前");
            Debug.WriteLine("Vertices: " + totalVertexCount.ToString());
            Debug.WriteLine("ProcessVertices: " + preCacheMissCount.ToString());
            Debug.WriteLine("ACMR: " + preACMR.ToString());

            //IfParallel.ForEach(polygonMesh, delegate(IfPolygonMesh pm)
            foreach (var pm in polygonMesh)
            {
                IfOptimizeAlgorithm algorithm = new IfOptimizeAlgorithmTipsify();
                algorithm.Analyze(pm);
            }

            var optimizedResult = new IfOptimizedResult();
            foreach (var pm in polygonMesh)
            {
                optimizedResult[pm] = new List<IfPolygonPrimitive>();
                optimizedResult[pm].Capacity = pm.AnalyzedTriangles.Count;
            }

            // 最も隣接するトライアングルが多い頂点／トライアングルの中から選択する。
            //IfParallel.ForEach(polygonMesh, delegate(IfPolygonMesh pm)
            foreach (var pm in polygonMesh)
            {
                IfOptimizeAlgorithm algorithm = new IfOptimizeAlgorithmTipsify();
                // 全ての MaxAdjacencyList を確認した場合でも大きな差はなかった(ACMR で 0.009 程度)
                // 最も隣接するトライアングルが多い最初の結果を採用することにしました。
                algorithm.Optimize(pm, optimizedResult[pm], pm.MaxAdjacencyList[0]);
            }

            IList<IfPseudoVertexShader> postVertexShaders
                = new List<IfPseudoVertexShader>(optimizedResult.Result.Count);

            postACMR = 0.0f;
            postCacheMissCount = 0;
            //IfParallel.ForEach(optimizedResult.Result,
            //    delegate(KeyValuePair<IfPolygonMesh, List<IfPolygonPrimitive>> pair)
            foreach (var pair in optimizedResult.Result)
            {
                IList<IfPolygonPrimitive> primitives = pair.Value;
                IfPseudoVertexShader vs =
                    new IfPseudoVertexShader(new IfPseudoVertexCacheFIFO());

                foreach (IfPolygonPrimitive primitive in primitives)
                {
                    vs.TransformVertices(primitive.Triangles);
                }

                postACMR += vs.AverageCacheMissRatio;
                postCacheMissCount += vs.CacheMissCount;
                postVertexShaders.Add(vs);
            }

            postACMR /= postVertexShaders.Count;

            // キャッシュ効率が向上している場合のみ適応する。
            int resultIndex = 0;
            bool isSwapIndexStream = false;
            foreach (var pair in optimizedResult.Result)
            {
                if (preVertexShaders[resultIndex].AverageCacheMissRatio
                    > postVertexShaders[resultIndex].AverageCacheMissRatio)
                {
                    IfPolygonMesh pm = pair.Key;
                    IList<IfPolygonPrimitive> primitives = pair.Value;

                    IfPolygonAnalyzer.Apply(pm, primitives, this.Target, this.Streams);
                    isSwapIndexStream = true;
                }

                ++resultIndex;
            }

            // インデクスの並び順に頂点ストリームを並び替えます。
            if (isSwapIndexStream)
            {
                AlignVertexStream(this.Target, this.Streams);
            }

            Debug.WriteLine("プリミティブ最適化後");
            Debug.WriteLine("Vertices: " + totalVertexCount.ToString());
            Debug.WriteLine("ProcessVertices: " + postCacheMissCount.ToString());
            Debug.WriteLine("ACMR: " + postACMR.ToString());
        }

        private static void AlignVertexStream(modelType model, List<G3dStream> streams)
        {
            if (model.vertex_array == null ||
                model.vertex_array.vertex == null) { return; }
            if (model.shape_array == null ||
                model.shape_array.shape == null) { return; }

            vertexType[] vertex_array = model.vertex_array.vertex;
            //IfParallel.ForEach(model.shape_array.shape, delegate(shapeType shape)
            foreach (var shape in model.shape_array.shape)
            {
                vertexType vertex = vertex_array[shape.shape_info.vertex_index];
                int vertexCount = vertex.vtx_attrib_array.vtx_attrib[0].count;
                int indexCount = shape.mesh_array.mesh[0].count;

                int[] vertexAlign = new int[vertexCount];
                int[] indexAlign = new int[indexCount];
                for (int i = 0; i < vertexCount; ++i)
                {
                    vertexAlign[i] = -1;
                }

                int vertexIndex = 0;
                List<int> indices = streams[shape.mesh_array.mesh[0].stream_index].IntData;
                int indexIndex = 0;
                foreach (int index in indices)
                {
                    bool isUsed = false;
                    for (int i = 0; i < vertexIndex; ++i)
                    {
                        if (vertexAlign[i] == index)
                        {
                            isUsed = true;
                            indexAlign[indexIndex] = i;
                            break;
                        }
                    }

                    if (!isUsed)
                    {
                        vertexAlign[vertexIndex] = index;
                        indexAlign[indexIndex] = vertexIndex;
                        ++vertexIndex;
                    }

                    ++indexIndex;
                }

                // サイズと型に応じて頂点ストリームを並び替えます。
                foreach (var attrib in vertex.vtx_attrib_array.vtx_attrib)
                {
                    G3dStream stream = streams[attrib.stream_index];

                    int attribSize = 0;
                    bool isFloat = false;
                    switch (attrib.type)
                    {
                        case vtx_attrib_typeType.@int:
                        case vtx_attrib_typeType.@uint:
                            attribSize = 1;
                            isFloat = false;
                            break;
                        case vtx_attrib_typeType.int2:
                        case vtx_attrib_typeType.uint2:
                            attribSize = 2;
                            isFloat = false;
                            break;
                        case vtx_attrib_typeType.int3:
                        case vtx_attrib_typeType.uint3:
                            attribSize = 3;
                            isFloat = false;
                            break;
                        case vtx_attrib_typeType.int4:
                        case vtx_attrib_typeType.uint4:
                            attribSize = 4;
                            isFloat = false;
                            break;
                        case vtx_attrib_typeType.@float:
                            attribSize = 1;
                            isFloat = true;
                            break;
                        case vtx_attrib_typeType.float2:
                            attribSize = 2;
                            isFloat = true;
                            break;
                        case vtx_attrib_typeType.float3:
                            attribSize = 3;
                            isFloat = true;
                            break;
                        case vtx_attrib_typeType.float4:
                            attribSize = 4;
                            isFloat = true;
                            break;
                    }

                    if (isFloat)
                    {
                        List<float> original = stream.FloatData;
                        float[] data = new float[original.Count];

                        int dataIndex = 0;
                        for (int i = 0; i < vertexAlign.Length; ++i)
                        {
                            for (int j = 0; j < attribSize; ++j)
                            {
                                data[dataIndex] = original[vertexAlign[i] * attribSize + j];
                                ++dataIndex;
                            }
                        }

                        for (int i = 0; i < original.Count; ++i)
                        {
                            stream.FloatData[i] = data[i];
                        }
                    }
                    else
                    {
                        List<int> original = stream.IntData;
                        int[] data = new int[original.Count];

                        int dataIndex = 0;
                        for (int i = 0; i < vertexAlign.Length; ++i)
                        {
                            for (int j = 0; j < attribSize; ++j)
                            {
                                data[dataIndex] = original[vertexAlign[i] * attribSize + j];
                                ++dataIndex;
                            }
                        }

                        for (int i = 0; i < original.Count; ++i)
                        {
                            stream.IntData[i] = data[i];
                        }
                    }
                }

                // 並び替わった頂点列に合わせてインデクスを並び替えます。
                for (int i = 0; i < indexCount; ++i)
                {
                    indices[i] = indexAlign[i];
                }
            }
        }
    }
}
