﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace nw.g3d.iflib
{
    public class IfAnalyzedTriangle
    {
        private static System.Random random = new Random();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public IfAnalyzedTriangle()
        {
            this.Triangle = new IfTriangle();
            this.IsUsed = false;
            this.SharedVertex1Triangles = new List<IfAnalyzedTriangle>();
            this.SharedVertex2Triangles = new List<IfAnalyzedTriangle>();
        }

        /// <summary>
        /// 三角形情報
        /// </summary>
        public IfTriangle Triangle { get; private set; }

        /// <summary>
        /// 使用済みチェック用フラグ
        /// </summary>
        public bool IsUsed { get; set; }

        /// <summary>
        /// 共有頂点を１つ持つトライアングルのリスト
        /// </summary>
        public IList<IfAnalyzedTriangle> SharedVertex1Triangles { get; private set; }

        /// <summary>
        /// 共有頂点を２つ持つトライアングルのリスト
        /// </summary>
        public IList<IfAnalyzedTriangle> SharedVertex2Triangles { get; private set; }

        //-----------------------------------------------------
        // 探索、くじ引き機能
        //-----------------------------------------------------

        /// <summary>
        /// 未使用のトライアングルのうち index 番目を見つけます。
        /// </summary>
        /// <param name="triangles">検索対象のトライアングル群です。</param>
        /// <param name="index">取得したい未使用トライアングルの番号です。</param>
        /// <returns>未使用のトライアングルのうち index 番目に見つけたトライアングルを返します。</returns>
        public static IfAnalyzedTriangle FindUnused(IList<IfAnalyzedTriangle> triangles, int index)
        {
            IfAnalyzedTriangle found = null;

            foreach (IfAnalyzedTriangle triangle in triangles)
            {
                if (!triangle.IsUsed)
                {
                    if (index == 0)
                    {
                        found = triangle;
                        break;
                    }

                    --index;
                }
            }

            return found;
        }

        /// <summary>
        /// 未使用のトライアングルから１つをランダムに選択します。
        /// </summary>
        /// <param name="triangles">検索対象のトライアングル群です。</param>
        /// <returns>ランダムに選ばれたトライアングルを返します。</returns>
        public static IfAnalyzedTriangle DrawLot(IList<IfAnalyzedTriangle> triangles)
        {
            int count = triangles.Count((triangle) => !triangle.IsUsed);
            return FindUnused(triangles, IfAnalyzedTriangle.random.Next(count));
        }

        /// <summary>
        /// 未使用のトライアングルで、隣接する未使用のトライアングル数が指定した数のトライアングルから、１つをランダムに選択します。
        /// </summary>
        /// <param name="triangles">検索対象のトライアングル群です。</param>
        /// <param name="count">隣接する未使用トライアングルの数です。</param>
        /// <returns>ランダムに選ばれたトライアングルを返します。</returns>
        public static IfAnalyzedTriangle DrawLotByAdjacentCount(IList<IfAnalyzedTriangle> triangles, int count)
        {
            IfAnalyzedTriangle result = null;

            if (0 <= count && count != int.MaxValue)
            {
                int size = triangles.Count((triangle) => !triangle.IsUsed && triangle.CountUnusedAdjacent() == count);
                result = FindAdjacentCount(triangles, count, IfAnalyzedTriangle.random.Next(size));
            }

            return result;
        }

        /// <summary>
        /// 隣接する未使用のトライアングルの数を調べます。
        /// </summary>
        /// <returns>隣接する未使用のトライアングルの数を返します。</returns>
        public int CountUnusedAdjacent()
        {
            int count = this.SharedVertex1Triangles.Count((triangle) => !triangle.IsUsed);
            count += this.SharedVertex2Triangles.Count((triangle) => !triangle.IsUsed);
            return count;
        }

        private static IfAnalyzedTriangle FindAdjacentCount(IList<IfAnalyzedTriangle> triangles, int adjacentCount, int index)
        {
            IfAnalyzedTriangle found = null;

            foreach (IfAnalyzedTriangle triangle in triangles)
            {
                if (!triangle.IsUsed && triangle.CountUnusedAdjacent() == adjacentCount)
                {
                    if (index == 0)
                    {
                        found = triangle;
                        break;
                    }

                    --index;
                }
            }

            return found;
        }
    }
}
