﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;
using System.Collections.ObjectModel;

namespace nw.g3d.iflib
{
    public class SubmeshCluster
    {
        private readonly Vector3 position = new Vector3(0.0f, 0.0f, 0.0f);
        private readonly Vector3 normal = new Vector3(0.0f, 0.0f, 0.0f);
        private readonly List<Triangle> triangles = new List<Triangle>();
        private readonly HashSet<uint> edgeKeys = new HashSet<uint>();
        private float area = 0.0f;
        private float score = 0.0f;

        public Vector3 Position
        {
            get
            {
                return this.position;
            }
        }

        public float Area
        {
            get
            {
                return this.area;
            }
        }

        public float Score
        {
            get
            {
                return this.score;
            }
        }

        public IList<Triangle> Triangles
        {
            get
            {
                return this.triangles;
            }
        }

        public void Add(Triangle triangle)
        {
            int edgeCount = 0;
            foreach (var key in triangle.EdgeKeys)
            {
                if (this.edgeKeys.Contains(key))
                {
                    ++edgeCount;
                }
                else
                {
                    this.edgeKeys.Add(key);
                }
            }
            this.triangles.Add(triangle);
        }

        public void Merge(SubmeshCluster cluster)
        {
            foreach (var triangle in cluster.triangles)
            {
                this.Add(triangle);
            }
        }

        /// <summary>
        /// 指定トライアングルが共有エッジ持つか判定します。
        /// </summary>
        /// <param name="triangle">判定対象の三角形</param>
        /// <returns>エッジが同一のものがあれば</returns>
        public bool HasSharedEdge(Triangle triangle)
        {
            bool result = false;
            foreach (var key in triangle.EdgeKeys)
            {
                if (this.edgeKeys.Contains(key))
                {
                    result = true;
                    break;
                }
            }
            return result;
        }

        /// <summary>
        /// 指定トライアングルが共有エッジ持つか判定します。
        /// </summary>
        /// <param name="triangle">判定対象の三角形</param>
        /// <returns>エッジが同一のものがあれば</returns>
        public bool HasSharedEdge(SubmeshCluster cluster)
        {
            foreach (var triangle in cluster.triangles)
            {
                if (this.HasSharedEdge(triangle))
                {
                    return true;
                }
            }
            return false;
        }

        public ReadOnlyCollection<int> GetVertexIndices()
        {
            List<int> indices = new List<int>();
            foreach (var triangle in this.triangles)
            {
                indices.AddRange(triangle.VertexIndices);
            }
            return indices.AsReadOnly();
        }

        public void CalcScore(Vector3 submeshPosition)
        {
            this.score = Vector3.Dot(this.position - submeshPosition, this.normal);
            if (this.score < -2e20f || this.score > 2e20f)
            {
                this.score = 0.0f;
            }
        }

        public void Setup()
        {
            this.position.X /= this.area * 3.0f;
            this.position.Y /= this.area * 3.0f;
            this.position.Z /= this.area * 3.0f;
            this.normal.SafeNormalize(this.normal);
        }

        public void CalcAreaAndNormal(G3dStream vertexStream)
        {
            foreach (var triangle in this.triangles)
            {
                List<Vector3> vertices = new List<Vector3>();
                for (int i = 0; i < 3; ++i)
                {
                    int vertexIndex = triangle.VertexIndices[i];
                    float[] data = { vertexStream.FloatData[vertexIndex], vertexStream.FloatData[vertexIndex + 1], vertexStream.FloatData[vertexIndex + 2] };
                    Vector3 vertex = new Vector3(data);
                    vertices.Add(vertex);
                }
                Vector3 ac = new Vector3(vertices[2] - vertices[0]);
                Vector3 ab = new Vector3(vertices[1] - vertices[0]);
                Vector3 triangleNormal = ac.Cross(ab);
                float clusterArea = triangleNormal.Length;
                if (clusterArea > 0.0f)
                {
                    triangleNormal.X /= clusterArea;
                    triangleNormal.Y /= clusterArea;
                    triangleNormal.Z /= clusterArea;
                }
                else
                {
                    clusterArea = 0.0f;
                    triangleNormal.SetZero();
                }

                for (int i = 0; i < 3; ++i)
                {
                    this.position.X += vertices[i].X * clusterArea;
                    this.position.Y += vertices[i].Y * clusterArea;
                    this.position.Z += vertices[i].Z * clusterArea;
                }
                this.normal.X += triangleNormal.X;
                this.normal.Y += triangleNormal.Y;
                this.normal.Z += triangleNormal.Z;
                this.area += clusterArea;
            }
        }

        public class Triangle
        {
            private readonly int index;
            private int[] vertexIndices = new int[3];
            private readonly HashSet<uint> edgeKeys = new HashSet<uint>();

            public Triangle(int index, int[] vertexIndices)
            {
                this.index = index;

                this.vertexIndices[0] = vertexIndices[0];
                this.vertexIndices[1] = vertexIndices[1];
                this.vertexIndices[2] = vertexIndices[2];
                ushort[] hashKeyIndices = new ushort[3]
                {
                    (ushort)vertexIndices[0],
                    (ushort)vertexIndices[1],
                    (ushort)vertexIndices[2]
                };

                edgeKeys.Add(GetEdgeHashKey(hashKeyIndices[0], hashKeyIndices[1]));
                edgeKeys.Add(GetEdgeHashKey(hashKeyIndices[1], hashKeyIndices[2]));
                edgeKeys.Add(GetEdgeHashKey(hashKeyIndices[2], hashKeyIndices[0]));
            }

            public int Index
            {
                get
                {
                    return this.index;
                }
            }

            public HashSet<uint> EdgeKeys
            {
                get
                {
                    return this.edgeKeys;
                }
            }

            public int[] VertexIndices
            {
                get
                {
                    return this.vertexIndices;
                }
            }
        }

        public static uint GetEdgeHashKey(ushort e1, ushort e2)
        {
            ushort edge1 = e1;
            ushort edge2 = e2;
            if (edge1 > edge2)
            {
                edge1 = e2;
                edge2 = e1;
            }
            uint result = (uint)((edge1 << 16) + edge2);
            return result;
        }
    }
}
